/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.expressions;

import java.io.*;
import java.util.HashMap;
import java.util.Iterator;
import oracle.toplink.essentials.exceptions.*;
import oracle.toplink.essentials.expressions.*;
import oracle.toplink.essentials.queryframework.*;
import oracle.toplink.essentials.internal.sessions.AbstractSession;
import java.util.Collection;
import oracle.toplink.essentials.internal.databaseaccess.DatabaseCall;
import oracle.toplink.essentials.internal.helper.DatabaseField;

/**
 * @author Guy Pelletier
 * @since TOPLink/Java 1.0
 */
public class SQLUpdateAllStatement extends SQLModifyStatement {
    protected HashMap m_updateClauses;
    protected HashMap databaseFieldsToTableAliases;

    protected SQLCall selectCallForExist;
    protected String tableAliasInSelectCallForExist;
    protected Collection primaryKeyFields;
    
    public void setSelectCallForExist(SQLCall selectCallForExist) {
        this.selectCallForExist = selectCallForExist;
    }
    public SQLCall getSelectCallForExist() {
        return selectCallForExist;
    }
    public void setTableAliasInSelectCallForExist(String tableAliasInSelectCallForExist) {
        this.tableAliasInSelectCallForExist = tableAliasInSelectCallForExist;
    }
    public String getTableAliasInSelectCallForExist() {
        return tableAliasInSelectCallForExist;
    }
    public void setPrimaryKeyFieldsForAutoJoin(Collection primaryKeyFields) {
        this.primaryKeyFields = primaryKeyFields;
    }
    public Collection getPrimaryKeyFieldsForAutoJoin() {
        return primaryKeyFields;
    }
    public void setUpdateClauses(HashMap updateClauses) {
        m_updateClauses = updateClauses;
    }
    public HashMap getUpdateClauses() {
        return m_updateClauses;
    }
    public void setDatabaseFieldsToTableAliases(HashMap databaseFieldsToTableAliases) {
        this.databaseFieldsToTableAliases = databaseFieldsToTableAliases;
    }
    public HashMap getDatabaseFieldsToTableAliases() {
        return databaseFieldsToTableAliases;
    }
    
    /**
     * Append the string containing the SQL insert string for the given table.
     */
    public DatabaseCall buildCall(AbstractSession session) {
        SQLCall call = buildSimple(session);
        if(selectCallForExist == null) {
            return call;
        }
        Writer writer = new CharArrayWriter(100);
        try {
            writer.write(call.getSQLString());

            if(selectCallForExist != null) {
                writer.write(" WHERE EXISTS(");
                // EXIST Example: selectCall.sqlString:
                // "SELECT t0.EMP_ID FROM EMPLOYEE t0, SALARY t1 WHERE (((t0.F_NAME LIKE 'a') AND (t1.SALARY = 0)) AND (t1.EMP_ID = t0.EMP_ID))"
                writeSelect(writer, selectCallForExist, tableAliasInSelectCallForExist, call);
                // closing bracket for EXISTS
                writer.write(")");
            }

            call.setSQLString(writer.toString());
            
        } catch (IOException exception) {
            throw ValidationException.fileError(exception);
        }
                
        return call;
    }
    
    protected SQLCall buildSimple(AbstractSession session) {
        SQLCall call = new SQLCall();
        call.returnNothing();
        Writer writer = new CharArrayWriter(100);
        ExpressionSQLPrinter printer = new ExpressionSQLPrinter(session, getTranslationRow(), call, false);
        printer.setWriter(writer);

        try {
            // UPDATE //
            writer.write("UPDATE ");
            writer.write(getTable().getQualifiedName());
            // SET CLAUSE //
            writer.write(" SET ");

            Iterator i = m_updateClauses.keySet().iterator();
            boolean commaNeeded = false;

            while (i.hasNext()) {
                if (commaNeeded) {
                    writer.write(", ");
                }

                DatabaseField field = (DatabaseField)i.next();
                Object value = m_updateClauses.get(field);

                writer.write(field.getName());
                writer.write(" = ");
                if(value instanceof Expression) {
                    printer.printExpression((Expression)value);
                } else {
                    // must be SQLCall
                    SQLCall selCall = (SQLCall)value;
                    String tableAlias = (String)getDatabaseFieldsToTableAliases().get(field);
                    // should be SQLCall select
                    writer.write("(");
                    writeSelect(writer, selCall, tableAlias, call);
                    writer.write(")");
                }

                commaNeeded = true;
            }

            // WHERE CLAUSE //
            if (getWhereClause() != null) {
                writer.write(" WHERE ");
                printer.printExpression(getWhereClause());
            }

            call.setSQLString(writer.toString());
            return call;
        } catch (IOException exception) {
            throw ValidationException.fileError(exception);
        }
    }

    protected void writeSelect(Writer writer, SQLCall selectCall, String tableAliasInSelectCall, SQLCall call) throws IOException {
        writer.write(selectCall.getSQLString());

        // Auto join
        // Example: AND t0.EMP_ID = EMP_ID
        Iterator it = getPrimaryKeyFieldsForAutoJoin().iterator();
        while(it.hasNext()) {
            writer.write(" AND ");
            String fieldName = ((DatabaseField)it.next()).getName();
            if(tableAliasInSelectCall != null) {
                writer.write(tableAliasInSelectCall);
                writer.write('.');
            }
            writer.write(fieldName);
            writer.write(" = ");
            writer.write(table.getQualifiedName());
            writer.write('.');
            writer.write(fieldName);
        }
        
        call.getParameters().addAll(selectCall.getParameters());
        call.getParameterTypes().addAll(selectCall.getParameterTypes());            
    }    
}
