/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2006, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.parsing;


// Java imports
import java.util.*;

// TopLink imports
import oracle.toplink.essentials.expressions.*;
import oracle.toplink.essentials.queryframework.ReportQuery;
import oracle.toplink.essentials.exceptions.EJBQLException;

/**
 * INTERNAL
 * <p><b>Purpose</b>: Represent an IN in EJBQL
 * <p><b>Responsibilities</b>:<ul>
 * <li> Generate the correct expression for an IN
 * </ul>
 *    @author Jon Driscoll and Joel Lucuik
 *    @since TopLink 4.0
 */
public class InNode extends SimpleConditionalExpressionNode {

    private List theObjects = null;

    //Was NOT indicated? "WHERE emp.lastName NOT IN (...)
    private boolean notIndicated = false;

    /**
     * InNode constructor comment.
     */
    public InNode() {
        super();
    }

    /**
     * INTERNAL
     * Add the passed node value to the collection of object for this node
     */
    public void addNodeToTheObjects(Node theNode) {
        getTheObjects().add(theNode);
    }

    /**
     * INTERNAL
     * Validate the current node and calculates its type.
     */
    public void validate(ParseTreeContext context) {
        Object leftType = null;
        TypeHelper typeHelper = context.getTypeHelper();

        if (left != null) {
            left.validate(context);
            leftType = left.getType();
        }
        for (Iterator i = getTheObjects().iterator(); i.hasNext();) {
            Node node = (Node)i.next();
            node.validate(context);
            node.validateParameter(context, leftType);
            Object nodeType = node.getType();
            if ((leftType != null) && !typeHelper.isAssignableFrom(leftType, nodeType))
                throw EJBQLException.invalidExpressionArgument("IN", node.getAsString(), typeHelper.getTypeName(leftType));
        }

        setType(typeHelper.getBooleanType());
    }

    /**
     * INTERNAL
     * Return the TopLink expression for this node
     */
    public Expression generateExpression(GenerationContext context) {
        Expression whereClause = getLeft().generateExpression(context);
        List arguments = getTheObjects();
        Node firstArg = (Node)arguments.get(0);
        if (firstArg.isSubqueryNode()) {
            SubqueryNode subqueryNode = (SubqueryNode)firstArg;
            ReportQuery reportQuery = subqueryNode.getReportQuery(context);
            if (notIndicated()) {
                whereClause = whereClause.notIn(reportQuery);
            }
            else {
                whereClause = whereClause.in(reportQuery);
            }
        }
        else {
            Vector inArguments = new Vector(arguments.size());
            for (Iterator iter = arguments.iterator(); iter.hasNext();) {
                Node nextNode = (Node)iter.next();
                inArguments.add(nextNode.generateExpression(context));
            }
            if (inArguments.size() > 0) {
                if (notIndicated()) {
                    whereClause = whereClause.notIn(inArguments);
                } else {
                    whereClause = whereClause.in(inArguments);
                }
            }
        }
        return whereClause;
    }

    /**
     * INTERNAL
     * Return the collection of the objects used as parameters for this node
     */
    public List getTheObjects() {
        if (theObjects == null) {
            setTheObjects(new Vector());
        }
        return theObjects;
    }

    /**
     * INTERNAL
     * Set this node's object collection to the passed value
     */
    public void setTheObjects(List newTheObjects) {
        theObjects = newTheObjects;
    }

    /**
     * INTERNAL
     * Indicate if a NOT was found in the WHERE clause.
     * Examples:
     *        ...WHERE ... NOT IN(...)
     */
    public void indicateNot() {
        notIndicated = true;
    }

    public boolean notIndicated() {
        return notIndicated;
    }
}
