/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004
 *					All rights reserved
 *
 *  This file is part of GPAC / command-line mp4 toolbox
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include <gpac/m4_author.h>
#include <gpac/m4_bifs.h>

#ifndef M4_READ_ONLY
M4File *import_file(char *inName, char *outName, Bool data_ref, Bool force_const_rate);
M4File *merge_files(char *inName1, char *inName2, char *outName);
void make_single_track_file(M4File *file, char *inName, u32 trackID);
#endif

void dump_file_text(char *file, char *inName, Bool xmt_dump);
void dump_scene_stats(M4File *file, char *inName, u32 stat_level);
void PrintNode(const char *name);


/*some global vars for swf import :(*/
u32 swf_flags;
Float swf_flatten_angle;

void PrintVersion()
{
	fprintf(stdout, "MP4Box - GPAC version %s\n", M4_VERSION);
}

void PrintGeneralUsage()
{
	fprintf(stdout, "General Options:\n"
			"-isma: rewrites the file as an ISMA 1.0 AV file (all systems info rewritten)\n"
			"-ismax: same as above but remove all clock references (streams not explicetly synchronized)\n"
			"-3gp: rewrites the file as a 3GP file (no more MPEG-4 Systems Info)\n"
			"-inter time_in_ms: interleaves file data, with time_in_ms max duration of continuous media. Cannot be used with -frag\n"
			"-frag time_in_ms: fragments file, with time_in_ms max duration of track fragments. Cannot be used with -inter\n"
			"-out filename: specifies output file name. Default output is out_$INPUT\n"
			"\n");
}
void PrintImportUsage()
{
	fprintf(stdout, "Import Options:\n"
			"-import file: imports file as ISMA mp4. Supported input NHNT, mp3 and\n"
			"\tavi (full import or track-based import: file#video or file#audio).\n"
			"\tAppending \"%%X\" to the file name will only import the first X seconds\n"
			"-dref: keeps media data in original file\n"
			"-nodrop: forces constant FPS when importing AVI video\n"
			"\n"
			"-merge file1 file2: merge both files into a single MP4\n"
			"\n"
			"-flat: forces flat storage (the imported media data is placed first in the file - no HTTP streaming possible).\n"
			"By default media importing uses 0.5s interleaving\n"
			);
}

void PrintEncodeUsage()
{
	fprintf(stdout, "Encode Options:\n"
			"-mp4: specify input file is for encoding. Supported inputs are BT and XMT-A files\n"
			"-def: encode DEF names\n"
			"-sync time_in_ms: forces BIFS sync sample generation every time_in_ms. Cannot be used with -shadow\n"
			"-shadow time_in_ms: forces BIFS sync shadow sample generation every time_in_ms. Cannot be used with -sync\n"
			"-log: generates BIFS encoder log file\n"
			"-ms file: specify file for track importing - by default FILE.mp4 is\n"
			"used when encoding FILE.bt (in-place rewrite)\n"
			);
}

void PrintHintUsage()
{
	fprintf(stdout, "Hinting Options:\n"
			"-hint: hint the file for RTP/RTSP sessions.\n"
			"-copy: forces hinted data to be copied to the track instead of referenced - speeds up server but takes much more space\n"
			"-mtu size: specifies MTU size in bytes. Default size is 1500\n"
			"-tight: performs tight interleaving (sample based) of hinted file - reduces server disk seek but increases file size\n"
			"-ocr: forces all streams to be synchronized - most RTSP servers don't support desynchronized streams\n"
			"\n"
			"MPEG-4 Generic Payload Options (Experts only)\n"
			"-rap: signals random access points in RTP packets\n"
			"-ts: signals AU Time Stamps in RTP packets\n"
			"-size: signals AU size in RTP packets\n"
			"-idx: signals AU indexes (sequence numbers) in RTP packets \n"
			"-multi: enables AU concatenation in RTP packets (ts, size and idx are selected if needed)\n"
			"-iod: prevents system tracks embedding in IOD (shouldn't be used with -isma option)\n"
			"\n");
}
void PrintExtractUsage()
{
	fprintf(stdout, "Extracting Options\n"
			"-raw TrackID: extracts track in raw format (cmp, aac, mp3, jpg, png) - does not work for systems tracks\n" 
			"-nhnt TrackID: extracts track in nhnt format\n" 
			"-single TrackID: extracts track to a new mp4 file\n"
			"-avi TrackID: extracts visual track to an avi file\n"
			"\n");
}
void PrintDumpUsage()
{
	fprintf(stdout, "Dumping Options\n"
			"-info [trackID]: prints movie/tracks info. If TrackID specified, dumps only track info\n"
			"\n"
			"-bt: decompress scene (mp4 or XMT-A) into bt format\n" 
			"-xmt: decompress scene (mp4 or BT) into XMT-A format\n" 
			"-dmp4: dumps mp4 file structure in XML output\n"
			"-drtp: dumps rtp hint samples structure in XML output\n"
			"-sdp: prints SDP description of hinted file\n"
			"-std: dumps to stdout instead of file\n"
			"\n"
			"-stat: generate stat report on node/field usage for the whole presentation\n"
			"-stats: generate stat report on node/field usage per BIFS Access Unit\n"
			"-statx: generate stat report on node/field usage in the scene graph after each BIFS Access Unit\n"
			"\n");
}
void PrintSWFUsage()
{
	fprintf(stdout, 
		"MP4Box can import simple Macromedia Flash files (\".SWF\")\n"
		"You can specify a SWF input file with \'-bt\', \'xmt\' and \'-mp4\' switches\n"
		"\n"
		"SWF Importer Options\n"
		"-static: all SWF defines are placed in first scene replace. By default SWF defines are sent when needed\n"
		"-ctrl: uses a dedicated stream for movie control (forces -static option)\n"
		"\n"
		"-notext: removes all SWF text\n"
		"-nofont: removes all embedded SWF Fonts, forcing usage of MPEG-4 Text and terminal fonts\n"
		"-noline: removes all lines from SWF shapes\n"
		"-nograd: removes all gradients from swf shapes\n"
		"-quad: uses quadratic bezier curves instead of cubic ones\n"
		"-xlp: support for lines transparency and scalability\n"
		"\n"
		"-flatten Value: replaces 2 consecutive lines by a single one when angle between lines is less than Value (expressed in radians). Value 0 disables flattening\n"
		"\n");
}

void PrintUsage()
{
	fprintf (stdout, "MP4Box [option] input\n"
#ifndef M4_READ_ONLY
			"-h general: prints general options\n"
			"-h hint: prints hinting options\n"
			"-h import: prints import options\n"
			"-h encode: prints encode options\n"
#else
			"\tREAD-ONLY VERSION\n"
#endif
			"\n"
			"-h extract: prints extraction options\n"
			"-h dump: prints dump options\n"
			"-h swf: prints flash options\n"
			"\n"
			"-version: gets build version\n"
			"-nodes: lists supported node in current build\n"
			"-node NodeName: gets node syntax and QP info\n"
			"\nWritten by Jean Le Feuvre - GPAC (c) 2000-2004\n");
}

void PrintBuiltInNodes()
{
	SFNode *node;
	LPSCENEGRAPH sg;
	u32 i, nb_nodes, nb_not_in;

	nb_nodes = nb_not_in = 0;
	sg = NewSceneGraph(NULL, NULL, NULL, NULL);
	fprintf(stdout, "Available nodes in this MP4Box build (encoding/decoding):\n");
	for (i=1; i<TAG_ProtoNode; i++) {
		node = SG_NewNode(sg, i);
		if (node) {
			Node_Register(node, NULL);
			fprintf(stdout, "\t%s\n", Node_GetName(node));
			Node_Unregister(node, NULL);
			nb_nodes++;
		} else {
			nb_not_in++;
		}
	}
	SG_Delete(sg);
	fprintf(stdout, "\n%d nodes supported - %d nodes not supported\n", nb_nodes, nb_not_in);
}




void DumpSDP(M4File *file, char *inName)
{
	const char *sdp;
	u32 size, i;
	FILE *dump;
	char szBuf[1024];

	if (inName) {
		strcpy(szBuf, inName);
		strcat(szBuf, "_sdp.txt");
		dump = fopen(szBuf, "wt");
	} else {
		dump = stdout;
		fprintf(dump, "* File SDP content *\n\n");
	}
	//get the movie SDP
	M4H_SDP_GetSDP(file, &sdp, &size);
	fprintf(dump, "%s", sdp);
	fprintf(dump, "\r\n");

	//then tracks
	for (i=0; i<M4_GetTrackCount(file); i++) {
		if (M4_GetMediaType(file, i+1) != M4_HintMediaType) continue;
		M4H_SDP_GetTrackSDP(file, i+1, &sdp, &size);
		fprintf(dump, "%s", sdp);
	}
	fprintf(dump, "\n\n");
	if (inName) fclose(dump);
}

void TrackInfo(M4File *file, u32 trackID, Bool full_dump)
{
	Float scale;
	u32 trackNum, j, di, size, dur, max_rate, rate, time_slice, ts;
	ESDescriptor *esd;
	char sType[5];

	sType[4] = 0;
	trackNum = M4_GetTrackByID(file, trackID);
	fprintf(stdout, "Track # %d Info - TrackID %d - TimeScale %d - Duration %d\n", trackNum, trackID, M4_GetMediaTimeScale(file, trackNum), (u32) M4_GetMediaDuration(file, trackNum));
	if (M4_IsTrackInRootOD(file, trackNum) ) fprintf(stdout, "Track is present in Root OD\n");
	MP4TypeToString(M4_GetMediaType(file, trackNum), sType);
	fprintf(stdout, "Media Type \"%s\" - ", sType);
	MP4TypeToString(M4_GetMediaSubType(file, trackNum, 1), sType);
	fprintf(stdout, "Media Sub Type \"%s\" - %d samples\n", sType, M4_GetSampleCount(file, trackNum));

	esd = M4_GetStreamDescriptor(file, trackNum, 1);
	if (esd) {
		const char *st = OD_GetStreamTypeName(esd->decoderConfig->streamType);
		if (st) {
			fprintf(stdout, "MPEG-4 Config%s%s Stream - ObjectTypeIndication %d\n",
						full_dump ? "\n\t" : ": ", st, esd->decoderConfig->objectTypeIndication);
		} else {
			fprintf(stdout, "MPEG-4 Config%sStream Type %d - ObjectTypeIndication %d\n",
						full_dump ? "\n\t" : ": ", esd->decoderConfig->streamType, esd->decoderConfig->objectTypeIndication);
		}
		if (esd->decoderConfig->streamType==M4ST_VISUAL) {
			if (esd->decoderConfig->objectTypeIndication==0x20) {
				M4VDecoderSpecificInfo dsi;
				M4V_GetConfig(esd->decoderConfig->decoderSpecificInfo->data, esd->decoderConfig->decoderSpecificInfo->dataLength, &dsi);
				if (full_dump) fprintf(stdout, "\t");
				fprintf(stdout, "MPEG-4 Visual Size %d x %d - %s\n", dsi.width, dsi.height, MP4T_VideoProfileName(dsi.VideoPL));
			} else {
				u32 w, h;
				M4_GetVisualEntrySize(file, trackNum, 1, &w, &h);
				if (full_dump) fprintf(stdout, "\t");
				fprintf(stdout, "Visual Size %d x %d\n", w, h);
			}
		} else if (esd->decoderConfig->streamType==M4ST_AUDIO) {
			u32 sr, nbCh, oti;
			switch (esd->decoderConfig->objectTypeIndication) {
			case 0x66:
			case 0x67:
			case 0x68:
			case 0x40:
				oti = M4A_GetCodecType(esd->decoderConfig->decoderSpecificInfo->data, esd->decoderConfig->decoderSpecificInfo->dataLength, &nbCh, &sr);
				if (full_dump) fprintf(stdout, "\t");
				fprintf(stdout, "MPEG-4 Audio %s - %d Channels - SampleRate %d\n", M4A_GetObjectTypeName(oti), nbCh, sr);
				break;
			case 0x69:
			case 0x6B:
				{
					M4Sample *samp = M4_GetSample(file, trackNum, 1, &oti);
					oti = FOUR_CHAR_INT((u8)samp->data[0], (u8)samp->data[1], (u8)samp->data[2], (u8)samp->data[3]);
					if (full_dump) fprintf(stdout, "\t");
					fprintf(stdout, "%s Audio - %d Channels - SampleRate %d - Layer %d\n",
						MP3_GetVersionName(oti),
						MP3_GetNumChannels(oti), 
						MP3_GetSamplingRate(oti), 
						MP3_GetLayer(oti)
					);
					M4_DeleteSample(&samp);
				}
				break;
			}
		}

		/*sync is only valid if we open all tracks to take care of default MP4 sync..*/
		if (!full_dump) {
			if (!esd->OCRESID || (esd->OCRESID == esd->ESID))
				fprintf(stdout, "Self-synchronized\n");
			else
				fprintf(stdout, "Synchronized on stream %d\n", esd->OCRESID);
		} else {
			fprintf(stdout, "\tDecoding Buffer size %d - Average bitrate %d - Max Bitrate %d\n", esd->decoderConfig->bufferSizeDB, esd->decoderConfig->avgBitrate, esd->decoderConfig->maxBitrate);
			if (esd->dependsOnESID)
				fprintf(stdout, "\tDepends on stream %d for decoding\n", esd->dependsOnESID);
			else
				fprintf(stdout, "\tNo stream dependencies for decoding\n");

			fprintf(stdout, "\tStreamPriority %d\n", esd->streamPriority);
			if (esd->URLString) fprintf(stdout, "\tRemote Data Source %s\n", esd->URLString);
		}

		OD_DeleteDescriptor((Descriptor **)&esd);
	}
	if (!full_dump) {
		fprintf(stdout, "\n");
		return;
	}

	dur = size = 0;
	max_rate = rate = time_slice = 0;
	ts = M4_GetMediaTimeScale(file, trackNum);
	for (j=0; j<M4_GetSampleCount(file, trackNum); j++) {
		M4Sample *samp = M4_GetSampleInfo(file, trackNum, j+1, &di, NULL);
		dur = samp->DTS+samp->CTS_Offset;
		size += samp->dataLength;
		rate += samp->dataLength;
		if (samp->DTS - time_slice>ts) {
			if (max_rate < rate) max_rate = rate;
			rate = 0;
			time_slice = samp->DTS;
		}
		M4_DeleteSample(&samp);
	}
	fprintf(stdout, "\nComputed info from media:\n");
	scale = 1000;
	scale /= ts;
	dur = (u32) (dur * scale);
	fprintf(stdout, "\tTotal size %d bytes - Total samples duration %d ms\n", size, dur);
	if (!dur) {
		fprintf(stdout, "\n");
		return;
	}
	dur /= 1000;
	rate = size / 1024 * 8 / dur ;
	max_rate *= 8;
	max_rate /= 1024;
	fprintf(stdout, "\tAverage rate %d kbps - Max Rate %d kbps\n", rate, max_rate);
	fprintf(stdout, "\n");
}

void MovieInfo(M4File *file)
{
	InitialObjectDescriptor *iod;
	u32 i, brand, min;
	char sType[5];

	fprintf(stdout, "* Movie Info *\n\tTimescale %d - Duration %d\n\tFragmented File %s - %d tracks\n",
		M4_GetTimeScale(file), (u32) M4_GetDuration(file), M4_IsMovieFragmented(file) ? "yes" : "no", M4_GetTrackCount(file));

	if (M4_GetMovieVersionInfo(file, &brand, &min, NULL) == M4OK) {
		sType[4] = 0;
		MP4TypeToString(brand, sType);
		fprintf(stdout, "\tFile Brand %s - version %d\n", sType, min);
	}

	iod = (InitialObjectDescriptor *) M4_GetRootOD(file);
	if (iod) {
		if (iod->tag == InitialObjectDescriptor_Tag) {
			fprintf(stdout, "File has root IOD\n");
			fprintf(stdout, "Scene PL %d - Graphics PL %d - OD PL %d - Visual PL %d - Audio PL %d\n", 
				iod->scene_profileAndLevel, iod->graphics_profileAndLevel, iod->OD_profileAndLevel, iod->visual_profileAndLevel, iod->audio_profileAndLevel);
			fprintf(stdout, "inline profiles included %s\n", iod->inlineProfileFlag ? "yes" : "no");
		} else {
			fprintf(stdout, "File has root OD\n");
		}
		OD_DeleteDescriptor((Descriptor **) &iod);
	} else {
		fprintf(stdout, "File has no MPEG4 IOD/OD\n");
	}

	fprintf(stdout, "\n");
	for (i=0; i<M4_GetTrackCount(file); i++) {
		TrackInfo(file, M4_GetTrackID(file, i+1), 0);
	}
}

void dump_file_mp4(M4File *file, char *inName)
{
	FILE *dump;
	char szBuf[1024];

	if (inName) {
		strcpy(szBuf, inName);
		strcat(szBuf, "_info.xml");
		dump = fopen(szBuf, "wt");
		M4_FileDump(file, dump);
		fclose(dump);
	} else {
		M4_FileDump(file, stdout);
	}
}



void dump_file_rtp(M4File *file, char *inName)
{
	u32 i, j;
	FILE *dump;
	char szBuf[1024];

	if (inName) {
		strcpy(szBuf, inName);
		strcat(szBuf, "_rtp.xml");
		dump = fopen(szBuf, "wt");
	} else {
		dump = stdout;
	}

	fprintf(dump, "<!-- MP4Box RTP trace -->\n");
	fprintf(dump, "<RTPFile>\n");

	for (i=0; i<M4_GetTrackCount(file); i++) {
		if (M4_GetMediaType(file, i+1) != M4_HintMediaType) continue;

		fprintf(dump, "<RTPHintTrack trackID=\"%d\">\n", M4_GetTrackID(file, i+1));
		for (j=0; j<M4_GetSampleCount(file, i+1); j++) {
			M4H_DumpHintSample(file, i+1, j+1, dump);
		}
		fprintf(dump, "</RTPHintTrack>\n");
	}
	fprintf(dump, "</RTPFile>\n");
	if (inName) fclose(dump);
}


#ifndef M4_READ_ONLY

void SetupClockReferences(M4File *file)
{
	u32 i, count, ocr_id;
	count = M4_GetTrackCount(file);
	if (count==1) return;
	ocr_id = 0;
	for (i=0; i<count; i++) {
		if (!M4_IsTrackInRootOD(file, i+1)) continue;
		ocr_id = M4_GetTrackID(file, i+1);
		break;
	}
	/*doesn't look like MP4*/
	if (!ocr_id) return;
	for (i=0; i<count; i++) {
		ESDescriptor *esd = M4_GetStreamDescriptor(file, i+1, 1);
		if (esd) {
			esd->OCRESID = ocr_id;
			M4_ChangeStreamDescriptor(file, i+1, 1, esd);
			OD_DeleteDescriptor((Descriptor **) &esd);
		}
	}
}

/*base RTP payload type used (you can specify your own types if needed)*/
#define BASE_PAYT		96

void OnHintProgress(void *cbk, u32 done, u32 total)
{
	if (done!=total) {
		fprintf(stdout, "Hinting sample %d/%d\r", done, total);
	} else {
		fprintf(stdout, "Hinting done - %d samples\n", total);
	}
}

void HintFile(M4File *file, u32 use_isma, u32 MTUSize, u32 base_flags, Bool copy_data, Bool interleave, Bool regular_iod)
{
	ESDescriptor *esd;
	u32 i, val, res, streamType;
	u32 sl_mode, prev_ocr, single_ocr;
	M4Err e;
	LPMP4RTPHINTER hinter;
	Bool copy;
	u8 init_payt = BASE_PAYT;
	u32 iod_mode;
	u32 media_group = 0;
	u8 media_prio = 0;

	prev_ocr = 0;
	single_ocr = 1;

	for (i=0; i<M4_GetTrackCount(file); i++) {
		sl_mode = base_flags;
		copy = copy_data;

		switch (M4_GetMediaType(file, i+1)) {
		case M4_VisualMediaType:
			if (use_isma) {
				media_group = ISMA_MEDIA_GROUP_ID;
				media_prio = MEDIA_VIDEO_PRIO;
				/*override sl config for isma (RFC 3016) */
				sl_mode = M4HF_AutoConf;
			}
			break;
		case M4_AudioMediaType:
			if (use_isma) {
				media_group = ISMA_MEDIA_GROUP_ID;
				media_prio = MEDIA_AUDIO_PRIO;
				sl_mode = M4HF_AutoConf;
			}
			break;
		case M4_HintMediaType:
			continue;
		default:
			/*no hinting of systems track on isma*/
			if (use_isma) continue;
		}

		if (!use_isma) {
			/*one media per group only (we should prompt user for group selection)*/
			media_group ++;
			media_prio = 1;
		}

		streamType = 0;
		esd = M4_GetStreamDescriptor(file, i+1, 1);
		if (esd) {
			streamType = esd->decoderConfig->streamType;
			if (!prev_ocr) {
				prev_ocr = esd->OCRESID;
				if (!esd->OCRESID) prev_ocr = esd->ESID;
			} else if (esd->OCRESID && prev_ocr != esd->OCRESID) {
				single_ocr = 0;
			}
			/*OD MUST BE WITHOUT REFERENCES*/
			if (streamType==1) copy = 1;
		}
		OD_DeleteDescriptor((Descriptor **) &esd);

		if (!regular_iod && M4_IsTrackInRootOD(file, i+1)) {
			/*single AU - check if base64 would fit in ESD (consider 33% overhead of base64), otherwise stream*/
			if (M4_GetSampleCount(file, i+1)==1) {
				M4Sample *samp = M4_GetSample(file, i+1, 1, &val);
				if (streamType) {
					res = MP4T_CanEmbbedData(samp->data, samp->dataLength, streamType);
				} else {
					/*not a system track, we shall hint it*/
					res = 0;
				}
				if (samp) M4_DeleteSample(&samp);
				if (res) continue;
			}
		}

		hinter = NewTrackHinter(file, i+1, MTUSize, sl_mode, init_payt, copy,
			OnHintProgress, NULL, interleave, NULL, media_group, media_prio, &e);

		if (e) {
			fprintf(stdout, "Cannot create hinter (%s)\n", M4ErrToString(e));
			continue;
		} else {
			fprintf(stdout, "Hinting track ID %d - streamType %x - %s\n", M4_GetTrackID(file, i+1), streamType, 
				(sl_mode & M4HF_AutoConf) ? "automatic payload configuration" : ((sl_mode & M4HF_UseMultiSL) ? "sl mode multiple AUs per packet" : "sl mode single AU per packet") );
		}
		e = MP4T_ProcessTrack(hinter);

		if (!e) e = MP4T_FinalizeHintTrack(hinter, 1);
		MP4T_DeleteHinter(hinter);
		
		if (e) {
			fprintf(stdout, "Error while hinting (%s)\n", M4ErrToString(e));
		}
		init_payt++;
	}

	iod_mode = SDP_IOD_ISMA;
	if (regular_iod) iod_mode = SDP_IOD_REGULAR;
	else if (use_isma==3) iod_mode = SDP_IOD_ISMA_STRICT;
	MP4T_FinalizeHintMovie(file, iod_mode, NULL, NULL);

	if (!single_ocr)
		fprintf(stdout, "Warning: at least 2 timelines found in the file\nThis may not be supported by servers/players\n\n");

}

M4Err EncodeFile(char *in, M4File *mp4, char *logFile, char *mediaSource, u32 flags, u32 rap_freq) 
{
	M4Err e;
	FILE *input;
	M4SceneManager *ctx;

	input = fopen(in, "rt");
	ctx = NewSceneManager();
	if (!input) {
		fprintf(stdout, "Cannot open %s\n", in);
		return M4BadParam;
	}
	if (strstr(in, ".bt") || strstr(in, ".wrl")) {
		e = M4SM_LoadContextFromBT(ctx, input, NULL, NULL, NULL);
	} else if (strstr(in, ".xml") || strstr(in, ".xmt")) {
		e = M4SM_LoadContextFromXMT(ctx, input, NULL, NULL, NULL);
	} else if (strstr(in, ".swf") || strstr(in, ".SWF")) {
		fclose(input);
		input = fopen(in, "rb");
		e = M4SM_LoadContextFromSWF(ctx, input, NULL, NULL, NULL, swf_flags, swf_flatten_angle);
	} else {
		e = M4NotSupported;
	}
	fclose(input);

	if (e) {
		fprintf(stdout, "Error parsing file %s\n", M4ErrToString(e));
	} else {
		e = M4SM_EncodeFile(ctx, mp4, logFile, mediaSource, flags, rap_freq);
	}
	M4SM_Delete(ctx);

	M4_SetMovieVersionInfo(mp4, MP4_V2_File, 1);
	M4_ModifyAlternateBrand(mp4, ISO_Media_File, 1);
	return e;
}

Bool needs_isma(M4File *file)
{
	if (M4_GetTrackCount(file) > 1) return 1;
	switch (M4_GetMediaType(file, 1)) {
	case M4_BIFSMediaType:
		M4_AddTrackToRootOD(file, 1);
		return 0;
	case M4_VisualMediaType:
		/*if 1 sample assume image*/
		if (M4_GetSampleCount(file, 1) == 1) return 2;
		return 1;
	default:
		return 1;
	}
}

#endif

/*return value:
	0: not supported 
	1: ISO media 
	2: input bt file (.bt, .wrl)
	3: input XML file (.xmt, .xml)
	4: input SWF file (.swf)
*/
u32 get_file_type_by_ext(char *inName)
{
	char *lowername;
	u32 type;
	lowername = strdup(inName);
	if (!lowername) return 0;
	strlwr(lowername);
	if (strstr(lowername, ".mp4") || strstr(lowername, ".3gp") || strstr(lowername, ".mov")) type = 1;
	else if (strstr(lowername, ".bt") || strstr(lowername, ".wrl")) type = 2;
	else if (strstr(lowername, ".xmt") || strstr(lowername, ".xml")) type = 3;
	else if (strstr(lowername, ".swf")) type = 4;
	else type = 0;

	/*try open file in read mode*/
	if (!type) {
		M4File *mp4 = M4_MovieOpen(inName, M4_OPEN_READ);
		if (mp4) {
			type = 1;
			M4_MovieDelete(mp4);
		}
	}
	
	free(lowername);
	return type;
}


int main (int argc, char **argv)
{
	char outfile[5000];
	M4Err e;
	Float InterTime;
	u32 i, MTUSize, stat_level, hint_flags, encode_flags, rap_freq, MakeIsma, Make3GP, info_track_id;
	Bool HintIt, needSave, FullInter, Frag, HintInter, dump_stderr, dump_rtp, dump_mode, regular_iod, trackID, HintCopy;
	Bool print_sdp, print_info, open_edit, track_dump_type, dump_mp4, import, use_dref, force_ocr, encode, do_log, no_frame_drop, do_merge, do_flat;
	char *inName, *outName, *arg, *mediaSource, *inName2;
	M4File *file;

	if (argc < 2) {
		PrintUsage();
		return 1;
	}

	e = M4OK;
	InterTime = 0;
	rap_freq = encode_flags = 0;
	MTUSize = 1500;
	import = use_dref = HintCopy = FullInter = HintInter = encode = do_log = do_merge = 0;
	dump_mode = Frag = force_ocr = no_frame_drop = 0;
	MakeIsma = Make3GP = HintIt = needSave = print_sdp = print_info = regular_iod = dump_stderr = open_edit = dump_mp4 = dump_rtp = 0;
	track_dump_type = 0;

	trackID = stat_level = hint_flags = 0;
	info_track_id = 0;
	do_flat = 0;
	inName = outName = mediaSource = inName2 = NULL;

	swf_flags = 0;
	swf_flatten_angle = 0.0f;
	
	/*parse our args*/
	for (i = 1; i < (u32) argc ; i++) {
		arg = argv[i];
		if (arg[0] != '-') {
			if (import || do_merge) {
				outName = arg;
			} else {
				inName = arg;
			}
			break;
		}		
		if (!stricmp(arg, "-?")) {
			PrintUsage();
			return 0;
		} else if (!stricmp(arg, "-version")) {
			PrintVersion();
			return 0;
		} else if (!stricmp(arg, "-sdp")) print_sdp = 1;
		else if (!stricmp(arg, "-info")) {
			print_info = 1;
			if (sscanf(argv[i+1], "%d", &info_track_id)==1) {
				i++;
			} else {
				info_track_id=0;
			}
		}
		else if (!stricmp(arg, "-raw")) {
			track_dump_type = DUMP_TRACK_NATIVE;
			trackID = atoi(argv[i+1]);
			i++;
		} else if (!stricmp(arg, "-nhnt")) {
			track_dump_type = DUMP_TRACK_NHNT;
			trackID = atoi(argv[i+1]);
			i++;
		} else if (!stricmp(arg, "-avi")) {
			track_dump_type = DUMP_TRACK_AVI;
			trackID = atoi(argv[i+1]);
			i++;
		}
		else if (!stricmp(arg, "-node")) {
			PrintNode(argv[i+1]);
			return (0);
		} else if (!stricmp(arg, "-nodes")) {
			PrintBuiltInNodes();
			return (0);
		} 
		else if (!stricmp(arg, "-std")) dump_stderr = 1;
		else if (!stricmp(arg, "-bt")) dump_mode = 1;
		else if (!stricmp(arg, "-xmt")) dump_mode = 2;
		else if (!stricmp(arg, "-stat")) stat_level = 1;
		else if (!stricmp(arg, "-stats")) stat_level = 2;
		else if (!stricmp(arg, "-statx")) stat_level = 3;
		else if (!stricmp(arg, "-dmp4")) dump_mp4 = 1;
		else if (!stricmp(arg, "-drtp")) dump_rtp = 1;
		/*SWF importer options*/
		else if (!stricmp(arg, "-static")) swf_flags |= M4SWF_StaticDictionary;
		else if (!stricmp(arg, "-ctrl")) swf_flags |= M4SWF_SplitTimeline;
		else if (!stricmp(arg, "-notext")) swf_flags |= M4SWF_NoText;
		else if (!stricmp(arg, "-nofont")) swf_flags |= M4SWF_NoFonts;
		else if (!stricmp(arg, "-noline")) swf_flags |= M4SWF_NoStrike;
		else if (!stricmp(arg, "-nograd")) swf_flags |= M4SWF_NoGradient;
		else if (!stricmp(arg, "-nograd")) swf_flags |= M4SWF_NoGradient;
		else if (!stricmp(arg, "-quad")) swf_flags |= M4SWF_UseXCurve;
		else if (!stricmp(arg, "-xlp")) swf_flags |= M4SWF_UseXLineProps;
		else if (!stricmp(arg, "-flatten")) {
			swf_flatten_angle = (Float) atof(argv[i+1]);
			i++;
		}

#ifndef M4_READ_ONLY
		else if (!stricmp(arg, "-isma")) {
			MakeIsma = 1;
			open_edit = 1;
		} else if (!stricmp(arg, "-3gp")) {
			Make3GP = 1;
			open_edit = 1;
		} else if (!stricmp(arg, "-ismax")) {
			MakeIsma = 3;
			open_edit = 1;
		} else if (!stricmp(arg, "-out")) {
			outName = argv[i+1];
			i++;
		} else if (!stricmp(arg, "-inter")) {
			if (InterTime) {
				PrintUsage();
				return 1;
			}
			InterTime = ( (Float) atof(argv[i+1]) ) / 1000;
			if (!InterTime) {
				PrintUsage();
				return 1;
			}
			open_edit = 1;
			i++;
		} else if (!stricmp(arg, "-frag")) {
			if (InterTime) {
				PrintUsage();
				return 1;
			}
			InterTime = ((Float) atof(argv[i+1]) ) / 1000;
			open_edit = 1;
			i++;
			Frag = 1;
		} else if (!stricmp(arg, "-hint")) {
			open_edit = 1;
			HintIt = 1;
		} else if (!stricmp(arg, "-copy")) {
			HintCopy = 1;
		} else if (!stricmp(arg, "-tight")) {
			FullInter = 1;
		} else if (!stricmp(arg, "-ocr")) {
			force_ocr = 1;
		} else if (!stricmp(arg, "-rap")) hint_flags |= M4HF_SignalRAP;
		else if (!stricmp(arg, "-ts")) hint_flags |= M4HF_SignalTS;
		else if (!stricmp(arg, "-size")) hint_flags |= M4HF_SignalSize;
		else if (!stricmp(arg, "-idx")) hint_flags |= M4HF_SignalIDX;
		else if (!stricmp(arg, "-multi")) hint_flags |= M4HF_UseMultiSL;
		else if (!stricmp(arg, "-mtu")) {
			MTUSize = atoi(argv[i+1]);
			i++;
		} else if (!stricmp(arg, "-single")) {
			track_dump_type = DUMP_TRACK_MP4;
			trackID = atoi(argv[i+1]);
			i++;
		} else if (!stricmp(arg, "-iod")) {
			regular_iod = 1;
		} else if (!stricmp(arg, "-flat")) {
			do_flat = 1;
		} else if (!stricmp(arg, "-import")) {
			import = 1;
			open_edit = 1;
			inName = argv[i+1];
			i++;
		} else if (!stricmp(arg, "-merge")) {
			if (i+3 > (u32) argc) {
				fprintf(stdout, "Usage: -merge file1.mp4 file2.mp4\n");
				return 1;
			}
			do_merge = 1;
			open_edit = 1;
			inName = argv[i+1];
			inName2 = argv[i+2];
			i+=2;
		} else if (!stricmp(arg, "-dref")) {
			use_dref = 1;
		} else if (!stricmp(arg, "-nodrop")) {
			no_frame_drop = 1;
		} else if (!stricmp(arg, "-ms")) {
			mediaSource = argv[i+1];
			i++;
		} else if (!stricmp(arg, "-mp4")) {
			encode = 1;
			open_edit = 1;
		} else if (!stricmp(arg, "-log")) {
			do_log = 1;
		} else if (!stricmp(arg, "-def")) {
			encode_flags |= M4SM_EncodeNames;
		} else if (!stricmp(arg, "-sync")) {
			encode_flags |= M4SM_RAPInBand;
			rap_freq = atoi(argv[i+1]);
			i++;
		} else if (!stricmp(arg, "-shadow")) {
			if (encode_flags & M4SM_RAPInBand) encode_flags ^= M4SM_RAPInBand;
			rap_freq = atoi(argv[i+1]);
			i++;
		} 
#endif
		else if (!stricmp(arg, "-h")) {
			if (i+1== (u32) argc) PrintUsage();
			else if (!strcmp(argv[i+1], "extract")) PrintExtractUsage();
			else if (!strcmp(argv[i+1], "dump")) PrintDumpUsage();
			else if (!strcmp(argv[i+1], "swf")) PrintSWFUsage();
#ifndef M4_READ_ONLY
			else if (!strcmp(argv[i+1], "general")) PrintGeneralUsage();
			else if (!strcmp(argv[i+1], "hint")) PrintHintUsage();
			else if (!strcmp(argv[i+1], "import")) PrintImportUsage();
			else if (!strcmp(argv[i+1], "encode")) PrintEncodeUsage();
#endif
			else PrintUsage();
			return 0;
		} else {
			fprintf(stdout, "Option %s unknown. Please check usage\n", arg);
			return 1;
		}
	}
	if (!inName) {
		PrintUsage();
		return 1;
	}

	if ((import || do_merge) && !do_flat) InterTime = 0.5f;

	if (import) {
#ifndef M4_READ_ONLY
		if (!outName) {
			fprintf(stdout, "import syntax: -import file.* file.mp4\n");
			return 1;
		}
		if (!HintIt && !Frag && !InterTime) {
			file = import_file(inName, outName, use_dref, no_frame_drop);
		} else {
			file = import_file(inName, NULL, use_dref, no_frame_drop);
		}
		if (!file) return 1;
		MakeIsma = needs_isma(file);
		needSave = 1;
#else
		fprintf(stdout, "Cannot import file - read-only release\n");
#endif
	} 
	else if (do_merge) {
#ifndef M4_READ_ONLY
		if (!outName || !inName || !inName2 ) {
			fprintf(stdout, "import syntax: -import file.* file.mp4\n");
			return 1;
		}
		if (!HintIt && !Frag && !InterTime) {
			file = merge_files(inName, inName2, outName);
		} else {
			file = merge_files(inName, inName2, NULL);
		}
		if (!file) return 1;
		MakeIsma = needs_isma(file);
		needSave = 1;
#else
		fprintf(stdout, "Cannot merge files - read-only release\n");
#endif
	}
	else if (encode) {
#ifndef M4_READ_ONLY
		char logfile[5000];
		if (do_log) {
			strcpy(logfile, inName);
			while (logfile[strlen(logfile)-1] != '.') logfile[strlen(logfile)-1] = 0;
			logfile[strlen(logfile)-1] = 0;
			strcat(logfile, "_log.txt");
		}

		if (outName) {
			strcpy(outfile, outName);
		} else {
			strcpy(outfile, inName);
			while (outfile[strlen(outfile)-1] != '.') outfile[strlen(outfile)-1] = 0;
			outfile[strlen(outfile)-1] = 0;
			strcat(outfile, ".mp4");
		}
		file = M4_MovieOpen(outfile, M4_WRITE_EDIT);
		e = EncodeFile(inName, file, do_log ? logfile : NULL, mediaSource ? mediaSource : outfile, encode_flags, rap_freq);
		if (e) goto err_exit;
#else
		fprintf(stdout, "Cannot encode file - read-only release\n");
#endif
		needSave = 1;
	} else {
		file = NULL;
		switch (get_file_type_by_ext(inName)) {
		case 1:
			file = M4_MovieOpen(inName, (u8) (open_edit ? M4_OPEN_EDIT : ( (dump_mp4>0) ? 0 : M4_OPEN_READ) ));
			if (!file) {
				fprintf(stdout, "Error opening file: %s\n", M4ErrToString(M4_GetLastError(NULL)));
				return 1;
			}
			break;
		/*allowed for bt<->xmt*/
		case 2:
		case 3:
		/*allowed for swf->bt, swf->xmt*/
		case 4:
			break;
		default:
			fprintf(stdout, "Cannot open %s - extension not supported\n", inName);
			return 1;
		}
	}

	if (outName)
		strcpy(outfile, outName);
	else
		strcpy(outfile, inName);
	while (outfile[strlen(outfile)-1] != '.') outfile[strlen(outfile)-1] = 0;
	outfile[strlen(outfile)-1] = 0;

	if (dump_mode) dump_file_text(inName, dump_stderr ? NULL : outfile, (dump_mode==2) ? 1 : 0);
	if (print_sdp) DumpSDP(file, dump_stderr ? NULL : outfile);
	if (print_info) {
		if (info_track_id) TrackInfo(file, info_track_id, 1);
		else MovieInfo(file);
	}
	if (dump_mp4) dump_file_mp4(file, dump_stderr ? NULL : outfile);
	if (dump_rtp) dump_file_rtp(file, dump_stderr ? NULL : outfile);
	if (track_dump_type) {
		char szFile[1024];
		M4TrackDumper m4dump;
		memset(&m4dump, 0, sizeof(m4dump));
		m4dump.file = file;
		m4dump.dump_type = track_dump_type;
		m4dump.trackID = trackID;
		sprintf(szFile, "%s_track%d", outfile, trackID);
		m4dump.out_name = szFile;
		e = MP4T_DumpTrack(&m4dump);
	}

	if (stat_level) dump_scene_stats(file, outfile, stat_level);

	if (!open_edit) {
		if (file) M4_MovieDelete(file);
		return 0;
	}

#ifndef M4_READ_ONLY
	if (!encode) {
		if (outName) {
			strcpy(outfile, outName);
		} else {
			char *rel_name = strrchr(inName, M4_PATH_SEPARATOR);
			if (rel_name) inName = rel_name + 1;
			sprintf(outfile, "out_%s", inName);
		}
		if (MakeIsma) {
			if (MakeIsma != 2) fprintf(stdout, "Converting to ISMA Audio-Video MP4 file...\n");
			e = MP4T_MakeISMA(file, (MakeIsma==2) ? 1 : 0, (MakeIsma==3) ? 1 : 0, NULL);
			if (e) goto err_exit;
			needSave = 1;
		}
		if (Make3GP) {
			fprintf(stdout, "Converting to 3GP file...\n");
			e = MP4T_Make3GPP(file, NULL);
			if (e) goto err_exit;
			needSave = 1;
		}
	}

	if (Frag) {
		if (HintIt) fprintf(stdout, "Warning: cannot hint and fragment - ignoring hint\n");
		fprintf(stdout, "Fragmenting file (%3f seconds fragments)...", InterTime);
		e = MP4T_FragmentMovie(file, outfile, InterTime);
		if (e) fprintf(stdout, "Error while fragmenting file: %s\n", M4ErrToString(e));
		else fprintf(stdout, " done\n");
		M4_MovieDelete(file);
		return (e!=M4OK) ? 1 : 0;
	}
	
	if (HintIt) {
		if (force_ocr) SetupClockReferences(file);
		fprintf(stdout, "Hinting file with Path-MTU %d Bytes\n", MTUSize);
		MTUSize -= 12;		
		if (!hint_flags) hint_flags = M4HF_AutoConf;
		HintFile(file, MakeIsma, MTUSize, hint_flags, HintCopy, HintInter, regular_iod);
		needSave = 1;
	}

	/*time interleave it if not hinted*/
	if (!HintIt && InterTime) {
		e = MP4T_MakeInterleaved(file, InterTime);
		needSave = 1;
	}
	/*full interleave (sample-based) if just hinted*/
	else if (HintIt && FullInter) {
		e = M4_SetStorageMode(file, M4_FULL_INTERLEAVED);
	} else {
		e = M4_SetStorageMode(file, M4_STREAMABLE);
	}
	if (e) goto err_exit;


	if (!encode) M4_SetFinalFileName(file, outfile);
	if (needSave) {
		fprintf(stdout, "Saving file into %s: ", outfile);
		if (!HintIt && InterTime) {
			fprintf(stdout, "Interleaving on (%.3f seconds)...", InterTime);
		}
		else if (HintIt && FullInter) {
			fprintf(stdout, "Hinted file - Full Interleaving...");
		} else {
			fprintf(stdout, "Flat storage...");
		}
		e = M4_MovieClose(file);
		if (e) goto err_exit;
		fprintf(stdout, " done\n");
	} else {
		M4_MovieDelete(file);
	}

	if (e) fprintf(stdout, "Error: %s\n", M4ErrToString(e));
	return (e!=M4OK) ? 1 : 0;

err_exit:
	M4_MovieDelete(file);
	fprintf(stdout, "\n\tError: %s\n", M4ErrToString(e));
	return 1;
#else
	M4_MovieDelete(file);
	fprintf(stdout, "Error: Read-only version of MP4Box.\n");
	return 1;
#endif
}

