/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Graph Generator sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <gpac/m4_tools.h>


#define COPYRIGHT "/*\n *			GPAC - MPEG-4 Systems C Development Kit\n *\n *			Copyright (c) Jean Le Feuvre 2000-2003\n *					All rights reserved\n *\n *  This file is part of GPAC / Scene Graph sub-project\n *\n *  GPAC is free software; you can redistribute it and/or modify\n *  it under the terms of the GNU General Public License as published by\n *  the Free Software Foundation; either version 2, or (at your option)\n *  any later version.\n *\n *  GPAC is distributed in the hope that it will be useful,\n *  but WITHOUT ANY WARRANTY; without even the implied warranty of\n *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n *  GNU General Public License for more details.	\n *\n *  You should have received a copy of the GNU General Public License\n *  along with GNU Make; see the file COPYING.  If not, write to\n *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.\n *\n */\n"

static char *CurrentLine;

void PrintUsage()
{
	printf("SGGen [-p file] template_file_v1 (template_file_v2 ...)\n"
			"\nGPAC Scene Graph generator. Usage:\n"
			"-p: listing file of nodes to exclude from tables\n"
			"Template files MUST be fed in order\n"
			"\n"
			"Generated Files are directly updated in the GPAC distribution - do NOT try to change this\n\n"
			"Written by Jean Le Feuvre - (c) 2000-2003\n"
			);
}

//a node field
typedef struct
{
	char type[50];
	//SFxxx, MFxxx
	char familly[50];
	//name
	char name[1000];
	//default value
	char def[100];
	//bounds
	u32 hasBounds;
	char b_min[20];
	char b_max[20];
	//Quant
	u32 hasQuant;
	char quant_type[50];
	char qt13_bits[50];
	//Anim
	u32 hasAnim;
	u32 AnimType;

} BField;

//NDTs

//a BIFS node
typedef struct
{
	char name[1000];
	//NDT info. NDT are created in alphabetical order
	Chain *NDT;
	//0: normal, 1: special
	u32 codingType;
	u32 version;

	Chain *Fields;

	//coding types
	u8 hasDef, hasIn, hasOut, hasDyn;
	u8 hasAQInfo;

	u8 hasDefault;

	u8 skip_impl;

	char Child_NDT_Name[1000];
} BNode;


void skip_sep(char *sep)
{
	//skip separaors
	while (*CurrentLine && strchr(sep, *CurrentLine)) {
		CurrentLine = CurrentLine + 1;
		//end of line - no token
		if (*CurrentLine == '\n') return;
	}
}

//note that we increment the line no matter what
u32 GetNextToken(char *token, char *sep)
{
	u32 i , j = 0;

	strcpy(token, "");
	
	//skip separaors
	while (*CurrentLine && strchr(sep, *CurrentLine)) {
		CurrentLine = CurrentLine + 1;
		j ++;
		//end of line - no token
		if (*CurrentLine == '\n') return 0;
	}

	//copy token untill next blank
	i=0;
	while (1) {
		//bad line
		if (! *CurrentLine) {
			token[i] = 0;
			return 0;
		}
		//end of token or end of line
		if (strchr(sep, *CurrentLine) || (*CurrentLine == '\n') ) {
			token[i] = 0;
			CurrentLine = CurrentLine + 1;
			return i;
		} else {
			token[i] = *CurrentLine;
		}
		CurrentLine = CurrentLine + 1;
		i++;
		j++;
	}
	return 1;
}

BField *BlankField()
{
	BField *n = malloc(sizeof(BField));
	memset(n, 0, sizeof(BField));
	return n;
}


BNode *BlankNode()
{
	BNode *n = malloc(sizeof(BNode));
	memset(n, 0, sizeof(BNode));
	n->NDT = NewChain();
	n->Fields = NewChain();
	return n;
}

u8 IsNDT(Chain *NDTs, char *famName)
{
	u32 i;
	char *ndtName;
	for (i=0; i<ChainGetCount(NDTs); i++) {
		ndtName = ChainGetEntry(NDTs, i);
		//remove SF / MF as we don't need that
		if (!strcmp(ndtName+2, famName+2)) return 1;
	}
	return 0;
}

void CheckInTable(char *token, Chain *NDTs)
{
	u32 i;
	char *p;
	for (i=0; i<ChainGetCount(NDTs); i++) {
		p = ChainGetEntry(NDTs, i);
		if (!strcmp(p, token)) return;
	}
	p = malloc(strlen(token)+1);
	strcpy(p, token);
	ChainAddEntry(NDTs, p);
}

/*type: 0: header, 1: source*/
FILE *BeginFile(char *name, u32 type)
{
	FILE *f;

	char sPath[MAX_FILE_PATH];

	if (!type) {
		if (!strcmp(name, "NDT")) {
			sprintf(sPath, "..%c..%cinclude%cgpac%cintern%cm4_node_tables.h", M4_PATH_SEPARATOR, M4_PATH_SEPARATOR, M4_PATH_SEPARATOR, M4_PATH_SEPARATOR, M4_PATH_SEPARATOR);
		}
		/*m4_nodes.h is exported*/
		else {
			sprintf(sPath, "..%c..%cinclude%cgpac%cm4_nodes.h", M4_PATH_SEPARATOR, M4_PATH_SEPARATOR, M4_PATH_SEPARATOR, M4_PATH_SEPARATOR);
		}
	} else {
		sprintf(sPath, "..%c..%cM4Systems%cSceneGraph%c%s.c", M4_PATH_SEPARATOR, M4_PATH_SEPARATOR, M4_PATH_SEPARATOR, M4_PATH_SEPARATOR, name);
	}
	
	f = fopen(sPath, "wt");
	fprintf(f, "%s\n", COPYRIGHT);

	fprintf(f, "\n/*FILE AUTOMATICALLY GENERATED BY SGGen - DO NOT MOFIFY*/\n\n");

	if (!type) {
		fprintf(f, "#ifndef _%s_H\n", name);
		fprintf(f, "#define _%s_H\n\n", name);
		if (!strcmp(name, "m4_nodes")) {
			fprintf(f, "#ifdef __cplusplus\nextern \"C\" {\n#endif\n\n");
		}
	}
	return f;
}

void EndFile(FILE *f, char *name, u32 type)
{
	if (!type) {
		if (!strcmp(name, "m4_nodes")) {
			fprintf(f, "#ifdef __cplusplus\n}\n#endif\n\n");
		}
		fprintf(f, "\n\n#endif\t\t/*_%s_H*/\n\n", name);
	}
	fclose(f);
}

void TranslateToken(char *token)
{
	if (!strcmp(token, "+I") || !strcmp(token, "I")) {
		strcpy(token, "M4_MAX_FLOAT");
	}
	else if (!strcmp(token, "-I")) {
		strcpy(token, "M4_MIN_FLOAT");
	}
}



void WriteNodesFile(Chain *BNodes, Chain *NDTs, u32 NumVersions)
{
	FILE *f;
	u32 i, j;
	char *NDTName;
	BNode *n;
	BField *bf;
	Bool hasViewport;

	f = BeginFile("m4_nodes", 0);

	fprintf(f, "#include <gpac/m4_scenegraph.h>\n\n");

	//write all tags
	fprintf(f, "\n\nenum {\n");
	fprintf(f, "\tTAG_NULL = 0");
	
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		fprintf(f, ",\n\tTAG_%s", n->name);
	}
	fprintf(f, ",\n\t/*all proto instances have this tag*/\n\tTAG_ProtoNode");
	fprintf(f, ",\n\t/*undefined nodes (when parsing) use this tag*/\n\tTAG_UndefinedNode\n};\n\n");

	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (n->skip_impl) continue;

		fprintf(f, "typedef struct _tag%s\n{\n", n->name);
		fprintf(f, "\tBASE_NODE\n");

		/*write children field*/
		for (j=0; j<ChainGetCount(n->Fields); j++) {
			bf = ChainGetEntry(n->Fields, j);
			if (!stricmp(bf->name, "addChildren") || !strcmp(bf->name, "removeChildren")) continue;
			if (!strcmp(bf->name, "children") && stricmp(n->name, "audioBuffer")) {
				fprintf(f, "\tCHILDREN\n");
				break;
			}
		}
		for (j=0; j<ChainGetCount(n->Fields); j++) {
			bf = ChainGetEntry(n->Fields, j);
			
			if (!strcmp(bf->name, "addChildren") || !strcmp(bf->name, "removeChildren")) continue;
			if (!strcmp(bf->name, "children") && stricmp(n->name, "audioBuffer")) continue;

			//write remaining fields
			//eventIn fields are handled as pointer to functions, called by the route manager
			if (!strcmp(bf->type, "eventIn")) {
				fprintf(f, "\t%s %s;\t/*eventIn*/\n", bf->familly, bf->name);
				fprintf(f, "\tvoid (*on_%s)(SFNode *pThis);\t/*eventInHandler*/\n", bf->name);
			} else if (!strcmp(bf->type, "eventOut")) {
				//eventOut fields are handled as an opaque stack pointing to the route manager
				//this will be refined once the route is in place
				//we will likely need a function such as:
				//	void SignalRoute(route_stack, node, par)
				fprintf(f, "\t%s %s;\t/*eventOut*/\n", bf->familly, bf->name);
			} else if (strstr(bf->familly, "Node")) {		
				//this is a POINTER to a node 
				if (strstr(bf->familly, "SF")) {
					fprintf(f, "\tSFNode *%s;\t/*%s*/\n", bf->name, bf->type);
				} else {
					//this is a POINTER to a chain
					fprintf(f, "\tChain *%s;\t/*%s*/\n", bf->name, bf->type);
				}
			} else {
				fprintf(f, "\t%s %s;\t/*%s*/\n", bf->familly, bf->name, bf->type);
			}
		}
		fprintf(f, "} B_%s;\n\n\n", n->name);
	}

	
	hasViewport = 0;
	//all NDTs are defined in v1
	fprintf(f, "/*NodeDataType tags*/\nenum {\n");
	for (i=0; i<ChainGetCount(NDTs); i++) {
		NDTName = ChainGetEntry(NDTs, i);
		if (!i) {
			fprintf(f, "\tNDT_%s = 1", NDTName);
		} else {
			fprintf(f, ",\n\tNDT_%s", NDTName);
		}

		if (strcmp(NDTName, "SFViewportNode")) hasViewport = 1;
	}
	//template fix: some node use NDT_SFViewportNode but the table is empty-> not generated
	if (!hasViewport) fprintf(f, ",\n\tNDT_SFViewportNode");
	fprintf(f, "\n};\n\n");


	fprintf(f, "/*All BIFS versions handled*/\n");
	fprintf(f, "#define NUM_BIFS_VERSION\t\t%d\n\n", NumVersions);
	fprintf(f, "enum {\n");
	for (i=0; i<NumVersions; i++) {
		if (!i) {
			fprintf(f, "\tBIFS_V1 = 1,\n");
		} else {
			fprintf(f, "\tBIFS_V%d,\n", i+1);
		}
	}
	fprintf(f, "\tLAST_BIFS_VERSION = BIFS_V%d\n};\n\n", i);

	/*write defined nodes*/
	fprintf(f, "/*All nodes generated*/\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (!n->skip_impl) fprintf(f, "#define M4_DEF_%s\n", n->name);
	}

	fprintf(f, "\n\n");

	EndFile(f, "m4_nodes", 0);

}


u32 IsNodeInTable(BNode *node, char *NDTName)
{
	u32 i;
	char *ndt;

	for (i=0; i<ChainGetCount(node->NDT); i++) {
		ndt = ChainGetEntry(node->NDT, i);
		if (!strcmp(ndt, NDTName)) return 1;
	}
	return 0;
}

u32 GetBitsCount(u32 MaxVal)
{
	u32 k=0;

	while ((s32) MaxVal > ((1<<k)-1) ) k+=1;
	return k;
}

u32 GetNDTCount(char *NDTName, Chain *BNodes, u32 Version)
{
	u32 i, nodeCount;
	BNode *n;

	//V1 begins at 0
	if (Version == 1) {
		nodeCount = 0;
	}
	//V2 at 2 (0 reserved + 1 proto)
	else if (Version == 2) {
		nodeCount = 2;
	}
	//other at 1 (0 reserved, no proto)
	else {
		nodeCount = 1;
	}
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (n->version != Version) continue;
		if (!IsNodeInTable(n, NDTName)) continue;
		nodeCount++;
	}
	return nodeCount;

}

void WriteNDT_H(FILE *f, Chain *BNodes, Chain *NDTs, u32 Version)
{
	u32 i, j, first, count;
	char *NDTName;
	BNode *n;


	fprintf(f, "\n\n/* NDT BIFS Version %d */\n\n", Version);

	//for all NDTs
	for (i=0; i<ChainGetCount(NDTs); i++) {
		NDTName = ChainGetEntry(NDTs, i);
		count = GetNDTCount(NDTName, BNodes, Version);
		if (Version == 2) {
			count -= 2;
		} else if (Version > 2) {
			count -= 1;
		}
		if (!count) continue;
		
		//numBits
		fprintf(f, "#define %s_V%d_NUMBITS\t\t%d\n", NDTName, Version, GetBitsCount(count + ( (Version == 2) ? 1 : 0) ) );
		fprintf(f, "#define %s_V%d_Count\t%d\n\n", NDTName, Version, count);
		
		fprintf(f, "static const u32 %s_V%d_TypeToTag[%d] = {\n", NDTName, Version, count);
		first = 1;
		//browse each node.
		for (j=0; j<ChainGetCount(BNodes); j++) {
			n = ChainGetEntry(BNodes, j);
			if (n->version != Version) continue;
			if (!IsNodeInTable(n, NDTName)) continue;
			
			if (first) {
				fprintf(f, " TAG_%s", n->name);
				first = 0;
			} else {
				fprintf(f, ", TAG_%s", n->name);
			}
		}
		fprintf(f, "\n};\n\n");

	}

	fprintf(f, "\nu32 NDT_V%d_GetNumBits(u32 NDT_Tag);\n", Version);
	fprintf(f, "u32 NDT_V%d_GetNodeTag(u32 Context_NDT_Tag, u32 NodeType);\n", Version);
	fprintf(f, "u32 NDT_V%d_GetNodeType(u32 NDT_Tag, u32 NodeTag);\n", Version);
	
	fprintf(f, "\n\n");
}

//write the NDTs functions for v1 nodes
//all our internal handling is in TAG_#nodename because we need an homogeneous
//namespace for all nodes (v1, v2, v3 and v4)
//the NDT functions will perform the translation from the NDT value to the absolute
//TAG of the node
void WriteNDT_Dec(FILE *f, Chain *BNodes, Chain *NDTs, u32 Version)
{
	char *NDTName;
	u32 i, count;

	//NodeTag complete translation
	fprintf(f, "\n\n\nu32 NDT_V%d_GetNodeTag(u32 Context_NDT_Tag, u32 NodeType)\n{\n\tif (!NodeType) return 0;\n", Version);

	//handle version
	fprintf(f, "\t/* adjust according to the table version */\n");
	if (Version == 2) {
		fprintf(f, "\t/* v2: 0 reserved for extensions, 1 reserved for protos */\n");
		fprintf(f, "\tif (NodeType == 1) return 0;\n");
		fprintf(f, "\tNodeType -= 2;\n");
	} else {
		fprintf(f, "\t/* v%d: 0 reserved for extensions */\n", Version);
		fprintf(f, "\tNodeType -= 1;\n");
	}

	fprintf(f, "\tswitch (Context_NDT_Tag) {\n");

	for (i=0; i<ChainGetCount(NDTs); i++) {
		NDTName = ChainGetEntry(NDTs, i);

		count = GetNDTCount(NDTName, BNodes, Version);
		if (Version == 2) {
			count -= 2;
		} else if (Version > 2) {
			count -= 1;
		}
		if (!count) continue;
		
		fprintf(f, "\tcase NDT_%s:\n\t\tif (NodeType >= %s_V%d_Count) return 0;\n", NDTName, NDTName, Version);
		fprintf(f, "\t\treturn %s_V%d_TypeToTag[NodeType];\n", NDTName, Version);
	}
	fprintf(f, "\tdefault:\n\t\treturn 0;\n\t}\n}");

	//NDT codec bits
	fprintf(f, "\n\n\nu32 NDT_V%d_GetNumBits(u32 NDT_Tag)\n{\n\tswitch (NDT_Tag) {\n", Version);

	for (i=0; i<ChainGetCount(NDTs); i++) {
		NDTName = ChainGetEntry(NDTs, i);

		count = GetNDTCount(NDTName, BNodes, Version);
		if (Version == 2) {
			count -= 2;
		} else if (Version > 2) {
			count -= 1;
		}
		if (!count) continue;
		
		fprintf(f, "\tcase NDT_%s:\n\t\treturn %s_V%d_NUMBITS;\n", NDTName, NDTName, Version);
	}
	/*all tables have 1 node in v2 for proto coding*/
	fprintf(f, "\tdefault:\n\t\treturn %d;\n\t}\n}\n\n", (Version==2) ? 1 : 0);
}


void WriteNDT_Enc(FILE *f, Chain *BNodes, Chain *NDTs, u32 Version)
{
	u32 i, count;
	char *NDTName;

	fprintf(f, "u32 NDT_V%d_GetNodeType(u32 NDT_Tag, u32 NodeTag)\n{\n\tif(!NDT_Tag || !NodeTag) return 0;\n\tswitch(NDT_Tag) {\n", Version);
	for (i=0; i<ChainGetCount(NDTs); i++) {
		NDTName = ChainGetEntry(NDTs, i);
		count = GetNDTCount(NDTName, BNodes, Version);
		if (Version == 2) {
			count -= 2;
		} else if (Version > 2) {
			count -= 1;
		}
		if (!count) continue;
		fprintf(f, "\tcase NDT_%s:\n\t\treturn ALL_GetNodeType(%s_V%d_TypeToTag, %s_V%d_Count, NodeTag, BIFS_V%d);\n", NDTName, NDTName, Version, NDTName, Version, Version);
	}
	fprintf(f, "\tdefault:\n\t\treturn 0;\n\t}\n}\n\n");
}



void WriteNodeFields(FILE *f, BNode *n)
{
	u32 i, first;
	BField *bf;
	u32 NbDef, NbIn, NbOut, NbDyn, hasAQ;

	NbDef = NbIn = NbOut = NbDyn = hasAQ = 0;
	for (i=0;i<ChainGetCount(n->Fields); i++) {
		bf = ChainGetEntry(n->Fields, i);
		if (!strcmp(bf->type, "field") || !strcmp(bf->type, "exposedField")) {
			NbDef += 1;
		}
		if (!strcmp(bf->type, "eventIn") || !strcmp(bf->type, "exposedField")) {
			NbIn += 1;
			//check for anim
			if (bf->hasAnim) NbDyn += 1;
		}
		if (!strcmp(bf->type, "eventOut") || !strcmp(bf->type, "exposedField")) {
			NbOut += 1;
		}
		if (bf->hasAnim || bf->hasQuant) hasAQ = 1;
	}

	n->hasAQInfo = hasAQ;

	//write the def2all table
	if (NbDef) {
		first = 1;
		fprintf(f, "static const u16 %s_Def2All[] = { ", n->name);
		for (i=0; i<ChainGetCount(n->Fields); i++) {
			bf = ChainGetEntry(n->Fields, i);
			if (strcmp(bf->type, "field") && strcmp(bf->type, "exposedField")) continue;
			if (first) {
				fprintf(f, "%d", i);
				first = 0;
			} else {
				fprintf(f, ", %d", i);
			}
		}
		fprintf(f, "};\n");
	}
	//write the in2all table
	if (NbIn) {
		first = 1;
		fprintf(f, "static const u16 %s_In2All[] = { ", n->name);
		for (i=0; i<ChainGetCount(n->Fields); i++) {
			bf = ChainGetEntry(n->Fields, i);
			if (strcmp(bf->type, "eventIn") && strcmp(bf->type, "exposedField")) continue;
			if (first) {
				fprintf(f, "%d", i);
				first = 0;
			} else {
				fprintf(f, ", %d", i);
			}
		}
		fprintf(f, "};\n");
	}
	//write the out2all table
	if (NbOut) {
		first = 1;
		fprintf(f, "static const u16 %s_Out2All[] = { ", n->name);
		for (i=0; i<ChainGetCount(n->Fields); i++) {
			bf = ChainGetEntry(n->Fields, i);
			if (strcmp(bf->type, "eventOut") && strcmp(bf->type, "exposedField")) continue;
			if (first) {
				fprintf(f, "%d", i);
				first = 0;
			} else {
				fprintf(f, ", %d", i);
			}
		}
		fprintf(f, "};\n");
	}
	//then write the dyn2all table
	if (NbDyn) {
		first = 1;
		fprintf(f, "static const u16 %s_Dyn2All[] = { ", n->name);
		for (i=0; i<ChainGetCount(n->Fields); i++) {
			bf = ChainGetEntry(n->Fields, i);
			if (strcmp(bf->type, "eventIn") && strcmp(bf->type, "exposedField")) continue;
			if (!bf->hasAnim) continue;
			if (first) {
				fprintf(f, "%d", i);
				first = 0;
			} else {
				fprintf(f, ", %d", i);
			}
		}
		fprintf(f, "};\n");
	}

	n->hasDef = NbDef;
	n->hasDyn = NbDyn;
	n->hasIn = NbIn;
	n->hasOut = NbOut;


	fprintf(f, "\nstatic u32 %s_get_field_count(SFNode *node, u8 IndexMode)\n{\n", n->name);
	fprintf(f, "\tswitch(IndexMode) {\n");

	fprintf(f, "\tcase FCM_IN: return %d;\n", NbIn);
	fprintf(f, "\tcase FCM_DEF: return %d;\n", NbDef);
	fprintf(f, "\tcase FCM_OUT: return %d;\n", NbOut);
	fprintf(f, "\tcase FCM_DYN: return %d;\n", NbDyn);
	fprintf(f, "\tdefault:\n");
	fprintf(f, "\t\treturn %d;\n", ChainGetCount(n->Fields));
	fprintf(f, "\t}");

	fprintf(f, "\n}\n");

	fprintf(f, "\nstatic M4Err %s_get_field_index(SFNode *n, u32 inField, u8 IndexMode, u32 *allField)\n{\n", n->name);
	fprintf(f, "\tswitch(IndexMode) {\n");

	if (NbIn) {
		fprintf(f, "\tcase FCM_IN:\n");
//		fprintf(f, "\t\tif (inField>=%d) return M4BadParam;\n", NbIn);
		fprintf(f, "\t\t*allField = %s_In2All[inField];\n\t\treturn M4OK;\n", n->name);
	}
	if (NbDef) {
		fprintf(f, "\tcase FCM_DEF:\n");
//		fprintf(f, "\t\tif (inField>=%d) return M4BadParam;\n", NbDef);
		fprintf(f, "\t\t*allField = %s_Def2All[inField];\n\t\treturn M4OK;\n", n->name);
	}
	if (NbOut) {
		fprintf(f, "\tcase FCM_OUT:\n");
//		fprintf(f, "\t\tif (inField>=%d) return M4BadParam;\n", NbOut);
		fprintf(f, "\t\t*allField = %s_Out2All[inField];\n\t\treturn M4OK;\n", n->name);
	}
	if (NbDyn) {
		fprintf(f, "\tcase FCM_DYN:\n");
//		fprintf(f, "\t\tif (inField>=%d) return M4BadParam;\n", NbDyn);
		fprintf(f, "\t\t*allField = %s_Dyn2All[inField];\n\t\treturn M4OK;\n", n->name);
	}

	fprintf(f, "\tdefault:\n");
	fprintf(f, "\t\treturn M4BadParam;\n");
	fprintf(f, "\t}");

	fprintf(f, "\n}\n");


	fprintf(f, "static M4Err %s_get_field(SFNode *node, FieldInfo *info)\n{\n\tswitch (info->allIndex) {\n", n->name);
	for (i=0;i<ChainGetCount(n->Fields); i++) {
		bf = ChainGetEntry(n->Fields, i);

		fprintf(f, "\tcase %d:\n", i);
		
		fprintf(f, "\t\tinfo->name = \"%s\";\n", bf->name);

		//skip all eventIn
		if (!strcmp(bf->type, "eventIn")) {
			fprintf(f, "\t\tinfo->eventType = ET_EventIn;\n");
			fprintf(f, "\t\tinfo->on_event_in = ((B_%s *)node)->on_%s;\n", n->name, bf->name);
		}
		else if (!strcmp(bf->type, "eventOut")) {
			fprintf(f, "\t\tinfo->eventType = ET_EventOut;\n");
		}
		else if (!strcmp(bf->type, "field")) {
			fprintf(f, "\t\tinfo->eventType = ET_Field;\n");
		}
		else {
			fprintf(f, "\t\tinfo->eventType = ET_ExposedField;\n");
		}

		if (strstr(bf->familly, "Node")) {
			if (strstr(bf->familly, "MF")) {
				fprintf(f, "\t\tinfo->fieldType = FT_MFNode;\n");
			} else {
				fprintf(f, "\t\tinfo->fieldType = FT_SFNode;\n");
			}
			//always remove the SF or MF, as all NDTs are SFXXX
			fprintf(f, "\t\tinfo->NDTtype = NDT_SF%s;\n", bf->familly+2);
			fprintf(f, "\t\tinfo->far_ptr = & ((B_%s *)node)->%s;\n", n->name, bf->name);
		} else {
			//no ext type
			fprintf(f, "\t\tinfo->fieldType = FT_%s;\n", bf->familly);
			fprintf(f, "\t\tinfo->far_ptr = & ((B_%s *) node)->%s;\n", n->name, bf->name);
		}
		fprintf(f, "\t\treturn M4OK;\n");
	}
	fprintf(f, "\tdefault:\n\t\treturn M4BadParam;\n\t}\n}\n\n");

}

//write the Quantization info for each node field(Quant and BIFS-Anim info)
void WriteNodeQuant(FILE *f, BNode *n)
{
	u32 i;
	BField *bf;
	fprintf(f, "static Bool %s_get_aq_info(SFNode *n, u32 FieldIndex, u8 *QType, u8 *AType, Float *b_min, Float *b_max, u32 *QT13_bits)\n{\n\tswitch (FieldIndex) {\n", n->name);
	
	for (i=0; i<ChainGetCount(n->Fields) ; i++ ) {
		bf = ChainGetEntry(n->Fields, i);
		if (!bf->hasAnim && !bf->hasQuant) continue;

		fprintf(f, "\tcase %d:\n", i);
		//Anim Type
		fprintf(f, "\t\t*AType = %d;\n", bf->AnimType);
		//Quant Type
		fprintf(f, "\t\t*QType = %s;\n", bf->quant_type);
		if (!strcmp(bf->quant_type, "13")) 
			fprintf(f, "\t\t*QT13_bits = %s;\n", bf->qt13_bits);

		//Bounds
		if (bf->hasBounds) {
			if (!strcmp(bf->b_min, "+I") || !strcmp(bf->b_min, " +I") || !strcmp(bf->b_min, "I")) {
				fprintf(f, "\t\t*b_min = M4_MAX_FLOAT;\n");
			} else if (!strcmp(bf->b_min, "-I")) {
				fprintf(f, "\t\t*b_min = M4_MIN_FLOAT;\n");
			} else {
				fprintf(f, "\t\t*b_min = (Float) %s;\n", bf->b_min);
			}
			if (!strcmp(bf->b_max, "+I") || !strcmp(bf->b_max, " +I") || !strcmp(bf->b_max, "I")) {
				fprintf(f, "\t\t*b_max = M4_MAX_FLOAT;\n");
			} else {
				fprintf(f, "\t\t*b_max = (Float) %s;\n", bf->b_max);
			}
		}
		fprintf(f, "\t\treturn 1;\n");
	}
	fprintf(f, "\tdefault:\n\t\treturn 0;\n\t}\n}\n\n");
}

#define USE_SWITCH_FUNC		1

void WriteNodeCode(Chain *BNodes)
{
	FILE *f;
	char token[20], tok[20];
	char *store;
	u32 i, j, k, go;
	BField *bf;
	BNode *n;

	f = BeginFile("Nodes", 1);

	fprintf(f, "\n#include <gpac/intern/m4_scenegraph_dev.h>\n");
	fprintf(f, "#include <gpac/intern/m4_node_tables.h>\n\n");

	for (k=0; k<ChainGetCount(BNodes); k++) {
		n = ChainGetEntry(BNodes, k);

		if (n->skip_impl) continue;

		fprintf(f, "\n/*\n\t%s Node deletion\n*/\n\n", n->name);
		fprintf(f, "static void %s_Del(SFNode *node)\n{\n\tB_%s *p = (B_%s *) node;\n", n->name, n->name, n->name);

		for (i=0; i<ChainGetCount(n->Fields); i++) {
			bf = ChainGetEntry(n->Fields, i);
			//nothing on child events
			if (!strcmp(bf->name, "addChildren")) continue;
			if (!strcmp(bf->name, "removeChildren")) continue;
			
			//delete all children node
			if (!strcmp(bf->name, "children") && stricmp(n->name, "audioBuffer")) {
				fprintf(f, "\tDestroyChildrenNode((SFNode *) p);\t\n");
				continue;
			}

			//delete ALL fields that must be deleted: this includes eventIn and out since 
			//all fields are defined in the node
			if (!strcmp(bf->familly, "MFInt32")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFInt")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFFloat")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFBool")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFColor")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFRotation")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFString")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFTime")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFVec2f")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFVec3f")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "SFString")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "SFImage")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "SFCommandBuffer")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "SFURL")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFURL")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFScript")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			}
			else if (!strcmp(bf->familly, "MFTime")) {
				fprintf(f, "\t%s_Del(p->%s);\n", bf->familly, bf->name);
			} else if (strstr(bf->familly, "Node")) {		
				//this is a POINTER to a node 
				if (strstr(bf->familly, "SF")) {
					fprintf(f, "\tNode_Delete((SFNode *) p->%s, (SFNode *) p);\t\n", bf->name);
				} else {
					//this is a POINTER to a chain
					fprintf(f, "\tNodeList_Delete((Chain *) p->%s, (SFNode *) p);\t\n", bf->name);
				}
			}
		}
		fprintf(f, "\tSFNode_Delete((SFNode *) p);\n}\n\n");

		//node fields
		WriteNodeFields(f, n);
		WriteNodeQuant(f, n);

		//
		//		Constructor
		//

		fprintf(f, "\n\nSFNode *%s_Create()\n{\n\tB_%s *p;\n\tSAFEALLOC(p, sizeof(B_%s));\n", n->name, n->name, n->name);
		fprintf(f, "\tif(!p) return NULL;\n");
		fprintf(f, "\tNode_Setup((SFNode *)p, TAG_%s);\n", n->name);

		for (i=0; i<ChainGetCount(n->Fields); i++) {
			bf = ChainGetEntry(n->Fields, i);
			//setup all children node
			if (!strcmp(bf->name, "children") && stricmp(n->name, "audioBuffer")) {
				fprintf(f, "\tSetupChildrenNode((SFNode *) p);\n");
				break;
			}
			else if ( strstr(bf->familly, "Node") && strncmp(bf->type, "event", 5) ) {		
				//this is a POINTER to a node 
				if (strstr(bf->familly, "MF")) {
					//this is a POINTER to a chain
					fprintf(f, "\tp->%s = NewChain();\t\n", bf->name);
				}
			}
			/*special case for SFCommandBuffer: we also create a command list*/
			if (!stricmp(bf->familly, "SFCommandBuffer")) {
				fprintf(f, "\tp->%s.commandList = NewChain();\t\n", bf->name);
			}
		}

		//setup pointers
		fprintf(f, "\n#ifdef NODE_USE_POINTERS\n\n");
		fprintf(f, "\t((SFNode *)p)->sgprivate->name = \"%s\";\n", n->name);
		fprintf(f, "\t((SFNode *)p)->sgprivate->node_del = %s_Del;\n", n->name);
		fprintf(f, "\t((SFNode *)p)->sgprivate->get_field_count = %s_get_field_count;\n", n->name);
		fprintf(f, "\t((SFNode *)p)->sgprivate->get_field_index = %s_get_field_index;\n", n->name);
		fprintf(f, "\t((SFNode *)p)->sgprivate->get_field = %s_get_field;\n", n->name);
		fprintf(f, "\t((SFNode *)p)->sgprivate->get_aq_info = %s_get_aq_info;\n", n->name);

		//check if we have a child node
		for (i=0; i<ChainGetCount(n->Fields); i++) {
			bf = ChainGetEntry(n->Fields, i);
			if ( !strcmp(bf->name, "children") || 
					( !strstr(bf->type, "event") && strstr(bf->familly, "MF") && strstr(bf->familly, "Node")) ) {

				fprintf(f, "\t((SFNode *)p)->sgprivate->child_ndt = NDT_SF%s;\n", bf->familly+2);
				sprintf(n->Child_NDT_Name, "NDT_SF%s", bf->familly+2);
				break;
			}
		}
		fprintf(f, "\n#endif\n\n");

		fprintf(f, "\n\t/*default field values*/\n");
		
		for (i=0; i<ChainGetCount(n->Fields); i++) {
			bf = ChainGetEntry(n->Fields, i);

			//nothing on eventIn or Out
			if (!strcmp(bf->type, "eventIn")) continue;
			if (!strcmp(bf->type, "eventOut")) continue;

			if (!strcmp(bf->def, "")) continue;

			//no default on nodes
			if (strstr(bf->familly, "Node")) continue;
			//extract default falue
			
			//
			//		SF Fields
			//
			
			//SFBool
			if (!strcmp(bf->familly, "SFBool")) {
				if (!strcmp(bf->def, "1") || !strcmp(bf->def, "TRUE")) 
					fprintf(f, "\tp->%s = 1;\n", bf->name);
			}
			//SFFloat
			else if (!strcmp(bf->familly, "SFFloat")) {
				fprintf(f, "\tp->%s = (SFFloat) %s;\n", bf->name, bf->def);
			}
			//SFTime
			else if (!strcmp(bf->familly, "SFTime")) {
				fprintf(f, "\tp->%s = %s;\n", bf->name, bf->def);
			}
			//SFInt32
			else if (!strcmp(bf->familly, "SFInt32")) {
				fprintf(f, "\tp->%s = %s;\n", bf->name, bf->def);
			}
			//SFColor
			else if (!strcmp(bf->familly, "SFColor")) {
				CurrentLine = bf->def;
				GetNextToken(token, " ");
				TranslateToken(token);

				fprintf(f, "\tp->%s.red = (SFFloat) %s;\n", bf->name, token);
				GetNextToken(token, " ");
				fprintf(f, "\tp->%s.green = (SFFloat) %s;\n", bf->name, token);
				GetNextToken(token, " ");
				fprintf(f, "\tp->%s.blue = (SFFloat) %s;\n", bf->name, token);
			}
			//SFVec2f
			else if (!strcmp(bf->familly, "SFVec2f")) {
				CurrentLine = bf->def;
				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.x = (SFFloat) %s;\n", bf->name, token);

				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.y = (SFFloat) %s;\n", bf->name, token);
			}
			//SFVec3f
			else if (!strcmp(bf->familly, "SFVec3f")) {
				CurrentLine = bf->def;
				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.x = (SFFloat) %s;\n", bf->name, token);

				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.y = (SFFloat) %s;\n", bf->name, token);

				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.z = (SFFloat) %s;\n", bf->name, token);
			}
			//SFVec4f
			else if (!strcmp(bf->familly, "SFVec4f")) {
				CurrentLine = bf->def;
				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.x = (SFFloat) %s;\n", bf->name, token);

				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.y = (SFFloat) %s;\n", bf->name, token);

				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.z = (SFFloat) %s;\n", bf->name, token);

				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.q = (SFFloat) %s;\n", bf->name, token);
			}
			//SFRotation
			else if (!strcmp(bf->familly, "SFRotation")) {
				CurrentLine = bf->def;
				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.xAxis = (SFFloat) %s;\n", bf->name, token);

				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.yAxis = (SFFloat) %s;\n", bf->name, token);

				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.zAxis = (SFFloat) %s;\n", bf->name, token);

				GetNextToken(token, " ");
				TranslateToken(token);
				fprintf(f, "\tp->%s.angle = (SFFloat) %s;\n", bf->name, token);
			}
			//SFString
			else if (!strcmp(bf->familly, "SFString")) {
				fprintf(f, "\tp->%s.buffer = malloc(sizeof(char) * %d);\n", bf->name, strlen(bf->def)+1);
				fprintf(f, "\tstrcpy(p->%s.buffer, \"%s\");\n", bf->name, bf->def);
			}
		
			//
			//		MF Fields
			//
			//MFFloat
			else if (!strcmp(bf->familly, "MFFloat")) {
				j = 0;
				CurrentLine = bf->def;
				while (GetNextToken(token, " ,")) j++;
				j+=1;
				fprintf(f, "\tp->%s.vals = malloc(sizeof(SFFloat)*%d);\n", bf->name, j);
				fprintf(f, "\tp->%s.count = %d;\n", bf->name, j);
				j = 0;
				go = 1;
				CurrentLine = bf->def;
				while (go) {
					if (!GetNextToken(token, " ,")) go = 0;
					TranslateToken(token);
					fprintf(f, "\tp->%s.vals[%d] = (SFFloat) %s;\n", bf->name, j, token);
					j+=1;
				}
			}
			//MFVec2f
			else if (!strcmp(bf->familly, "MFVec2f")) {
				j = 0;
				CurrentLine = bf->def;
				while (GetNextToken(token, ",")) j++;
				j+=1;
				fprintf(f, "\tp->%s.vals = malloc(sizeof(SFVec2f)*%d);\n", bf->name, j);
				fprintf(f, "\tp->%s.count = %d;\n", bf->name, j);
				j = 0;
				go = 1;
				CurrentLine = bf->def;
				while (go) {
					if (!GetNextToken(token, ",")) go = 0;
					store = CurrentLine;
					CurrentLine = token;
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].x = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].y = (SFFloat) %s;\n", bf->name, j, tok);
					j+=1;
					CurrentLine = store;
				}
			}
			//MFVec3f
			else if (!strcmp(bf->familly, "MFVec3f")) {
				j = 0;
				CurrentLine = bf->def;
				while (GetNextToken(token, ",")) j++;
				j+=1;
				fprintf(f, "\tp->%s.vals = malloc(sizeof(SFVec3f)*%d);\n", bf->name, j);
				fprintf(f, "\tp->%s.count = %d;\n", bf->name, j);
				j = 0;
				go = 1;
				CurrentLine = bf->def;
				while (go) {
					if (!GetNextToken(token, ",")) go = 0;
					store = CurrentLine;
					CurrentLine = token;
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].x = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].y = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].z = (SFFloat) %s;\n", bf->name, j, tok);
					j+=1;
					CurrentLine = store;
				}
			}
			//MFVec4f
			else if (!strcmp(bf->familly, "MFVec4f")) {
				j = 0;
				CurrentLine = bf->def;
				while (GetNextToken(token, ",")) j++;
				j+=1;
				fprintf(f, "\tp->%s.vals = malloc(sizeof(SFVec4f)*%d);\n", bf->name, j);
				fprintf(f, "\tp->%s.count = %d;\n", bf->name, j);
				j = 0;
				go = 1;
				CurrentLine = bf->def;
				while (go) {
					if (!GetNextToken(token, ",")) go = 0;
					store = CurrentLine;
					CurrentLine = token;
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].x = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].y = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].z = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].q = (SFFloat) %s;\n", bf->name, j, tok);
					j+=1;
					CurrentLine = store;
				}
			}
			//MFInt32
			else if (!strcmp(bf->familly, "MFInt32")) {
				j = 0;
				CurrentLine = bf->def;
				while (GetNextToken(token, ",")) j++;
				j+=1;
				fprintf(f, "\tp->%s.vals = malloc(sizeof(SFInt32)*%d);\n", bf->name, j);
				fprintf(f, "\tp->%s.count = %d;\n", bf->name, j);
				j = 0;
				go = 1;
				CurrentLine = bf->def;
				while (go) {
					if (!GetNextToken(token, ",")) go = 0;
					store = CurrentLine;
					CurrentLine = token;
					GetNextToken(tok, " ");
					fprintf(f, "\tp->%s.vals[%d] = %s;\n", bf->name, j, tok);
					j+=1;
					CurrentLine = store;
				}
			}
			//MFColor
			else if (!strcmp(bf->familly, "MFColor")) {
				j = 0;
				CurrentLine = bf->def;
				while (GetNextToken(token, ",")) j++;
				j+=1;
				fprintf(f, "\tp->%s.vals = malloc(sizeof(SFColor)*%d);\n", bf->name, j);
				fprintf(f, "\tp->%s.count = %d;\n", bf->name, j);
				j = 0;
				go = 1;
				CurrentLine = bf->def;
				while (go) {
					if (!GetNextToken(token, ",")) go = 0;
					store = CurrentLine;
					CurrentLine = token;
					GetNextToken(tok, " ");
					fprintf(f, "\tp->%s.vals[%d].red = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					fprintf(f, "\tp->%s.vals[%d].green = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					fprintf(f, "\tp->%s.vals[%d].blue = (SFFloat) %s;\n", bf->name, j, tok);
					j+=1;
					CurrentLine = store;
				}
			}
			//MFRotation
			else if (!strcmp(bf->familly, "MFRotation")) {
				j = 0;
				CurrentLine = bf->def;
				while (GetNextToken(token, ",")) j++;
				j+=1;
				fprintf(f, "\tp->%s.vals = malloc(sizeof(SFRotation)*%d);\n", bf->name, j);
				fprintf(f, "\tp->%s.count = %d;\n", bf->name, j);
				j = 0;
				go = 1;
				CurrentLine = bf->def;
				while (go) {
					if (!GetNextToken(token, ",")) go = 0;
					store = CurrentLine;
					CurrentLine = token;
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].xAxis = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].yAxis = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].zAxis = (SFFloat) %s;\n", bf->name, j, tok);
					GetNextToken(tok, " ");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d].angle = (SFFloat) %s;\n", bf->name, j, tok);
					j+=1;
					CurrentLine = store;
				}
			}
			//MFString
			else if (!strcmp(bf->familly, "MFString")) {
				j = 0;
				CurrentLine = bf->def;
				while (GetNextToken(token, ",")) j++;
				j+=1;
				fprintf(f, "\tp->%s.vals = malloc(sizeof(SFString)*%d);\n", bf->name, j);
				fprintf(f, "\tp->%s.count = %d;\n", bf->name, j);
				j = 0;
				go = 1;
				CurrentLine = bf->def;
				while (go) {
					if (!GetNextToken(token, ",")) go = 0;
					store = CurrentLine;
					CurrentLine = token;
					GetNextToken(tok, " \"");
					fprintf(f, "\tp->%s.vals[%d] = malloc(sizeof(char) * %d);\n", bf->name, j, strlen(tok)+1);
					fprintf(f, "\tstrcpy(p->%s.vals[%d], \"%s\");\n", bf->name, j, tok);
					j+=1;
					CurrentLine = store;
				}
			}
			//MFTime
			else if (!strcmp(bf->familly, "MFTime")) {
				j = 0;
				CurrentLine = bf->def;
				while (GetNextToken(token, ",")) j++;
				j+=1;
				fprintf(f, "\tp->%s.vals = malloc(sizeof(SFTime)*%d);\n", bf->name, j);
				fprintf(f, "\tp->%s.count = %d;\n", bf->name, j);
				j = 0;
				go = 1;
				CurrentLine = bf->def;
				while (go) {
					if (!GetNextToken(token, ",")) go = 0;
					store = CurrentLine;
					CurrentLine = token;
					GetNextToken(tok, " \"");
					TranslateToken(tok);
					fprintf(f, "\tp->%s.vals[%d] = %s;\n", bf->name, j, tok);
					j+=1;
					CurrentLine = store;
				}
			}

			//other nodes
			else if (!strcmp(bf->familly, "SFImage")) {
				//we currently only have SFImage, with NO texture so do nothing
			}
			//unknown init (for debug)
			else {
				fprintf(f, "UNKNOWN FIELD (%s);\n", bf->familly);

			}
		}
		fprintf(f, "\treturn (SFNode *)p;\n}\n\n");

	}

	fprintf(f, "\n\n\n");

#if USE_SWITCH_FUNC
	//creator function
	fprintf(f, "SFNode *CreateNode(u32 NodeTag)\n{\n\tswitch (NodeTag) {\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (!n->skip_impl) {
			fprintf(f, "\tcase TAG_%s:\n\t\treturn %s_Create();\n", n->name, n->name);
		}
	}
	fprintf(f, "\tdefault:\n\t\treturn NULL;\n\t}\n}\n\n");
#else
	fprintf(f, "typedef SFNode *(*node_const)();\n");
	fprintf(f, "node_const node_constructors[%d] = \n{\n", ChainGetCount(BNodes) + 1);
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (n->skip_impl) {
			fprintf(f, "\tNULL,\n");
		} else {
			fprintf(f, "\t%s_Create,\n", n->name);
		}
	}
	fprintf(f, "\tNULL\n};\n\n");
	fprintf(f, "SFNode *CreateNode(u32 NodeTag)\n{\n\tif (!NodeTag && (node_constructors[NodeTag-1]==NULL) ) return NULL;\n\treturn node_constructors[NodeTag-1]();\n}\n\n");
#endif	

	fprintf(f, "#ifndef NODE_USE_POINTERS\n");

	fprintf(f, "const char *GetNodeName(u32 NodeTag)\n{\n\tswitch (NodeTag) {\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (!n->skip_impl) fprintf(f, "\tcase TAG_%s:\n\t\treturn \"%s\";\n", n->name, n->name);
	}
	fprintf(f, "\tdefault:\n\t\treturn \"Unknown Node\";\n\t}\n}\n\n");

#if USE_SWITCH_FUNC
	fprintf(f, "void Node_Del(SFNode *node)\n{\n\tswitch (node->sgprivate->tag) {\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (!n->skip_impl) {
			fprintf(f, "\tcase TAG_%s:\n\t\t%s_Del(node); return;\n", n->name, n->name);
		}
	}
	fprintf(f, "\tdefault:\n\t\treturn;\n\t}\n}\n\n");
#else
	fprintf(f, "typedef void (*node_dest)(SFNode *node);\n");
	fprintf(f, "node_dest node_destructors[%d] = \n{\n", ChainGetCount(BNodes) + 1);
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (n->skip_impl) {
			fprintf(f, "\tNULL,\n");
		} else {
			fprintf(f, "\t%s_Del,\n", n->name);
		}
	}
	fprintf(f, "\tNULL\n};\n\n");
	fprintf(f, "void Node_Del(SFNode *n)\n{\n\tassert(node_destructors[n->sgprivate->tag-1]!=NULL);\n\tnode_destructors[n->sgprivate->tag-1](n);\n}\n\n");
#endif

#if USE_SWITCH_FUNC
	fprintf(f, "u32 SFNode_GetFieldCount(SFNode *node, u8 code_mode)\n{\n\tswitch (node->sgprivate->tag) {\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (!n->skip_impl) {
			fprintf(f, "\tcase TAG_%s:return %s_get_field_count(node, code_mode);\n", n->name, n->name);
		}
	}
	fprintf(f, "\tdefault:\n\t\treturn 0;\n\t}\n}\n\n");
#else
	fprintf(f, "typedef u32 (*node_getcount)(SFNode *node, u8 code_mode);\n");
	fprintf(f, "node_getcount node_getcounts[%d] = \n{\n", ChainGetCount(BNodes) + 1);
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (n->skip_impl) {
			fprintf(f, "\tNULL,\n");
		} else {
			fprintf(f, "\t%s_get_field_count,\n", n->name);
		}
	}
	fprintf(f, "\tNULL\n};\n\n");
	fprintf(f, "u32 SFNode_GetFieldCount(SFNode *n, u8 code_mode)\n{\n\tassert(node_getcounts[n->sgprivate->tag-1]!=NULL);\n\treturn node_getcounts[n->sgprivate->tag-1](n, code_mode);\n}\n\n");
#endif

#if USE_SWITCH_FUNC
	fprintf(f, "M4Err SFNode_GetFieldIndex(SFNode *node, u32 inField, u8 code_mode, u32 *allIndex)\n{\n\tswitch (node->sgprivate->tag) {\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (!n->skip_impl) {
			fprintf(f, "\tcase TAG_%s: return %s_get_field_index(node, inField, code_mode, allIndex);\n", n->name, n->name);
		}
	}
	fprintf(f, "\tdefault:\n\t\treturn M4BadParam;\n\t}\n}\n\n");
#else
	fprintf(f, "typedef M4Err (*node_getindex)(SFNode *node, u32 inField, u8 code_mode, u32 *allIndex);\n");
	fprintf(f, "node_getindex node_getindexes[%d] = \n{\n", ChainGetCount(BNodes) + 1);
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (n->skip_impl) {
			fprintf(f, "\tNULL,\n");
		} else {
			fprintf(f, "\t%s_get_field_index,\n", n->name);
		}
	}
	fprintf(f, "\tNULL\n};\n\n");
	fprintf(f, "M4Err SFNode_GetFieldIndex(SFNode *n, u32 inField, u8 code_mode, u32 *allIndex)\n{\n\tassert(node_getindexes[n->sgprivate->tag-1]!=NULL);\n\treturn node_getindexes[n->sgprivate->tag-1](n, inField, code_mode, allIndex);\n}\n\n");
#endif

#if USE_SWITCH_FUNC
	fprintf(f, "M4Err SFNode_GetField(SFNode *node, FieldInfo *field)\n{\n\tswitch (node->sgprivate->tag) {\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (!n->skip_impl) {
			fprintf(f, "\tcase TAG_%s: return %s_get_field(node, field);\n", n->name, n->name);
		}
	}
	fprintf(f, "\tdefault:\n\t\treturn M4BadParam;\n\t}\n}\n\n");
#else
	fprintf(f, "typedef M4Err (*node_getfield)(SFNode *node, FieldInfo *field);\n");
	fprintf(f, "node_getfield node_getfields[%d] = \n{\n", ChainGetCount(BNodes) + 1);
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (n->skip_impl) {
			fprintf(f, "\tNULL,\n");
		} else {
			fprintf(f, "\t%s_get_field,\n", n->name);
		}
	}
	fprintf(f, "\tNULL\n};\n\n");
	fprintf(f, "M4Err SFNode_GetField(SFNode *n, FieldInfo *field)\n{\n\tassert(node_getfields[n->sgprivate->tag-1]!=NULL);\n\treturn node_getfields[n->sgprivate->tag-1](n, field);\n}\n\n");
#endif


#if USE_SWITCH_FUNC
	fprintf(f, "Bool SFNode_GetAQInfo(SFNode *node, u32 FieldIndex, u8 *QType, u8 *AType, Float *b_min, Float *b_max, u32 *QT13_bits)\n{\n\tswitch (node->sgprivate->tag) {\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (!n->skip_impl) {
			fprintf(f, "\tcase TAG_%s: return %s_get_aq_info(node, FieldIndex, QType, AType, b_min, b_max, QT13_bits);\n", n->name, n->name);
		}
	}
	fprintf(f, "\tdefault:\n\t\treturn 0;\n\t}\n}\n\n");
#else
	fprintf(f, "typedef Bool (*node_getaq)(SFNode *node, u32 FieldIndex, u8 *QType, u8 *AType, Float *b_min, Float *b_max, u32 *QT13_bits);\n");
	fprintf(f, "node_getaq node_getaqps[%d] = \n{\n", ChainGetCount(BNodes) + 1);
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (n->skip_impl) {
			fprintf(f, "\tNULL,\n");
		} else {
			fprintf(f, "\t%s_get_aq_info,\n", n->name);
		}
	}
	fprintf(f, "\tNULL\n};\n\n");
	fprintf(f, "Bool SFNode_GetAQInfo(SFNode *n, u32 FieldIndex, u8 *QType, u8 *AType, Float *b_min, Float *b_max, u32 *QT13_bits)\n{\n\tassert(node_getaqps[n->sgprivate->tag-1]!=NULL);\n\treturn node_getaqps[n->sgprivate->tag-1](n, FieldIndex, QType, AType, b_min, b_max, QT13_bits);\n}\n\n");
#endif


	fprintf(f, "u32 Node_GetChildNDT(SFNode *node)\n{\n\tswitch (node->sgprivate->tag) {\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (!n->skip_impl && strlen(n->Child_NDT_Name) ) {
			fprintf(f, "\tcase TAG_%s: return %s;\n", n->name, n->Child_NDT_Name);
		}
	}
	fprintf(f, "\tdefault:\n\t\treturn 0;\n\t}\n}\n\n");

	fprintf(f, "\n\n#endif\n");



	EndFile(f, "", 1);
}

void ParseTemplateFile(FILE *nodes, Chain *BNodes, Chain *NDTs, u32 version)
{
	char sLine[2000];
	char token[100];
	char *p;
	BNode *n;
	BField *f;
	u32 j, i, k;

	//get lines one by one
	n = NULL;
	while (!feof(nodes)) {
		fgets(sLine, 2000, nodes);
		//skip comment and empty lines
		if (sLine[0] == '#') continue;
		if (sLine[0] == '\n') continue;

		CurrentLine = sLine;
		
		//parse the line till end of line
		while (GetNextToken(token, " \t")) {

			//this is a new node
			if (!strcmp(token, "PROTO") ) {
				n = BlankNode();
				n->version = version;
				ChainAddEntry(BNodes, n);

				//get its name
				GetNextToken(n->name, " \t[");
				if (!strcmp(n->name, "TimeSensor")) {
					n = n;
				}

				//extract the NDTs
				GetNextToken(token, "\t[ %#=");
				if (strcmp(token, "NDT")) {
					printf("Corrupted template file\n");
					return;
				}
				while (1) {
					GetNextToken(token, "=, \t");
					//done with NDTs
					if (token[0] == '%') break;

					//update the NDT list
					CheckInTable(token, NDTs);
					p = malloc(strlen(token)+1);
					strcpy(p, token);
					ChainAddEntry(n->NDT, p);
				}

				//extract the coding type
				if (strcmp(token, "%COD")) {
					printf("Corrupted template file\n");
					return;
				} else {
					GetNextToken(token, "= ");
					if (token[0] == 'N') {
						n->codingType = 0;
					} else {
						n->codingType = 1;
					}
				}
			}
			//this is NOT a field
			else if (token[0] == ']' || token[0] == '{' || token[0] == '}' ) {
				break;
			}
			//parse a field
			else {
				if (!n) {
					printf("Corrupted template file\n");
					return;
				}
				f = BlankField();
				ChainAddEntry(n->Fields, f);
				
				//get the field type
				strcpy(f->type, token);
				GetNextToken(f->familly, " \t");
				GetNextToken(f->name, " \t");
				//fix for our own code :(
				if (!strcmp(f->name, "tag")) strcpy(f->name, "_tag");

				//has default
				skip_sep(" \t");
				if (GetNextToken(token, "#\t")) {
					j=0;
					while (token[j] == ' ') j+=1;
					if (token[j] == '[') j+=1;
					if (token[j] == '"') j+=1;

					if (token[j] != '"' && token[j] != ']') {
						strcpy(f->def, token+j);
						j=1;
						while (j) {
							switch (f->def[strlen(f->def)-1]) {
							case ' ':
							case '"':
							case ']':
								f->def[strlen(f->def)-1] = 0;
								break;
							default:
								j=0;
								break;
							}
						}
					} else {
						strcpy(f->def, "");
					}
					if (!strcmp(f->familly, "SFFloat")) {
						if (!strcmp(f->def, "+I") || !strcmp(f->def, "I")) {
							strcpy(f->def, "M4_MAX_FLOAT");
						} else if (!strcmp(f->def, "-I")) {
							strcpy(f->def, "M4_MIN_FLOAT");
						}
					} else if (!strcmp(f->familly, "SFTime")) {
						if (!strcmp(f->def, "+I") || !strcmp(f->def, "I")) {
							strcpy(f->def, "M4_MAX_FLOAT");
						} else if (!strcmp(f->def, "-I")) {
							strcpy(f->def, "M4_MIN_FLOAT");
						}
					} else if (!strcmp(f->familly, "SFInt32")) {
						if (!strcmp(f->def, "+I") || !strcmp(f->def, "I")) {
							strcpy(f->def, "2 << 31");
						} else if (!strcmp(f->def, "-I")) {
							strcpy(f->def, "- (2 << 31)");
						}
					}
				}
				//has other
				while (GetNextToken(token, " \t#%=")) {
					switch (token[0]) {
					//bounds
					case 'b':
						f->hasBounds = 1;
						GetNextToken(f->b_min, "[(,");
						GetNextToken(f->b_max, ")]");
						break;
					case 'q':
						f->hasQuant = 1;
						GetNextToken(f->quant_type, " \t");
						if (!strcmp(f->quant_type, "13"))
							GetNextToken(f->qt13_bits, " \t");
						break;
					case 'a':
						f->hasAnim = 1;
						GetNextToken(token, " \t");
						f->AnimType = atoi(token);
						break;
					default:
						break;
					}
				}
			}
		}
	}


	for (k=0; k<ChainGetCount(BNodes); k++) {
		n = ChainGetEntry(BNodes, k);

		for (i=0; i<ChainGetCount(n->Fields); i++) {
			f = ChainGetEntry(n->Fields, i);
			//nothing on events
			if (!strcmp(f->type, "eventIn")) continue;
			if (!strcmp(f->type, "eventOut")) continue;
			if (!strcmp(f->def, "")) continue;
			if (strstr(f->familly, "Node")) continue;
			n->hasDefault = 1;
		}
	}
}


void WriteNodeDump(FILE *f, BNode *n)
{
	BField *bf;
	u32 i;

	fprintf(f, "static const char *%s_FieldName[] = {\n", n->name);
	for (i=0; i<ChainGetCount(n->Fields); i++) {
		bf = ChainGetEntry(n->Fields, i);
		if (!i) {
			fprintf(f, " \"%s\"", bf->name);
		} else {
			fprintf(f, ", \"%s\"", bf->name);
		}
	}
	fprintf(f, "\n};\n\n");
}

void parse_profile(Chain *nodes, FILE *prof)
{
	char sLine[2000];
	BNode *n;
	Bool found;
	u32 i;

	while (!feof(prof)) {
		fgets(sLine, 2000, prof);
		//skip comment and empty lines
		if (sLine[0] == '#') continue;
		if (sLine[0] == '\n') continue;
		if (strstr(sLine, "Proximity")) 
			found = 0;
		found = 1;
		while (found) {
			switch (sLine[strlen(sLine)-1]) {
			case '\n':
			case '\r':
			case ' ':
				sLine[strlen(sLine)-1] = 0;
				break;
			default:
				found = 0;
				break;
			}
		}

//		if (0 && !stricmp(sLine, "Appearance") || !stricmp(sLine, "Shape") || !stricmp(sLine, "Sound2D") ) {
		if (0) {
			printf("Warning: cannot disable node %s (required in all BIFS profiles)\n", sLine);
		} else {
			found = 0;
			for (i=0; i<ChainGetCount(nodes); i++) {
				n = ChainGetEntry(nodes, i);
				if (!stricmp(n->name, sLine)) {
					n->skip_impl = 1;
					found = 1;
					break;
				}
			}
			if (!found) printf("cannot disable %s: node not found\n", sLine);
		}
	}
}

int main (int argc, char **argv)
{
	FILE *nodes, *ndt_dec_c, *ndt_enc_c, *ndt_h, *fskip;
	Chain *BNodes, *NDTs;
	u32 i, j, nbVersion;
	BNode *n;
	BField *bf;

	if (argc < 2) {
		PrintUsage();
		return 0;
	}

	BNodes = NewChain();
	NDTs = NewChain();



	fskip = NULL;
	i=1;
	if (argv[i][0]=='-') {
		fskip = fopen(argv[i+1], "rt");
		if (!fskip) {
			printf("file %s not found\n", argv[i+1]);
			return 0;
		}
		i+=2;
	}
	nbVersion = 1;
	for (; i<argc; i++) {
		nodes = fopen(argv[i], "rt");
		//all nodes are in the same list but we keep version info
		ParseTemplateFile(nodes, BNodes, NDTs, nbVersion);
		

		//special case for viewport: it is present in V1 but empty
		if (nbVersion==1) CheckInTable("SFViewportNode", NDTs);
		nbVersion++;
		fclose(nodes);
	}
	nbVersion--;
	printf("BIFS tables parsed: %d versions\n", nbVersion);
	
	if (fskip) {
		parse_profile(BNodes, fskip);
		fclose(fskip);
	}

	//write the nodes def
	WriteNodesFile(BNodes, NDTs, nbVersion);
	//write all nodes init stuff
	WriteNodeCode(BNodes);

	//write all NDTs
	ndt_h = BeginFile("NDT", 0);
	ndt_dec_c = BeginFile("NDT_dec", 1);
	ndt_enc_c = BeginFile("NDT_enc", 1);

	fprintf(ndt_h, "#include <gpac/m4_nodes.h>\n\n");
	fprintf(ndt_dec_c, "\n\n#include <gpac/intern/m4_node_tables.h>\n");
	fprintf(ndt_enc_c, "\n\n#include <gpac/intern/m4_node_tables.h>\n");

	
	//prepare the encoding file
	fprintf(ndt_h, "\n\nu32 ALL_GetNodeType(const u32 *table, const u32 count, u32 NodeTag, u32 Version);\n\n");
	fprintf(ndt_enc_c, "\n\nu32 ALL_GetNodeType(const u32 *table, const u32 count, u32 NodeTag, u32 Version)\n{\n\tu32 i = 0;");
	fprintf(ndt_enc_c, "\n\twhile (i<count) {\n\t\tif (table[i] == NodeTag) goto found;\n\t\ti++;\n\t}\n\treturn 0;\nfound:\n\tif (Version == 2) return i+2;\n\treturn i+1;\n}\n\n");

	//write the NDT
	for (i=0; i<nbVersion; i++) {
		//write header
		WriteNDT_H(ndt_h, BNodes, NDTs, i+1);
		//write decoding code
		WriteNDT_Dec(ndt_dec_c, BNodes, NDTs, i+1);
		//write encoding code
		WriteNDT_Enc(ndt_enc_c, BNodes, NDTs, i+1);
	}



	fprintf(ndt_dec_c, "\n\nu32 NDT_GetNodeTag(u32 NDT_Tag, u32 NodeType, u32 Version)\n{\n\tswitch (Version) {\n");
	for (i=0; i<nbVersion; i++) {
		fprintf(ndt_dec_c, "\tcase BIFS_V%d:\n\t\treturn NDT_V%d_GetNodeTag(NDT_Tag, NodeType);\n", i+1, i+1);
	}
	fprintf(ndt_dec_c, "\tdefault:\n\t\treturn 0;\n\t}\n}");

	fprintf(ndt_dec_c, "\n\nu32 NDT_GetNumBits(u32 NDT_Tag, u32 Version)\n{\n\tswitch (Version) {\n");
	for (i=0; i<nbVersion; i++) {
		fprintf(ndt_dec_c, "\tcase BIFS_V%d:\n\t\treturn NDT_V%d_GetNumBits(NDT_Tag);\n", i+1, i+1);
	}
	fprintf(ndt_dec_c, "\tdefault:\n\t\treturn 0;\n\t}\n}");

	fprintf(ndt_enc_c, "\n\nu32 NDT_GetNodeType(u32 NDT_Tag, u32 NodeTag, u32 Version)\n{\n\tswitch (Version) {\n");
	for (i=0; i<nbVersion; i++) {
		fprintf(ndt_enc_c, "\tcase BIFS_V%d:\n\t\treturn NDT_V%d_GetNodeType(NDT_Tag, NodeTag);\n", i+1, i+1);
	}
	fprintf(ndt_enc_c, "\tdefault:\n\t\treturn 0;\n\t}\n}");

	fprintf(ndt_enc_c, "u32 Node_GetTagByName(const char *node_name)\n{\n\tif(!node_name) return 0;\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (n->skip_impl) continue;
		fprintf(ndt_enc_c, "\tif (!stricmp(node_name, \"%s\")) return TAG_%s;\n", n->name, n->name);
	}
	fprintf(ndt_enc_c, "\treturn 0;\n}\n\n");

	

	fprintf(ndt_h, "\nu32 NDT_GetChildTable(u32 NodeTag);\n");
	fprintf(ndt_h, "\n\n");
	
	//NDT checking
	fprintf(ndt_dec_c, "u32 GetChildrenNDT(SFNode *node)\n{\n\tif (!node) return 0;\n\tswitch (Node_GetTag(node)) {\n");
	for (i=0; i<ChainGetCount(BNodes); i++) {
		n = ChainGetEntry(BNodes, i);
		if (n->skip_impl) continue;
		for (j=0; j<ChainGetCount(n->Fields); j++) {
			bf = ChainGetEntry(n->Fields, j);
			if (!strcmp(bf->name, "children")) {
				fprintf(ndt_dec_c, "\tcase TAG_%s:\n\t\treturn NDT_SF%s;\n", n->name, bf->familly+2);
				break;
			}
		}
	}
	fprintf(ndt_dec_c, "\tdefault:\n\t\treturn 0;\n\t}\n}\n\n");

	EndFile(ndt_h, "NDT", 0);
	EndFile(ndt_dec_c, "", 1);
	EndFile(ndt_enc_c, "", 1);
	
	//free NDTs
	while (ChainGetCount(NDTs)) {
		char *tmp = ChainGetEntry(NDTs, 0);
		free(tmp);
		ChainDeleteEntry(NDTs, 0);
	}
	DeleteChain(NDTs);
	//free nodes
	while (ChainGetCount(BNodes)) {
		n = ChainGetEntry(BNodes, 0);
		ChainDeleteEntry(BNodes, 0);
		while (ChainGetCount(n->NDT)) {
			char *tmp = ChainGetEntry(n->NDT, 0);
			free(tmp);
			ChainDeleteEntry(n->NDT, 0);
		}
		DeleteChain(n->NDT);
		while (ChainGetCount(n->Fields)) {
			bf = ChainGetEntry(n->Fields, 0);
			free(bf);
			ChainDeleteEntry(n->Fields, 0);
		}
		DeleteChain(n->Fields);
		free(n);
	}
	DeleteChain(BNodes);

	return 0;
}

