/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / command-line mp4 toolbox
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */


#include <gpac/m4_author.h>
#include <gpac/m4_bifs.h>
#include <gpac/m4_scenegraph.h>
#include <gpac/intern/avilib.h>

M4Err dump_message(M4TrackDumper *dumper, M4Err e, char *format, ...)
{
	va_list args;
	va_start(args, format);
	if (dumper->dump_message) {
		char szMsg[1024];
		vsprintf(szMsg, format, args);
		dumper->dump_message(dumper, e, szMsg);
	} else {
		vfprintf(stderr,format,args);
		if (e) fprintf(stderr, " Error: %s", M4ErrToString(e));
		fprintf(stderr, "\n");
	}
	va_end(args);
	return e;
}

void dump_progress(M4TrackDumper *dumper, u32 cur_samp, u32 count)
{
	if (dumper->dump_progress) {
		dumper->dump_progress(dumper, cur_samp, count);
	} else {
		fprintf(stdout, "Extracting sample %d / %d (%.2f)\r", cur_samp, count, ((Float)cur_samp)*100 / count);
		if (cur_samp==count) fprintf(stderr, "\n");
	}
}

M4Err MP4T_DumpTrackNative(M4TrackDumper *dumper)
{
	ESDescriptor *esd;
	char szName[1000];
	FILE *out;
	u32 track, i, di, count;
	Bool dump_dsi;

	track = M4_GetTrackByID(dumper->file, dumper->trackID);
	esd = M4_GetStreamDescriptor(dumper->file, track, 1);
	if (!esd) 
		return dump_message(dumper, M4NonCompliantBitStream, "Invalid MPEG-4 stream in track ID %d\n", dumper->trackID);

	dump_dsi = 0;
	strcpy(szName, dumper->out_name);
	switch (esd->decoderConfig->streamType) {
	case M4ST_VISUAL:
		switch (esd->decoderConfig->objectTypeIndication) {
		case 0x20:
			dump_dsi = 1;
			strcat(szName, ".cmp");
			dump_message(dumper, M4OK, "Extracting MPEG-4 Visual stream to cmp");
			break;
		case 0x6C:
			strcat(szName, ".jpg");
			dump_message(dumper, M4OK, "Extracting JPEG image");
			break;
		case 0x6D:
			strcat(szName, ".png");
			dump_message(dumper, M4OK, "Extracting PNG image");
			break;
		default:
			OD_DeleteDescriptor((Descriptor **) &esd);
			return dump_message(dumper, M4NotSupported, "Unknown media in track ID %d - use NHNT instead", dumper->trackID);
		}
		break;
	case M4ST_AUDIO:
		switch (esd->decoderConfig->objectTypeIndication) {
		case 0x66:
		case 0x67:
		case 0x68:
			dump_dsi = 1;
			strcat(szName, ".aac");
			dump_message(dumper, M4OK, "Extracting MPEG-2 AAC");
			break;
		case 0x40:
			dump_dsi = 1;
			strcat(szName, ".aac");
			dump_message(dumper, M4OK, "Extracting MPEG-4 AAC");
			break;
		case 0x69:
		case 0x6B:
			strcat(szName, ".mp3");
			dump_message(dumper, M4OK, "Extracting MPEG-1/2 Audio (MP3)");
			break;
		default:
			OD_DeleteDescriptor((Descriptor **) &esd);
			return dump_message(dumper, M4NotSupported, "Unknown audio in track ID %d - use NHNT", dumper->trackID);
		}
		break;
	default:
		OD_DeleteDescriptor((Descriptor **) &esd);
		return dump_message(dumper, M4NotSupported, "Cannot extract systems track ID %d in raw format - use NHNT", dumper->trackID);
	}

	out = fopen(szName, "wb");
	if (!out) {
		OD_DeleteDescriptor((Descriptor **) &esd);
		return dump_message(dumper, M4IOErr, "Error opening %s for writing - check disk access & permissions", szName);
	}

	if (dump_dsi && esd->decoderConfig->decoderSpecificInfo  && esd->decoderConfig->decoderSpecificInfo->data) {
		fwrite(esd->decoderConfig->decoderSpecificInfo->data, esd->decoderConfig->decoderSpecificInfo->dataLength, 1, out);
	}
	OD_DeleteDescriptor((Descriptor **) &esd);

	count = M4_GetSampleCount(dumper->file, track);
	for (i=0; i<count; i++) {
		M4Sample *samp = M4_GetSample(dumper->file, track, i+1, &di);
		if (!samp) break;
		fwrite(samp->data, samp->dataLength, 1, out);
		M4_DeleteSample(&samp);
		dump_progress(dumper, i+1, count);
		if (dumper->do_abort) break;
	}
	fclose(out);
	return dump_message(dumper, M4OK, "Extracting done");
}



M4Err MP4T_DumpTrackNHNT(M4TrackDumper *dumper)
{
	ESDescriptor *esd;
	char szName[1000];
	FILE *out_med, *out_inf;
	FILE *out_nhnt;
	BitStream *bs;
	u32 track, i, di, count, pos;

	track = M4_GetTrackByID(dumper->file, dumper->trackID);
	esd = M4_GetStreamDescriptor(dumper->file, track, 1);
	if (!esd) return dump_message(dumper, M4NonCompliantBitStream, "Invalid MPEG-4 stream in track ID %d", dumper->trackID);

	sprintf(szName, "%s.media", dumper->out_name);
	out_med = fopen(szName, "wb");
	if (!out_med) {
		OD_DeleteDescriptor((Descriptor **) &esd);
		return dump_message(dumper, M4IOErr, "Error opening %s for writing - check disk access & permissions", szName);
	}

	sprintf(szName, "%s.nhnt", dumper->out_name);
	out_nhnt = fopen(szName, "wb");
	if (!out_nhnt) {
		fclose(out_med);
		OD_DeleteDescriptor((Descriptor **) &esd);
		return dump_message(dumper, M4IOErr, "Error opening %s for writing - check disk access & permissions", szName);
	}


	bs = NewBitStreamFromFile(out_nhnt, BS_FILE_WRITE_ONLY);

	if (esd->decoderConfig->decoderSpecificInfo  && esd->decoderConfig->decoderSpecificInfo->data) {
		sprintf(szName, "%s.info", dumper->out_name);
		out_inf = fopen(szName, "wb");
		fwrite(esd->decoderConfig->decoderSpecificInfo->data, esd->decoderConfig->decoderSpecificInfo->dataLength, 1, out_inf);
		fclose(out_inf);
	}

	/*write header*/
	/*'NHnt' format*/
	BS_WriteInt(bs, 'N', 8);
	BS_WriteInt(bs, 'H', 8);
	BS_WriteInt(bs, 'n', 8);
	BS_WriteInt(bs, 't', 8);
	/*version 1*/
	BS_WriteInt(bs, 0, 8);
	/*streamType*/
	BS_WriteInt(bs, esd->decoderConfig->streamType, 8);
	/*OTI*/
	BS_WriteInt(bs, esd->decoderConfig->objectTypeIndication, 8);
	/*reserved*/
	BS_WriteInt(bs, 0, 16);
	/*bufferDB*/
	BS_WriteInt(bs, esd->decoderConfig->bufferSizeDB, 24);
	/*avg BitRate*/
	BS_WriteInt(bs, esd->decoderConfig->avgBitrate, 32);
	/*max bitrate*/
	BS_WriteInt(bs, esd->decoderConfig->maxBitrate , 32);
	/*timescale*/
	BS_WriteInt(bs, esd->slConfig->timestampResolution, 32);

	OD_DeleteDescriptor((Descriptor **) &esd);

	pos = 0;
	count = M4_GetSampleCount(dumper->file, track);
	for (i=0; i<count; i++) {
		M4Sample *samp = M4_GetSample(dumper->file, track, i+1, &di);
		if (!samp) break;
		fwrite(samp->data, samp->dataLength, 1, out_med);
		
		/*dump nhnt info*/
		BS_WriteInt(bs, samp->dataLength, 24);
		BS_WriteInt(bs, samp->IsRAP, 1);
		/*AU start & end flag always true*/
		BS_WriteInt(bs, 1, 1);
		BS_WriteInt(bs, 1, 1);
		/*reserved*/
		BS_WriteInt(bs, 0, 3);
		/*type (ignored)*/
		BS_WriteInt(bs, 0, 2);
		BS_WriteInt(bs, pos, 32);
		BS_WriteInt(bs, samp->DTS + samp->CTS_Offset, 32);
		BS_WriteInt(bs, samp->DTS, 32);

		pos += samp->dataLength;
		M4_DeleteSample(&samp);
		dump_progress(dumper, i+1, count);
		if (dumper->do_abort) break;
	}
	fclose(out_med);
	DeleteBitStream(bs);
	fclose(out_nhnt);
	return dump_message(dumper, M4OK, "Extracting done");
}


M4Err MP4T_DumpTrackMP4(M4TrackDumper *dumper)
{
	M4File *outfile;
	M4Err e;
	ESDescriptor *esd;
	char szName[1000];
	u32 track;
	u8 mode;

	track = M4_GetTrackByID(dumper->file, dumper->trackID);

	esd = M4_GetStreamDescriptor(dumper->file, track, 1);
	if (esd && (esd->decoderConfig->streamType == M4ST_OD)) {
		if (esd) OD_DeleteDescriptor((Descriptor **) &esd);
		return dump_message(dumper, M4BadParam, "Cannot extract OD track, result is  meaningless");
	}
	if (esd) OD_DeleteDescriptor((Descriptor **) &esd);

	sprintf(szName, "%s.mp4", dumper->out_name);

	mode = M4_WRITE_EDIT;
	if (dumper->merge_tracks) {
		FILE *t = fopen(szName, "rb");
		if (t) {
			mode = M4_OPEN_EDIT;
			fclose(t);
		}
	}
	outfile = M4_MovieOpen(szName, mode);

	if (mode == M4_WRITE_EDIT) {
		M4_SetMoviePLIndication(outfile, M4_PL_AUDIO, 0xFF);
		M4_SetMoviePLIndication(outfile, M4_PL_VISUAL, 0xFF);
		M4_SetMoviePLIndication(outfile, M4_PL_GRAPHICS, 0xFF);
		M4_SetMoviePLIndication(outfile, M4_PL_SCENE, 0xFF);
		M4_SetMoviePLIndication(outfile, M4_PL_OD, 0xFF);
		M4_SetMoviePLIndication(outfile, M4_PL_MPEGJ, 0xFF);
	}

	e = MP4T_CopyTrack(dumper->file, track, outfile, 1, 1);
	M4_MovieClose(outfile);

	return dump_message(dumper, e, "Extracting done");
}

M4Err MP4T_DumpTrackAVI(M4TrackDumper *dumper)
{
	ESDescriptor *esd;
	M4Sample *samp;
	char szName[1000];
	avi_t *avi_out;
	char dumdata[1];
	u32 track, i, di, count, w, h, frame_d;
	M4VDecoderSpecificInfo dsi;
	Double FPS;

	track = M4_GetTrackByID(dumper->file, dumper->trackID);
	esd = M4_GetStreamDescriptor(dumper->file, track, 1);
	if (!esd) return dump_message(dumper, M4NonCompliantBitStream, "Invalid MPEG-4 stream in track ID %d", dumper->trackID);

	if ((esd->decoderConfig->streamType!=M4ST_VISUAL) || (esd->decoderConfig->objectTypeIndication!=0x20)) {
		OD_DeleteDescriptor((Descriptor**)&esd);
		dump_message(dumper, M4NonCompliantBitStream, "Track ID %d is not MPEG-4 Visual - cannot extract to AVI", dumper->trackID);
	}
	if (!esd->decoderConfig->decoderSpecificInfo) {
		OD_DeleteDescriptor((Descriptor**)&esd);
		dump_message(dumper, M4NonCompliantBitStream, "Missing decoder config for track ID %d", dumper->trackID);
	}

	sprintf(szName, "%s.avi", dumper->out_name);
	avi_out = AVI_open_output_file(szName);
	if (!avi_out) {
		OD_DeleteDescriptor((Descriptor **)&esd);
		return dump_message(dumper, M4IOErr, "Error opening %s for writing - check disk access & permissions", szName);
	}
	/*ignore visual size info, get it from dsi*/
	M4V_GetConfig(esd->decoderConfig->decoderSpecificInfo->data, esd->decoderConfig->decoderSpecificInfo->dataLength, &dsi);
	w = dsi.width;
	h = dsi.height;

	/*compute FPS - note we assume constant frame rate without droped frames...*/
	count = M4_GetSampleCount(dumper->file, track);
	FPS = M4_GetMediaTimeScale(dumper->file, track);
	FPS *= (count-1);
	samp = M4_GetSample(dumper->file, track, count, &di);
	FPS /= samp->DTS;
	M4_DeleteSample(&samp);

	frame_d = 0;
	if (M4_TrackHasTimeOffsets(dumper->file, track)) {
		u32 DTS, max_CTSO;
		u64 dataoff;
		DTS = max_CTSO = 0;
		for (i=0; i<count; i++) {
			samp = M4_GetSampleInfo(dumper->file, track, i+1, &di, &dataoff);
			if (!samp) break;
			if (samp->CTS_Offset>max_CTSO) max_CTSO = samp->CTS_Offset;
			DTS = samp->DTS;
			M4_DeleteSample(&samp);
		}
		DTS /= (count-1);
		frame_d = max_CTSO / DTS;
		frame_d -= 1;
		/*dummy delay frame for xvid unpacked bitstreams*/
		dumdata[0] = 127;
	}

	AVI_set_video(avi_out, w, h, FPS, "XVID");
	dump_message(dumper, M4OK, "Creating AVI file %d x %d @ %.2f FPS - 4CC \"XVID\"", w, h, FPS);
	if (frame_d) dump_message(dumper, M4OK, "B-Frames detected - using unpacked bitstream with max B-VOP delta %d", frame_d);

	for (i=0; i<count; i++) {
		samp = M4_GetSample(dumper->file, track, i+1, &di);
		if (!samp) break;

		if (!i) {
			char *data = malloc(sizeof(char) * (samp->dataLength + esd->decoderConfig->decoderSpecificInfo->dataLength));
			memcpy(data, esd->decoderConfig->decoderSpecificInfo->data, esd->decoderConfig->decoderSpecificInfo->dataLength);
			memcpy(data + esd->decoderConfig->decoderSpecificInfo->dataLength, samp->data, samp->dataLength);
			AVI_write_frame(avi_out, data, samp->dataLength + esd->decoderConfig->decoderSpecificInfo->dataLength, 1);
			free(data);
		} else {
			AVI_write_frame(avi_out, samp->data, samp->dataLength, samp->IsRAP);
		}
		M4_DeleteSample(&samp);
		while (frame_d) {
			AVI_write_frame(avi_out, dumdata, 1, 0);
			frame_d--;
		}
		dump_progress(dumper, i+1, count);
		if (dumper->do_abort) break;
	}

	OD_DeleteDescriptor((Descriptor **) &esd);
	AVI_close(avi_out);
	return dump_message(dumper, M4OK, "Extracting done");
}


M4Err MP4T_DumpTrack(M4TrackDumper *dumper)
{
	if (!dumper) return M4BadParam;
	switch (dumper->dump_type) {
	case DUMP_TRACK_NATIVE:
		return MP4T_DumpTrackNative(dumper);
	case DUMP_TRACK_NHNT:
		return MP4T_DumpTrackNHNT(dumper);
	case DUMP_TRACK_AVI:
		return MP4T_DumpTrackAVI(dumper);
	case DUMP_TRACK_MP4:
		return MP4T_DumpTrackMP4(dumper);
	default:
		return M4BadParam;
	}
}

