/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "stacks3d.h"

/*FIXME - no more gradients*/
#if 0

void GradientGetMatrix(SFNode *transform, M4Matrix2D *mat)
{
	MX2D_init(*mat);
	if (transform) {
		switch (Node_GetTag(transform) ) {
#ifdef M4_DEF_Transform2D
		case TAG_Transform2D:
		{
			B_Transform2D *tr = (B_Transform2D *)transform;
			MX2D_add_scale_at(mat, 0, 0, tr->scale.x, tr->scale.y, tr->scaleOrientation);
			MX2D_add_rotation(mat, tr->center.x, tr->center.y, tr->rotationAngle);
			MX2D_add_translation(mat, tr->translation.x, tr->translation.y);
		}
			break;
#endif
		
#ifdef M4_DEF_TransformMatrix2D
		case TAG_TransformMatrix2D:
			TM2D_GetMatrix(transform, mat);
			break;
#endif
		default:
			break;
		}
	}
}


#ifdef M4_DEF_LinearGradient

/*
		linear gradient
*/

static void DestroyLinearGradient(SFNode *node)
{
	GradientStack *st = (GradientStack *) Node_GetPrivate(node);
	texture_destroy(&st->txh);
	free(st);
}

static Bool UpdateLinearGradient(TextureHandler *txh)
{
	u32 i;
	B_LinearGradient *lg = (B_LinearGradient *) txh->owner;
	GradientStack *st = (GradientStack *) Node_GetPrivate(txh->owner);

	if (!txh->texture) {
		Graphics2DDriver *graph_hw = txh->compositor->video_render->graph_hw;
		if (!graph_hw) return 0;
		txh->texture = graph_hw->new_stencil(graph_hw, M4StencilLinearGradient);
	}

	if (!Node_GetDirty(txh->owner)) return;
	Node_ClearDirty(txh->owner);

	st->txh.transparent = 0;
	for (i=0; i<lg->opacity.count; i++) {
		if (lg->opacity.vals[i] != 1.0) {
			st->txh.transparent = 1;
			break;
		}
	}
	return 1;
}

static void LG_ComputeMatrix(TextureHandler *txh, M4Rect *bounds, M4Matrix *mat)
{
	SFVec2f start, end;
	u32 i, *cols;
	Float a;
	Bool const_a;
	Graphics2DDriver *dr;
	B_LinearGradient *lg = (B_LinearGradient *) txh->owner;

	if (lg->key.count<2) return;
	if (lg->key.count != lg->keyValue.count) return;

	start = lg->startPoint;
	end = lg->endPoint;

	/*create gradient brush if needed*/
	dr = txh->compositor->video_render->graph_hw;
	if (!txh->texture) return;


	GradientGetMatrix((SFNode *) lg->transform, mat);
	
	/*move line to object space*/
	start.x *= bounds->width;
	end.x *= bounds->width;
	start.y *= bounds->height;
	end.y *= bounds->height;

	/*move transform to object space*/
	mat->m[2] *= bounds->width;
	mat->m[5] *= bounds->height;
	mat->m[1] *= bounds->width / bounds->height;
	mat->m[3] *= bounds->height / bounds->width;

	/*translate to the center of the bounds*/
	mat_add_translation(mat, bounds->x, bounds->y - bounds->height);


	dr->set_linear_gradient(txh->texture, start.x, start.y, end.x, end.y, 0xFFFF0000, 0xFFFF00FF);
	
	const_a = (lg->opacity.count == 1) ? 1 : 0;
	cols = malloc(sizeof(u32) * lg->key.count);
	for (i=0; i<lg->key.count; i++) {
		a = (const_a ? lg->opacity.vals[0] : lg->opacity.vals[i]);
		cols[i] = MAKE_ARGB_FLOAT(a, lg->keyValue.vals[i].red, lg->keyValue.vals[i].green, lg->keyValue.vals[i].blue);
	}
	dr->set_gradient_interpolation(txh->texture, lg->key.vals, cols, lg->key.count);
	free(cols);
	dr->set_gradient_mode(txh->texture, lg->spreadMethod);

}

void R3D_InitLinearGradient(Render3D *sr, SFNode *node)
{
	GradientStack *st = malloc(sizeof(GradientStack));
	memset(st, 0, sizeof(GradientStack));

	texture_setup(&st->txh, sr, node);
	st->txh.update_texture_fcnt = UpdateLinearGradient;

	st->txh.compute_gradient_matrix = LG_ComputeMatrix;
	st->txh.is_gradient = 1;

	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyLinearGradient);
}

TextureHandler *r3d_lg_get_texture(SFNode *node)
{
	GradientStack *st = (GradientStack*) Node_GetPrivate(node);
	return &st->txh;
}

#endif

#ifdef M4_DEF_RadialGradient

/*
		radial gradient
*/


static void DestroyRadialGradient(SFNode *node)
{
	GradientStack *st = (GradientStack *) Node_GetPrivate(node);
	texture_destroy(&st->txh);
	free(st);
}

static Bool UpdateRadialGradient(TextureHandler *txh)
{
	u32 i;
	B_RadialGradient *rg = (B_RadialGradient*) txh->owner;
	GradientStack *st = (GradientStack *) Node_GetPrivate(txh->owner);

	if (!txh->texture) {
		Graphics2DDriver *graph_hw = txh->compositor->video_render->graph_hw;
		if (!graph_hw) return 0;
		txh->texture = graph_hw->new_stencil(graph_hw, M4StencilRadialGradient);
	}

	if (!Node_GetDirty(txh->owner)) return;
	Node_ClearDirty(txh->owner);

	st->txh.transparent = 0;
	for (i=0; i<rg->opacity.count; i++) {
		if (rg->opacity.vals[i] != 1.0) {
			st->txh.transparent = 1;
			break;
		}
	}
	return 1;
}

static void RG_ComputeMatrix(TextureHandler *txh, M4Rect *bounds, M4Matrix *mat)
{
	SFVec2f center, focal;
	u32 i, *cols;
	Float a;
	Bool const_a;
	Graphics2DDriver *dr;
	B_RadialGradient *rg = (B_RadialGradient *) txh->owner;

	if (rg->key.count<2) return;
	if (rg->key.count != rg->keyValue.count) return;

	/*create gradient brush if needed*/
	dr = txh->compositor->video_render->graph_hw;
	if (!txh->texture) return;

	GradientGetMatrix((SFNode *) rg->transform, mat);

	center = rg->center;
	focal = rg->focalPoint;

	/*move circle to object space*/
	center.x *= bounds->width;
	center.y *= bounds->height;
	focal.x *= bounds->width;
	focal.y *= bounds->height;


	/*move transform to object space*/
	mat->m[2] *= bounds->width;
	mat->m[5] *= bounds->height;
	mat->m[1] *= bounds->width / bounds->height;
	mat->m[3] *= bounds->height / bounds->width;

	dr->set_radial_gradient(txh->texture, center.x, center.y, focal.x, focal.y, rg->radius * bounds->width, rg->radius * bounds->height);

	const_a = (rg->opacity.count == 1) ? 1 : 0;
	cols = malloc(sizeof(u32) * rg->key.count);
	for (i=0; i<rg->key.count; i++) {
		a = (const_a ? rg->opacity.vals[0] : rg->opacity.vals[i]);
		cols[i] = MAKE_ARGB_FLOAT(a, rg->keyValue.vals[i].red, rg->keyValue.vals[i].green, rg->keyValue.vals[i].blue);
	}
	dr->set_gradient_interpolation(txh->texture, rg->key.vals, cols, rg->key.count);
	free(cols);

	dr->set_gradient_mode(txh->texture, rg->spreadMethod);
	mat_add_translation(mat, bounds->x, bounds->y - bounds->height);
}

void R3D_InitRadialGradient(Render3D *sr, SFNode *node)
{
	GradientStack *st = malloc(sizeof(GradientStack));
	memset(st, 0, sizeof(GradientStack));

	texture_setup(&st->txh, sr, node);
	st->txh.update_texture_fcnt = UpdateRadialGradient;

	st->txh.compute_gradient_matrix = RG_ComputeMatrix;
	st->txh.is_gradient = 1;

	Node_SetPrivate(node, st);
	Node_SetPreDestroyFunction(node, DestroyRadialGradient);
}

TextureHandler *r3d_rg_get_texture(SFNode *node)
{
	GradientStack *st = (GradientStack*) Node_GetPrivate(node);
	return &st->txh;
}

#endif


#else

void R3D_InitLinearGradient(Render3D *sr, SFNode *node)
{
}
TextureHandler *r3d_lg_get_texture(SFNode *node)
{
	return NULL;
}
void R3D_InitRadialGradient(Render3D *sr, SFNode *node)
{
}
TextureHandler *r3d_rg_get_texture(SFNode *node)
{
	return NULL;
}

#endif

