/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004 
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#include "m4_mesh.h"
#include "visual_surface.h"


#define MESH_STEP_ALLOC		50
#define MESH_CHECK_VERTEX(m)		\
	if (m->v_count == m->v_alloc) {	\
		m->v_alloc += MESH_STEP_ALLOC;	\
		m->vertices = realloc(m->vertices, sizeof(M4Vertex)*m->v_alloc);	\
	}	\

#define MESH_CHECK_IDX(m)		\
	if (m->i_count == m->i_alloc) {	\
		m->i_alloc += MESH_STEP_ALLOC;	\
		m->indices = realloc(m->indices, sizeof(M4Vertex)*m->i_alloc);	\
	}	\

void mesh_reset(M4Mesh *mesh)
{
	mesh->v_count = 0;
	mesh->i_count = 0;
	mesh->flags = 0;
	mesh->mesh_type = 0;
	memset(&mesh->bounds.min_edge, 0, sizeof(SFVec3f));
	memset(&mesh->bounds.max_edge, 0, sizeof(SFVec3f));
}

void mesh_free(M4Mesh *mesh)
{
	if (mesh->vertices) free(mesh->vertices);
	if (mesh->indices) free(mesh->indices);
	free(mesh);
}

M4Mesh *new_mesh()
{
	M4Mesh *mesh = malloc(sizeof(M4Mesh));
	if (mesh) {
		memset(mesh, 0, sizeof(M4Mesh));
		mesh->v_alloc = MESH_STEP_ALLOC;
		mesh->vertices = malloc(sizeof(M4Vertex)*mesh->v_alloc);
		mesh->i_alloc = MESH_STEP_ALLOC;
		mesh->indices = malloc(sizeof(u32)*mesh->i_alloc);
	}
	return mesh;
}

void mesh_update_bounds(M4Mesh *mesh)
{
	u32 i;
	Float mx, my, mz, Mx, My, Mz;
	mx = my = mz = M4_MAX_FLOAT;
	Mx = My = Mz = M4_MIN_FLOAT;
	for (i=0; i<mesh->v_count; i++) {
		SFVec3f *v = &mesh->vertices[i].pos;
		if (mx>v->x) mx=v->x;
		if (my>v->y) my=v->y;
		if (mz>v->z) mz=v->z;
		if (Mx<v->x) Mx=v->x;
		if (My<v->y) My=v->y;
		if (Mz<v->z) Mz=v->z;
	}
	mesh->bounds.min_edge.x = mx; mesh->bounds.min_edge.y = my; mesh->bounds.min_edge.z = mz;
	mesh->bounds.max_edge.x = Mx; mesh->bounds.max_edge.y = My; mesh->bounds.max_edge.z = Mz;
	bbox_refresh(&mesh->bounds);
}

static M4INLINE M4Vertex set_vertex(Float x, Float y, Float z, Float nx, Float ny, Float nz, Float u, Float v)
{
	M4Vertex res;
	res.pos.x = x; res.pos.y = y; res.pos.z = z;
	res.normal.x = nx; res.normal.y = ny; res.normal.z = nz;
	vec_norm(&res.normal);
	res.texcoords.x = u; res.texcoords.y = v;
	res.color.blue = res.color.red = res.color.green = 1.0f;
	return res;
}
void mesh_set_vertex(M4Mesh *mesh, Float x, Float y, Float z, Float nx, Float ny, Float nz, Float u, Float v)
{
	MESH_CHECK_VERTEX(mesh);
	mesh->vertices[mesh->v_count] = set_vertex(x, y, z, nx, ny, nz, u, v);
	mesh->v_count++;
}

void mesh_set_vertex_v(M4Mesh *mesh, SFVec3f pt, SFVec3f nor, SFVec2f tx, SFColor col)
{
	MESH_CHECK_VERTEX(mesh);
	mesh->vertices[mesh->v_count].pos = pt;
	mesh->vertices[mesh->v_count].texcoords = tx;
	mesh->vertices[mesh->v_count].color = col;
	/*always normalize normal vector*/
	vec_norm(&nor);
	mesh->vertices[mesh->v_count].normal = nor;
	mesh->v_count++;
}

void mesh_set_vertex_vx(M4Mesh *mesh, M4Vertex *vx)
{
	MESH_CHECK_VERTEX(mesh);
	mesh->vertices[mesh->v_count] = *vx;
	mesh->v_count++;
}

void mesh_set_point(M4Mesh *mesh, Float x, Float y, Float z, SFColor col)
{
	MESH_CHECK_VERTEX(mesh);
	mesh->vertices[mesh->v_count].pos.x = x;
	mesh->vertices[mesh->v_count].pos.y = y;
	mesh->vertices[mesh->v_count].pos.z = z;
	mesh->vertices[mesh->v_count].normal.x = mesh->vertices[mesh->v_count].normal.y = mesh->vertices[mesh->v_count].normal.z = 0;
	mesh->vertices[mesh->v_count].texcoords.x = mesh->vertices[mesh->v_count].texcoords.y = 0;
	mesh->vertices[mesh->v_count].color = col;
	mesh->v_count++;
}
void mesh_set_index(M4Mesh *mesh, u32 idx)
{
	MESH_CHECK_IDX(mesh);
	mesh->indices[mesh->i_count] = idx;
	mesh->i_count++;
}
void mesh_set_triangle(M4Mesh *mesh, u32 v1_idx, u32 v2_idx, u32 v3_idx)
{
	mesh_set_index(mesh, v1_idx);
	mesh_set_index(mesh, v2_idx);
	mesh_set_index(mesh, v3_idx);
}
void mesh_set_line(M4Mesh *mesh, u32 v1_idx, u32 v2_idx)
{
	mesh_set_index(mesh, v1_idx);
	mesh_set_index(mesh, v2_idx);
}

#if 0
static void mesh_recompute_normals(M4Mesh *mesh)
{
	u32 i;
	if (mesh->mesh_type) return;

	for (i=0; i<mesh->i_count; i+=3) {
		SFVec3f v1 = vec_diff(&mesh->vertices[mesh->indices[i+1]].pos, &mesh->vertices[mesh->indices[i]].pos);
		SFVec3f v2 = vec_diff(&mesh->vertices[mesh->indices[i+2]].pos, &mesh->vertices[mesh->indices[i+1]].pos);
		SFVec3f v3 = vec_cross(&v1, &v2);
		vec_norm(&v3);
		mesh->vertices[mesh->indices[i]].normal = v3;
		mesh->vertices[mesh->indices[i+1]].normal = v3;
		mesh->vertices[mesh->indices[i+2]].normal = v3;
	}
}
#endif


void mesh_new_box(M4Mesh *mesh, SFVec3f size)
{
	Float hx = size.x / 2.0f;
	Float hy = size.y / 2.0f;
	Float hz = size.z / 2.0f;

	mesh_reset(mesh);
	/*back face (horiz flip of texcoords)*/
	mesh_set_vertex(mesh,  hx, -hy, -hz,  0.0f,  0.0f, -1.0f, 0.0f, 0.0f);
	mesh_set_vertex(mesh, -hx, -hy, -hz,  0.0f,  0.0f, -1.0f, 1.0f, 0.0f);
	mesh_set_vertex(mesh, -hx,  hy, -hz,  0.0f,  0.0f, -1.0f, 1.0f, 1.0f);
	mesh_set_vertex(mesh,  hx,  hy, -hz,  0.0f,  0.0f, -1.0f, 0.0f, 1.0f);
	mesh_set_triangle(mesh, 0, 1, 2); mesh_set_triangle(mesh, 0, 2, 3);
	/*top face*/
	mesh_set_vertex(mesh, -hx,  hy,  hz,  0.0f,  1.0f,  0.0f, 0.0f, 0.0f);
	mesh_set_vertex(mesh,  hx,  hy,  hz,  0.0f,  1.0f,  0.0f, 1.0f, 0.0f);
	mesh_set_vertex(mesh,  hx,  hy, -hz,  0.0f,  1.0f,  0.0f, 1.0f, 1.0f);
	mesh_set_vertex(mesh, -hx,  hy, -hz,  0.0f,  1.0f,  0.0f, 0.0f, 1.0f);
	mesh_set_triangle(mesh, 4, 5, 6); mesh_set_triangle(mesh, 4, 6, 7);
	/*front face*/
	mesh_set_vertex(mesh, -hx, -hy,  hz,  0.0f,  0.0f,  1.0f, 0.0f, 0.0f);
	mesh_set_vertex(mesh,  hx, -hy,  hz,  0.0f,  0.0f,  1.0f, 1.0f, 0.0f);
	mesh_set_vertex(mesh,  hx,  hy,  hz,  0.0f,  0.0f,  1.0f, 1.0f, 1.0f);
	mesh_set_vertex(mesh, -hx,  hy,  hz,  0.0f,  0.0f,  1.0f, 0.0f, 1.0f);
	mesh_set_triangle(mesh, 8, 9, 10); mesh_set_triangle(mesh, 8, 10, 11);
	/*left face*/
	mesh_set_vertex(mesh, -hx, -hy, -hz, -1.0f,  0.0f,  0.0f, 0.0f, 0.0f);
	mesh_set_vertex(mesh, -hx, -hy,  hz, -1.0f,  0.0f,  0.0f, 1.0f, 0.0f);
	mesh_set_vertex(mesh, -hx,  hy,  hz, -1.0f,  0.0f,  0.0f, 1.0f, 1.0f);
	mesh_set_vertex(mesh, -hx,  hy, -hz, -1.0f,  0.0f,  0.0f, 0.0f, 1.0f);
	mesh_set_triangle(mesh, 12, 13, 14); mesh_set_triangle(mesh, 12, 14, 15);
	/*bottom face*/
	mesh_set_vertex(mesh, -hx, -hy, -hz,  0.0f, -1.0f,  0.0f, 0.0f, 0.0f);
	mesh_set_vertex(mesh,  hx, -hy, -hz,  0.0f, -1.0f,  0.0f, 1.0f, 0.0f);
	mesh_set_vertex(mesh,  hx, -hy,  hz,  0.0f, -1.0f,  0.0f, 1.0f, 1.0f);
	mesh_set_vertex(mesh, -hx, -hy,  hz,  0.0f, -1.0f,  0.0f, 0.0f, 1.0f);
	mesh_set_triangle(mesh, 16, 17, 18); mesh_set_triangle(mesh, 16, 18, 19);
	/*right face*/
	mesh_set_vertex(mesh,  hx, -hy,  hz,  1.0f,  0.0f,  0.0f, 0.0f, 0.0f);
	mesh_set_vertex(mesh,  hx, -hy, -hz,  1.0f,  0.0f,  0.0f, 1.0f, 0.0f);
	mesh_set_vertex(mesh,  hx,  hy, -hz,  1.0f,  0.0f,  0.0f, 1.0f, 1.0f);
	mesh_set_vertex(mesh,  hx,  hy,  hz,  1.0f,  0.0f,  0.0f, 0.0f, 1.0f);
	mesh_set_triangle(mesh, 20, 21, 22); mesh_set_triangle(mesh, 20, 22, 23);


    mesh->flags |= MESH_IS_SOLID;
	mesh->bounds.min_edge.x = -hx; mesh->bounds.min_edge.y = -hy; mesh->bounds.min_edge.z = -hz;
	mesh->bounds.max_edge.x = hx; mesh->bounds.max_edge.y = hy; mesh->bounds.max_edge.z = hz;
	bbox_refresh(&mesh->bounds);
}

void mesh_new_unit_bbox(M4Mesh *mesh)
{
	SFColor col;
	Float s = 0.5f;

	memset(&col, 0, sizeof(SFColor));
	mesh_reset(mesh);
	mesh->mesh_type = MESH_LINESET;
	mesh_set_point(mesh, -s, -s, -s,  col);
	mesh_set_point(mesh, s, -s, -s,  col);
	mesh_set_point(mesh, s, s, -s,  col);
	mesh_set_point(mesh, -s, s, -s,  col);
	mesh_set_point(mesh, -s, -s, s,  col);
	mesh_set_point(mesh, s, -s, s,  col);
	mesh_set_point(mesh, s, s, s,  col);
	mesh_set_point(mesh, -s, s, s,  col);

	mesh_set_line(mesh, 0, 1); mesh_set_line(mesh, 1, 2); mesh_set_line(mesh, 2, 3); mesh_set_line(mesh, 3, 0);
	mesh_set_line(mesh, 4, 5); mesh_set_line(mesh, 5, 6); mesh_set_line(mesh, 6, 7); mesh_set_line(mesh, 7, 4);
	mesh_set_line(mesh, 0, 4); 
	mesh_set_line(mesh, 1, 5); 
	mesh_set_line(mesh, 2, 6); 
	mesh_set_line(mesh, 3, 7);
	bbox_refresh(&mesh->bounds);
}


static void compute_cylinder(Float height, Float radius, s32 numFacets, SFVec3f *coords, SFVec2f *texcoords)
{
    Float angle, t, u;
    s32 i;

	t = 0.5f * height;
    for (i=0; i<numFacets; ++i) {
        angle = i * 2 * M_PI / numFacets;
        coords[i].x = radius * (Float) cos(angle);
        coords[i].y = t;
        coords[i].z = radius * (Float) sin(angle);
        u = 0.75f - ((Float) i) / numFacets;
        texcoords[i].x = u;
        texcoords[i].y = 1.0;
    }
}

#define CYLINDER_SUBDIV	24
void mesh_new_cylinder(M4Mesh *mesh, Float height, Float radius, Bool bottom, Bool side, Bool top, Bool low_res)
{
	u32 nfacets, i, c_idx;
	SFVec3f *coords;
	SFVec2f *texcoords;

	mesh_reset(mesh);
	if (!bottom && !side && !top) return;

	nfacets = CYLINDER_SUBDIV;
	if (low_res) nfacets /= HIGH_SPEED_RATIO;
	coords = malloc(sizeof(SFVec3f) * nfacets);
	texcoords = malloc(sizeof(SFVec2f) * nfacets);

	compute_cylinder(height, radius, nfacets, coords, texcoords);

	if (side) {
		for (i=0; i<nfacets; ++i) {
			/*top*/
			mesh_set_vertex(mesh, coords[i].x, coords[i].y, coords[i].z,
								coords[i].x, 0.0, coords[i].z, 
								texcoords[i].x, 1.0f);

			/*bottom*/
			mesh_set_vertex(mesh, coords[i].x, -1*coords[i].y, coords[i].z,
								coords[i].x, 0.0, coords[i].z, 
								texcoords[i].x, 0.0f);


			/*top circle is counterclockwise, reverse coords*/
			if (i) {
				mesh_set_triangle(mesh, mesh->v_count-4, mesh->v_count-1, mesh->v_count-3); 
				mesh_set_triangle(mesh, mesh->v_count-4, mesh->v_count-2, mesh->v_count-1); 
			}
		}

		/*top*/
		mesh_set_vertex(mesh, coords[0].x, coords[0].y, coords[0].z,
							coords[0].x, 0.0, coords[0].z, 
							texcoords[0].x - 1.0f, 1.0f);

		/*bottom*/
		mesh_set_vertex(mesh, coords[0].x, -1*coords[0].y, coords[0].z,
							coords[0].x, 0.0, coords[0].z, 
							texcoords[0].x - 1.0f, 0.0f);

		mesh_set_triangle(mesh, mesh->v_count-4, mesh->v_count-1, mesh->v_count-3); 
		mesh_set_triangle(mesh, mesh->v_count-4, mesh->v_count-2, mesh->v_count-1); 
	}

	if (bottom) {
		Float angle = 0.5f * M_PI;
		Float aincr = 2.0f * M_PI / nfacets;

		mesh_set_vertex(mesh, 0.0, -0.5f * height, 0.0, 0.0, -1.0, 0.0, 0.5, 0.5);
		c_idx = mesh->v_count-1;
		for (i=0; i<nfacets; ++i, angle += aincr) {
			mesh_set_vertex(mesh, coords[i].x, -1*coords[i].y, coords[i].z,
							0.0, -1.0, 0.0, 
							0.5f * (1.0f + (Float) sin(angle)), 1.0f - 0.5f * (1.0f + (Float) cos(angle)));
			if (i) mesh_set_triangle(mesh, c_idx, mesh->v_count-2, mesh->v_count-1);
		}

		mesh_set_vertex(mesh, coords[0].x, -1*coords[0].y, coords[0].z,
						0.0, -1.0, 0.0, 
						0.5f * (1.0f + (Float) sin(angle)), 1.0f - 0.5f * (1.0f + (Float) cos(angle)));
		mesh_set_triangle(mesh, c_idx, mesh->v_count-2, mesh->v_count-1);
	}

    if (top) {
        Float aincr = 2.0f * M_PI / nfacets;
        Float angle = 0.5f * M_PI + aincr;

		mesh_set_vertex(mesh, 0.0, 0.5f * height, 0.0, 0.0, 1.0, 0.0, 0.5f, 0.5f);
        c_idx = mesh->v_count-1;
        for (i=nfacets; i>0; --i, angle += aincr) {

			mesh_set_vertex(mesh, coords[i - 1].x, coords[i - 1].y, coords[i - 1].z,
							0.0, 1.0, 0.0,
							0.5f * (1.0f + (Float) sin(angle)), 1.0f - 0.5f * (1.0f + (Float) cos(angle)));
			if (i) mesh_set_triangle(mesh, c_idx, mesh->v_count-2, mesh->v_count-1);
        }
		mesh_set_vertex(mesh, coords[nfacets - 1].x, coords[nfacets - 1].y, coords[nfacets - 1].z,
						0.0, 1.0, 0.0,
						0.5f * (1.0f + (Float) sin(angle)), 1.0f - 0.5f * (1.0f + (Float) cos(angle)));
		mesh_set_triangle(mesh, c_idx, mesh->v_count-2, mesh->v_count-1);
    }
	free(texcoords);
	free(coords);

    if (top && bottom && side) mesh->flags |= MESH_IS_SOLID;

	mesh->bounds.min_edge.x = mesh->bounds.min_edge.z = -radius;
	mesh->bounds.max_edge.x = mesh->bounds.max_edge.z = radius;
	mesh->bounds.max_edge.y = (side || (top && bottom)) ? height/2 : 0;
	mesh->bounds.min_edge.y = -mesh->bounds.max_edge.y;
	bbox_refresh(&mesh->bounds);
}

#define CONE_SUBDIV	24
void mesh_new_cone(M4Mesh *mesh, Float height, Float radius, Bool bottom, Bool side, Bool low_res)
{
    u32 nfacets, i, c_idx;
    SFVec3f *coords;
	SFVec2f *texcoords;

	mesh_reset(mesh);
    if (!bottom && !side) return;

    nfacets = CONE_SUBDIV;
    if (low_res) nfacets /= HIGH_SPEED_RATIO;
    coords = malloc(sizeof(SFVec3f) * nfacets);
	texcoords = malloc(sizeof(SFVec2f) * nfacets);
	
	compute_cylinder(height, radius, nfacets, coords, texcoords);

	if (side) {
		Float Ny = radius * radius / height;

		for (i = 0; i < nfacets; ++i) {
			/*top*/
			mesh_set_vertex(mesh, 0.0f, coords[i].y, 0.0f, 
							coords[i].x, Ny, coords[i].z,
							texcoords[i].x, 1.0f);

			/*base*/
			mesh_set_vertex(mesh, coords[i].x, -1*coords[i].y, coords[i].z, 
							coords[i].x, Ny, coords[i].z,
							texcoords[i].x, 0.0f);
			if (i) {
				mesh_set_triangle(mesh, mesh->v_count-4, mesh->v_count-1, mesh->v_count-3); 
				mesh_set_triangle(mesh, mesh->v_count-4, mesh->v_count-2, mesh->v_count-1); 
			}
		}
		/*top*/
		mesh_set_vertex(mesh, 0.0f, coords[0].y, 0.0f, coords[0].x, Ny, coords[0].z, texcoords[0].x - 1.0f, 1.0f);
		/*base*/
		mesh_set_vertex(mesh, coords[0].x, -1*coords[0].y, coords[0].z, 
						coords[0].x, Ny, coords[0].z,
						texcoords[0].x - 1.0f, 0.0f);

		mesh_set_triangle(mesh, mesh->v_count-4, mesh->v_count-1, mesh->v_count-3); 
		mesh_set_triangle(mesh, mesh->v_count-4, mesh->v_count-2, mesh->v_count-1); 
	}

	if (bottom) {
		Float angle = 0.5f * M_PI;
		Float aincr = 2.0f * M_PI / nfacets;

		mesh_set_vertex(mesh, 0.0, -0.5f * height, 0.0, 0.0, -1.0, 0.0, 0.5, 0.5);
		c_idx = mesh->v_count - 1;
		for (i=0; i<nfacets; ++i, angle += aincr) {
			mesh_set_vertex(mesh, coords[i].x, -1*coords[i].y, coords[i].z,
									0.0, -1.0, 0.0,
									0.5f * (1.0f + (Float) sin(angle)), 1.0f - 0.5f * (1.0f + (Float) cos(angle)));

			if (i) 
				mesh_set_triangle(mesh, c_idx, mesh->v_count-2, mesh->v_count-1); 
		}
		mesh_set_vertex(mesh, coords[0].x, -1*coords[0].y, coords[0].z,
									0.0, -1.0, 0.0,
									0.5f * (1.0f + (Float) sin(angle)), 1.0f - 0.5f * (1.0f + (Float)  cos(angle)));
		mesh_set_triangle(mesh, c_idx, mesh->v_count-2, mesh->v_count-1); 
	}
	free(texcoords);
	free(coords);

    if (bottom && side) mesh->flags |= MESH_IS_SOLID;

	mesh->bounds.min_edge.x = mesh->bounds.min_edge.z = -radius;
	mesh->bounds.max_edge.x = mesh->bounds.max_edge.z = radius;
	mesh->bounds.max_edge.y = height/2;
	mesh->bounds.min_edge.y = -mesh->bounds.max_edge.y;
	bbox_refresh(&mesh->bounds);
}

void compute_sphere(Float radius, SFVec3f *coords, SFVec2f *texcoords, u32 num_steps)
{
	Float r, angle, x, y, z;
	u32 i, j;

    for (i=0; i<num_steps; i++) {
        angle = (i * M_PI / (num_steps-1) ) - M_PI/2.0f;
        y = (Float) sin(angle);
        r = (Float) sqrt(1.0 - y*y);
        for (j = 0; j < num_steps; j++) {
            angle = 2* M_PI * j / num_steps;
            x = -1.0f * (Float) sin(angle)*r;
            z = -1.0f * (Float) cos(angle)*r;
            coords[i * num_steps + j].x = radius * x;
            coords[i * num_steps + j].y = radius * y;
            coords[i * num_steps + j].z = radius * z;
			texcoords[i * num_steps + j].x = (Float) (j) / num_steps;
			texcoords[i * num_steps + j].y = (Float) (i) / num_steps;
        }
    }
}

#define SPHERE_SUBDIV	16
void mesh_new_sphere(M4Mesh *mesh, Float radius, Bool low_res)
{
	u32 i, j, num_steps, npts;
	SFVec3f *coords;
	SFVec2f *texcoords;
	
	num_steps = SPHERE_SUBDIV;
	if (low_res) num_steps /= 2;
    npts = num_steps * num_steps;

	coords = malloc(sizeof(SFVec3f)*npts);
	texcoords = malloc(sizeof(SFVec2f)*npts);
	compute_sphere(radius, coords, texcoords, num_steps);

    for (i=0; i<num_steps-1; i++) {
        u32 n = i * num_steps;

        for (j=0; j<num_steps; j++) {
			mesh_set_vertex(mesh, coords[n + j + num_steps].x, coords[n + j + num_steps].y, coords[n + j + num_steps].z, 
								coords[n + j + num_steps].x, coords[n + j + num_steps].y, coords[n + j + num_steps].z,
								texcoords[n + j + num_steps].x, texcoords[n + j + num_steps].y);

			mesh_set_vertex(mesh, coords[n + j].x, coords[n + j].y, coords[n + j].z, 
								coords[n + j].x, coords[n + j].y, coords[n + j].z,
								texcoords[n + j].x, texcoords[n + j].y);

			if (j) {
				mesh_set_triangle(mesh, mesh->v_count-4, mesh->v_count-3, mesh->v_count-2);
				mesh_set_triangle(mesh, mesh->v_count-2, mesh->v_count-3, mesh->v_count-1);
			}

        }

		mesh_set_vertex(mesh, coords[n + num_steps].x, coords[n + num_steps].y, coords[n + num_steps].z, 
							coords[n + num_steps].x, coords[n + num_steps].y, coords[n  + num_steps].z,
							1.0, texcoords[n + num_steps].y);
		mesh_set_vertex(mesh, coords[n].x, coords[n].y, coords[n].z, 
							coords[n].x, coords[n].y, coords[n].z,
							1.0, texcoords[n].y);

		mesh_set_triangle(mesh, mesh->v_count-4, mesh->v_count-3, mesh->v_count-2);
		mesh_set_triangle(mesh, mesh->v_count-2, mesh->v_count-3, mesh->v_count-1);
    }

	free(coords);
	free(texcoords);
	mesh->flags |= MESH_IS_SOLID;
	mesh->bounds.min_edge.x = mesh->bounds.min_edge.y = mesh->bounds.min_edge.z = -radius;
	mesh->bounds.max_edge.x = mesh->bounds.max_edge.y = mesh->bounds.max_edge.z = radius;
	bbox_refresh(&mesh->bounds);
}


void mesh_new_rectangle(M4Mesh *mesh, SFVec2f size)
{
	Float hx = size.x / 2.0f;
	Float hy = size.y / 2.0f;
	mesh_reset(mesh);

	mesh_set_vertex(mesh, -hx, -hy,  0,  0.0f,  0.0f,  1.0f, 0.0f, 0.0f);
	mesh_set_vertex(mesh,  hx, -hy,  0,  0.0f,  0.0f,  1.0f, 1.0f, 0.0f);
	mesh_set_vertex(mesh,  hx,  hy,  0,  0.0f,  0.0f,  1.0f, 1.0f, 1.0f);
	mesh_set_vertex(mesh, -hx,  hy,  0,  0.0f,  0.0f,  1.0f, 0.0f, 1.0f);
	mesh_set_triangle(mesh, 0, 1, 2); mesh_set_triangle(mesh, 0, 2, 3);
	
	mesh->flags |= MESH_IS_2D;

	mesh->bounds.min_edge.x = -hx; mesh->bounds.min_edge.y = -hy; mesh->bounds.min_edge.z = 0;
	mesh->bounds.max_edge.x = hx; mesh->bounds.max_edge.y = hy; mesh->bounds.max_edge.z = 0;
	bbox_refresh(&mesh->bounds);
}

#define ELLIPSE_SUBDIV		32
void mesh_new_ellipse(M4Mesh *mesh, Float a_dia, Float b_dia, Bool low_res)
{
	Float step, cur, end, cosa, sina;
	a_dia /= 2;
	b_dia /= 2;

	/*no begin/end draw since we always use generic 2D node drawing methods*/
	end = 2*M_PI;
	step = end / ELLIPSE_SUBDIV;
    if (low_res) step *= HIGH_SPEED_RATIO;


	mesh_reset(mesh);

	/*center*/
	mesh_set_vertex(mesh, 0, 0, 0, 0.0f, 0.0f, 1.0f, 0.5f, 0.5f);
	for (cur=0; cur<end; cur += step) {
		cosa = (Float) (cos(cur) );
		sina = (Float) (sin(cur) );

		mesh_set_vertex(mesh, a_dia*cosa, b_dia*sina, 0, 
								0.0f, 0.0f, 1.0f, 
								0.5f * (1.0f + cosa), 0.5f * (1.0f + sina));

		if (cur) mesh_set_triangle(mesh, 0, mesh->v_count-2, mesh->v_count-1); 
	}
	mesh_set_vertex(mesh, a_dia, 0, 0, 0.0f, 0.0f, 1.0f, 1.0f, 0.5f);
	mesh_set_triangle(mesh, 0, mesh->v_count-2, mesh->v_count-1); 

	mesh->flags |= MESH_IS_2D;
	mesh->bounds.min_edge.x = -a_dia; mesh->bounds.min_edge.y = -b_dia; mesh->bounds.min_edge.z = 0;
	mesh->bounds.max_edge.x = a_dia; mesh->bounds.max_edge.y = b_dia; mesh->bounds.max_edge.z = 0;
	bbox_refresh(&mesh->bounds);
}

void mesh_from_path(M4Mesh *mesh, M4Path *path)
{
	u32 i, nbPts;
	Float w, h;
	Bool isCW = 0;

	mesh_reset(mesh);
	if (path->subpathlen==1) {
		u32 type = polygon2D_check_convexity(path->subpath[0]->point, path->subpath[0]->pointlen);
		switch (type) {
		/*degenrated polygon - skip*/
		case M4_PolyConvexLine:
			return;
		case M4_PolyConvexCW:
			isCW = 1;
		case M4_PolyConvexCCW:
			w = path->max_x - path->min_x;
			h = path->max_y - path->min_y;

			/*add all vertices*/
			for (i=0; i<path->subpath[0]->pointlen - 1; i++) {
				M4Point2D pt = path->subpath[0]->point[i];
				mesh_set_vertex(mesh, pt.x, pt.y, 0, 0, 0, 1, (pt.x - path->min_x) / w, (path->max_y - pt.y) / h);
			}
			nbPts = path->subpath[0]->pointlen - 1;
			/*take care of already closed path*/
			if ( (path->subpath[0]->point[i].x != path->subpath[0]->point[0].x) || (path->subpath[0]->point[i].y != path->subpath[0]->point[0].y)) {
				M4Point2D pt = path->subpath[0]->point[i];
				mesh_set_vertex(mesh, pt.x, pt.y, 0, 0, 0, 1, (pt.x - path->min_x) / w, (path->max_y - pt.y) / h);
				nbPts = path->subpath[0]->pointlen;
			}
			/*make it CCW*/
			for (i=1; i<nbPts-1; i++) {
				if (isCW) {
					mesh_set_triangle(mesh, 0, nbPts-i, nbPts-i-1);
				} else {
					mesh_set_triangle(mesh, 0, i, i+1);
				}
			}
			mesh->bounds.min_edge.x = path->min_x; mesh->bounds.min_edge.y = path->min_y; mesh->bounds.min_edge.z = 0;
			mesh->bounds.max_edge.x = path->max_x; mesh->bounds.max_edge.y = path->max_y; mesh->bounds.max_edge.z = 0;
			bbox_refresh(&mesh->bounds);
			return;
		default:
			break;
		}
	}
	/*we need to tesselate the path*/
	TesselatePath(mesh, path, 0);
}

void mesh_get_outline(M4Mesh *mesh, M4Path *path)
{
	u32 i, j;
	mesh_reset(mesh);

	mesh->mesh_type = MESH_LINESET;
	mesh->flags |= (MESH_IS_2D | MESH_NO_TEXTURE);

	for (i=0; i<path->subpathlen; i++) {
		M4SubPath *p = path->subpath[i];
		for (j=0; j<p->pointlen; j++) {
			M4Point2D pt = p->point[j];
			if (j) mesh_set_line(mesh, mesh->v_count-1, mesh->v_count);
			mesh_set_vertex(mesh, pt.x, pt.y, 0, 0, 0, 1, 0, 0);
		}
	}
	mesh_update_bounds(mesh);
}


void mesh_new_ils(M4Mesh *mesh, SFNode *__coord, MFInt32 *coordIndex, SFNode *__color, MFInt32 *colorIndex, Bool colorPerVertex, Bool do_close)
{
	u32 i, n, count, c_count, col_count;
	u32 index;
	u32 first_idx, last_idx;
	Bool move_to;
	SFVec3f pt;
	SFColor col;
	Bool has_color, has_coord;
	B_Coordinate2D *coord2D = (B_Coordinate2D*) __coord;
	B_Coordinate *coord = (B_Coordinate*) __coord;
	B_Color *color = (B_Color *) __color;

	if (Node_GetTag(__coord) != TAG_Coordinate2D) coord2D = NULL;
	if (Node_GetTag(__coord) != TAG_Coordinate) coord = NULL;

	if (!coord2D && !coord) return;
	c_count = coord2D ? coord2D->point.count : coord->point.count;
	if (!c_count) return;

	count = coordIndex->count;
	has_coord = count ? 1 : 0;
	if (!has_coord) count = c_count; 

	if (!colorIndex->vals) colorIndex = coordIndex;
	col_count = colorIndex->count ? colorIndex->count : c_count;
	/*not enough color indices, use coord ones*/
	if (colorPerVertex && (col_count<count) ) {
		colorIndex = coordIndex;
		col_count = count;
	}

	mesh_reset(mesh);
	mesh->mesh_type = MESH_LINESET;
	has_color = (color && color->color.count) ? 1 : 0;
	if (has_color) mesh->flags |= MESH_HAS_COLOR;

	n = 0;
	if (has_color && !colorPerVertex) {
		index = colorIndex->count ? colorIndex->vals[0] : 0;
		if ((u32) index < col_count) col = color->color.vals[index];
	}
	move_to = 1;

	first_idx = last_idx = 0;
	for (i=0; i<count; i++) {
		if (has_coord && coordIndex->vals[i] == -1) {
			/*close color per vertex*/
			if (!move_to && do_close && !vec_equal(&mesh->vertices[first_idx].pos, &mesh->vertices[last_idx].pos) ) {
				mesh_set_line(mesh, last_idx, first_idx);
			}
			move_to = 1;
			n++;
			if (has_color && !colorPerVertex) {
				if (n<colorIndex->count) index = colorIndex->vals[n];
				else if (n<col_count) index = n;
				else index = 0;
				if ((u32) index < col_count) col = color->color.vals[index];
			}
		} else {
			if (has_color && colorPerVertex) {
				if (i<colorIndex->count) index = colorIndex->vals[i];
				else if (i<col_count) index = i;
				else index=0;
				if ((u32) index < color->color.count) col = color->color.vals[index];
			}
			if (has_coord) index = coordIndex->vals[i];
			else index = i;
			if (index < c_count) {
				if (coord2D) {
					pt.x = coord2D->point.vals[index].x;
					pt.y = coord2D->point.vals[index].y;
					pt.z = 0;
				} else {
					pt = coord->point.vals[index];
				}
				mesh_set_point(mesh, pt.x, pt.y, pt.z, col);
				last_idx = mesh->v_count - 1;
				if (move_to) {
					first_idx = last_idx;
					move_to = 0;
				} else {
					mesh_set_line(mesh, last_idx-1, last_idx);
				}
			}
		}
	}
	/*close color per vertex*/
	if (do_close && !vec_equal(&mesh->vertices[first_idx].pos, &mesh->vertices[last_idx].pos) ) {
		mesh_set_line(mesh, last_idx, first_idx);
	}
	if (coord2D) mesh->flags |= MESH_IS_2D;
	mesh_update_bounds(mesh);
}

void mesh_new_ps(M4Mesh *mesh, SFNode *__coord, SFNode *__color)
{
	u32 c_count, i;
	Bool has_color;
	SFVec3f pt;
	SFColor col;
	B_Coordinate2D *coord2D = (B_Coordinate2D*) __coord;
	B_Coordinate *coord = (B_Coordinate*) __coord;
	B_Color *color = (B_Color *) __color;

	if (Node_GetTag(__coord) != TAG_Coordinate2D) coord2D = NULL;
	if (Node_GetTag(__coord) != TAG_Coordinate) coord = NULL;
	if (!coord2D && !coord) return;
	c_count = coord2D ? coord2D->point.count : coord->point.count;
	if (!c_count) return;

	mesh_reset(mesh);
	mesh->mesh_type = MESH_POINTSET;

	has_color = (color && color->color.count) ? 1 : 0;
	if (has_color) mesh->flags |= MESH_HAS_COLOR;

	col.red = col.green = col.blue = 1.0f;

	for (i=0; i<c_count; ++i) {
		if (has_color && i < color->color.count) col = color->color.vals[i];
		if (coord2D) {
			pt.x = coord2D->point.vals[i].x;
			pt.y = coord2D->point.vals[i].y;
			pt.z = 0;
		} else {
			pt = coord->point.vals[i];
		}
		mesh_set_point(mesh, pt.x, pt.y, pt.z, col);
		mesh_set_index(mesh, mesh->v_count-1);
	}
	mesh_update_bounds(mesh);
}


#define GET_IDX(idx, array) \
		if (idx<array->count) index = array->vals[idx];	\
		else if (idx<c_count) index = idx;	\
		else index = 0;	\


void mesh_new_ifs_intern(M4Mesh *mesh, SFNode *__coord, MFInt32 *coordIndex, SFNode *__color, MFInt32 *colorIndex, Bool colorPerVertex, SFNode *__normal, MFInt32 *normalIndex, Bool normalPerVertex, SFNode *__texCoords, MFInt32 *texCoordIndex)
{
	u32 i, n, count, c_count, col_count, nor_count;
	u32 index;
	u32 first_idx, last_idx;
	Bool move_to;
	SFVec2f tx;
	u32 s_axis, t_axis;
	SFVec3f pt, nor, bounds, center;
	SFColor col;
	Bool has_color, has_coord, has_normal, has_tex;
	M4Mesh **faces;
	u32 face_count, cur_face;
	B_Coordinate2D *coord2D = (B_Coordinate2D*) __coord;
	B_Coordinate *coord = (B_Coordinate*) __coord;
	B_Color *color = (B_Color *) __color;
	B_Normal *normal = (B_Normal*) __normal;
	B_TextureCoordinate *txcoord = (B_TextureCoordinate*) __texCoords;

	if (Node_GetTag(__coord) != TAG_Coordinate2D) coord2D = NULL;
	if (Node_GetTag(__coord) != TAG_Coordinate) coord = NULL;

	if (!coord2D && !coord) return;
	c_count = coord2D ? coord2D->point.count : coord->point.count;
	if (!c_count) return;

	if (!colorIndex->vals) colorIndex = coordIndex;
	col_count = colorIndex->count ? colorIndex->count : c_count;
	if (normal && normalIndex) {
		if (!normalIndex->vals) normalIndex = coordIndex;
		nor_count = normalIndex->count ? normalIndex->count : c_count;
		has_normal = normal->vector.count ? 1 : 0;
	} else {
		nor_count = 0;
		nor.x = nor.y = 0.0;
		nor.z = 1.0;
		has_normal = 0;
	}

	has_tex = txcoord ? 1 : 0;
	if (has_tex && !texCoordIndex->vals) texCoordIndex = coordIndex;

	mesh_reset(mesh);
	memset(&col, 0, sizeof(SFColor));
	/*compute bounds - note we assume all points in the IFS coordinate are used*/
	s_axis = t_axis = 0;
	if (!has_tex) {
		for (i=0; i<c_count; i++) {
			if (coord2D) mesh_set_point(mesh, coord2D->point.vals[i].x, coord2D->point.vals[i].y, 0, col);
			else mesh_set_point(mesh, coord->point.vals[i].x, coord->point.vals[i].y, coord->point.vals[i].z, col);
		}
		mesh_update_bounds(mesh);
		bounds = vec_diff(&mesh->bounds.max_edge, &mesh->bounds.min_edge);
		center = mesh->bounds.min_edge;
		if ( (bounds.x >= bounds.y) && (bounds.x >= bounds.z) ) {
			s_axis = 0;
			t_axis = (bounds.y >= bounds.z) ? 1 : 2;
		}
		else if ( (bounds.y >= bounds.x) && (bounds.y >= bounds.z) ) {
			s_axis = 1;
			t_axis = (bounds.x >= bounds.z) ? 0 : 2;
		}
		else {
			s_axis = 2;
			t_axis = (bounds.x >= bounds.y) ? 0 : 1;
		}
	}

	has_color = (color && color->color.count) ? 1 : 0;
	n = 0;
	if (has_color && !colorPerVertex) {
		index = colorIndex->count ? colorIndex->vals[0] : 0;
		if (index < col_count) col = color->color.vals[index];
	}
	if (has_normal && !normalPerVertex) {
		index = normalIndex->count ? normalIndex->vals[0] : 0;
		if (index < nor_count) nor = normal->vector.vals[index];
	}

	move_to = 1;
	count = coordIndex->count;
	has_coord = count ? 1 : 0;
	if (!has_coord) count = c_count; 

	/*build face list*/
	if (!has_coord) {
		face_count = 1;
	} else {
		face_count = 0;
		for (i=0; i<count; i++) {
			if (coordIndex->vals[i] == -1) face_count++;
		}
		/*don't forget last face*/
		if (coordIndex->vals[count-1] != -1) face_count++;
	}
	faces = malloc(sizeof(M4Mesh *)*face_count);
	for (i=0; i<face_count; i++) faces[i] = new_mesh();
	
	cur_face = 0;
	first_idx = last_idx = 0;
	for (i=0; i<count; i++) {
		if (has_coord && coordIndex->vals[i] == -1) {
			move_to = 1;
			n++;
			if (has_color && !colorPerVertex) {
				GET_IDX(n, colorIndex);
				if (index < col_count) col = color->color.vals[index];
			}
			if (has_normal && !normalPerVertex) {
				GET_IDX(n, normalIndex);
				if (index < nor_count) nor = normal->vector.vals[index];
			}
			if (faces[cur_face]->v_count<3) faces[cur_face]->v_count=0;
			cur_face++;
		} else {
			if (has_color && colorPerVertex) {
				GET_IDX(i, colorIndex);
				if (index < color->color.count) col = color->color.vals[index];
			}
			if (has_normal && normalPerVertex) {
				GET_IDX(i, normalIndex);
				if (index < normal->vector.count) {
					nor = normal->vector.vals[index];
				}
			}

			if (has_coord) index = coordIndex->vals[i];
			else index = i;
			if (index < c_count) {
				if (coord2D) {
					pt.x = coord2D->point.vals[index].x;
					pt.y = coord2D->point.vals[index].y;
					pt.z = 0;
				} else {
					pt = coord->point.vals[index];
				}
			}

			if (has_tex) {
				GET_IDX(i, texCoordIndex);
				if (index < txcoord->point.count) tx = txcoord->point.vals[index];
			} else {
				SFVec3f v = vec_diff(&pt, &center);
				tx.x = tx.y = 0;
				if (s_axis==0) tx.x = v.x/bounds.x;
				else if (s_axis==1) tx.x = v.y/bounds.y;
				else if (s_axis==2) tx.x = v.z/bounds.z;
				if (t_axis==0) tx.y = v.x/bounds.x;
				else if (t_axis==1) tx.y = v.y/bounds.y;
				else if (t_axis==2) tx.y = v.z/bounds.z;
			}
			
			mesh_set_vertex_v(faces[cur_face], pt, nor, tx, col);
		}
	}

	/*generate normals in 3D - FIXME: smooth normals with creaseAngle*/
	if (!has_normal && coord) {
		for (i=0; i<face_count; i++) {
			SFVec3f v1, v2;
			u32 j;
			v1 = vec_diff(&faces[i]->vertices[1].pos, &faces[i]->vertices[0].pos);
			v2 = vec_diff(&faces[i]->vertices[2].pos, &faces[i]->vertices[1].pos);
			nor = vec_cross(&v1, &v2);
			for (j=0; j<faces[i]->v_count; j++) {
				faces[i]->vertices[j].normal = nor;
			}
		}
	}

	mesh_reset(mesh);
	mesh->mesh_type = MESH_TRIANGLES;
	if (has_color) mesh->flags |= MESH_HAS_COLOR;

	for (i=0; i<face_count; i++) {
		if (faces[i]->v_count) TesselateFaceMesh(mesh, faces[i]);
	}


	/*destroy faces*/
	for (i=0; i<face_count; i++) mesh_free(faces[i]);
	free(faces);
	mesh_update_bounds(mesh);
}

void mesh_new_ifs2d(M4Mesh *mesh, SFNode *node)
{
	B_IndexedFaceSet2D *ifs2D = (B_IndexedFaceSet2D *)node;
	mesh_new_ifs_intern(mesh, ifs2D->coord, &ifs2D->coordIndex,
							ifs2D->color, &ifs2D->colorIndex, ifs2D->colorPerVertex,
							NULL, NULL, 0, ifs2D->texCoord, &ifs2D->texCoordIndex);

	mesh->flags |= MESH_IS_2D;
}

void mesh_new_ifs(M4Mesh *mesh, SFNode *node)
{
	B_IndexedFaceSet *ifs = (B_IndexedFaceSet *)node;
	mesh_new_ifs_intern(mesh, ifs->coord, &ifs->coordIndex, ifs->color, &ifs->colorIndex, ifs->colorPerVertex,
						ifs->normal, &ifs->normalIndex, ifs->normalPerVertex, ifs->texCoord, &ifs->texCoordIndex);

	if (ifs->solid) mesh->flags |= MESH_IS_SOLID;
	if (!ifs->ccw) mesh->flags |= MESH_IS_CW;
	/*note we don't use the convex flag since we always translate down to triangles*/
}
