/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2004
 *					All rights reserved
 *
 *  This file is part of GPAC / Scene Rendering sub-project
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

#ifndef VISUALSURFACE_H_
#define VISUALSURFACE_H_

#include "render3d.h"
#include "m4_mesh.h"


/*when highspeeds, amount of subdivisions for bezier and ellipse will be devided by this factor*/
#define HIGH_SPEED_RATIO	2

/*navigation info flags*/
enum
{
	/*no navigation*/
	NAV_NONE = 1,
	/*walk navigation*/
	NAV_WALK = 1<<1,
	/*fly navigation*/
	NAV_FLY = 1<<2,
	/*examine navigation*/
	NAV_EXAMINE = 1<<3,
	/*any navigation (eg, user-interface navigation control allowed)*/
	NAV_ANY = (1<<4),
	/*headlight is on*/
	NAV_HEADLIGHT = (1<<5),
};

typedef struct _visual_surface
{
	/*background, viewport and fog stacks*/
	Chain *back_stack;
	Chain *view_stack;
	Chain *navigation_stack;
	Chain *fog_stack;

	/*renderer*/
	Render3D *render;

	/*FIXME - THIS IS AN INLINE SCENE THING, NOT THE TOP LEVEL ONE*/
	Bool is_pixel_metrics;
	/*surface size as used by form, layout or layer2D*/
	u32 width, height;

	Bool tx_transparent;

	/*navigation info - overwridden by any bindable NavigationInfo node*/
	SFVec3f avatar_size;
	Float visibility, speed;
	/*cf above*/
	u32 navigation_flags;

	/*top-level modelview and projection*/
	M4Matrix modelview;
	M4Matrix projection;
	M4Frustum frustum;
} VisualSurface;

VisualSurface *VS_New(Bool for_3D);
void VS_Delete(VisualSurface *surf);
/*viewport setup and all bindables handling*/
void VS_InitRender(VisualSurface *surf, RenderEffect *eff);


/*base drawable stack:
	BASERENDERSTACK
	mesh: 3D object for drawable - all objects (2D and 3D) use this
*/
typedef struct
{
	SFNode *owner;
	struct scene_renderer *compositor;
	M4Mesh *mesh;
} DrawableStack;

#define	stack_setup(_object, _owner, _compositor)	\
		_object->owner = _owner;		\
		_object->compositor = _compositor; \
		_object->mesh = new_mesh(); \

#define	drawablestack_predestroy(_object) \
	if (_object->mesh) mesh_free(_object->mesh);	\


DrawableStack *new_drawable(SFNode *owner, SceneRenderer *compositor);
void delete_drawable(DrawableStack *d);
/*can be used for Node_PreDestroy with all nodes using only a the base 3D stack*/
void drawable_Node_PreDestroy(SFNode *n);
/*default setup for box/cylinder...*/
void BaseDrawableStack(SceneRenderer *sr, SFNode *node);

/*returns 1 if node is (partially) in view frustrum, false otherwise, and update effect cull flag*/
Bool node_cull_or_render(RenderEffect *eff, M4BBox *bbox);

/*stored at compositor level and in each 2D drawable node*/
typedef struct
{
	/*if set, mesh is the mesh version of the vectorial outline (triangles)
	otherwise, mesh is the line version of the path (lines)
	*/
	Bool is_vectorial;
	/*outline*/
	M4Mesh *outline;
	SFNode *lineProps;
	SFNode *node2D;
	u32 last_update_time;
	/*we also MUST handle width changes due to scale transforms, most content is designed with width=cst 
	whatever the transformation*/
	Float line_scale;
	Bool has_texcoords;
} StrikeInfo;

void delete_strikeinfo(StrikeInfo *info);

/*base 2D stack - same + extensions for striking - ALL 2D NODES SHALL USE IT*/
typedef struct
{
	SFNode *owner;
	struct scene_renderer *compositor;
	M4Mesh *mesh;
	M4Path *path;
	Chain *strike_list;
} stack2D;

typedef struct
{
	/*including alpha*/
	SFColor fill_color, line_color;
	Float alpha; /*1-transp*/
	Float line_alpha; /*1-transp*/
	Bool filled;
	M4PenSettings pen_props;
	/*current modelview scale for line = max abs(scaleX), abs(scaleY) - not set if no line*/
	Float line_scale;

	/*cached lp node - do not touch*/
	SFNode *lp;
	/*cached texture handler - do not touch*/
	void *txh;
	/*cached texture transform*/
	SFNode *tx_trans;
} Aspect2D;
/*returns aspect*/
void VS_GetAspect2D(RenderEffect *eff, Aspect2D *asp);
/*returns line width updated with view scale if needed*/
Float Aspect_GetLineWidth(Aspect2D *asp);

/*setup strike (check for texture)*/
void VS_Set2DStrikeAspect(VisualSurface *surf, Aspect2D *asp);

/*get strike info - recomputes outline if needed*/
StrikeInfo *VS_GetStrikeInfo(stack2D *st, Aspect2D *asp, RenderEffect *eff);
/*get strike info for IFS2D - reset outline if needed but doesn't recomputes it*/
StrikeInfo *VS_GetStrikeInfoIFS(stack2D *st, Aspect2D *asp, RenderEffect *eff);

stack2D *new_stack2D(SFNode *owner, SceneRenderer *compositor);
void delete_stack2D(stack2D *d);
/*use to destroy stacks inheriting from stack2D - doesn't free structure. Must be called
by all 2D nodes*/
void stack2D_predestroy(stack2D *d);
/*can be used for Node_PreDestroy with all nodes using only a the base 2D stack*/
void stack2D_node_predestroy(SFNode *n);
/*reset the path/dlist*/
void stack2D_reset(stack2D *d);
/*default setup for rect/ellipse/circle/etc*/
void BaseStack2D(SceneRenderer *sr, SFNode *node);
/*setup and draw 2D primitive*/
void stack2D_draw(stack2D *d, RenderEffect *eff);
/*setup base 2D stack*/
void stack2D_setup(stack2D *st, SceneRenderer *sr, SFNode *node);

/*setup appearance (material & texture)*/
void VS_setup_appearance(RenderEffect *eff);

/*setup texture - return 0 if texture could not be setup (not available or error)*/
Bool VS_setup_texture(RenderEffect *eff);

/*converts 2D path into a polygon - 
for_outline:
	 0, regular odd/even windining rule with texCoords
	 1, zero-non-zero windining rule without texCoords
	 2, zero-non-zero windining rule with texCoords
*/
void TesselatePath(M4Mesh *mesh, M4Path *path, u32 outline_style);
/*appends given face (and tesselate if needed) to the mesh. Only vertices are used in the face
indices are ignored. */
void TesselateFaceMesh(M4Mesh *mesh, M4Mesh *face);


/*
	till end of file: all 3D specific calls
*/

/*load driver extensions (mainly pixel formats and texture size support)*/
void R3D_LoadExtensions(Render3D *sr);

/*setup surface (hint & co)*/
void VS3D_Setup(VisualSurface *surf);
/*store / restore enabled feature states*/
void VS3D_PushState(VisualSurface *surf);
void VS3D_PopState(VisualSurface *surf);
/*turns 2D AA on/off*/
void VS3D_SetAntiAlias(VisualSurface *surf, Bool bOn);
/*turns headlight on/off*/
void VS3D_SetHeadlight(VisualSurface *surf, Bool bOn);

enum
{
	/*lighting flag*/
	F3D_LIGHT = 1,
	/*blending flag*/
	F3D_BLEND = (1<<1),
	/*color (material3D) flag*/
	F3D_COLOR = (1<<2)
};

/*enable/disable one of the above feature*/
void VS3D_SetState(VisualSurface *surf, u32 flag_mask, Bool setOn);
/*clear surface with given color - alpha should only be specified for composite textures*/
void VS3D_ClearSurface(VisualSurface *surf, SFColor color, Float alpha);

/*matrix mode types*/
enum
{
	MAT_MODELVIEW,
	MAT_PROJECTION,
	MAT_TEXTURE,
};
/*set current matrix type*/
void VS3D_SetMatrixMode(VisualSurface *surf, u32 mat_type);
/*push matrix stack*/
void VS3D_PushMatrix(VisualSurface *surf);
/*reset current matrix (identity)*/
void VS3D_ResetMatrix(VisualSurface *surf);
/*multiply current matrix with given matrix (16 coefs)*/
void VS3D_MultMatrix(VisualSurface *surf, Float *mat);
/*pop matrix stack*/
void VS3D_PopMatrix(VisualSurface *surf);
/*get matrix for the desired mode*/
void VS3D_GetMatrix(VisualSurface *surf, u32 mat_type, Float *mat);
/*setup viewport (vp: top-left, width, height)*/
void VS3D_SetViewport(VisualSurface *surf, M4Rect vp);
/*setup rectangular cliper (clip: top-left, width, height)*/
void VS3D_SetClip2D(VisualSurface *surf, M4Rect clip);
/*draw mesh*/
void VS3D_DrawMesh(VisualSurface *surf, M4Mesh *mesh);
/*only used for ILS/ILS2D or IFS2D outline*/
void VS3D_StrikeMesh(VisualSurface *surf, M4Mesh *mesh, Float width, u32 dash_style);

/*material types*/
enum
{
	/*default material*/
	MATERIAL_NONE,
	MATERIAL_AMBIENT,
	MATERIAL_DIFFUSE,
	MATERIAL_SPECULAR,
	MATERIAL_EMISSIVE,
};
/*set material*/
void VS3D_SetMaterial(VisualSurface *surf, u32 material_type, Float *rgba);
/*set shininess (between 0 and 1.0)*/
void VS3D_SetShininess(VisualSurface *surf, Float shininess);
/*set 2D material (eq to disable lighting and set material (none))*/
void VS3D_SetMaterial2D(VisualSurface *surf, SFColor col, Float alpha);

/*draws image data:
	pos_x, pos_y: top-left pos of image
	width, height: size of image
	pixelformat: image pixel format
	data: image data
	scale_x, scale_y: x & y scale
*/
void VS3D_DrawImage(VisualSurface *surf, Float pos_x, Float pos_y, u32 width, u32 height, u32 pixelformat, char *data, Float scale_x, Float scale_y);


#endif

