/*
 * @(#)AccessControlModel.java	1.7 05/11/09
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */
package com.sun.messaging.jmq.auth.api.server.model;

import java.util.Properties;
import java.security.Principal;
import java.security.AccessControlException;
import javax.security.auth.Subject;
import com.sun.messaging.jmq.auth.api.server.*;

/**
 * An AccessControlModel contains access controls which guards access
 * JMQ resources (connections, destinations)
 *
 * @version     JMQ 2.0
 * @author      Amy Kang
 *
 */

public interface AccessControlModel 
{
    /**
     * @return the type of this access control model
     */
    public String getType();

    /**
     * This method is called immediately after this AccessControlModel
     * has been instantiated and prior to any calls to its other public
     * methods.
     *
	 * @param type The jmq.accesscontrol.type value in authProperties
     * @param authProperties The broker authentication/access control properties
     *
     * @exception AccessControlException
     */
    public void initialize(String type, 
                           Properties authProperties)
                           throws AccessControlException;

    /**
     * load the access control model 
     *
     * @exception AccessControlException
     */
    public void load() throws AccessControlException; 

   /**
    * Check connection permission for the subject
    *
    * @param mqUser The Principal represents the client user 
    *               that associated with the subject
    * @param serviceName The service instance name  (eg. "broker", "admin")
    * @param serviceType The service type for the service instance <BR>
    *                    ("NORMAL" or "ADMIN") <BR>
    * @param subject The subject
    *
    * @exception AccessControlException 
    */
    public void checkConnectionPermission(Principal clientUser,
                                          String serviceName, 
                                          String serviceType,
                                          Subject subject) 
                                          throws AccessControlException ;

   /**
    * Check permission for an operation on a destination for the subject
    *
    * @param clientUser The Principal represents the client user
    *                   that associated with the subject
    * @param serviceName The service instance name  (eg. "broker", "admin")
    * @param serviceType The service type for the service instance  <BR>
    *                    ("NORMAL" or "ADMIN") <BR>
    * @param subject The subject
    * @param operation The operaction ("send", "receive", "browse","publish", "subscribe")
    * @param destination The destination name
    * @param destinationType The destination Type ("queue" or "topic")
    *
    * @exception AccessControlException 
    */
    public void checkDestinationPermission(Principal clientUser,
                                           String serviceName,
                                           String serviceType,
                                           Subject subject,
                                           String operation,
                                           String destination,
                                           String destinationType)
                                           throws AccessControlException; 
}
