/*
 * @(#)MQAddress.java	1.4 02/13/06
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.io;

import java.util.*;
import java.net.*;
import java.io.Serializable;

/**
 * This class represents broker address URL.
 */
public class MQAddress implements Serializable 
{
    public static final String isHostTrusted = "isHostTrusted";

    protected static final String DEFAULT_SCHEME_NAME = "mq";
    protected static final String DEFAULT_HOSTNAME = "localhost";
    protected static final int DEFAULT_PORTMAPPER_PORT = 7676;
    protected static final String DEFAULT_SERVICENAME = "jms";

    private String addr = null;

    protected String schemeName = null;
    protected String addrHost = null;
    protected int port = -1;
    protected String serviceName = null;
    protected boolean isHTTP = false;
    protected Properties props = new Properties();

    protected transient String tostring = null;

    //This flag is used to indicate if 'isHostTrusted' prop is set in
    //the imqAddressList.  If set, it over rides 'imqSSLIsHostTrusted'
    //property.  If not set, 'imqSSLIsHostTrusted' value is used.
    protected transient boolean isSSLHostTrustedSet = false;


    protected MQAddress() {}

    protected void initialize(String addr) 
        throws MalformedURLException
    {
        this.addr = addr;
        this.init();
        this.parseAndValidate();
    }

    private void init() {
        // Set the default value for isHostTrusted attribute.
        props.setProperty(isHostTrusted, "false"); //TCR #3 default to false
    }

    protected void parseAndValidate() throws MalformedURLException {
        String tmp = new String(addr);

        // Find scheme name.
        // Possible values : mq, mqtcp, mqssl, http, https
        schemeName = DEFAULT_SCHEME_NAME;
        int i = tmp.indexOf("://");
        if (i > 0) {
            schemeName = tmp.substring(0, i);
            tmp = tmp.substring(i + 3);
        }

        if (schemeName.equalsIgnoreCase("mq")) {
            /*
             * Typical example -
             * mq://jpgserv:7676/ssljms?isHostTrusted=true
             */
            i = tmp.indexOf('?');
            if (i >= 0) {
                String qs = tmp.substring(i+1);
                parseQueryString(qs);
                tmp = tmp.substring(0, i);
            }

            i = tmp.indexOf('/');
            if (i >= 0) {
                serviceName = tmp.substring(i+1);
                tmp = tmp.substring(0, i);
            }

            i = tmp.indexOf(':');
            if (i >= 0) {
                port = Integer.parseInt(tmp.substring(i+1));
                tmp = tmp.substring(0, i);
            }

            addrHost = tmp;
            if (addrHost == null || addrHost.equals(""))
                addrHost = DEFAULT_HOSTNAME;

            if (port == -1)
                port = DEFAULT_PORTMAPPER_PORT;

            if (serviceName == null || serviceName.equals(""))
                serviceName = getDefaultServiceName();
        }
        else if (schemeName.equalsIgnoreCase("mqssl") ||
            schemeName.equalsIgnoreCase("mqtcp")) {
            /*
             * Typical example -
             * mqtcp://jpgserv:12345/jms
             * mqssl://jpgserv:23456/ssladmin
             */
            i = tmp.indexOf('?');
            if (i >= 0) {
                String qs = tmp.substring(i+1);
                parseQueryString(qs);
                tmp = tmp.substring(0, i);
            }

            i = tmp.indexOf('/');
            if (i >= 0) {
                serviceName = tmp.substring(i+1);
                tmp = tmp.substring(0, i);
            }

            i = tmp.indexOf(':');
            if (i >= 0) {
                port = Integer.parseInt(tmp.substring(i+1));
                tmp = tmp.substring(0, i);
            }

            addrHost = tmp;
            if (addrHost == null || addrHost.equals("")) {
                throw new MalformedURLException(
                    "Illegal address. Hostname missing : " + addr);
            }

            if (port == -1) {
                throw new MalformedURLException(
                    "Illegal address. Port missing : " + addr);
            }
        }
        else if (schemeName.equalsIgnoreCase("http") ||
            schemeName.equalsIgnoreCase("https")) {
            isHTTP = true;
            return;
        }
        else {
            throw new MalformedURLException(
                "Illegal address. Unknown address scheme : " + addr);
        }
    }

    protected void parseQueryString(String qs) throws MalformedURLException {
        String tmp = new String(qs);

        while (tmp.length() > 0) {
            String pair = tmp;

            int i = tmp.indexOf('&');
            if (i >= 0) {
                pair = tmp.substring(0, i);
                tmp = tmp.substring(i+1);
            }
            else {
                tmp = "";
            }

            int n = pair.indexOf('=');
            if (n <= 0)
                throw new MalformedURLException(
                    "Illegal address. Bad query string : " + addr);

            String name = pair.substring(0, n);
            String value = pair.substring(n+1);
            props.setProperty(name, value);

            if ( isHostTrusted.equals(name) ) {
                isSSLHostTrustedSet = true;
            }
        }
    }

    public boolean isServicePortFinal() {
        return (isHTTP || schemeName.equalsIgnoreCase("mqtcp") ||
            schemeName.equalsIgnoreCase("mqssl"));
    }

    public String getProperty(String pname) {
        return props.getProperty(pname);
    }

    public boolean getIsSSLHostTrustedSet() {
        return this.isSSLHostTrustedSet;
    }



    public String getSchemeName() {
        return schemeName;
    }

    public String getHostName() {
        return addrHost;
    }

    public int getPort() {
        return port;
    }

    public String getServiceName() {
        return serviceName;
    }

    public boolean getIsHTTP() {
        return isHTTP;
    }

    public String getURL() {
        return addr;
    }

    public String toString() {

        if (tostring != null)
            return tostring;

        if (isHTTP) {
            tostring = addr;
            return addr;
        }

        tostring = schemeName + "://" + addrHost + ":" + port + "/" + serviceName;
        return tostring;
    }

    public int hashCode() {
        return toString().hashCode();
    }

    public boolean equals(Object obj) {
        if (! (obj instanceof MQAddress)) {
            return false;
        }
        return this.toString().equals(((MQAddress)obj).toString());
    }

    public String getDefaultServiceName()  {
	return (DEFAULT_SERVICENAME);
    }


     /**
     * Parses the given MQ Message Service Address and creates an
     * MQAddress object.
     */
    public static MQAddress getMQAddress(String addr) 
        throws MalformedURLException {
        MQAddress ret = new MQAddress();
        ret.initialize(addr);
        return ret;
    }
       

}
