/*
 * @(#)SysMessageID.java	1.13 05/11/30
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 *
 */

package com.sun.messaging.jmq.io;

import java.io.*;
import com.sun.messaging.jmq.net.IPAddress;
import com.sun.messaging.jmq.util.Bits;

/**
 * This class encapsulates a system message id. This message id uniquely
 * identifies a message in the JMQ system. It consists of four values:
 * the message sequence number, source IP address, source port and timestamp.
 * Note that this is not the JMS MessageID that can be set by the client.
 * <P>
 * The Swift packet format specifies that the IP address be in IPv6 (128 bit)
 * format, so we canonicalize all addresses to IPv6. Initially we will only
 * be dealing with IPv4 addresses since that is all the JDK supports -- those
 * get translated into an "IPv4-mapped IPv6" address.
 *
 * @version 1.13 05/11/30
 */
public class SysMessageID implements Cloneable {

    /**
     * Size of a SysMessageID when externalized.
     */
    public static final int ID_SIZE = 4 + 4 + 8 + IPAddress.IPV6_SIZE;

    protected int  sequence;
    protected int  port;
    protected long timestamp;
    protected IPAddress ip;

    private String uniqueName = null;
    private boolean dirty = true;

    /**
     * Construct an unititialized system message ID. It is assumed
     * the caller will set the fields either explicitly or via
     * readID()
     */
    public SysMessageID() {
	ip = new IPAddress();
	clear();
    }

    /**
     * Clears the message id
     */
    public void clear() {
	sequence = port = 0;
	timestamp = 0;
	ip.clear();
        dirty = true;
    }

    /**
     * Check if the passed SysMessageID equals this one
     *
     * @param    o    the object to compare
     *
     * @return   true if the objects are equivalent, else false
     */
    public boolean equals(Object o) {

        if (this == o) return true;

        if (!(o instanceof SysMessageID)) {
            return false;
        }

        SysMessageID id = (SysMessageID)o;

	/*
	 * We try to optimize the comparison so if it fails it fails 
	 * quickly. Most message id comparisions will be done when 
	 * the router handles acknowledgements from a client. In that
	 * case the sequence number is the most likely to determine 
	 * uniqueness
	 */
	return (
	     this.sequence == id.sequence &&
	    this.timestamp == id.timestamp &&
	         this.port == id.port &&
	         ip.equals(id.ip)
            );
    }

    public int hashCode() {
	// This should generate enough uniqueness without messing with
	// the IP address.
	return (int)timestamp * port * sequence;
    }


    /**
     * Return a string description. The string will be of the format:
     *
     *    nnn-nnn.nnn.nnn.nnn-nnnn-nnnnnnnnnnnn
     *    ^   ^               ^    ^
     *    seq IP              port timestamp
     *
     * @return    String description of message id
     *
     */
    public String toString() {

	/*
	 * Since the data fields are public we allocate a new string each
	 * time just in case somebody change one.
	 */
	return (
	    sequence + "-" +
	    ip.toString() + "-" + 
	    port + "-" +
	    timestamp
	);
    }

    /**
     * Generates a unique string that is suitable for use
     * as a key.
     */
    public String getUniqueName() {
        if (uniqueName == null || dirty) {
            uniqueName = String.valueOf(sequence) +
			 String.valueOf(timestamp) +
			 String.valueOf(port) +
                IPAddress.rawIPToString(ip.getAddress(), false);
            dirty = false;
        }
        return uniqueName;
    }

    public int getSequence() {
	return sequence;
    }

    public int getPort() {
	return port;
    }

    public long getTimestamp() {
	return timestamp;
    }

    /**
     * Get the IP address. 
     *
     * @return   The raw IP address. This will always be an IPv6 (128 bit)
     *           address. It will contain an embedded Mac address if one
     *           was provided to setIPAddress().
     */
    public byte[] getIPAddress() {
	return ip.getAddress();
    }

    public void setSequence(int n) {
        dirty = true;
	sequence = n;
    }

    public void setPort(int n) {
        dirty = true;
	port = n;
    }

    public void setTimestamp(long n) {
        dirty = true;
	timestamp = n;
    }

    /**
     * Set the IP address. 
     *
     * @param    newip    IP address in network byte order. This can be the
     *                    buffer returned by InetAddress.getAddress().
     *                    A null value results in the IP address being cleared.
     *
     */
    public void setIPAddress(byte[] newip)
	throws IllegalArgumentException {

	ip.setAddress(newip);
        dirty = true;
    }

    /**
     * Set the IP address and a 48 bit mac addres.
     *
     * @param    newip    IP address in network byte order. This can be the
     *                    buffer returned by InetAddress.getAddress().
     *                    A null value results in the IP address being cleared.
     *
     * @param   mac       6 byte MAC address (or random psuedo address. You
     *                    can get it from IPAddress.getRandomMac())
     *
     */
    public void setIPAddress(byte[] newip, byte[] mac)
	throws IllegalArgumentException {

	ip.setAddress(newip);

        if (mac != null && ip.getType() != IPAddress.IPV6) {
            // Only set mac if we are not IPV6
	    ip.setMac(mac);
        }
        dirty = true;
    }

    /**
     * Write the ID to the specified DataOutput.
     * The format of the written data will be:
     *
     *   <PRE>
     *    0                   1                   2                   3
     *   |0 1 2 3 4 5 6 7|8 9 0 1 2 3 4 5|6 7 8 9 0 1 2 3|4 5 6 7 8 9 0 1|
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   |                                                               |
     *   +                        timestamp                              +
     *   |                                                               |
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   |                                                               |
     *   +                                                               +
     *   |                     source IP                                 |
     *   +                                                               +
     *   |                                                               |
     *   +                                                               +
     *   |                                                               |
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   |                      source port                              |
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   |                     sequence number                           |
     *   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
     *   </PRE>
     *
     * This format matches the format specified in the ACKNOWLEDGE message
     * of the JMQ protocol spec.
     *
     * We don't use ObjectSerialization to do this because we don't want
     * the class version cruft that would be prefixed to the data.
     *
     * @param    out    DataOutput to write ID to
     */
    public void writeID(DataOutput out)
	throws IOException {

	out.writeLong(timestamp);
	ip.writeAddress(out);
	out.writeInt(port);
	out.writeInt(sequence);

        if (out instanceof DataOutputStream) {
            ((DataOutputStream)out).flush();
        }
    }

    /**
     * Returns the ID in a raw format. The format will be as described
     * in writeID, and the size of the byte[] will be ID_SIZE
     */
    public byte[] getRawID() {
        byte[] buf = new byte[ID_SIZE];

        int i = 0;
        i = Bits.put(buf, i, timestamp);
        i = Bits.put(buf, i, ip.getAddressUnsafe());
        i = Bits.put(buf, i, port);
            Bits.put(buf, i, sequence);

        return buf;
    }

    /**
     * Read the ID from the specified DataInput. The format of
     * the data is assumed to match that generated by writeID.
     *
     * @param    in    DataInput to write ID to
     *
     */
    public void readID(DataInput in)
	throws IOException {

	timestamp = in.readLong();
	            ip.readAddress(in);
	port      = in.readInt();
	sequence  = in.readInt();
    }

    /**
     * Make a deep copy of this object
     */
    public Object clone() {
	try {
	    SysMessageID newID = (SysMessageID)super.clone();
	    newID.ip           = (IPAddress)this.ip.clone();
	    return newID;
	} catch (CloneNotSupportedException e) {
	    // Should never get this, but don't fail silently
	    System.out.println("SysMessageID: Could not clone: " + e);
	    return null;
	}
    }
}
