/**
 * @(#)VRecordMap.java	1.3 12/18/03
 *
 * Copyright 2003 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.io;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.*;

/**
 * A VRecordMap encapsulates a slice of mapped buffer allocated by VRFileMap.
 */
public class VRecordMap extends VRecord {

    private VRFileMap vrfile;
    private ByteBuffer bbuf;
    private ByteBuffer databuf; // slice after the header

    private MappedByteBuffer parent;

    // instantiate with an existing record (sanity checked by caller)
    VRecordMap(VRFileMap v, MappedByteBuffer p, ByteBuffer buf) {
	vrfile = v;
	parent = p;
	bbuf = buf;

	// read header
	magic = bbuf.getInt();
	capacity = bbuf.getInt();
	state = bbuf.getShort();
	cookie = bbuf.getShort();

	bbuf.limit(capacity);
	bbuf.position(VRFile.RECORD_HEADER_SIZE);
	databuf = bbuf.slice();
    }

    // instantiate with an uninitialized record
    VRecordMap(VRFileMap v, MappedByteBuffer p, ByteBuffer buf, int size) {
	vrfile = v;
	parent = p;
	bbuf = buf;

	capacity = size;
	state = VRFile.STATE_ALLOCATED;

	// write header
	bbuf.putInt(magic);
	bbuf.putInt(capacity);
	bbuf.putShort(state);
	bbuf.putShort(cookie);

	bbuf.limit(capacity);
	bbuf.position(VRFile.RECORD_HEADER_SIZE);
	databuf = bbuf.slice();
    }

    /**
     * Get the record buffer. Its 'capacity' may be larger than what
     * was requested. Its 'limit' will match what was requested.
     * Whatever is written to the buffer may be written to the backing
     * file, but is not guaranteed to be until force() is called
     * or the VRfile is closed.
     */
    public ByteBuffer getBuffer() {
	return databuf;
    }

    /*
     * Force any modifications made to the buffer to be written
     * to physical storage.
     */
    public void force() throws IOException {
	if (VRFile.DEBUG) {
	    System.out.println("will do force on "+parent);
	}

	parent.force();
    }

    public void setCookie(short c) throws IOException {
	this.cookie = c;
	bbuf.putShort(VRFile.RECORD_COOKIE_OFFSET, cookie);

	if (vrfile.getSafe()) {
	    force();
	}
    }

    public short getCookie() {
	return cookie;
    }

    public String toString() {
	return "VRecordMap: "+bbuf.toString();
    }

    MappedByteBuffer getParent() {
	return parent;
    }

    void free() {
	state = VRFile.STATE_FREE;
	bbuf.putShort(VRFile.RECORD_STATE_OFFSET, state);
	bbuf.putShort(VRFile.RECORD_COOKIE_OFFSET, VRFile.RESERVED_SHORT);
	databuf.rewind();
    }

    void allocate(short s) {
	state = s;
	bbuf.putShort(VRFile.RECORD_STATE_OFFSET, state);
    }
}


