/*
 * @(#)PortMapperClient.java	1.28 05/05/23
 *
 * Copyright 2003 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsclient;

import javax.jms.*;
import com.sun.messaging.jmq.Version;
import com.sun.messaging.jmq.io.PortMapperTable;
import com.sun.messaging.jmq.io.PortMapperEntry;
import com.sun.messaging.jmq.io.Packet;
import com.sun.messaging.jmq.io.ReadOnlyPacket;
import com.sun.messaging.AdministeredObject;
import com.sun.messaging.ConnectionConfiguration;
import java.io.*;
import java.net.Socket;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Hashtable;
import java.util.Map;

/**
 *
 * This class provide a way for the client to recover itself if the
 * connection goes away.
 *
 * <p>The API user is transparent to the recovering attempt.  All
 * consumers are registered to the broker with their original IDs.
 * All producers are added with the same parameters.
 *
 *
 * @version     JMQ 2.0
 * @author      Chiaming Yang
 *
 */


public class PortMapperClient {

    protected ConnectionImpl connection = null;
    protected PortMapperTable portMapperTable = null;

    protected boolean useMQAddress = false;
    protected MQAddress addr = null;

    private static final Version version = com.sun.messaging.jmq.jmsclient.ConnectionImpl.version;

    private boolean debug = Debug.debug;

    public PortMapperClient (ConnectionImpl connection) throws JMSException{
        this.connection = connection;
        init();
    }

    public PortMapperClient (MQAddress addr, ConnectionImpl connection)
        throws JMSException {
        this.addr = addr;
        this.useMQAddress = true;
        this.connection = connection;
        init();
    }

    public int getPortForProtocol(String protocol){
        String type = connection.getConnectionType();
        return getPort(protocol, type, null);
    }

    //bug 4959114.
    public int
    getPortForService(String protocol, String service) throws JMSException {
        String type = connection.getConnectionType();
        int port = getPort(protocol, type, service);

        if ( port == -1 ) {
            String errorString =
            AdministeredObject.cr.getKString(AdministeredObject.cr.X_UNKNOWN_BROKER_SERVICE, service);
            JMSException jmse =new com.sun.messaging.jms.JMSException
            (errorString, AdministeredObject.cr.X_UNKNOWN_BROKER_SERVICE);

            ExceptionHandler.throwJMSException(jmse);

        }

        return port;
    }

    //bug 4959114.
    private int getPort(String protocol, String type, String servicename) {
        //int port = 25374;
        int port = -1;
        Map table = portMapperTable.getServices();
        PortMapperEntry pme = null;

        Iterator it = table.values().iterator();

        while (it.hasNext()){
            pme = (PortMapperEntry) it.next();
            if (pme.getProtocol().equals(protocol)){
                if (pme.getType().equals(type)){
                    if (servicename == null){
                        port = pme.getPort();
                        break;
                    } else {
                        if (pme.getName().equals(servicename)){
                            port = pme.getPort();
                            break;
                        }
                    }
                }
            }
        }

        return port;
    }

    protected void init() throws JMSException {
        try {
            readBrokerPorts();

            checkBrokerVersion();
        } catch (JMSException jmse) {

            String str = this.getHostName() + ":" + this.getHostPort();
            connection.setLastContactedBrokerAddress(str);

            ExceptionHandler.throwJMSException(jmse);
        }
    }

    protected void checkBrokerVersion() throws JMSException {
        String bkrversion = portMapperTable.getBrokerVersion();
        String clientMVersion = version.getImplementationVersion();

        // Raptor (3.5) clients can talk to a Falcon (3.0) broker.
        if (Version.compareVersions(bkrversion, "3.0") < 0) {
            String errorString = AdministeredObject.cr.getKString(
                AdministeredObject.cr.X_VERSION_MISMATCH,
                 clientMVersion, bkrversion);

            JMSException jmse =
            new com.sun.messaging.jms.JMSException
            (errorString,AdministeredObject.cr.X_VERSION_MISMATCH);

            ExceptionHandler.throwJMSException(jmse);
        }

        // Use Packet version 200 for brokers older than 3.5
        if (Version.compareVersions(bkrversion, "3.0.1", false) < 0) {
            ReadOnlyPacket.setDefaultVersion(Packet.VERSION2);
        }
    }

    private String getHostName() {
        if (useMQAddress)
            return addr.getHostName();

        return connection.getProperty(
            ConnectionConfiguration.imqBrokerHostName);
    }

    public int getHostPort() {
        if (useMQAddress)
            return addr.getPort();

        String prop = connection.getProperty(
            ConnectionConfiguration.imqBrokerHostPort);
        return Integer.parseInt(prop);
    }

    protected void readBrokerPorts() throws JMSException {

        String host = getHostName();
        //port mapper port
        int port = getHostPort();

        if ( debug ) {
            Debug.println("PortMapper connecting to host: " + host + "  port: " + port);
        }

        try {
            String version =
                String.valueOf(PortMapperTable.PORTMAPPER_VERSION) + "\n";

            Socket socket = new Socket (host, port);
            InputStream is = socket.getInputStream();
            OutputStream os = socket.getOutputStream();

            // Write version of portmapper we support to broker
            try {
                os.write(version.getBytes());
                os.flush();
            } catch (IOException e) {
                // This can sometimes fail if the server already wrote
                // the port table and closed the connection
            }

            portMapperTable = new PortMapperTable();
            portMapperTable.read(is);

            is.close();
            socket.close();

        } catch ( Exception e ) {
            connection.getExceptionHandler().handleConnectException (
                e, host, port);
        }
    }

    public static void main (String args[]) {
        try {
            PortMapperClient pmc = new PortMapperClient (null);
            String protocol = "tcp";

            String prop = System.getProperty("protocol");
            if ( prop != null ) {
                protocol = prop;
            }

            int port = pmc.getPortForProtocol(protocol);

            if ( Debug.debug ) {
                Debug.println ("port = " + port );
            }

        } catch (Exception e) {
            Debug.printStackTrace(e);
        }
    }
}

