/*
 * @(#)XASessionImpl.java	1.5 04/06/08
 *
 * Copyright 2000 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsclient;

import javax.jms.*;
import javax.transaction.xa.XAResource;

import com.sun.messaging.AdministeredObject;
import com.sun.messaging.jms.ra.ManagedConnection;

/**
 * An XASession provides a regular Session which can be used to
 * create QueueReceivers, QueueSenders and QueueBrowsers (optional).
 *
 * <P>XASession extends the capability of Session by adding access to a JMS
 * provider's support for JTA (optional). This support takes the form of a
 * <CODE>javax.transaction.xa.XAResource</CODE> object. The functionality of
 * this object closely resembles that defined by the standard X/Open XA
 * Resource interface.
 *
 * <P>An application server controls the transactional assignment of an
 * XASession by obtaining its XAResource. It uses the XAResource to assign
 * the session to a transaction; prepare and commit work on the
 * transaction; etc.
 *
 * <P>An XAResource provides some fairly sophisticated facilities for
 * interleaving work on multiple transactions; recovering a list of
 * transactions in progress; etc. A JTA aware JMS provider must fully
 * implement this functionality. This could be done by using the services
 * of a database that supports XA or a JMS provider may choose to implement
 * this functionality from scratch.
 *
 * <P>A client of the application server is given what it thinks is a
 * regular JMS Session. Behind the scenes, the application server controls
 * the transaction management of the underlying XASession.
 *
 * @version     04/06/08 1.5
 * @author      George Tharakan
 *
 * @see         javax.jms.XASession javax.jms.XASession
 * @see         javax.jms.XAQueueSession javax.jms.XAQueueSession
 */

public class XASessionImpl extends UnifiedSessionImpl implements XASession {

    private XAResourceImpl xar;

    public XASessionImpl
            (ConnectionImpl connection, boolean transacted, int ackMode) throws JMSException {

        super (connection, transacted, ackMode);
        xar = new XAResourceImpl(this);
    }    
 
    public XASessionImpl
            (ConnectionImpl connection, boolean transacted, int ackMode, ManagedConnection mc) throws JMSException {

        super (connection, transacted, ackMode, mc);
        xar = new XAResourceImpl(this);
    }    
 
    /**
     * Return an XA resource to the caller.
     * 
     * @return an XA resource to the caller
    */
    public XAResource
    getXAResource() {
         return (XAResource) xar;
    }
 
    /**
     * Is the session in transacted mode?
     *
     * @return true
     *
     * @exception JMSException if JMS fails to return the transaction
     *                         mode due to internal error in JMS Provider.
     */
    public boolean
    getTransacted() throws JMSException {
        checkSessionState();
        if (xaTxnMode) {
            return  true;
        }
        return isTransacted;
    }
 
    /**
     * Throws TransactionInProgressException since it should not be called
     * for an XASession object.
     *
     * @exception TransactionInProgressException if method is called on
     *                         a XASession.
     *
     */
    public void
    commit() throws JMSException {
        if (xaTxnMode) {
            String error = AdministeredObject.cr.getKString(AdministeredObject.cr.X_COMMIT_ROLLBACK_XASESSION);
            throw new TransactionInProgressException(error);
        }
        super.commit();
    }
 
    /**
     * Throws TransactionInProgressException since it should not be called
     * for an XASession object.
     *
     * @exception TransactionInProgressException if method is called on
     *                         a XASession.
     *
     */
    public void
    rollback() throws JMSException {
        if (xaTxnMode) {
            String error = AdministeredObject.cr.getKString(AdministeredObject.cr.X_COMMIT_ROLLBACK_XASESSION);
            throw new TransactionInProgressException(error);
        }
        super.rollback();
    }
 
    /**
     * Get the queue session associated with this XAQueueSession.
     *  
     * @return the queue session object.
     *  
     * @exception JMSException if a JMS error occurs.
     */ 
    public Session
    getSession() throws JMSException {
        return (Session) this;
    }

}
 
