/*
 * @(#)EventHandler.java	1.7 05/09/13
 *
 * Copyright 2003 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */
package com.sun.messaging.jmq.jmsclient.notification;

import java.util.Date;
import javax.jms.JMSException;
import javax.jms.ExceptionListener;

import com.sun.messaging.jmq.jmsclient.ConnectionImpl;
import com.sun.messaging.jmq.jmsclient.SequentialQueue;
import com.sun.messaging.jmq.jmsclient.resources.ClientResources;

import com.sun.messaging.jmq.jmsclient.Debug;
import com.sun.messaging.jms.notification.*;

import com.sun.messaging.ConnectionConfiguration;

/**
 * MQ Client Runtime event handler.
 *
 * The event handler receives event from the client runtime and notifies the
 * connectiion's event listener.
 * <p>
 *
 *
 * @author Chiaming Yang
 * @version 1.0
 */
public class EventHandler implements Runnable {

    private ConnectionImpl connection = null;

    private Thread handlerThread = null;
    
    protected static final String iMQEventHandler = "iMQEventHandler-";

    private SequentialQueue eventQueue = null;

    private boolean isClosed = false;

    /**
     * This flag is used to prevent duplicate delivery of ConnectionClosedEvent.
     * This is set to true after each delivery for the closed event.  It is
     * set to false after reconnected.
     */
    private boolean closedEventdelivered = false;

    private ExceptionListener exlistener = null;

    public static long WAIT_TIMEOUT = 2 * 60 * 1000; // 2 minutes.

    private boolean debug = Debug.debug;

    private static  boolean debugEvent =
        Boolean.getBoolean("imq.debug.notification");

    public EventHandler (ConnectionImpl conn) {
        this.connection = conn;

        init();
    }

    private void init() {
        eventQueue = new SequentialQueue (2);
    }

    private synchronized void onEvent (Event event) {

        if ( debugEvent ) {
            Debug.getPrintStream().println(new Date() +
        "-- event triggerred, code = " + event.getEventCode() +
        ", msg = " + event.getEventMessage() );
        }

        if ( isClosed ) {
            return;
        }

        eventQueue.enqueue(event);

        if (handlerThread == null) {
            createHandlerThread();
        }

        notifyAll();
    }

    public synchronized void close() {
        isClosed = true;
        notifyAll();
    }

    private void createHandlerThread() {

        synchronized (this) {

            if (handlerThread == null) {

                handlerThread = new Thread(this);
                
                if ( connection.hasDaemonThreads() ) {
                	handlerThread.setDaemon(true);
                }
                
                handlerThread.setName(iMQEventHandler + connection.getLocalID());
                
                handlerThread.start();
            }
        }

    }

    public void run() {

        boolean timeoutExit = false;
        boolean keepRunning = true;

        while ( keepRunning ) {

            timeoutExit = false;

            synchronized (this) {

                if ( shouldWait() ) {
                    try {
                        wait(WAIT_TIMEOUT);
                    } catch (InterruptedException inte) {
                        ;
                    }
                }
            }

            if ( isClosed ) {
                /**
                 * when close() is called, we simply exit the handler.
                 */
                return;
            } else if (eventQueue.isEmpty()) {
                //timeout occurred.
                timeoutExit = true;
            } else {
                /**
                 * get the event from the queue.
                 */
                Event event = (Event) eventQueue.dequeue();

                /**
                 * call connection exception listener if this is a
                 * ConnectionExitEvent
                 */
                if ( event instanceof ConnectionExitEvent ) {
                   deliverException (event);
                } else {
                    /**
                     * regular connection event.
                     */
                    deliverEvent(event);
                }
            }

            /**
             * check if we need to continue.
             */
            keepRunning = shouldContinue( timeoutExit);
        }
    }

    private void deliverException (Event event) {

        try {

            if (exlistener != null  && isClosed == false) {

                ConnectionExitEvent exitEvent = (ConnectionExitEvent) event;

                JMSException jmse = exitEvent.getJMSException();

                exlistener.onException(jmse);

                if ( debugEvent ) {
                   Debug.getPrintStream().println (new Date() +
                   " Exception is delivered to the listener: " + jmse);
                }


            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            isClosed = true;
        }
    }

    private boolean shouldWait() {

        if ( eventQueue.isEmpty() && (connection.getEventListener() != null)
             && (isClosed == false) ) {
            return true;
        } else {
            return false;
        }
    }

    private synchronized boolean shouldContinue (boolean timeoutExit) {

        boolean keepRunning = true;
        //exit if closed or timeout.
        if ( isClosed || (timeoutExit && eventQueue.isEmpty()) ) {

            this.handlerThread = null;

            keepRunning = false;

        }

        return keepRunning;
    }

    private void deliverEvent(Event event) {

        EventListener listener = connection.getEventListener();

        try {

            if ( shouldDeliver (listener, event) ) {
                listener.onEvent(event);

                if ( debugEvent ) {
                   Debug.getPrintStream().println( new Date() +
                   "*** Delivered event, code = " + event.getEventCode() +
                   ", msg = " + event.getEventMessage() );
                }

            }

        } catch (Throwable t) {
            t.printStackTrace();
        } finally {
            postEventProcess(event);
        }
    }

    private boolean shouldDeliver (EventListener listener, Event event) {

        boolean canDeliver = true;

        if ( listener == null || isClosed ) {
            canDeliver = false;
        } else {
            if ( event instanceof ConnectionClosedEvent && closedEventdelivered ) {
               canDeliver = false;
            }
        }

        return canDeliver;
    }

    /**
     * Perform action based on event type and connection state.
     */
    private void postEventProcess (Event event) {
        String eid = event.getEventCode();

        if ( event instanceof ConnectionReconnectedEvent ) {
            //wake up waiting threads on reconnecting
            connection.setReconnecting(false);
            //reset flag
            closedEventdelivered = false;

            startConnection();

        } else if ( event instanceof ConnectionClosedEvent ) {
            //set delivered flag
            closedEventdelivered = true;
        }
    }

    private void startConnection() {

        try {
            //if the connection is not in stop mode, restart the connection
            if ( connection.getIsStopped() == false ) {
                connection.getProtocolHandler().start();
            }

        } catch (Exception e) {

            if ( this.debug ) {
               e.printStackTrace( Debug.getPrintStream() );
            }
        }
    }

    public void triggerConnectionClosedEvent
        (String evCode, JMSException jmse) {

        if ( connection.getEventListener() == null ) {
            return;
        }

        String evMessage = ClientResources.getResources().getKString(evCode,
                           connection.getLastContactedBrokerAddress());

        if ( evCode.equals(ClientResources.E_CONNECTION_CLOSED_NON_RESPONSIVE)) {
            evMessage = evMessage + ", "  +
                        ConnectionConfiguration.imqAckTimeout + ": " +
                        connection.getTimeout();
        }

        ConnectionClosedEvent event =
        new ConnectionClosedEvent (connection, evCode, evMessage, jmse);

        this.onEvent(event);
    }

    public void triggerConnectionClosingEvent
        (String evCode, long timePeriod) {

        if ( connection.getEventListener() == null ) {
            return;
        }

        String millisecs = String.valueOf(timePeriod);

        String evMessage = ClientResources.getResources().getKString
                           (evCode, millisecs, connection.getLastContactedBrokerAddress());

        ConnectionClosingEvent event =
        new ConnectionClosingEvent (connection, evCode, evMessage, timePeriod);

        this.onEvent(event);
    }


    public void triggerConnectionReconnectFailedEvent (JMSException jmse, String brokerAddr) {

        if ( connection.getEventListener() == null ) {
            return;
        }

        String evCode =
        ConnectionReconnectFailedEvent.CONNECTION_RECONNECT_FAILED;

        String evMessage =
        ClientResources.getResources().getKString(evCode, brokerAddr);

        ConnectionReconnectFailedEvent event =
        new ConnectionReconnectFailedEvent (connection, evCode, evMessage, jmse);

        this.onEvent(event);
    }

    public void triggerConnectionReconnectedEvent () {

        if ( connection.getEventListener() == null ) {
            return;
        }

        String brokerAddr = connection.getBrokerAddress();

        String evCode =
        ConnectionReconnectedEvent.CONNECTION_RECONNECTED;

        String evMessage =
        ClientResources.getResources().getKString(evCode, brokerAddr);

        ConnectionReconnectedEvent event =
        new ConnectionReconnectedEvent (connection, evCode, evMessage);

        this.onEvent(event);
    }

    public void triggerConnectionExitEvent(JMSException jmse, ExceptionListener listener) {

        try {

            if ( connection.getEventListener() == null ) {
                return;
            }

            this.exlistener = listener;

            //this event is for MQ internal use only.  This triggered the event
            //handler to call connection exception handler.
            ConnectionExitEvent event =
            new ConnectionExitEvent(connection, ConnectionExitEvent.CONNECTION_EXIT,
                                        jmse.getMessage(), jmse);

            this.onEvent(event);

        } catch (Exception ex) {
            ex.printStackTrace( Debug.getPrintStream() );
        }

    }

    public void triggerConnectionAddressListChangedEvent (String addrList) {

        if ( connection.getEventListener() == null ) {
            return;
        }

        String evCode =
        BrokerAddressListChangedEvent.CONNECTION_ADDRESS_LIST_CHANGED;

        String evMessage =
        ClientResources.getResources().getKString(evCode, addrList);

        BrokerAddressListChangedEvent event =
        new BrokerAddressListChangedEvent (connection, evCode, evMessage, addrList);

        this.onEvent(event);
    }

}
