/*
 * @(#)JMQFileUserRepository.java	1.29 05/12/06
 *
 * Copyright 2003 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */
package com.sun.messaging.jmq.jmsserver.auth.file;

import java.io.*;
import java.util.*;
import javax.security.auth.Subject;
import javax.security.auth.Refreshable;
import javax.security.auth.login.LoginException;
import com.sun.messaging.jmq.util.MD5;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.util.StringUtil;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.auth.AccessController;
import com.sun.messaging.jmq.auth.jaas.MQUser;
import com.sun.messaging.jmq.auth.jaas.MQGroup;
import com.sun.messaging.jmq.auth.jaas.MQAdminGroup;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.auth.api.server.model.*;

/**
 * JMQ passwd user repository plugin
 *
 * @version     JMQ 2.0
 * @author      Amy Kang
 */
public class JMQFileUserRepository implements UserRepository
{
    public static final String TYPE = "file";

    public static final String PROP_FILENAME_SUFFIX = TYPE + ".filename";
    public static final String PROP_DIRPATH_SUFFIX = TYPE + ".dirpath";
    public static final String DEFAULT_PW_FILENAME = "passwd";

    public static boolean DEBUG = false;
	private transient Logger logger = Globals.getLogger();

    private static String ADMINGROUP = "admin";
    private String authType;
    private Properties authProps = null;

    public void JMQFileUserRepository() { }

    public String getType() {
        return TYPE;
    }

    public void open(String authType, Properties authProperties,
                     Refreshable cacheData) throws LoginException {
        this.authType = authType;
        this.authProps = authProperties;
    }

    /**
     * Find the user in the repository and compare the credential with
     * the user's  credential in database
     *
     * @param user the user name
     * @param credential password (String type) for "basic" is the password
     * @param extra null for basic, nonce if digest
     * @param matchType must be "basic" or "digest"
     *        
     * @return the authenticated subject  <BR>
     *         null if no match found <BR>
     *
     * @exception LoginException
     */
    public Subject findMatch(String user, Object credential,
                             Object extra, String matchType)
                             throws LoginException {
        if (matchType == null ||
            (!matchType.equals(AccessController.AUTHTYPE_BASIC) &&
             !matchType.equals(AccessController.AUTHTYPE_DIGEST))) {
            String matchtyp = (matchType == null) ? "null": matchType;
            String[] args = {matchtyp, authType, getType(),
                AccessController.AUTHTYPE_BASIC+":"+AccessController.AUTHTYPE_DIGEST};
            throw new LoginException(Globals.getBrokerResources().getKString(
                BrokerResources.X_UNSUPPORTED_USER_REPOSITORY_MATCHTYPE, args));
        }

        HashMap userPTable = new HashMap();
        HashMap userRTable = new HashMap();
        try {
            loadUserTable(userPTable, userRTable);
        } catch (IOException e) {
            logger.logStack(logger.ERROR, e.getMessage(), e);
            userPTable = null;
            userRTable = null;
            throw new LoginException(e.getMessage());
        }

        Subject subject = null; 
        if (matchType.equals(AccessController.AUTHTYPE_BASIC)) {
            subject = basicFindMatch(user, (String)credential, userPTable, userRTable);
        }
        else if (matchType.equals(AccessController.AUTHTYPE_DIGEST)) {
            subject = digestFindMatch(user, (String)credential, (String)extra, userPTable, userRTable); 
        }
        userPTable = null;
        userRTable = null;
        return subject;
    }
      
    private Subject basicFindMatch(String user, String userpwd, 
                      HashMap userPTable, HashMap userRTable) throws LoginException {
        if (DEBUG) {
        logger.log(Logger.DEBUG, "basic matching "+user+":"+userpwd);
        }
        Subject subject = null;
        String passwd = (String)userPTable.get(user);
        if (passwd != null) {
            if (passwd.equals(MD5.getHashString(user+":"+userpwd))) {
                subject = getSubject(user, userRTable);
            }
        }
        return subject;
    }

    private Subject digestFindMatch(String user, String credential, String nonce,
                      HashMap userPTable, HashMap userRTable) throws LoginException {
        if (DEBUG) {
        logger.log(Logger.DEBUG, "digest matching "+credential);
        }
        Subject subject = null;
        String passwd = (String)userPTable.get(user);
        if (passwd != null) {
            String passwdnonce = MD5.getHashString(passwd+":"+nonce);
            if (credential.equals(passwdnonce)) {
                subject =  getSubject(user, userRTable);
            }
        }
        return subject;
    }
            
    private Subject getSubject(String user, HashMap userRTable) {  
        Subject subject = null;
        String rolestr = (String)userRTable.get(user);
        if (rolestr != null && !rolestr.trim().equals("")) {
            subject = new Subject();
            subject.getPrincipals().add(new MQUser(user));
            subject.getPrincipals().add(new MQGroup(rolestr));
            if (rolestr.equals(ADMINGROUP)) {
                subject.getPrincipals().add(new MQAdminGroup(ADMINGROUP));
            }
        }
        return subject;
    }

    private void loadUserTable(HashMap userPTable, HashMap userRTable) throws IOException {
        String rep = authProps.getProperty(
                          AccessController.PROP_AUTHENTICATION_PREFIX +
                          authType +AccessController.PROP_USER_REPOSITORY_SUFFIX);
        if (rep == null) {
        throw new IOException(Globals.getBrokerResources().getKString(
                              BrokerResources.X_USER_REPOSITORY_NOT_DEFINED, authType));
        }
		if (!rep.equals(TYPE)) {
        String[] args = {rep, TYPE, this.getClass().getName()};
        throw new IOException(Globals.getBrokerResources().getKString(
                              BrokerResources.X_REPOSITORY_TYPE_MISMATCH, args));
        }

        File pwdfile = getPasswordFile(authProps);

        InputStreamReader fr = new InputStreamReader(new FileInputStream(pwdfile), "UTF8");
        BufferedReader br = new BufferedReader(fr);

        try {

        String line, name, passwd, role, active;
        while ((line = br.readLine()) != null) {
            name = passwd = role = active = null;
            StringTokenizer st = new StringTokenizer(line, ":", false);
            if (st.hasMoreTokens()) {
                name = st.nextToken(); 
            }
            if (st.hasMoreTokens()) {
                passwd = st.nextToken(); 
            }
            if (st.hasMoreTokens()) {
                role = st.nextToken(); 
            }
            if (st.hasMoreTokens()) {
                active = st.nextToken(); 
            }
            if (DEBUG) {
            logger.log(Logger.DEBUG, "passwd entry "+name+":"+passwd+":"+role+":"+active);
            }
            if (name !=null && passwd != null && role != null 
                    && active != null && active.equals("1")) {
                userPTable.put(name, passwd);
                userRTable.put(name, role);
            }
        }

        br.close();
        fr.close();

        } catch (IOException ioe) {
        try {
        if (br != null) br.close();
        if (fr != null) fr.close();
        } catch (IOException e) {}
        throw ioe;
        }
    }

    public Refreshable getCacheData() {
       return null;  
    }

    public void close() throws LoginException { }

    public static File getPasswordFile(Properties props) {
        String passwd_loc = props.getProperty(
				AccessController.PROP_USER_REPOSITORY_PREFIX
					+ PROP_DIRPATH_SUFFIX,
				Globals.getInstanceEtcDir());
        passwd_loc = StringUtil.expandVariables(passwd_loc, props);

        String f = props.getProperty(
				AccessController.PROP_USER_REPOSITORY_PREFIX
					+ PROP_FILENAME_SUFFIX,
				DEFAULT_PW_FILENAME);

        String pwdfile = passwd_loc +File.separator + f;
        return new File(pwdfile);
    }
}
