
/*
 * @(#)ClusteredBroker.java	1.9 12/01/05
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.cluster;

import com.sun.messaging.jmq.io.MQAddress;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.util.UID;


/**
 * represents an instance of a broker in a cluster
 */
public interface ClusteredBroker
{
    /**
     * A unique identifier assigned to the broker
     * (randomly assigned).<P>
     *
     * This name is only unique to this broker. The
     * broker at this URL may be assigned a different name
     * on another broker in the cluster.
     *
     * @return the name of the broker
     */
    public String getBrokerName();  

    /**
     * Returns the URL to the portmapper of this broker.
     * @return the URL of this broker
     */
    public MQAddress getBrokerURL();  

    /**
     * Sets the URL to the portmapper of this broker
     * @param addr the URL of this broker
     */
    public void setBrokerURL(MQAddress addr);  

    /**
     * Returns if this is the address of the broker running in this
     * VM.
     * @return true if this is the broker running in the
     *         current vm
     */
    public boolean isLocalBroker();  

    /**
     * Retrieves the status of the broker.
     * @return the status of the broker
     */
    public int getStatus();  

    /**
     * Gets the protocol version of the broker.
     * @return the protocol version (if known) or 0 if
     *     not known.
     */
    public int getVersion();  

    /**
     * Sets the protocol version of the broker.
     * @param version the protocol version
     * @throws UnsupportedOperationException if the version can
     *         not be set for this broker
     */
    public void setVersion(int version)
         throws UnsupportedOperationException;

    /**
     * Sets the status of the broker. Do not hold locks while calling
     * this routine.
     *
     * @param status the broker status to set for this broker
     * @param userData optional data associated with the change
     */
    public void setStatus(int status, Object userData);


    /**
     * Updates the BROKER_UP bit flag on status.
     * 
     * @param up setting for the bit flag (true/false)
     * @param userData optional data associated with the change
     */
    public void setBrokerIsUp(boolean up, Object userData);

    /**
     * Updates the BROKER_LINK_UP bit flag on status.
     * 
     * @param up setting for the bit flag (true/false)
     * @param userData optional data associated with the change
     */
    public void setBrokerLinkUp(boolean up, Object userData);

    /**
     * Updates the BROKER_INDOUBT bit flag on status.
     * 
     * @param up setting for the bit flag (true/false)
     * @param userData optional data associated with the change
     */
    public void setBrokerInDoubt(boolean indoubt, Object userData);


    /**
     * Destroys the ClusteredBroker.
     */
    public void destroy();

    /**
     * Gets the state of the broker.
     *
     * @return the broker state
     * @throws BrokerException if the state can not be retrieved
     */
    public BrokerState getState()
        throws BrokerException;

    /**
     * Sets the state of the broker.     * @throws IllegalAccessException if the broker does not have
     *               permission to change the broker (e.g. one broker
     *               is updating anothers state).
     * @throws IllegalStateException if the broker state changed
     *               unexpectedly.
     * @throws IllegalArgumentException if the state is not supported
     *               for this cluster type.
     * @param state the new broker state
     */
    public void setState(BrokerState state)
         throws IllegalAccessException, IllegalStateException,
                IllegalArgumentException;

    
    /**
     * Is the broker static or dynmically configured
     */
    public boolean isConfigBroker();


    /**
     * equals method
     */
    public boolean equals(Object o);

    /**
     *  hashcode method
     */
    public int hashCode();


    /**
     * Gets the UID associated with the broker session.
     *
     * @return the broker session uid (if known)
     */
    public UID getBrokerSessionUID();

    /**
     * Sets the UID associated with the broker session.
     *
     * @param uid the new broker session uid 
     */
    public void setBrokerSessionUID(UID uid);

    /**
     * returns if the brokerID was generated.
     * @return true if the ID was generated
     */
    public boolean isBrokerIDGenerated();

}
