/*
 * @(#)PacketReference.java	1.172 02/08/06
 *
 * Copyright 2000-2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.core;

import java.util.*;
import java.lang.ref.*;
import java.io.*;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.DMQ;
import com.sun.messaging.jmq.jmsserver.GlobalProperties;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.jmsserver.service.ConnectionUID;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.config.*;
import com.sun.messaging.jmq.jmsserver.persist.Store;
import com.sun.messaging.jmq.jmsserver.service.Connection;
import com.sun.messaging.jmq.util.DestType;
import com.sun.messaging.jmq.util.SizeString;
import com.sun.messaging.jmq.util.lists.*;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.util.memory.MemoryGlobals;

/**
 * This class contains the references for a packet which is passed
 * around the system.
 *
 * An optimization of this class would allow messages to be 
 * retrieved from the persistent store (if they are persistent)
 *
 */

public class PacketReference implements Sized
{

    public static ConsumerUID queueUID = null;


    /**
     * Controls if we should enable the fix for bug 6196233 and prepend the
     * message ID with "ID:"
     */
    private static boolean PREPEND_ID =
        Globals.getConfig().getBooleanProperty(
            Globals.IMQ + ".fix.JMSMessageID", false);

    static {
        queueUID = new ConsumerUID(true /* empty */);
        queueUID.setShouldStore(true);
        
    }

    public static boolean DEBUG = false;


    /**
     * set once store() is called
     */
    private boolean isStored = false;

    private boolean neverStore = false;

    /**
     * ets once store + interests are saved
     */
    private boolean isStoredWithInterest = false;

    private HashMap attachedData = null;

    /**
     * has packet been destroyed
     */
    private boolean destroyed = false;


    /**
     * message id for the packet
     */
    private SysMessageID msgid;


    transient ConnectionUID con_uid;

    /**
     * time the packet reference was created (entered the system)
     */
    private long creationtime;

    /**
     * time the packet reference was last accessed
     */
    private long lastaccesstime;

    /**
     * sequence # (used for sorting queue browsers)
     */
    private int sequence;

    /**
     * if the message is persisted
     */
    private boolean persist;


    /**
     * determines if the message should remain even if its expired
     * (if the lbit count is > 0)
     */
    private Set lbit_set = null;

    private boolean sendMessageDeliveredAck = false;

    /**
     * size of the packet data 
     */
    private long size;

    /**
     * message properties (null if swapped)
     */
    private Hashtable props;

    private HashMap headers;

    /**
     * original packet or SoftReference (null if swapped)
     */
    private Object pktPtr;

    /**
     * priority of the packet
     */
    private int priority = 4;

    /**
     * isQueue setting
     */
    boolean isQueue = false;


    /**
     * flag called when a packet "should" be removed
     * but is is remaining in memory (because the lbit
     * is set on the packet
     */ 
    boolean invalid = false;


    /**
     * destination uid
     */
    DestinationUID destination = null;

    ExpirationInfo expire = null;

    /**
     * expired flag
     */
    boolean isExpired = false;

    /**
     * timestamp on the packet (for sorting)
     */
    long timestamp = 0;

    /**
     * transaction id
     */
    TransactionUID transactionid;


    /**
     * override redeliver
     */
    transient boolean overrideRedeliver = false;


    transient BrokerAddress addr = null;


    transient Destination d = null;

    transient String clientID = null;


    int interestCnt;
    int deliveredCnt;
    int ackCnt;
    int deadCnt;

    Map ackInfo = null;

    private static final int INITIAL = 0;
    private static final int ROUTED = 1;
    private static final int DELIVERED = 2;
    private static final int CONSUMED = 3;
    private static final int ACKED = 4;
    private static final int DEAD = 5;


    ConsumerUID lastDead = null;

    static class ConsumerMessagePair
    {
        private ConsumerUID uid;
        private int state = INITIAL;
        private int redeliverCnt = 0;
        private boolean stored;
        private String deadComment = null;
        private Reason deadReason = null;
        private Throwable deadException = null;
        private long timestamp = 0;
        private String deadBroker = null;


        public ConsumerMessagePair(ConsumerUID uid,
                  boolean stored) {
            this.uid = uid;
            this.stored = stored;
        }

        public synchronized int incrementRedeliver(){
            redeliverCnt ++;
            return redeliverCnt;
        }

        public synchronized int decrementRedeliver() {
            redeliverCnt --;
            return redeliverCnt;
        }
        
        public synchronized boolean setState(int state) {
            if (this.state == state) return false;
            this.state = state;
            timestamp = System.currentTimeMillis();
            return true;
        }

        public synchronized int getState() {
            return state;
        }

        public synchronized boolean compareAndSetState(int state, int expected) {
            if (this.state != expected) {
                   return false;
            }
            timestamp = System.currentTimeMillis();
            this.state = state;
            return true;
        }

        public synchronized boolean compareState(int expected) {
            return this.state == expected;
        }

        public synchronized boolean compareStateGT(int expected) {
            return this.state > expected;
        }

        public synchronized boolean compareStateLT(int expected) {
            return this.state < expected;
        }

        public synchronized boolean setStateIfLess(int state, int max) {
            if (compareStateLT(max))
                return setState(state);
            return false;
        }

        public boolean isStored() {
            return stored;
        }

        public int getRedeliverCount() {
            return redeliverCnt;
        }

        public void setRedeliverCount(int cnt) {
            redeliverCnt = cnt;
        }

        public void setDeadComment(String str) {
             deadComment = str;
        }
        public void setDeadReason(Reason r) {
             deadReason = r;
        }
        public String getDeadComment() {
             return deadComment;
        }
        public Reason getDeadReason() {
            return deadReason;
        }
        public Throwable getDeadException() {
            return deadException;
        }

        public void setDeadException(Throwable thr) {
             deadException = thr;
        }
        public void setDeadBroker(String bkr) {
             deadBroker = bkr;
        }
        public String getDeadBroker() {
            return deadBroker;
        }

        public String toString() {
            return ": CMPair["+ uid.longValue()  + ":" + stateToString(state) 
                    + ":" + stored + "] - " + timestamp;
        }
    }


    public static PacketReference createReference(Packet p, 
             DestinationUID duid, 
             Connection con) {
        PacketReference pr = new PacketReference(p,duid, con);
        if (con != null && pr.getExpiration() != null) {
            con.checkClockSkew(pr.getTime(), pr.getTimestamp(),
                pr.getExpireTime());
        }
            
        return pr;
    }

    public static PacketReference createReference(Packet p, 
             Connection con) {
        return createReference(p, null, con);
    }

    public static void moveMessage(PacketReference oldLoc,
        PacketReference newLoc, Set targets)
            throws BrokerException, IOException
    {
        ConsumerUID[] uids = null;
        int[] states = null;
        if (targets == null) {
            // get from the original message
            throw new RuntimeException("Internal error: moving message"
                      + " to null targets not supported");
        } 
        ReturnInfo info = calculateConsumerInfo(targets, true);
        newLoc.ackInfo = info.ackInfo;
        newLoc.ackCnt = info.ackInfo.size();
        if (info.uids == null || info.uids.length == 0) {
            // nothing to store
            newLoc.neverStore=true;
            return;
        }
        if (oldLoc.isStored && oldLoc.persist && !oldLoc.neverStore) {
            Globals.getStore().moveMessage(newLoc.getPacket(),
                 oldLoc.getDestinationUID(), 
                 newLoc.getDestinationUID(),
                 info.uids, info.states, Destination.PERSIST_SYNC);
        }

        newLoc.isStored=true;
        newLoc.isStoredWithInterest=true;
        oldLoc.isStored=false;
    }


    static class ReturnInfo
    {
        ConsumerUID uids[];
        int states[];
        int interestCnt;
        Map ackInfo;
    }

// LKS- BUG SOMEWHERE BELOW
/*
 * this method is called when a set of consumers are being routed
*/
    private static ReturnInfo calculateConsumerInfo(Collection consumers, boolean checkStored)
    {
 
        if (consumers.isEmpty())
            return null;

        ReturnInfo ri = new ReturnInfo(); 
        ri.ackInfo = Collections.synchronizedMap(new HashMap());
        ArrayList storedConsumers = (checkStored ? new ArrayList() : null);
        Iterator itr = consumers.iterator();

        int count = 0;
        while (itr.hasNext()) {
           Object o = itr.next();
           ConsumerUID cuid = null;
           if (o instanceof Consumer) {
               cuid = ((Consumer)o).getStoredConsumerUID();
           } else {
               cuid = (ConsumerUID)o;
           }
           ri.interestCnt ++;

           boolean store = false;
           if (storedConsumers != null && cuid.shouldStore()) {
               storedConsumers.add(cuid);
               store = true;
            }

           ConsumerMessagePair cmp = new ConsumerMessagePair(cuid,
                  store);
           ri.ackInfo.put(cuid, cmp);

        }
      
        if (storedConsumers != null) {
            ConsumerUID [] type = new ConsumerUID[0];
            ri.uids = (ConsumerUID[])
                   storedConsumers.toArray(type);
            ri.states = new int[ri.uids.length];
            for (int i=0; i < ri.states.length; i ++) {
                ri.states[i] = Store.INTEREST_STATE_ROUTED;
            }
         }
         return ri;
    }


    /**
     * create a new PacketReference object
     */
    private PacketReference(Packet pkt, DestinationUID duid,
             Connection con) {
        this.creationtime = System.currentTimeMillis();
        this.lastaccesstime = creationtime;
        this.msgid = (SysMessageID)pkt.getSysMessageID().clone();
        this.isQueue = pkt.getIsQueue();
        this.persist = pkt.getPersistent();
        this.priority = pkt.getPriority();
        this.sequence = pkt.getSequence();
        this.timestamp = pkt.getTimestamp();
        if (pkt.getRedelivered())
            overrideRedeliver = true;

        if (con != null)
            this.clientID = (String)con.getClientData(IMQConnection.CLIENT_ID);
        setExpireTime(pkt.getExpiration());
        this.size = pkt.getPacketSize();
        String d  = pkt.getDestination();
        this.con_uid = (con == null ? null : con.getConnectionUID());
        if (duid != null) {
            destination = duid;
        } else {
            destination = DestinationUID.getUID(d, 
               isQueue);
        }
        long tid = pkt.getTransactionID();
        synchronized(this) {
            setPacketObject(false, pkt);
        }
        if (tid != 0) {
            transactionid = new TransactionUID(tid);
        } else {
            transactionid = null;
        }
    }

    public void setSequence(int seq) {
        this.sequence = seq;
    }


    public String getClientID() {
        return clientID;
    }


    private ConsumerMessagePair getAck(Object obj) {
        ConsumerUID cuid = null;
        if (obj instanceof ConsumerUID) {
            cuid = (ConsumerUID)obj;
        } else if (obj instanceof Consumer) {
            cuid = ((Consumer)obj).getConsumerUID();
        } else {
            throw new RuntimeException("Bogus ID");
        }
        return (ConsumerMessagePair) ackInfo.get(cuid);
    }




/*
 *-------------------------------------------------------------------
 * 
 *   ACCESS METHODS
 *
 *--------------------------------------------------------------------
 */

    public void setDestination(Destination d) {
        this.d = d;
    }

    public Destination getDestination() {
         if (d == null) {
             d = Destination.getDestination(destination);
         }
         return d;
    }

    boolean isStored() {
        return isStored;
    }


    public void setBrokerAddress(BrokerAddress a)
    {
        addr = a;
    }

    public BrokerAddress getAddress() {
        return addr;
    }

    public boolean isLocal() {
        return addr == null || addr == Globals.getMyAddress();
    }

    public String toString() {
        return "PacketReference["+msgid+"]";
    }

    Set deliveredMsgAcks = new HashSet();

    public boolean getMessageDeliveredAck(ConsumerUID uid) {
        return !deliveredMsgAcks.isEmpty() && 
              deliveredMsgAcks.contains(uid);
    }

    public ConsumerUID[] getConsumersForMsgDelivered()
    {
        return (ConsumerUID [])deliveredMsgAcks.toArray(
                   new ConsumerUID[0]);
    }

    public void removeMessageDeliveredAck(ConsumerUID uid) {
        deliveredMsgAcks.remove(uid);
    }

    public void addMessageDeliveredAck(ConsumerUID uid) {
        deliveredMsgAcks.add(uid);
    }

    private void setExpireTime(long time) {
        if (time == 0) {
            expire = null;
        } else {
            expire = new ExpirationInfo(msgid, time);
        }
    }
    public long getExpireTime() {
        if (expire == null) return 0;
        return expire.getExpireTime();
    }

    public ExpirationInfo getExpiration() {
        return expire;
    }

    public void setInvalid() {
        invalid = true;
    }

    private Packet getPacketObject() {

        assert (pktPtr ==  null ||
               pktPtr instanceof SoftReference
               || pktPtr instanceof Packet) : pktPtr;

        Object ptr = pktPtr;
        if (ptr == null) return null;
        if (ptr instanceof SoftReference) {
            return (Packet)((SoftReference)pktPtr).get();
        }
        return (Packet)pktPtr;
    }

    private void setPacketObject(boolean soft, Packet p) {

        assert Thread.holdsLock(this);
        if (soft) {
            pktPtr = new SoftReference(p);
        } else {
            pktPtr = p;
        }
    }

    private void makePacketSoftRef() {
        assert Thread.holdsLock(this);
        Object ptr = pktPtr;
        if (ptr != null && ptr instanceof Packet) {
            pktPtr = new SoftReference(ptr);
        }
    }

    public void setNeverStore(boolean s) {
        neverStore = s;
    }

    public long byteSize() {
        return size;
    }


    public synchronized void setLoaded() {
        isStoredWithInterest = true;
        isStored = true;
        makePacketSoftRef();
    }


    public ConnectionUID getProducingConnectionUID()
    {
        return con_uid;
    }

    public int getPriority() {
        return priority;
    }

    public TransactionUID getTransactionID() {
        return transactionid;
    }

    public boolean getIsQueue() {
        return this.isQueue;
    }

    public long getTime() {
        return creationtime;
    }

    public long getSequence() {
        return sequence;
    }

    public synchronized void setLastBit(ConsumerUID id)
        throws IllegalStateException
    {
        if (isInvalid() && isDestroyed()) {
            throw new IllegalStateException(
                   Globals.getBrokerResources().getString(
                    BrokerResources.X_INTERNAL_EXCEPTION, 
                        "reference has been destroyed"));
        }
        if (lbit_set == null) {
            lbit_set = new HashSet();
        }
        lbit_set.add(id);

    }

    public Hashtable getDebugState() {
        Hashtable ht = new Hashtable();
        ht.put("AckCount", String.valueOf(ackCnt));
        ht.put("DeadCount", String.valueOf(deadCnt));
        ht.put("ackInfo[#]", String.valueOf( ackInfo.size()));
        ht.put("interestCount", String.valueOf(interestCnt));
        Vector vt = new Vector();
        synchronized(ackInfo) {
            Iterator itr = ackInfo.keySet().iterator();
            while (itr.hasNext()) {
                Object key = itr.next();
                ConsumerMessagePair cmp = getAck(key);
                vt.add(cmp.toString());
            }    
        }
        if (!vt.isEmpty())
            ht.put("Acks",vt);
        return ht;
    }
        

    public synchronized boolean isLast(ConsumerUID id) {
        if (lbit_set == null) {
            return false;
        }
        return lbit_set.contains(id);
    }

    public synchronized void removeIsLast(ConsumerUID id) {
        if (lbit_set == null) {
            return;
        }
        lbit_set.remove(id);
        if (lbit_set.isEmpty() && invalid) { // clean up, lbit is gone
            destroy(); // at next expiration check, we will be
                       // removed from the packet store
        }
    }

    public synchronized boolean getLBitSet() {
        return lbit_set != null && !lbit_set.isEmpty();
    }

    public DestinationUID getDestinationUID() {
        return destination;
    }

    public String getDestinationName() {
        return destination.getName();
    }

    public synchronized Packet getPacket()
    {
        Packet pkt = getPacketObject();
        if (pkt != null || destroyed) {
            return pkt;
        }

        assert persist;

        if (!persist) {
            return null;
        }

        pkt = recoverPacket();

        assert pkt != null;

        setPacketObject(true, pkt);
        return pkt;
    }

    private Packet recoverPacket() 
    {
        // recover from the database
        assert Thread.holdsLock(this);
        assert pktPtr == null || 
            (pktPtr instanceof SoftReference &&
              ((Reference)pktPtr).get() == null);

        try {
            Packet p = Globals.getStore().getMessage(destination, msgid);
            assert p != null;
            return p;

        } catch (BrokerException ex) {
            assert false :ex;
            Globals.getLogger().logStack(Logger.ERROR,
                 BrokerResources.E_LOAD_MSG_ERROR,
                 msgid.toString(), ex);
        }
            
        return null;
    }

    public synchronized Hashtable getProperties() 
        throws ClassNotFoundException
    {
        if (destroyed || invalid) {
            return new Hashtable();
        }
        this.lastaccesstime = System.currentTimeMillis();
        Packet pkt = getPacketObject();
        if (props == null && !destroyed) {
            if (pkt == null) {
                pkt = getPacket();
            }
            try {
                props = pkt.getProperties(); 
            } catch (IOException ex) {
                // no properties
                Globals.getLogger().log(Logger.INFO,"Internal Exception: " , ex);
                props = new Hashtable();
            } catch (ClassNotFoundException ex) {
                assert false; // should not happen
                throw ex;
            }
        }
        return props;
     }

     public synchronized HashMap getHeaders() {
         if (headers == null) {
             if (destroyed || invalid) {
                 return new HashMap();
             }
             Packet pkt = getPacketObject();
             assert pkt != null;
             headers = new HashMap();
             if (pkt == null) {
                 if (DEBUG) {
                     Globals.getLogger().log(Logger.DEBUG,"no packet for "+
                        "non-destroyed message " + msgid);
                 }
                 return headers;
             }
             headers.put("JMSPriority", new Integer(priority));

             /*
              * XXX If the fix for bug 6196233 is enabled, then 
              * prepend the messageid with "ID:".
              */
             headers.put("JMSMessageID",
                (PREPEND_ID ? "ID:" : "") + msgid.toString());
             headers.put("JMSTimestamp", new Long(timestamp));
             headers.put("JMSDeliveryMode",
                  (pkt.getPersistent() ? "PERSISTENT" :
                      "NON_PERSISTENT"));
             headers.put("JMSCorrelationID", pkt.getCorrelationID());
             headers.put("JMSType", pkt.getMessageType());
         }
         return headers;
     }
 

    public  SysMessageID getSysMessageID() 
    {
        return msgid;
    }

    public  long getCreateTime() 
    {
        return creationtime;
    }

    public  long getLastAccessTime() 
    {
        return lastaccesstime;
    }

    public  long getTimestamp() 
    {
        return timestamp;
    }

    public void setTimestamp(long time)
    {
        timestamp = time;
    }


    public  boolean isPersistent() 
    {
        return persist;
    }

    public  void overridePersistence(boolean persist) 
    {
        this.persist = persist;
    }

    public  long getSize() 
    {
        return size;
    }


    public boolean isDestroyed()
    {
        return destroyed;
    }


    public synchronized boolean isInvalid()
    {
        return invalid;
    }

    public boolean mayExpire() {
        return expire == null;
    }

    public boolean isExpired() {
        if (expire == null) {
            return false;
        }
        return isExpired(System.currentTimeMillis());

    }

    public void overrideExpireTime(long expire)
    {
        setExpireTime(expire);
    }

    public boolean isExpired(long curtime) {
        if (isExpired) {
            return true;
        }
        if (expire == null) {
            return false;
        }
        boolean expiring = expire.isExpired(curtime);
        if (expiring) {
           // change to soft reference
           synchronized(this) {
               makePacketSoftRef();
           }
        }
        return expiring;
    }

    void clearExpirationInfo() {
        isExpired = true;
        expire = null;
    }



    public boolean equals(Object obj) {
        if (msgid == null)
            return msgid == obj;
        if (obj instanceof SysMessageID)
            return msgid.equals(obj);
        if (obj instanceof Packet)
            return msgid.equals(((Packet)obj).getSysMessageID());
        if (obj instanceof PacketReference)
            return msgid.equals(((PacketReference)obj).msgid);
        return false;
    }

    public int hashCode() {
        return msgid == null ? 0 : msgid.hashCode();
    }
        
      

    public boolean matches(DestinationUID uid)
    {
        return true;
    }  

/*
 *-------------------------------------------------------------------
 * 
 *   Acknowledgement handling methods
 *
 *--------------------------------------------------------------------
 */


    public static String stateToString(int state) {
        switch (state) {
            case INITIAL:
                return "INITIAL";
            case ROUTED:
                return "ROUTED";
            case DELIVERED:
                return "DELIVERED";
            case CONSUMED:
                return "CONSUMED";
            case ACKED:
                return "ACKED";
            case DEAD:
                return "DEAD";
        }
        return "UNKNOWN";
    }


    /**
     * stores the persistent message (if necessary)
     * (may be called by transactions to store the 
     * message)
     */

    public synchronized void store()  
        throws BrokerException
    {
        if (!destroyed && persist && !neverStore && !isStored) {
            // persist indicated IF we want to
            // store the message and may be
            // different from the actual
            // state on the message
            assert pktPtr instanceof Packet;
            try {
                Globals.getStore().storeMessage(destination,
                    (Packet)pktPtr, Destination.PERSIST_SYNC);
                makePacketSoftRef();
            } catch (IOException ex) {
                throw new BrokerException(
                        ex.toString(), ex);
            } catch (Exception ex) {
                Globals.getLogger().logStack(Logger.ERROR,
                     BrokerResources.W_MESSAGE_STORE_FAILED,
                     msgid.toString(), ex);
                throw new BrokerException(
                        ex.toString(), ex);
            }
            isStored = true;
            
        }
    }

    public void store(Collection consumers) 
        throws BrokerException
    {        
        if (destroyed || pktPtr == null) {
            return;
        }

        if (isStoredWithInterest) {
            // done already
            return;
        }

        boolean botherToStore=!neverStore && persist;

        ReturnInfo info = calculateConsumerInfo(consumers, botherToStore);
        if (ackInfo != null)  {
            ackInfo.putAll(info.ackInfo);
        } else {
            ackInfo = info.ackInfo;
        }
        interestCnt = info.ackInfo.size();

        if (!botherToStore) return;
        if (info.uids == null || info.uids.length == 0) {
            // nothing to store
            neverStore = true;
            return;
        }
            
        try {
            if (isStored && !neverStore && persist) {
                Globals.getStore().storeInterestStates(
                    destination,
                    msgid, info.uids, info.states, 
                    Destination.PERSIST_SYNC);
            } else {
                assert pktPtr instanceof Packet || (pktPtr == null && destroyed) : "PktPtr is " + pktPtr.getClass();

                Globals.getStore().storeMessage(destination,
                        (Packet)pktPtr,
                         info.uids, info.states, Destination.PERSIST_SYNC);
                synchronized(this) {
                    makePacketSoftRef();
                }
            }
        } catch (IOException ex) {
                throw new BrokerException(
                    ex.toString(), ex);
        } catch (Exception ex) {
                Globals.getLogger().logStack(Logger.ERROR,
                    BrokerResources.W_MESSAGE_STORE_FAILED,
                    msgid.toString(), ex);
                throw new BrokerException(
                    ex.toString(), ex);
        } 
        isStored = true;
        isStoredWithInterest = true;
        assert interestCnt != 0;
               
    }

    /**
     * this method is called from ???
     */
    public void add(Collection uids) {
        Iterator itr = uids.iterator();
        while (itr.hasNext()) {
           Object o = itr.next();
           ConsumerUID cuid = null;

           if (o instanceof ConsumerUID) {
               cuid = (ConsumerUID)o;
           } else {
               cuid = ((Consumer)o).getStoredConsumerUID(); 
           }

           interestCnt ++;

           if (ackInfo == null)
               ackInfo = Collections.synchronizedMap(new HashMap());

           ConsumerMessagePair cmp = new
               ConsumerMessagePair(cuid, false);
           cmp.setState(ROUTED);
           ackInfo.put(cuid, cmp);
        }
   }

    /**
     * called for messages which have already been
     * loaded from the database
     */
    public void update(ConsumerUID[] uids, int[] states) {
        update(uids, states, false);
    }

    public void update(ConsumerUID[] uids, int[] states, boolean store) {
        assert isStored;
        assert uids != null;
        assert states != null;
        assert uids.length == states.length;
        assert ackInfo == null;
        assert uids.length != 0;


        synchronized (this) {
            interestCnt +=uids.length;
        }
          

        for (int i=0; i < uids.length; i ++) {
           ConsumerUID cuid = uids[i];

           assert uids[i] != null;

           assert states[i] >= Store.INTEREST_STATE_ROUTED &&
                  states[i] <= Store.INTEREST_STATE_ACKNOWLEDGED;

           if (states[i] == Store.INTEREST_STATE_ACKNOWLEDGED) {
               // left over
               continue;
           }

           if (ackInfo == null)
               ackInfo = Collections.synchronizedMap(new HashMap());

           ConsumerMessagePair cmp = new
               ConsumerMessagePair(cuid, true);

           ackInfo.put(cuid, cmp);
           cmp.setState(states[i] == Store.INTEREST_STATE_ROUTED
                 ? ROUTED : CONSUMED);

           if (store) {
                try {
                   Globals.getStore().storeInterestStates(destination,
                         msgid,
                         uids, states, Destination.PERSIST_SYNC);
                } catch (Exception ex) {
                }
            }

        }

        assert interestCnt != 0;
            
    }

    public void debug(String prefix)
    {
        if (prefix == null)
            prefix = "";
        Globals.getLogger().log(Logger.INFO,prefix +"Message " + msgid);
        Globals.getLogger().log(Logger.INFO,prefix + "size " + ackInfo.size());
        Iterator itr = ackInfo.values().iterator();
        while (itr.hasNext()) {
            ConsumerMessagePair ae = (ConsumerMessagePair)itr.next();
            Globals.getLogger().log(Logger.INFO,prefix + "\t " + ae);
        }
    }
    

    /**
     * called when a consumer received
     * the specific message for delivery
     */
    public void routed(ConsumerUID intid) 
        throws BrokerException, IOException
    {

        // NOTE: the caller is responsible for
        // passing the "right" intid - for
        // queues this is the generic ID, for
        // durable subscribers, it is the subscription
        // ID

        if (destroyed || invalid) {
            Globals.getLogger().log(Logger.DEBUG,"route on destroyed ref "
                 + msgid + ":" + intid);
            return; // destroyed
        }


        // nothing to store
        // this just indicates that we've passed
        // the message off to a consumer

        // the state of the message should have
        // already been stored w/ routeTable

        // additional logic may be added in the future

    }

    public int getCompleteCnt() {
        return ackCnt + deadCnt;
    }

    public int getDeliverCnt() {
        return deliveredCnt;
    }

    /**
     * called just before the message is written
     * to the wire for a consumer
     */
    public boolean delivered(ConsumerUID intid, ConsumerUID storedid,
             boolean sync, boolean store) 
        throws BrokerException, IOException
    {
        // NOTE: the caller is responsible for
        // passing the "right" storedid - for
        // queues this is the generic ID, for
        // durable subscribers, it is the subscription
        // ID
        if (destroyed || invalid) {
            Globals.getLogger().log(Logger.DEBUG,"delivered on destroyed ref "
                 + msgid + ":" + storedid);
            return true; // destroyed
        }

        if (intid.isNoAck()) {
            // immediately ack message when delivered
            return acknowledged(intid, 
                   storedid,
                   sync, store);
        }

        ConsumerMessagePair cmp = getAck(storedid);

        if (cmp == null) {
            // nothing to do
            Globals.getLogger().log(Logger.DEBUG,"Received Unknown delivered:" 
                   +"\n\tStoreUID: " +  storedid 
                   +"\n\tConsumerUID: " + intid 
                   +"\n\tConsumer: " +  Consumer.getConsumer(intid));
            //if (DEBUG)
                debug("\t- ");
            return false;
        }

        // if we are greater than delivered
        if (!cmp.compareStateLT(DELIVERED)) {
            synchronized (this) {
                deliveredCnt ++;
             }
        }
        cmp.setStateIfLess(DELIVERED, DELIVERED);        
        if (cmp.isStored() && store) {
                Globals.getStore().updateInterestState(
                    destination,
                    msgid, storedid, 
                    Store.INTEREST_STATE_DELIVERED, 
                    Destination.PERSIST_SYNC && sync);
        }

        synchronized (this) {
            // in low memory, free ref explicity
            if (deliveredCnt >= interestCnt &&
                isStored &&
                ((MemoryGlobals.MEM_FREE_P_ACKED 
                   && persist ) ||
                  (MemoryGlobals.MEM_FREE_NP_ACKED
                   && (!persist)))) {
                   unload();
            }
        }

        return false;

    }

    public int getRedeliverCount(ConsumerUID intid) {
        ConsumerMessagePair cmp = getAck(intid);
        return cmp == null ? 0 : cmp.getRedeliverCount();
    }


    /**
     * called when the client indicates that
     * the message has been consumed (delivered
     * to the specific client side consumer).
     * This method may be triggered by a specific
     * message from the consumer OR when an ack
     * is received in autoack mode
     */
    public void consumed(ConsumerUID intid, boolean sync, boolean delivered) 
        throws BrokerException, IOException
    {
        if (destroyed || invalid) {
            Globals.getLogger().log(Logger.DEBUG,"consumed on destroyed ref "
                 + msgid + ":" + intid);
            return; // destroyed
        }

        if (delivered) {
            delivered(intid, intid, sync, true);
        }

        assert ackInfo != null;

        ConsumerMessagePair cmp = getAck(intid);

        assert cmp != null;

        // something went wrong, ignore
        if (cmp == null) {
           // nothing to do
           Globals.getLogger().log(Logger.ERROR,"Internal Error: unknown interest for " 
               + " consumed on " + msgid
               + intid);
           return;
        }

        cmp.incrementRedeliver();
        cmp.setStateIfLess(CONSUMED, CONSUMED);

    }


    public boolean matches(ConsumerUID id) {
        return getAck(id) != null;
    }

    public boolean isAcknowledged(ConsumerUID id) {
        ConsumerMessagePair cmp = getAck(id);
        if (cmp == null) return true;
        return cmp.compareState(ACKED);
    }

    public boolean isDelivered(ConsumerUID id) {
        ConsumerMessagePair cmp = getAck(id);
        if (cmp == null) return true;
        return cmp.compareState(DELIVERED) || cmp.compareState(CONSUMED);
    }

    public boolean removeDelivered(ConsumerUID storedid, boolean decrementCounter)
    {
        if (destroyed || invalid) {
            return true; // destroyed
        }

        ConsumerMessagePair cmp = getAck(storedid);

        assert cmp != null;

        // something went wrong, ignore
        if (cmp == null) {
           // nothing to do
           Globals.getLogger().log(Logger.ERROR,"Internal Error: unknown interest for " 
               + " remove consumed on " + msgid
               + storedid);
           return false;
        }

        if (decrementCounter)
            cmp.decrementRedeliver();

        cmp.compareAndSetState(ROUTED, DELIVERED);

        //XXX use destination setting
        return (cmp.getRedeliverCount() >= 20);

    }


    public boolean hasConsumerAcked(ConsumerUID storedid)
    {
        try {
            if (destroyed || invalid) {
                return true; // destroyed
            }    
            ConsumerMessagePair cmp = getAck(storedid);

            
            if ( cmp != null && cmp.getState() != ACKED) {
                return false;
            }
        } catch (Throwable ex) {
            Globals.getLogger().logStack(Logger.ERROR,"Internal Error checking ack" +
                         " on " + msgid + " for " + storedid, ex);
            return false;
        }
        return true;
    }

    public boolean prepare(ConsumerUID intid, TransactionUID tuid, boolean sendid)
        throws BrokerException, IOException
    {
        Long txn = (tuid == null ? null : new Long(tuid.longValue()));
        if (!isLocal() && sendid ) { // not local
            Globals.getClusterBroadcast().
                acknowledgeMessage(getAddress(),
                getSysMessageID(), intid, 
                ClusterBroadcast.MSG_PREPARED,
                true, null, txn);
        }
        return true;
    }

    public boolean txnAcknowledge(ConsumerUID intid, TransactionUID tuid)
        throws BrokerException, IOException
    {
        Long txn = (tuid == null ? null : new Long(tuid.longValue()));
        if (!isLocal() && tuid != null ) { // not local
            Globals.getClusterBroadcast().
                acknowledgeMessage(getAddress(),
                getSysMessageID(), intid, 
                ClusterBroadcast.MSG_TXN_ACKNOWLEDGED,
                true, null, txn);
        }
        return true;
    }

    public boolean rollback(ConsumerUID intid, TransactionUID tuid, boolean sendid)
        throws BrokerException, IOException
    {
        Long txn = (tuid == null ? null : new Long(tuid.longValue()));
        if (!isLocal() && sendid ) { // not local
            Globals.getClusterBroadcast().
                acknowledgeMessage(getAddress(),
                getSysMessageID(), intid, 
                ClusterBroadcast.MSG_ROLLEDBACK,
                true /* wait for ack */,
                null, txn);
        }
        return true;
    }

    public boolean commit(ConsumerUID intid, TransactionUID tuid, boolean sendid)
        throws BrokerException, IOException
    {
        Long txn = (tuid == null ? null : new Long(tuid.longValue()));
        if (!isLocal() && sendid ) { // not local
            Globals.getClusterBroadcast().
                acknowledgeMessage(getAddress(),
                getSysMessageID(), intid, 
                ClusterBroadcast.MSG_ACKNOWLEDGED,
                true,
                null, (sendid ? txn : null));
        }
        return true;
    }

    /**
     * handle updating acknowledgement of the message
     *
     * @returns if the message should be removed from the persistent list
     */
    public boolean acknowledged(ConsumerUID intid,ConsumerUID storedid,
                   boolean sync, boolean notIgnored) 
        throws BrokerException, IOException
    {
        return acknowledged(intid, storedid, sync, notIgnored, null);
    }


    public boolean acknowledged(ConsumerUID intid,ConsumerUID storedid,
                   boolean sync, boolean notIgnored, TransactionUID tuid) 
        throws BrokerException, IOException
    {
        Long txn = (tuid == null ? null : new Long(tuid.longValue()));
        try {
            if (destroyed || invalid) {
                return true; // destroyed
            }    
    
            ConsumerMessagePair cmp = getAck(storedid);
        
            // something went wrong, ignore
            if (cmp == null) {
               // nothing to do
               Globals.getLogger().log(Logger.ERROR,"Internal Error: Received Unknown ack " 
                   + intid);
               Globals.getLogger().log(Logger.ERROR, "AckInfo" + ackInfo.toString());
               synchronized (this) {
                   return (ackCnt+deadCnt) >= interestCnt;
               }
            }

            // ok ... if setState == false, we were already
            // acked so do nothing
            if (cmp.setState(ACKED)) {
                    if (cmp.isStored()) {
                        boolean acked = false;
                        Store store = Globals.getStore();
                        if (store.isJDBCStore()) {
                            acked = store.hasMessageBeenAcked(msgid);
                        }
                        if (!acked) {
                            try {
                                store.updateInterestState(
                                    destination,
                                    msgid, storedid, 
                                    Store.INTEREST_STATE_ACKNOWLEDGED, 
                                    Destination.PERSIST_SYNC && sync);
                            } catch (Exception ex) {
                                Globals.getLogger().log(Logger.DEBUGHIGH,"Message has been acked");
                           }
                       }
                    }
                    if (!isLocal() ) { // not local
                        if (notIgnored) {
                            if (txn == null) { 
                                int type = ClusterBroadcast.MSG_ACKNOWLEDGED;
                                Globals.getClusterBroadcast().
                                    acknowledgeMessage(getAddress(),
                                    getSysMessageID(), intid, 
                                    type,
                                    true /* wait for ack */, null, txn);
                            }
                        } else {
                            Globals.getClusterBroadcast().
                                acknowledgeMessage(getAddress(),
                                getSysMessageID(), intid, 
                                ClusterBroadcast.MSG_IGNORED,
                                false /* dont wait for ack */,
                                null, txn);
                        }
                    }
                    
            } else {
                    Consumer c = Consumer.getConsumer(intid);
                    if (c == null || !c.isValid()) {
// do we want to add a debug level message here
                            // got it while cleaning up
                            // this can only happen on topics and is a
                            // non-fatal error
                            // fixing the timing problem through synchronization
                            // adds too much overhead
                           synchronized (this) {
                               return (ackCnt+deadCnt) >= interestCnt;
                           }
                    } else {
                        Exception e = new Exception("double ack " + cmp);
                        e.fillInStackTrace();
                        Globals.getLogger().logStack(Logger.ERROR,"Internal Error: received ack twice " +
                             " on " + msgid + " for " + intid + " state is = " + cmp, e );
                        synchronized (this) {
                           return (ackCnt+deadCnt) >= interestCnt;
                        }
                    }
            }

            synchronized (this) {
                ackCnt ++;
                return (ackCnt+deadCnt) >= interestCnt;
            }
        } catch (Throwable thr) {
            Globals.getLogger().logStack(Logger.ERROR,
                        BrokerResources.E_INTERNAL_BROKER_ERROR,
                        "processing ack" +
                         " on " + msgid + " for " + intid, thr);
            if (thr instanceof BrokerException)
                throw (BrokerException)thr;
            throw new BrokerException("Unable to process ack", thr);
        }
        
    }

    public void overrideRedeliver() {
        if (!overrideRedeliver) overrideRedeliver = true;
    }

    public boolean getRedeliverFlag(ConsumerUID intid)
    {
        if (destroyed || invalid) {
            Globals.getLogger().log(Logger.DEBUG,"redeliver for destroyed "
                 + msgid + ":" + intid);
            return true; // doesnt matter
        }

        ConsumerMessagePair cmp = getAck(intid);

        assert cmp != null;

        // something went wrong, ignore
        if (cmp == null) {
            return false;
        }

        if (overrideRedeliver)
            return true;

        // return true if our state is greater or equal to
        // DELIVERED
        return !cmp.compareStateLT(DELIVERED);
    }


    public boolean getConsumed(ConsumerUID intid) {
        if (destroyed || invalid) {
            Globals.getLogger().log(Logger.DEBUG,"getConsumed for destroyed "
                 + msgid + ":" + intid);
            return true; // doesnt matter
        }

        ConsumerMessagePair cmp = getAck(intid);

        // something went wrong, ignore
        if (cmp == null) {
           // nothing to do
           Globals.getLogger().log(Logger.ERROR,"Internal Error: unknown interest for " 
               + " getConsumed on " + msgid
               + intid);
           return true;
        }

        return cmp.compareState(CONSUMED);
    }



    public synchronized void clear() {
        props = null;
        if (pktPtr instanceof Reference) {
            ((Reference)pktPtr).clear();
            ((Reference)pktPtr).enqueue();
        }
        pktPtr = null;
        msgid = null;
    }

    public void remove() {
        if (isStored && !neverStore && persist) {
            try {
                //
                // removes synchronization when we remove
                // the message (we really dont need it)
                //
                // in the future we may evaulate changing the
                // code be smarter about syncing (if two
                // threads both change the file within a short
                // period of time, we really only need to sync once

                Globals.getStore().removeMessage(destination,
                       msgid, 
                      false /*Destination.PERSIST_SYNC*/ );
                isStored = false;

            } catch (IOException ex) {
                Globals.getLogger().logStack(Logger.ERROR,
                     BrokerResources.E_REMOVE_MSG_ERROR,
                     msgid.toString(), ex);
            } catch (BrokerException ex) {
                Globals.getLogger().logStack(Logger.ERROR,
                     BrokerResources.E_REMOVE_MSG_ERROR,
                     msgid.toString(), ex);
            }
            isStored = false;
        }
    }

    public void destroy() {
        assert getLBitSet() == false;
        synchronized (this) {
            if (destroyed) return;
            destroyed = true;
        }
        if (isStored) {
            try {
                // initial fix for bug 5025164
                // removes synchronization when we remove
                // the message (we really dont need it)
                //
                // in the future we may evaulate changing the
                // code be smarter about syncing (if two
                // threads both change the file within a short
                // period of time, we really only need to sync once
                Globals.getStore().removeMessage(destination, msgid, 
                      false /*Destination.PERSIST_SYNC*/ );
                isStored = false;

            } catch (IOException ex) {
                Globals.getLogger().logStack(Logger.ERROR,
                     BrokerResources.E_REMOVE_MSG_ERROR,
                     msgid.toString(), ex);
            } catch (BrokerException ex) {
                Globals.getLogger().logStack(Logger.ERROR,
                     BrokerResources.E_REMOVE_MSG_ERROR,
                     msgid.toString(), ex);
            }
        }
        props = null;
        if (pktPtr instanceof Reference) {
            ((Reference)pktPtr).clear();
            ((Reference)pktPtr).enqueue();
        }
        pktPtr = null;
    }


    void unload() {
        // clears out the reference
        if (pktPtr instanceof SoftReference) {
            ((SoftReference)pktPtr).clear();
        }
    }
    

//------------------------------------------------------------------
//
//          DMQ methods
//------------------------------------------------------------------

    public String getDeadComment() {
        if (lastDead == null)
            return null;
        ConsumerMessagePair cmp = getAck(lastDead);
        return cmp == null ? null: cmp.getDeadComment();
    }
    public int getDeadDeliverCnt() {
        if (lastDead == null)
            return getDeliverCnt();
        ConsumerMessagePair cmp = getAck(lastDead);
        return cmp == null ? getDeliverCnt(): cmp.getRedeliverCount();
    }
    public Reason getDeadReason() {
        if (lastDead == null)
            return null;
        ConsumerMessagePair cmp = getAck(lastDead);
        return cmp == null ? null: cmp.getDeadReason();
    }
    public String getDeadBroker() {
        if (lastDead == null)
            return null;
        ConsumerMessagePair cmp = getAck(lastDead);
        return cmp == null ? null: cmp.getDeadBroker();
    }
    public Throwable getDeadException() {
        if (lastDead == null)
            return null;
        ConsumerMessagePair cmp = getAck(lastDead);
        return cmp == null ? null: cmp.getDeadException();
    }

    public boolean markDead(ConsumerUID intid, String comment,
        Throwable ex, Reason reason, int redeliverCnt, String broker) 
        throws BrokerException
   {
        ConsumerMessagePair cmp = getAck(intid);

        if (cmp == null) {
            // nothing to do
            Globals.getLogger().log(Logger.DEBUG,"Received unknown dead message " 
                   + intid);
            return false;
        }

        if (!isLocal()) { // send remotely and ack
            Hashtable props = new Hashtable();
            if (comment != null)
                props.put(DMQ.UNDELIVERED_COMMENT, comment);
            if (redeliverCnt != -1)
                props.put(Destination.TEMP_CNT, new Integer(redeliverCnt));
            if (ex != null)
                props.put(DMQ.UNDELIVERED_EXCEPTION, ex);
            if (reason != null)
                props.put("REASON", new Integer(reason.intValue()));
            if (broker != null)
                props.put(DMQ.DEAD_BROKER, broker);

            Globals.getClusterBroadcast().
                acknowledgeMessage(getAddress(),
                getSysMessageID(), intid, 
                ClusterBroadcast.MSG_DEAD, true,
                props, null /* txnID */);
            cmp.setState(ACKED);
        } else {
            lastDead = intid;
            cmp.setState(DEAD);
            cmp.setDeadComment(comment);
            cmp.setDeadReason(reason);
            cmp.setDeadException(ex);
            cmp.setDeadBroker(broker);
            if (redeliverCnt > -1)
               cmp.setRedeliverCount(redeliverCnt);
        }
        synchronized (this) {
            deadCnt ++;
            return (ackCnt + deadCnt >= interestCnt);
        }
    }

    public boolean isDead() {
        synchronized( this ) {
            return deadCnt > 0 && (ackCnt + deadCnt >= interestCnt);
        }
    }
    

}

