/*
 * @(#)RedeliverHandler.java	1.36 07/21/04
 *
 * Copyright 2000-2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data.handlers;

import java.io.*;
import java.util.*;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.data.PacketHandler;
import com.sun.messaging.jmq.io.Packet;
import com.sun.messaging.jmq.jmsserver.core.PacketReference;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.service.Connection;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.Session;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.Globals;

import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;


/**
 * Handler class which deals with requests delivering messages
 */
public class RedeliverHandler extends PacketHandler 
{
    // An Ack block is a 4 byte interest ID and a SysMessageID
    static final int REDELIVER_BLOCK_SIZE =  8 + SysMessageID.ID_SIZE;

    private Logger logger = Globals.getLogger();
    public static boolean DEBUG = false;

    public RedeliverHandler() {
    }



    /**
     * Method to handle DELIVER  messages
     */
    public boolean handle(IMQConnection con, Packet msg) 
        throws BrokerException
   {
        Hashtable props = null;
        try {
            props = msg.getProperties();
        } catch (Exception ex) {
            logger.log(Logger.INFO,"Internal Error: unable to retrieve "+
                " properties from redeliver message " + msg, ex);
            props = new Hashtable();
        }

        boolean redeliver = false;

        if (props != null) {
            Boolean bool = (Boolean)props.get("JMQSetRedelivered");
            if (bool != null)
                redeliver = bool.booleanValue();
        } 

        int size = msg.getMessageBodySize();
        int ackcount = size/REDELIVER_BLOCK_SIZE;
        int mod = size%REDELIVER_BLOCK_SIZE;


        if (ackcount == 0 ) {
            return true;
        }
        if (mod != 0) {
            throw new BrokerException(Globals.getBrokerResources().getString(
                BrokerResources.X_INTERNAL_EXCEPTION,"Invalid Redeliver Message Size: " + size +
		". Not multiple of " + REDELIVER_BLOCK_SIZE));
        }

        if (DEBUG) {
            logger.log(Logger.DEBUG,"RedeliverMessage: processing message {0} {1}",
                     msg.toString(), 
                     con.getConnectionUID().toString());
        }
        SysMessageID sysid = new SysMessageID();
        DataInputStream is = new DataInputStream(
		msg.getMessageBodyStream());
        try {

            Set sessions = new HashSet(); //really should only have one
                             // but client to broker protocol doesnt specify
            HashMap cToM = new HashMap();

            int position = 0;
            for (int i = 0; i < ackcount; i ++ ) {
                ConsumerUID id = new ConsumerUID(is.readLong());
                id.setConnectionUID(con.getConnectionUID());
                sysid.readID(is); 
  
                PacketReference ref = Destination.get(sysid);

  
                if (ref == null || ref.isInvalid()) {
                    continue;
                }
                Session s= Session.getSession(id);
                if (!sessions.contains(s)) {
                    s.pause("redeliver");
                    sessions.add(s);
                }
                Consumer c = s.getConsumerOnSession(id);

                if (c == null) {
                    //ok, make sure the consumer has really gone away
                    //if not, something is really wrong
                    // otherwise ...
                    // consumer has gone away but session is still
                    // valid -> 
                    c = Consumer.getConsumer(id);
                    
                    if (c != null) {
                        logger.log(Logger.WARNING,"Internal Error " +
                           " consumer with id of " + id + " is unavailable "
                           + " on session " + s + "[conuid,sess conuid] ="
                           + "[" + con.getConnectionUID().longValue() +
                             "," + (s == null ? 0 : s.getConnectionUID().longValue())
                           + "] consumer session is : " +c.getSessionUID()) ;
                        continue;
                    } else {
                        // the consumer for this message has been 
                        // closed before the redeliver was requested
                        // (this means the message has not been acked and
                        // the session is open)
                        // we dont need to deliver this message, ignore it
                        logger.log(Logger.DEBUG,"Internal Error " +
                           " consumer with id of " + id + " is unavailable "
                           + " on session " + s + "[conuid,sess conuid] ="
                           + "[" + con.getConnectionUID().longValue() +
                             "," + (s == null ? 0 : s.getConnectionUID().longValue())
                           + "] it has been closed") ;
                        continue;
                    }
                }
 
                      
                Set set = (Set)cToM.get(c);
                if (set == null) {
                    set = new LinkedHashSet();
                    cToM.put(c, set);
                }
                set.add(ref);
                if (redeliver) {
                    ref.consumed(c.getStoredConsumerUID(),
                        s.isDupsOK(c.getConsumerUID()), false);
                } else {
                    ref.removeDelivered(c.getStoredConsumerUID(), true);
                }
            }
            Iterator itr = cToM.keySet().iterator();
            while (itr.hasNext()) {
                Consumer c = (Consumer)itr.next();
                Set msgs = (Set)cToM.get(c);
                c.pause("start redeliver");
                c.routeMessages(msgs, true);
                c.resume("end redeliver");
            }
            cToM.clear(); // help gc
                
            itr = sessions.iterator();
            while (itr.hasNext()) {
                Session s = (Session)itr.next();
                s.resume("redeliver");
            }

        } catch (Exception ex) {
            throw new BrokerException(Globals.getBrokerResources().getString(
                BrokerResources.X_INTERNAL_EXCEPTION,"Invalid Redeliver Packet", ex), ex);
        }
        return true;
    }

}
