/*
 * @(#)TransactionHandler.java	1.95 02/13/06
 *
 * Copyright 2000-2005 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.data.handlers;

import java.util.*;
import java.nio.ByteBuffer;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import javax.transaction.xa.XAResource;
import com.sun.messaging.jmq.jmsserver.data.PacketHandler;
import com.sun.messaging.jmq.jmsserver.data.TransactionState;
import com.sun.messaging.jmq.jmsserver.data.TransactionList;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.jmsserver.data.AutoRollbackType;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.ClusterBroadcast;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.core.SessionUID;
import com.sun.messaging.jmq.jmsserver.core.Session;
import com.sun.messaging.jmq.jmsserver.core.BrokerAddress;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.util.UID;
import com.sun.messaging.jmq.jmsserver.service.Connection;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.util.lists.RemoveReason;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.Broker;
import com.sun.messaging.jmq.jmsserver.FaultInjection;
import com.sun.messaging.jmq.jmsserver.BrokerStateHandler;
import com.sun.messaging.jmq.jmsserver.core.PacketReference;
import com.sun.messaging.jmq.jmsserver.management.agent.Agent;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.util.JMQXid;
import com.sun.messaging.jmq.util.CacheHashMap;

import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;

/**
 * Handler class for transaction packets.
 */
public class TransactionHandler extends PacketHandler 
{

    private TransactionList translist = null;

    public static boolean DEBUG = false;

    FaultInjection fi = null;

    static boolean useNewProtocol = Globals.useNewTxnProtocol();

    public TransactionHandler(TransactionList list)
    {
        fi = FaultInjection.getInjection();
        this.translist = list;
    }

    static boolean useNewProtocol()
    {
        return useNewProtocol;
    }

    public void sendReply(IMQConnection con, Packet msg, int type, int status, long tid, String reason)
    {
         if (fi.FAULT_INJECTION)
            checkFIAfterProcess(msg.getPacketType());
          Packet pkt = new Packet(con.useDirectBuffers());
          pkt.setPacketType(type);
          pkt.setConsumerID(msg.getConsumerID());
          Hashtable hash = new Hashtable();
          hash.put("JMQStatus", new Integer(status));
          if (reason != null)
              hash.put("JMQReason", reason);
          if (tid != 0) {
            hash.put("JMQTransactionID", new Long(tid));
          }
          pkt.setProperties ( hash );
	      con.sendControlMessage(pkt);
          if (fi.FAULT_INJECTION)
            checkFIAfterReply(msg.getPacketType());
    }

    /**
     * Send a reply packet with a body and properties
     */
    public void sendReplyBody(IMQConnection con, Packet msg, int type, int status, Hashtable hash, byte[] body)
    {
          Packet pkt = new Packet(con.useDirectBuffers());
          pkt.setPacketType(type);
          pkt.setConsumerID(msg.getConsumerID());

          if (hash == null) {
            hash = new Hashtable();
          }
          hash.put("JMQStatus", new Integer(status));
          if (con.DUMP_PACKET || con.OUT_DUMP_PACKET)
              hash.put("JMQReqID", msg.getSysMessageID().toString());

          pkt.setProperties ( hash );

          if (body != null) {
            pkt.setMessageBody(body);
          }
	  con.sendControlMessage(pkt);
    }

    /**
     * Get the value of the JMQTransactionID property. For 2.0 clients
     * the property is an Integer. For Falcon it is a Long
     */
    public long getJMQTransactionID(Hashtable props) {
        if (props != null) {
            Object obj = props.get("JMQTransactionID");
            if (obj != null && obj instanceof Integer) {
                // old protocol
                return ((Integer)obj).intValue();
            } else if (obj != null) {
                // new protocol
                return ((Long)obj).longValue();
            }
        }
        return 0;
    }

    /**
     * Convert the transaction ID on an iMQ 2.0 packet to the new style.
     */
    public static void convertPacketTid(IMQConnection con, Packet p) {
        long messagetid = p.getTransactionID();
        HashMap tidMap =
            (HashMap)con.getClientData(IMQConnection.TRANSACTION_IDMAP);

        if (tidMap == null) {
            // No transactions have been started yet. Can't convert ID
            return;
        }

        // Lookup old style ID in table
        TransactionUID id = (TransactionUID)tidMap.get(new Long(messagetid));
        if (id == null) {
            return;
        }

        // Convert the old ID to the corresponding new ID
        p.setTransactionID(id.longValue());
    }


    /**
     * Method to handle Transaction Messages
     */
    public boolean handle(IMQConnection con, Packet msg) 
        throws BrokerException
    {

        long    messagetid = 0;
        TransactionUID id = null;
        TransactionState ts = null;
        JMQXid xid = null;
        Integer xaFlags = null;
         
        boolean redeliverMsgs = false;
        boolean setRedeliverFlag = true;
        boolean isIndemp = msg.getIndempontent();
        boolean replay = false;

        Hashtable props = null;
        String reason = null;

        try {
            props = msg.getProperties();
            if (props == null)
                props = new Hashtable();
        } catch (Exception ex) {
            logger.log(Logger.INFO,"Internal Error: unable to retrieve "+
                " properties from transaction message " + msg, ex);
            props = new Hashtable();

        }

        Boolean redeliverMsgBool = (Boolean)props.get("JMQRedeliver");
        redeliverMsgs = (redeliverMsgBool == null ? false : 
                      redeliverMsgBool.booleanValue());
   
        Boolean redeliverFlag = (Boolean)props.get("JMQSetRedelivered");
        setRedeliverFlag = (redeliverFlag == null ? true :
                         redeliverFlag.booleanValue());
        if (DEBUG) {
            logger.log(Logger.DEBUG,"JMQRedeliver flag is " + 
                  redeliverMsgBool);
        }

        List conlist = (List)con.getClientData(IMQConnection.TRANSACTION_LIST);
        if (conlist == null) {
            conlist = new ArrayList();
             con.addClientData(IMQConnection.TRANSACTION_LIST, conlist);
        }

        // If there is a message body, then it should contain an Xid.
        ByteBuffer body = msg.getMessageBody();
        if (body != null) {
            JMQByteBufferInputStream  bbis = new JMQByteBufferInputStream(body);
            try {
                xid = JMQXid.read(new DataInputStream(bbis));
            } catch (IOException e) {
                logger.log(Logger.ERROR,
                        BrokerResources.E_INTERNAL_BROKER_ERROR,
                        "Could not decode xid from packet: " +  e +
                        " Ignoring " +
                        PacketType.getString(msg.getPacketType()));
                reason = e.getMessage();
                sendReply(con, msg, msg.getPacketType() + 1,
                          Status.BAD_REQUEST, 0, reason);
                return true;
            }
        }

        // Get XAFlags. Note, not all packets will have this -- that's OK.
        if (props != null) {
            xaFlags = (Integer)props.get("JMQXAFlags");
        }

        // tidMap maps an old style transaction identifier to a TransactionUID.
        // In iMQ2.0 the transaction identifier in the packet was an int
        // generated by the client and only unique on the connection.
        // In Falcon it is a long that is unique accross the cluster.
        // So for 2.0 clients we allocate a TransactionUID and map the old
        // style identifier to it.
        //
        // Get tidMap
        HashMap tidMap = null;
        synchronized(con) {
            tidMap = (HashMap)con.getClientData(IMQConnection.TRANSACTION_IDMAP);
            if (tidMap == null) {
                tidMap = new HashMap();
                con.addClientData(IMQConnection.TRANSACTION_IDMAP, tidMap);
            }
        }

        // Go ahead and get the value of "JMQTransactionID" from the packet.
        // may not be used in all cases.
        messagetid = getJMQTransactionID(props);


        // handle initial fault injections

        if (fi.FAULT_INJECTION)
            checkFIBeforeProcess(msg.getPacketType());

        // If it is a new transaction then create a new transaction id.
        // else wrap the one specified in the packet
        if (msg.getPacketType() == PacketType.START_TRANSACTION && 
            (xaFlags == null ||
            TransactionState.isFlagSet(XAResource.TMNOFLAGS, xaFlags) )) {

            // It's a START_TRANSACTION that is not a Join, or Resume.
            // Allocate a new TransactionUID


            if (isIndemp) { // deal with indemp flag
                id = translist.getTransaction(msg.getSysMessageID());
                if (id != null) {
                    replay = true;
                } else {
                    id = new TransactionUID();
                }
            } else {
                id = new TransactionUID();
            }
        } else if (msg.getPacketType() == PacketType.RECOVER_TRANSACTION) {
            if (messagetid != 0) {
                // Recovering a specific transaction.
                id = new TransactionUID(messagetid);
            }
            xid = null;
        } else {
            // Operation on an existing transaction
            // Get TransactionUID out of packet
            // If only Xid was specified need to lookup TransactionUID
            if (messagetid == 0 && xid != null) {
                id = translist.xidToUID(xid);

                if (id != null) {
                    messagetid = id.longValue();
                } else {
                    // Hmmm...haven't seen this Xid before.
                    // XXX I18N
                    logger.log(Logger.WARNING,
                        PacketType.getString(msg.getPacketType()) +
                        ": Ignoring unknown XID=" + xid +
                         " broker will " 
                         + (msg.getSendAcknowledge() ? 
                               "notify the client" :
                               " not notify the client" ));
                    if (msg.getSendAcknowledge()) {
                        reason = "Uknown XID " + xid;
                        sendReply(con, msg, msg.getPacketType() + 1,
                            Status.NOT_FOUND, 0, reason);
                    }
                    return true;
                }
            } else if (messagetid != 0) {
                if (con.getClientProtocolVersion() == PacketType.VERSION1) {
                    // Map old style to new
                    synchronized(tidMap) {
                        id = (TransactionUID)tidMap.get(new Long(messagetid));
                    }
                } else {
                    // Wrap new style
                    id = new TransactionUID(messagetid);
                }
            }

            // Get the state of the transaction
            if (id == null) {
                logger.log(Logger.INFO,"InternalError: "
                         + "Transaction ID was not passed by "
                         +"the jms api on a method that reqires an "
                         + "existing transaction ");
                sendReply(con, msg, msg.getPacketType() + 1,
                          Status.ERROR, 0, "Internal Error: bad MQ protocol,"
                               + " missing TransactionID");
                return true;

                
            } else {
                ts = translist.retrieveState(id);
            }

            if (ts == null) {
                // Check if we've recently operated on this transaction

                if (isIndemp && ( msg.getPacketType() == 
                          PacketType.ROLLBACK_TRANSACTION ||
                       msg.getPacketType() == 
                          PacketType.COMMIT_TRANSACTION  )) {
                    if (msg.getSendAcknowledge()) {
                        sendReply(con, msg, msg.getPacketType() + 1,
                            Status.OK, id.longValue(), reason);
                        return true;
                    } else {
                        if (fi.FAULT_INJECTION) {
                            checkFIAfterProcess(msg.getPacketType());
                            checkFIAfterReply(msg.getPacketType());
                        }
                    }

                } else {
                    ts = cacheGetState(id, con);
                    if (ts != null) {
                        // XXX l10n
                        logger.log(Logger.ERROR,
                            "Transaction ID " + id +
                            " has already been resolved. Ignoring request: " +
                            PacketType.getString(msg.getPacketType()) + 
                            ". Last state of this transaction: " + 
                            ts.toString() +
                             " broker will " 
                             + (msg.getSendAcknowledge() ? 
                                   "notify the client" :
                                  " not notify the client" ));
                    } else {
                    logger.log((BrokerStateHandler.shuttingDown? Logger.DEBUG : Logger.ERROR),
                            BrokerResources.E_INTERNAL_BROKER_ERROR,
                            "Unknown transaction: " + id + "(" + messagetid + ")" +
                             " broker will "
                             + (msg.getSendAcknowledge() ? 
                                   "notify the client" :
                                  " not notify the client" ) +
                            " Ignoring " +
                            PacketType.getString(msg.getPacketType()) + "\n" +
                           com.sun.messaging.jmq.jmsserver.util.PacketUtil.dumpPacket(msg));
                    }

                    // Only send reply if A bit is set
                    if (msg.getSendAcknowledge()) {
                        reason = "Unknown transaction " + id;
                        sendReply(con, msg, msg.getPacketType() + 1,
                        Status.NOT_FOUND, id.longValue(), reason);
                    }
                    return true;
                }
            }
        }


        if (DEBUG) {
            logger.log(Logger.DEBUG, this.getClass().getName() + ": " +
                PacketType.getString(msg.getPacketType()) + ": " +
                "TUID=" + id +
                " XAFLAGS=" + TransactionState.xaFlagToString(xaFlags) +
                " State=" + ts + " Xid=" + xid);
        }

        // If a packet came with an Xid, make sure it matches what
        // we have in the transaction table.
        if (xid != null && ts != null) {
            if (ts.getXid() == null || !xid.equals(ts.getXid())) {
                // This should never happen
                logger.log(Logger.ERROR,
                        BrokerResources.E_INTERNAL_BROKER_ERROR,
                        "Transaction Xid mismatch. " +
                        PacketType.getString(msg.getPacketType()) +
                        " Packet has tuid=" +
                        id + " xid=" + xid + ", transaction table has tuid=" +
                        id + " xid=" + ts.getXid() +
                        ". Using values from table.");
                xid = ts.getXid();
            }
        }

        if (xid == null && ts != null && ts.getXid() != null &&
            msg.getPacketType() != PacketType.RECOVER_TRANSACTION) {
            // Client forgot to put Xid in packet.
            xid = ts.getXid();
            logger.log(Logger.WARNING,
                        BrokerResources.E_INTERNAL_BROKER_ERROR,
                        "Transaction Xid not found in " +
                        PacketType.getString(msg.getPacketType()) +
                        " packet for tuid " +
                        id + ". Will use " + xid);
        }


        int status = Status.OK; 

        // retrieve new 4.0 properties
        AutoRollbackType type = null;
        long lifetime = 0;
        boolean sessionLess = false;
        Integer typeValue = (Integer)props.get("JMQAutoRollback");
        Long lifetimeValue = (Long)props.get("JMQLifetime");
        Boolean sessionLessValue = (Boolean)props.get("JMQSessionLess");

        if (typeValue != null) {
            type = AutoRollbackType.getType(typeValue.intValue());
        }

        if (lifetimeValue != null) {
            lifetime = lifetimeValue.longValue();
        }

        if (sessionLessValue != null) {
            sessionLess = sessionLessValue.booleanValue();
        } else {
            sessionLess = xid != null;
        }


        switch (msg.getPacketType())  {
            case PacketType.START_TRANSACTION:
            {
                SessionUID suid = null;
                Long sessionID = (Long)props.get("JMQSessionID");
                if (sessionID != null) {
                    suid = new SessionUID(sessionID.longValue());
                }
 
                if (type == AutoRollbackType.NEVER || lifetime > 0) {
                    // not supported
                    status = Status.NOT_IMPLEMENTED;
                    reason = "AutoRollbackType of NEVER not supported";
                } else if (xid != null && !sessionLess) {
                    // not supported yet
                    status = Status.NOT_IMPLEMENTED;
                    reason = "XA transactions only supported on sessionless "
                              + "connections";
                } else if (xid == null && sessionLess) {
                    // not supported yet
                    status = Status.ERROR;
                    reason = "non-XA transactions only supported on "
                              + " non-sessionless connections";

                } else {
        
                    if (replay) {
                        // do nothing it already happened
                    } else if (xaFlags != null &&
                        !TransactionState.isFlagSet(XAResource.TMNOFLAGS, xaFlags))
                    {
                        // This is either a TMJOIN or TMRESUME. We just need to
                        // update the transaction state
                        try {
                            int s = ts.nextState(msg.getPacketType(), xaFlags);
                            translist.updateState(id, s, true);
                        } catch (BrokerException ex) {
                            logger.log(Logger.ERROR,
                                BrokerResources.E_INTERNAL_BROKER_ERROR,
                                ex.toString() + ": TUID=" + id + " Xid=" + xid);
                            reason = ex.getMessage();
                            status = ex.getStatusCode();
                        }
                    } else {
                      // Brand new transaction
                        try {
                            if (con.getClientProtocolVersion() ==
                                                        PacketType.VERSION1) {
                                // If 2.0 client Map old style ID to new
                                tidMap.put(new Long(messagetid), id);
                            }

                            ts = new TransactionState(type, lifetime, sessionLess);
                            ts.setState(TransactionState.STARTED);
                            ts.setUser(con.getUserName());
                            ts.setCreator((SysMessageID)
                                  msg.getSysMessageID().clone());
                            ts.setClientID(
                                (String)con.getClientData(IMQConnection.CLIENT_ID));
                            ts.setXid(xid);

                            if (con instanceof IMQConnection) {
                                ts.setConnectionString(
                                    ((IMQConnection)con).userReadableString());
                            }
                            translist.addTransactionID(id, ts);

                            conlist.add(id);
                        } catch (BrokerException ex) {
                            // XXX I18N
                            logger.log(Logger.WARNING,
                                "Exception starting new transaction: " +
                                ex.toString());
                            reason = ex.getMessage();
                            status = ex.getStatusCode(); 
                        }
                  }
                }
                sendReply(con, msg, PacketType.START_TRANSACTION_REPLY,
                            status, id.longValue(), reason);
                break;

            }

            case PacketType.END_TRANSACTION:
            case PacketType.PREPARE_TRANSACTION:
                // XATransaction only
                // These just alter the transaction state.
                try {
                    int s = ts.nextState(msg.getPacketType(), xaFlags);
                    translist.updateState(id, s, true);
		            if (msg.getPacketType() == PacketType.PREPARE_TRANSACTION)  {
                        HashMap cmap = null;
                        cmap = translist.retrieveConsumedMessages(id);
                        HashSet processedBrokers = new HashSet();
                        if (cmap != null && cmap.size() > 0) {
                            Iterator itr = cmap.keySet().iterator();
                            while (itr.hasNext()) {
                                SysMessageID sysid = (SysMessageID)itr.next();
                                if (sysid == null) continue;
                                PacketReference ref = Destination.get(sysid);
                                List interests = (List) cmap.get(sysid);
                                for (int i = 0; i < interests.size(); i ++) { 
                                    if (!ref.isLocal() && useNewProtocol()) {
                                        BrokerAddress addr = ref.getAddress();
                                        if (processedBrokers.contains(addr))
                                            continue;
                                        processedBrokers.add(addr);
                                    }
                                    ConsumerUID intid = (ConsumerUID) interests.get(i);
                                    ref.prepare(intid, id, useNewProtocol());
                                }
                            }
                        }
	                Agent agent = Globals.getAgent();
	                if (agent != null)  {
		            agent.notifyTransactionPrepare(id);
	                }
		    }
                } catch (Exception ex) {
                    logger.logStack(Logger.ERROR,
                            BrokerResources.E_INTERNAL_BROKER_ERROR,
                            ex.toString() + ": TUID=" + id + " Xid=" + xid, ex);
                    reason = ex.getMessage();
                    if (ex instanceof BrokerException)
                        status = ((BrokerException)ex).getStatusCode();
                }
                sendReply(con, msg,  msg.getPacketType() + 1, status, id.longValue(), reason);
                break;

            case PacketType.RECOVER_TRANSACTION:

                Vector v = null;

                if (id != null) {
                    // Check if specified transaction is in PREPARED state
                    v = new Vector();
                    ts = translist.retrieveState(id);
                    if (ts.getState() == TransactionState.PREPARED) {
                        v.add(id);
                    }
                } else {
                    // XA Transaction only. We return all Xids on a STARTRSCAN
                    // and nothing on ENDRSCAN or NOFLAGS
                    if (xaFlags == null ||
                        !TransactionState.isFlagSet(XAResource.TMSTARTRSCAN,
                                                                    xaFlags)) {
                        Hashtable hash = new Hashtable();
                        hash.put("JMQQuantity", new Integer(0));
    
                        sendReplyBody(con, msg,
                            PacketType.RECOVER_TRANSACTION_REPLY,
                            Status.OK, hash, null);
                        break;
                    }
                    // Get list of transactions in PENDING state and marshal
                    // the Xid's to a byte array.
                    v = translist.getTransactions(TransactionState.PREPARED);
                }

                int nIDs = v.size();
                int nWritten = 0;
                ByteArrayOutputStream bos =
                            new ByteArrayOutputStream(nIDs * JMQXid.size());
                DataOutputStream dos = new DataOutputStream(bos);
                for (int n = 0; n < nIDs; n++) {
                    TransactionUID tuid = (TransactionUID)v.get(n);
                    TransactionState _ts = translist.retrieveState(tuid);
                    if (_ts == null) {
                        // Should never happen
                        logger.log(Logger.ERROR,
                                BrokerResources.E_INTERNAL_BROKER_ERROR,
                                "Could not find state for TUID " + tuid);
                        continue;
                    }
                    JMQXid _xid = _ts.getXid();
                    if (_xid != null) {
                        try {
                            _xid.write(dos);
                            nWritten++;
                        } catch (Exception e) {
                            logger.log(Logger.ERROR,
                                BrokerResources.E_INTERNAL_BROKER_ERROR,
                                "Could not write Xid " +
                                _xid + " to message body: " +
                                e.toString());
                        }
                    }
                }

                Hashtable hash = new Hashtable();
                hash.put("JMQQuantity", new Integer(nWritten));

                if (id != null) {
                    hash.put("JMQTransactionID", new Long(id.longValue()));
                }

                // Send reply with serialized Xids as the body
                sendReplyBody(con, msg, PacketType.RECOVER_TRANSACTION_REPLY,
                    Status.OK, hash, bos.toByteArray());
                break;


            case PacketType.COMMIT_TRANSACTION:

                //XXX-LKS clean up any destination locks ???

                try {
                    // doCommit will send reply if successful
                    doCommit(id, xid, xaFlags, ts, conlist,
                        true, con, msg);
                } catch (BrokerException ex) {
                    // doCommit has already logged error
                    status = ex.getStatusCode();
                    reason = ex.getMessage();
                    if (msg.getSendAcknowledge()) {
                        sendReply(con, msg,  msg.getPacketType() + 1,
                            status, id.longValue(), reason);
                    } else {
                        if (fi.FAULT_INJECTION) {
                            checkFIAfterProcess(msg.getPacketType());
                            checkFIAfterReply(msg.getPacketType());
                        }
                    }

                }
                break;
                
            case PacketType.ROLLBACK_TRANSACTION:
            {

                try {
                    // once we fix bug 4970642 we will also want
                    // to call this method if redeliverMsgs is NOT
                    // true
                    if (redeliverMsgs) {
                        if (DEBUG) {
                            logger.log(Logger.INFO,"Redelivering msgs for "
                                + id);
                        }
                        // if redeliverMsgs is true, we want to redeliver
                        // to both active and inactive consumers 
                        boolean processActiveConsumers = redeliverMsgs;
                        redeliverUnacked(id, processActiveConsumers, setRedeliverFlag);
                    }
                } catch (BrokerException ex) {
                    logger.logStack(Logger.ERROR,
                        BrokerResources.E_INTERNAL_BROKER_ERROR,
                        "REDELIVER: " + 
                        ex.toString() + ": TUID=" + id + " Xid=" + xid, ex);
                    reason = ex.getMessage();
                    status = ex.getStatusCode();
                }
                try {
                    doRollback(id, xid, xaFlags, ts, conlist, con);
                } catch (BrokerException ex) {
                    // doRollback has already logged error
                    reason = ex.getMessage();
                    status = ex.getStatusCode();
                }
                if (msg.getSendAcknowledge()) {
                    sendReply(con, msg,  msg.getPacketType() + 1, status,
                        id.longValue(), reason);
                } else {
                    if (fi.FAULT_INJECTION) {
                        checkFIAfterProcess(msg.getPacketType());
                        checkFIAfterReply(msg.getPacketType());
                    }
                }
                break;
            }
        }

        return true;
    }

    public void checkFIBeforeProcess(int type)
    {
        switch (type) {
            case PacketType.START_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_START_1,
                      null, 2, false);
                break;
            case PacketType.END_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_END_1,
                      null, 2, false);
                break;
            case PacketType.PREPARE_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_PREPARE_1,
                      null, 2, false);
                break;
            case PacketType.ROLLBACK_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_ROLLBACK_1,
                      null, 2, false);
                break;
            case PacketType.COMMIT_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_COMMIT_1,
                      null, 2, false);
                break;
        }
    }

    public void checkFIAfterProcess(int type)
    {
        switch (type) {
            case PacketType.START_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_START_2,
                      null, 2, false);
                break;
            case PacketType.END_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_END_2,
                      null, 2, false);
                break;
            case PacketType.PREPARE_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_PREPARE_2,
                      null, 2, false);
                break;
            case PacketType.ROLLBACK_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_ROLLBACK_2,
                      null, 2, false);
                break;
            case PacketType.COMMIT_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_COMMIT_2,
                      null, 2, false);
                break;
        }
    }

    public void checkFIAfterDB(int type)
    {
        switch (type) {
            case PacketType.ROLLBACK_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_ROLLBACK_4,
                      null, 2, false);
                break;
            case PacketType.COMMIT_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_COMMIT_4,
                      null, 2, false);
                break;
        }
    }

    public void checkFIAfterReply(int type)
    {
        switch (type) {
            case PacketType.START_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_START_3,
                      null, 2, false);
                break;
            case PacketType.END_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_END_3,
                      null, 2, false);
                break;
            case PacketType.PREPARE_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_PREPARE_3,
                      null, 2, false);
                break;
            case PacketType.ROLLBACK_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_ROLLBACK_3,
                      null, 2, false);
                break;
            case PacketType.COMMIT_TRANSACTION:
                fi.checkFaultAndExit(FaultInjection.FAULT_TXN_COMMIT_3,
                      null, 2, false);
                break;
        }
    }

    /**
     * We cache the last N transactions to be committed or rolledback.
     * This is simply to let us more easily debug problems when the
     * client sends a bogus transaction ID
     */
    private void cacheSetState(TransactionUID id,
                            TransactionState ts,
                            IMQConnection con) {

        if (con == null) return;
        CacheHashMap cache =
            (CacheHashMap)con.getClientData(IMQConnection.TRANSACTION_CACHE);
        if (cache == null) {
            cache = new CacheHashMap(4);
            con.addClientData(IMQConnection.TRANSACTION_CACHE, cache);
        }

        cache.put(id, ts);
    }

    private TransactionState cacheGetState(TransactionUID id,
                                           IMQConnection con) {

        CacheHashMap cache =
            (CacheHashMap)con.getClientData(IMQConnection.TRANSACTION_CACHE);
        if (cache == null) {
            return null;
        } else {
           return (TransactionState)cache.get(id);
        }
    }


    /**
     * Commit a transaction. This method is invoked from two places:
     * 1) From TransactionHandler.handle() when handling a client
     *    COMMIT packet. This is the common case.
     * 2) From the admin handler when an admin commit request has been
     *    issued on a PREPARED XA transaction.
     *
     * @param id  The TransactionUID to commit
     * @param xid The Xid of the transaction to commit. Required if transaction
     *            is an XA transaction. Must be null if it is not an XA
     *            transaction.
     * @param xaFlags  xaFlags passed on COMMIT operation. Used only if
     *                 an XA transaction.
     * @param ts       Current TransactionState of this transaction.
     * @param conlist  List of transactions on this connection. Will be null
     *                 if commit is trigger by an admin request.
     * @param sendReply True to have method send a Status.OK reply while
     *                  processing transaction. This should be "true" for
     *                  client initiated commits, "false" for admin
     *                  initiated commits.
     * @param con       Connection client commit packet came in on or, for
     *                  admin, the connection the admin request came in on.
     * @param msg       Client commit packet. Should be "null" for admin
     *                  initiated commits.
     *
     * @throws BrokerException on an error. The method will have logged a
     *         message to the broker log.
     */
    public void doCommit(TransactionUID id, JMQXid xid,
                          Integer xaFlags, TransactionState ts, List conlist,
                          boolean sendReply, IMQConnection con, Packet msg)
                          throws BrokerException {

        int status = Status.OK;

        HashMap cmap = null;
        HashMap sToCmap = null;
        BrokerException bex = null;
        List plist = null;


        // let acks get handled at a lower level since the
        // lower level methods assumes only 1 ack per message
            plist = translist.retrieveSentMessages(id);
            cmap = translist.retrieveConsumedMessages(id);
            sToCmap = translist.retrieveStoredConsumerUIDs(id);
            cacheSetState(id, ts, con);

            // remove from our active connection list
            if (conlist != null) {
                conlist.remove(id);
            }

        // Update transaction state
        try {
            int s;
            if (xid == null) {
                // Plain JMS transaction.
                s = TransactionState.COMMITTED;
            } else {
                // XA Transaction. 
                s = ts.nextState(PacketType.COMMIT_TRANSACTION, xaFlags);
            }
            ts = translist.updateState(id, s, true);
            if (fi.FAULT_INJECTION)
                checkFIAfterDB(PacketType.COMMIT_TRANSACTION);

            // ok - two choices:
            //  if >= 4.0 broker, send one null packet to both
            //  otherwise, send all
            HashSet processedBrokers = new HashSet();
            if (cmap != null && cmap.size() > 0) {
                Iterator itr = cmap.keySet().iterator();
                while (itr.hasNext()) {
                SysMessageID sysid = (SysMessageID)itr.next();
                    if (sysid == null) continue;
                    PacketReference ref = Destination.get(sysid);
                    // ok, now deal with the remote brokers
                    // if we are running in "new protocol" mode
                    // only send one COMMIT per broker
                    if (ref == null) continue;
                    List interests = (List) cmap.get(sysid);
                    for (int i = 0; i < interests.size(); i ++) { 
                        ConsumerUID intid = (ConsumerUID) interests.get(i);
                        if (!ref.isLocal() && useNewProtocol()) {
                            BrokerAddress addr = ref.getAddress();
                            if (processedBrokers.contains(addr))
                                continue;
                            processedBrokers.add(addr);
                        }

                        ref.commit(intid, id, useNewProtocol());
                    }
                }
            }
	    /*
	     * Can't really call the JMX notification code at the end of doCommit()
	     * because the call to translist.removeTransactionID(id) removes the
	     * MBean.
	     */
	    Agent agent = Globals.getAgent();
	    if (agent != null)  {
		agent.notifyTransactionCommit(id);
	    }
        } catch (BrokerException ex) {
            logger.logStack(Logger.ERROR,
                    BrokerResources.E_INTERNAL_BROKER_ERROR,
                    ex.toString() + ": TUID=" + id + " Xid=" + xid, ex);
            throw ex;
        } catch (IOException ex) {
            logger.logStack(Logger.ERROR,
                    BrokerResources.E_INTERNAL_BROKER_ERROR,
                    ex.toString() + ": TUID=" + id + " Xid=" + xid, ex);
            throw new BrokerException("Error on commit", ex);
        }



        if (status != Status.OK) {
            throw bex;
        } else if (sendReply) {
            //chiaming
            sendReply(con, msg, PacketType.COMMIT_TRANSACTION_REPLY, status,
                id.longValue(), null);
        }

        // OK .. handle producer transaction
        for (int i =0; plist != null && i < plist.size(); i ++ ) {
            SysMessageID sysid = (SysMessageID)plist.get(i);
            PacketReference ref = Destination.get(sysid);
            if (ref == null) {
                logger.log(Logger.ERROR,BrokerResources.E_INTERNAL_BROKER_ERROR,  "transacted message removed too early "+sysid);
                continue;
            }
                     
            // handle forwarding the message
            try {
                Destination d = Destination.getDestination(ref.getDestinationUID());
                Set s = d.routeNewMessage(ref);
                d.forwardMessage(s,ref);

             } catch (Exception ex) {
                 logger.logStack((BrokerStateHandler.shuttingDown? Logger.DEBUG : Logger.ERROR),BrokerResources.E_INTERNAL_BROKER_ERROR, "unable to route/send transaction message " + sysid , ex);
             }

        }

        // handle consumer transaction
        if (cmap != null && cmap.size() > 0) {
            Iterator itr = cmap.keySet().iterator();
            while (itr.hasNext()) {
                SysMessageID sysid = (SysMessageID)itr.next();
                // CANT just pull from connection
                if (sysid == null) continue;
                PacketReference ref = Destination.get(sysid);
                if (ref == null) continue;
                if (!ref.isLocal()) {
                    // should we do anything ?
                }
                List interests = (List) cmap.get(sysid);
                for (int i = 0; i < interests.size(); i ++) {
                    ConsumerUID intid = (ConsumerUID) interests.get(i);
                    try {
                        if (ref == null || ref.isDestroyed() ||
                            ref.isInvalid())  {
                            // already been deleted .. ignore
                            continue;
                        }
                        Session s = Session.getSession(intid);
                        if (s != null) {
                            PacketReference r1 =s.ackMessage(intid, sysid, 
(useNewProtocol() ? id : null) );
                            if (r1 != null) {
                                 Destination d=Destination.getDestination(
                                     ref.getDestinationUID());
                                 d.removeMessage(ref.getSysMessageID(),
                                     RemoveReason.ACKNOWLEDGED);
                             }

                        } else {
                            //OK the session has been closed, we need
                            // to retrieve the message and acknowledge it
                            // with the stored UID
                            ConsumerUID sid = (ConsumerUID)
                                   sToCmap.get(intid);
                            if (sid == null)
                               sid = intid;
                            try {
                                if (ref.acknowledged(intid, sid, 
                                    true, true, id)) {
                                      Destination d=Destination.getDestination(
                                         ref.getDestinationUID());
                                      d.removeMessage(ref.getSysMessageID(),
                                          RemoveReason.ACKNOWLEDGED);
                                }
                            } catch (BrokerException ex) {
                                // XXX improve internal error
                                logger.log(Logger.WARNING,"Internal error", ex);
                            }
                        }
                    } catch (Exception ex) {
                        logger.logStack(Logger.ERROR,BrokerResources.E_INTERNAL_BROKER_ERROR,
                            "-------------------------------------------" +
                            "Processing Acknowledgement during committ [" +
                            sysid + ":" + intid + ":" + con.getConnectionUID()+
                            "]\nReference is " + (ref == null ? null : ref.getSysMessageID())
                            + "\n" + com.sun.messaging.jmq.jmsserver.util.PacketUtil.dumpPacket(msg)
                            + "--------------------------------------------",
                            ex);
                    }
                }
            }
        }

       
        // OK .. now remove the acks .. and free up the id for ues
        // XXX Fixed 6383878, memory leaks because txn ack can never be removed
        // from the store if the txn is removed before the ack; this is due
        // to the fack that in 4.0 when removing the ack, the method check
        // to see if the txn still exits in the cache. This temporary fix
        // will probably break some HA functionality and need to be revisited.
        translist.removeTransactionAck(id);
        translist.removeTransactionID(id);
        return;
    }

    /**
     * Rollback a transaction. This method is invoked from two places:
     * 1) From TransactionHandler.handle() when handling a client
     *    ROLLBACK packet. This is the common case.
     * 2) From the admin handler when an admin rollback request has been
     *    issued on a PREPARED XA transaction.
     *
     * @param id  The TransactionUID to commit
     * @param xid The Xid of the transaction to commit. Required if transaction
     *            is an XA transaction. Must be null if it is not an XA
     *            transaction.
     * @param xaFlags  xaFlags passed on COMMIT operation. Used only if
     *                 an XA transaction.
     * @param ts       Current TransactionState of this transaction.
     * @param conlist  List of transactions on this connection. Will be null
     *                 if commit is trigger by an admin request.
     * @param con       Connection client commit packet came in on or, for
     *                  admin, the connection the admin request came in on.
     *
     * @throws BrokerException on an error. The method will have logged a
     *         message to the broker log.
     */
    public void doRollback(TransactionUID id, JMQXid xid,
                          Integer xaFlags, TransactionState ts, List conlist,
                          IMQConnection con)
                          throws BrokerException {

        int status = Status.OK;
        int s;

        // Update transaction state
        try {
            if (xid == null) {
                // Plain JMS transaction.
                s = TransactionState.ROLLEDBACK;
            } else {
                // XA Transaction. 
                s = ts.nextState(PacketType.ROLLBACK_TRANSACTION, xaFlags);
            }
        } catch (BrokerException ex) {
            logger.log(Logger.ERROR,
                    BrokerResources.E_INTERNAL_BROKER_ERROR,
                    ex.toString() + ": TUID=" + id + " Xid=" + xid);
            throw ex;
        }
        ts = translist.updateState(id, s, true);
        if (fi.FAULT_INJECTION)
            checkFIAfterDB(PacketType.ROLLBACK_TRANSACTION);

        try {
            // send out rollbacks
            HashMap cmap = null;
            cmap = translist.retrieveConsumedMessages(id);
            // hashmap of brokers->packet references
            HashSet processedBrokers = new HashSet();
            if (cmap != null && cmap.size() > 0) {
                Iterator itr = cmap.keySet().iterator();
                while (itr.hasNext()) {
                    SysMessageID sysid = (SysMessageID)itr.next();
                    if (sysid == null) continue;
                    PacketReference ref = Destination.get(sysid);
                    List interests = (List) cmap.get(sysid);
                    for (int i = 0; i < interests.size(); i ++) { 
                        ConsumerUID intid = (ConsumerUID) interests.get(i);
                        if (!ref.isLocal() && useNewProtocol()) {
                            BrokerAddress addr = ref.getAddress();
                            if (processedBrokers.contains(addr))
                                continue;
                            processedBrokers.add(addr);
                        }
                        ref.rollback(intid, id, useNewProtocol());
                    }
                }
            }
        } catch (Exception ex) {
               logger.logStack(Logger.ERROR,
                    BrokerResources.E_INTERNAL_BROKER_ERROR,
                    ex.toString() + ": TUID=" + id + " Xid=" + xid, ex);
        }
        List list = new ArrayList(translist.retrieveSentMessages(id));
        for (int i =0; list != null && i < list.size(); i ++ ) {
            SysMessageID sysid = (SysMessageID)list.get(i);
            if (DEBUG) {
                logger.log(Logger.DEBUG,"Removing " +
                      sysid + " because of rollback");
            }
            PacketReference ref = Destination.get(sysid);
            if (ref == null) continue;
            DestinationUID duid = ref.getDestinationUID();
            Destination d = Destination.getDestination(duid);
            d.removeMessage(sysid, RemoveReason.ROLLBACK);
            
            // remove from our active connection list
            if (conlist != null) {
                conlist.remove(id);
            }
        }

        // re-queue any orphan messages

        // how we handle the orphan messages depends on a couple
        // of things:
        //   - has the session closed ?
        //     if the session has closed the messages are "orphan"
        //   - otherwise, the messages are still "in play" and
        //     we dont do anything with them
        //
        Map m = translist.getOrphanAck(id);
        if (m != null) {
            // returns a sysid -> cuid mapping
            Iterator itr = m.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry me = (Map.Entry)itr.next();
                SysMessageID sysid = (SysMessageID)me.getKey();
                PacketReference ref = Destination.get(sysid);
                if (ref == null) {
                    logger.log(Logger.DEBUG,id + ":Unknown orphan " + sysid);
                    continue;
                }
                List l = (List)me.getValue();
                Iterator ciditr = l.iterator();
                while (ciditr.hasNext()) {
                    ConsumerUID cuid = (ConsumerUID)ciditr.next();
                    ref.getDestination().forwardOrphanMessage(ref, cuid);
                }
            }
        }

        // OK .. now remove the acks
        translist.removeTransactionAck(id);

	/*
	* Can't really call the JMX notification code at the end of doRollback()
	* because the call to translist.removeTransactionID(id) removes the
	* MBean.
	*/
	Agent agent = Globals.getAgent();
	if (agent != null)  {
	    agent.notifyTransactionRollback(id);
	}

       
        try {
            ts.setState(s);
            cacheSetState(id, ts, con);
            translist.removeTransactionID(id);
        } catch (BrokerException ex) {
            logger.logStack(Logger.ERROR, BrokerResources.E_INTERNAL_BROKER_ERROR, "exception removing transaction", ex);
            throw ex;
        }
    }


    /**
     * this method triggers a redeliver of all unacked
     * messages in the transaction.
     * Since there may be multiple connections for the various
     * consumers .. we guarentee that the messages for this
     * transaction are resent in order but make no guarentee
     * about other messages arriving before these messages.
     */
    void redeliverUnacked(TransactionUID tid, boolean processActiveConsumers,
             boolean redeliver) 
        throws BrokerException
    {
        List plist = null;
        HashMap cmap = null;
        HashMap sToCmap = null;
        BrokerException bex = null;
        cmap = translist.retrieveConsumedMessages(tid);
        sToCmap = translist.retrieveStoredConsumerUIDs(tid);
        /* OK .. we've retrieved the messages .. now go through
           them and get References and order them
         */

        HashMap sendMap = new HashMap();

        /*
         * transaction ack data is stored as message -> list of IDs
         */
        // loop through the transaction ack list
        if (cmap != null && cmap.size() > 0) {
            Iterator itr = cmap.keySet().iterator();
            while (itr.hasNext()) {
                // for each message, get the ref

                SysMessageID sysid = (SysMessageID)itr.next();
                // CANT just pull from connection
                if (sysid == null) continue;
                PacketReference ref = Destination.get(sysid);

                if (ref == null || ref.isDestroyed() ||
                      ref.isInvalid())  {
                      // already been deleted .. ignore
                      continue;
                }

                // now get the interests and stick each message
                // in a list specific to the consumer
                // the list is automatically sorted
                List interests = (List) cmap.get(sysid);
                for (int i = 0; i < interests.size(); i ++) {
                    ConsumerUID intid = (ConsumerUID) interests.get(i);
                    ConsumerUID stored = (ConsumerUID)sToCmap.get(intid);
                    if (stored == null)
                        stored = intid;

                    SortedSet ss = (SortedSet)sendMap.get(intid);
                    if (ss == null) {
                        ss = new TreeSet(new RefCompare());
                        sendMap.put(intid, ss);
                    }
                    try {
                        if (redeliver) {
                            ref.consumed(stored, false, false);
                        } else {
                            ref.removeDelivered(stored, true);
                        }
                    } catch (IOException ex) {
                        logger.log(Logger.WARNING,"Internal error",
                           ex);
                    }
                    ss.add(ref);
                }
            }
        }

        // OK -> now we have a NEW hashmap id -> messages (sorted)
        // lets resend the ones we can to the consumer
        Iterator sitr = sendMap.keySet().iterator();
        while (sitr.hasNext()) {
            ConsumerUID intid = (ConsumerUID)sitr.next();
            Consumer cs = Consumer.getConsumer(intid);
            if (cs == null) {
                if (DEBUG) {
                    logger.log(Logger.INFO,tid + ":Can not redeliver messages to "
                       + intid + " consumer is gone");
                }
                continue;
            }
            SortedSet ss = (SortedSet)sendMap.get(intid);
            if (DEBUG) {
                logger.log(Logger.INFO,tid + ":Redelivering " + 
                       ss.size() + " msgs to "
                       + intid );
            }
            if (!processActiveConsumers) {
               // we dont want to process open consumers
               // if the boolen is false ->
               // remove them so that we only process inactive
               // consumers later
                sitr.remove();
            } else if (cs.routeMessages(ss, true /* toFront*/)) {
                if (DEBUG) {
                    logger.log(Logger.INFO,"Sucessfully routed msgs to "
                          + cs);
                }
                sitr.remove();
            } else { // couldnt route messages ..invalid consumer
                if (DEBUG) {
                    logger.log(Logger.INFO,"Could not route messages to "
                          + cs);
                }
            }
        }

        // finally deal w/ any remaining messages (those are the
        // ones that dont have consumers)
        if (DEBUG) {
            logger.log(Logger.INFO,tid + ":after redeliver, " +
                sendMap.size() + " inactive consumers remaining");
        }

        // loop through remaining messages
        sitr = sendMap.keySet().iterator();
        while (sitr.hasNext()) {
            ConsumerUID intid = (ConsumerUID)sitr.next();
            ConsumerUID storedID =(ConsumerUID) sToCmap.get(intid);
            if (storedID == null || intid == storedID) { 
                    // non-durable subscriber
                    // ignore
                    sitr.remove();
                   continue;
            }
            // see if we are a queue
            if (storedID == PacketReference.queueUID) {
                // queue message on 
                SortedSet ss = (SortedSet)sendMap.get(intid);
                if (ss.isEmpty()) {
                    if (DEBUG) {
                        logger.log(Logger.INFO,"Internal Error: "
                           + " empty set");
                    }
                    continue;
                }
                // queues are complex ->
                PacketReference ref = (PacketReference)ss.first();

                if (ref == null) {
                    if (DEBUG) {
                        logger.log(Logger.INFO,"Internal Error: "
                           + " null reterence");
                    }
                    continue;
                }
                try {
                    if (redeliver) {
                        ref.consumed(storedID, false, false);
                    } else {
                        ref.removeDelivered(storedID, false);
                    }
                } catch (IOException ex) {
                    logger.log(Logger.WARNING,"Internal error",
                           ex);
                }

                Destination d= ref.getDestination();
                if (d == null) {
                    if (DEBUG) {
                        logger.log(Logger.INFO,"Internal Error: "
                           + " unknown destination for reference: "
                           + ref);
                    }
                    continue;
                }

                Iterator ditr = ss.iterator();

                // route each message to the destination
                // this puts it on the pending list
                while (ditr.hasNext()) {
                    ref = (PacketReference)ditr.next();
                    try {
                        Set s = d.routeNewMessage(ref);
                        d.forwardMessage(s,ref);
                    } catch (Exception ex) {
                        logger.log(Logger.INFO,"Internal Error: "
                             + "Unable to re-queue message "
                             + " to queue " + d, ex);
                    }
                }
            } else { // durable 
                // ok - requeue the message on the subscription
                Consumer cs = Consumer.getConsumer(storedID);
                if (cs == null) {
                    if (DEBUG) {
                        logger.log(Logger.INFO,"Internal Error: "
                           + " unknown consumer " + storedID);
                    }
                    continue;
                }
                SortedSet ss = (SortedSet)sendMap.get(intid);
                if (ss != null && !ss.isEmpty() &&
                     cs.routeMessages(ss, true /* toFront*/)) {
                    // successful
                    sitr.remove();
                }
            }
            
        }
        if (DEBUG && sendMap.size() > 0) {
            logger.log(Logger.INFO,tid + ":after all processing, " +
                sendMap.size() + " inactive consumers remaining");
        }

    }
}


