/*
 * @(#)StoreManager.java	1.17 09/16/05
 *
 * Copyright 2000-2002 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.persist;

import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.persist.jdbc.JDBCStore;
import com.sun.messaging.jmq.jmsserver.persist.jdbc.DBConnectionPool;
import com.sun.messaging.jmq.jmsserver.config.*;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;

import java.lang.reflect.*;

/**
 * This class contains static methods to obtain a singleton Store instance
 * for storing and retrieving data needed by the broker.
 */

public class StoreManager {

    static private final String PERSIST_PROP = Globals.IMQ + ".persist.";
    static private final String CLASS_PROP = ".class";
    static private final String STORE_TYPE_PROP =
					Globals.IMQ + ".persist.store";

    static private final String DEFAULT_STORE_TYPE = Store.FILE_STORE_TYPE;

    static private final String DEFAULT_FILESTORE_CLASS =
	"com.sun.messaging.jmq.jmsserver.persist.file.FileStore";

    static private final String DEFAULT_JDBCSTORE_CLASS =
	"com.sun.messaging.jmq.jmsserver.persist.jdbc.JDBCStore";

    // Singleton Store instance
    static private Store store = null;

    private Logger logger = Globals.getLogger();
    private BrokerResources br = Globals.getBrokerResources();

    /**
     * Return a singleton instance of a Store object.
     * <p>
     * The type of store to use is defined in the property:<br>
     * jmq.persist.store=<type>
     * <p>
     * The class to use for the specified type is defined in:<br>
     * jmq.persist.<type>.class=<classname>
     * <p>
     * If the type property is not defined, the default file based
     * store will be instantiated and returned.
     * If 'jdbc' type is defined, and no class is defined, the default
     * jdbc based store will be instantiated and returned.
     * <p>
     * If the type property is defined but we fail to instantiate the
     * correspoinding class, a BrokerException will be thrown
     * @return	a Store
     * @exception BrokerException	if it fails to instantiate a Store
     *					instance
     */
    public static synchronized Store getStore() throws BrokerException {
        Logger logger = Globals.getLogger();
        BrokerResources br = Globals.getBrokerResources();

	if (store == null) {
	    BrokerConfig config = Globals.getConfig();

	    String type = config.getProperty(STORE_TYPE_PROP,
					DEFAULT_STORE_TYPE);
	    if (Store.DEBUG) {
		logger.log(logger.DEBUG, STORE_TYPE_PROP + "=" + type);
	    }

	    String classname = config.getProperty(
				PERSIST_PROP + type + CLASS_PROP);
	    
	    if (classname == null || classname.equals("")) {
		if (type.equals(Store.FILE_STORE_TYPE))
		    classname = DEFAULT_FILESTORE_CLASS;
		else if (type.equals(Store.JDBC_STORE_TYPE))
		    classname = DEFAULT_JDBCSTORE_CLASS;
		else
		    classname = null;
	    }

	    if (classname == null) {
		logger.log(logger.ERROR, br.E_BAD_STORE_TYPE, type);
		throw new BrokerException(
				br.getString(br.E_BAD_STORE_TYPE, type));
	    } else {
		if (Store.DEBUG) {
		    logger.log(logger.DEBUG,
			PERSIST_PROP + type + CLASS_PROP + "=" + classname);
		}
	    }

	    Exception exception = null;
	    try {
		store = (Store)Class.forName(classname).newInstance();
	    } catch (ClassNotFoundException e) {
		exception = e;
	    } catch (InstantiationException e) {
		exception = e;
	    } catch (IllegalAccessException e) {
		exception = e;
	    } finally {
		if (exception != null) {
		    if (exception instanceof BrokerException) {
			throw (BrokerException)exception;
		    } else {
			throw new BrokerException(
				br.getString(br.E_OPEN_STORE_FAILED),
				exception);
		    }
		}
	    }
	}

	return store;
    }

    /**
     * Release the singleton instance of a Store object.
     * The store will be closed and any resources used by it released.
     * <p>
     * The next time <code>getStore()</code> is called, a new instance will
     * be instantiaed.
     * @param	cleanup	if true, the store will be cleaned up, i.e.
     *			redundant data removed.
     */
    public static synchronized void releaseStore(boolean cleanup)
	throws BrokerException {

	if (store != null) {
	    // this check is for tonga test so that the store
	    // is not closed twice
	    if (!store.closed()) {
		store.close(cleanup);
	    }

	    store = null;
	}
    }

    public static Store getStoreForTonga() throws BrokerException {

        // For tonga test, get a store without a lock otherwise we'll
        // an error because the store is already locked by the broker!
        BrokerConfig config = Globals.getConfig();
        String type = config.getProperty(STORE_TYPE_PROP, DEFAULT_STORE_TYPE);
        if (type.equals(Store.JDBC_STORE_TYPE)) {
            config.put(JDBCStore.LOCK_STORE_PROP, "false");
            // Limit the # of conn in the pool to 2
            config.put(DBConnectionPool.NUM_CONN_PROP, "2");
        }
        return getStore();
    }
}
