/*
 * @(#)TidList.java	1.28 12/02/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms. 
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.file;

import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.util.SizeString;
import com.sun.messaging.jmq.util.PHashMap;
import com.sun.messaging.jmq.util.PHashMapLoadException;
import com.sun.messaging.jmq.util.PHashMapMMF;
import com.sun.messaging.jmq.io.VRFileWarning;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.jmsserver.data.TransactionState;
import com.sun.messaging.jmq.jmsserver.data.TransactionAcknowledgement;
import com.sun.messaging.jmq.jmsserver.config.*;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.persist.Store;
import com.sun.messaging.jmq.jmsserver.persist.LoadException;

import java.io.*;
import java.util.*;


/**
 * Keep track of all persisted transaction states by using PHashMap.
 */
class TidList {

    Logger logger = Globals.getLogger();
    BrokerResources br = Globals.getBrokerResources();
    BrokerConfig config = Globals.getConfig();

    // initial size of backing file
    static final String TXN_USE_MEMORY_MAPPED_FILE_PROP
        = FileStore.FILE_PROP_PREFIX + "transaction.memorymappedfile.enabled";
    static final String TXN_UPDATE_OPTIMIZATION_PROP
        = FileStore.FILE_PROP_PREFIX +
            "transaction.memorymappedfile.updateoptimization.enabled";
    static final String TXN_FILE_SIZE_PROP
        = FileStore.FILE_PROP_PREFIX + "transaction.file.size";

    static final boolean DEFAULT_TXN_USE_MEMORY_MAPPED_FILE = true;
    static final boolean DEFAULT_TXN_UPDATE_OPTIMIZATION = true;
    static final long DEFAULT_TXN_FILE_SIZE = 1024; // 1024k = 1M

    static final int CLIENT_DATA_SIZE = 1; // 1 byte for modified txn state

    static final String BASENAME = "txn"; // basename of data file

    // cache all persisted transaction ids
    // maps tid -> txn state
    private PHashMap tidMap = null;

    private boolean useMemoryMappedFile = true;
    private boolean updateOptimization = true;
    private File backingFile = null;

    // object encapsulates persistence of all transactions' ack lists
    private TxnAckList txnAckList = null;

    private LoadException loadException = null;

    // when instantiated, all data are loaded
    TidList(File topDir, boolean clear) throws BrokerException {

	SizeString filesize = config.getSizeProperty(TXN_FILE_SIZE_PROP,
					DEFAULT_TXN_FILE_SIZE);

	backingFile = new File(topDir, BASENAME);
	try {
            useMemoryMappedFile = config.getBooleanProperty(
                TXN_USE_MEMORY_MAPPED_FILE_PROP,
                DEFAULT_TXN_USE_MEMORY_MAPPED_FILE);
            updateOptimization = useMemoryMappedFile &&
                config.getBooleanProperty(
                    TXN_UPDATE_OPTIMIZATION_PROP,
                    DEFAULT_TXN_UPDATE_OPTIMIZATION);

            // safe = false; caller controls data synchronization
            if (useMemoryMappedFile) {
                tidMap = new PHashMapMMF(
                    backingFile, filesize.getBytes(), false, clear);
                if (updateOptimization) {
                    ((PHashMapMMF)tidMap).intClientData(CLIENT_DATA_SIZE);
                }
            } else {
                tidMap = new PHashMap(
                    backingFile, filesize.getBytes(), false, clear);
            }
	} catch (IOException e) {
	    logger.log(logger.ERROR, br.X_LOAD_TRANSACTIONS_FAILED, e);
	    throw new BrokerException(
			br.getString(br.X_LOAD_TRANSACTIONS_FAILED), e);
	}

	try {
	    tidMap.load();

            // Process client data
            loadClientData();
	} catch (IOException e) {
	    logger.log(logger.ERROR, br.X_LOAD_TRANSACTIONS_FAILED, e);
	    throw new BrokerException(
			br.getString(br.X_LOAD_TRANSACTIONS_FAILED), e);
	} catch (ClassNotFoundException e) {
	    logger.log(logger.ERROR, br.X_LOAD_TRANSACTIONS_FAILED, e);
	    throw new BrokerException(
			br.getString(br.X_LOAD_TRANSACTIONS_FAILED), e);
	} catch (PHashMapLoadException le) {
	
	    while (le != null) {
		logger.log(Logger.WARNING, br.X_FAILED_TO_LOAD_A_TXN, le);

		// save info in LoadException
		LoadException e = new LoadException(le.getMessage(),
						le.getCause());
		e.setKey(le.getKey());
		e.setValue(le.getValue());
		e.setKeyCause(le.getKeyCause());
		e.setValueCause(le.getValueCause());
		e.setNextException(loadException);
		loadException = e;

		// get the chained exception
		le = le.getNextException();
	    }
	}

	VRFileWarning w = tidMap.getWarning();
	if (w != null) {
	    logger.log(logger.WARNING,
			"possible loss of transaction data", w);
	}

	if (clear && Store.DEBUG) {
	    logger.log(logger.DEBUGHIGH,
				"TidList initialized with clear option");
	}

	if (Store.DEBUG) {
	    logger.log(logger.DEBUG, "TidList: loaded "+
					tidMap.size() + " transactions");
	}

	// load transaction acknowledgements
	txnAckList = new TxnAckList(topDir, clear);
    }

    // when instantiated, old data are upgraded
    TidList(FileStore p, File topDir, File oldTop)
	throws BrokerException {

	File oldFile = new File(oldTop, BASENAME);
	PHashMap olddata = null;

	backingFile = new File(topDir, BASENAME);
	try {
	    // load old data
	    // safe=false; reset=false
	    olddata = new PHashMap(oldFile, false, false);
	} catch (IOException e) {
	    logger.log(logger.ERROR, br.X_UPGRADE_TRANSACTIONS_FAILED,
			oldFile, backingFile, e);
	    throw new BrokerException(
			br.getString(br.X_UPGRADE_TRANSACTIONS_FAILED,
				oldFile, backingFile), e);
	}

	try {
	    olddata.load();
	} catch (IOException e) {
	    logger.log(logger.ERROR, br.X_UPGRADE_TRANSACTIONS_FAILED,
			oldFile, backingFile, e);
	    throw new BrokerException(
			br.getString(br.X_UPGRADE_TRANSACTIONS_FAILED,
				oldFile, backingFile), e);
	} catch (ClassNotFoundException e) {
	    logger.log(logger.ERROR, br.X_UPGRADE_TRANSACTIONS_FAILED,
			oldFile, backingFile, e);
	    throw new BrokerException(
			br.getString(br.X_UPGRADE_TRANSACTIONS_FAILED,
				oldFile, backingFile), e);
	} catch (PHashMapLoadException le) {
	
	    while (le != null) {
		logger.log(Logger.WARNING,
			br.X_FAILED_TO_LOAD_A_TXN_FROM_OLDSTORE, le);

		// save info in LoadException
		LoadException e = new LoadException(le.getMessage(),
						le.getCause());
		e.setKey(le.getKey());
		e.setValue(le.getValue());
		e.setKeyCause(le.getKeyCause());
		e.setValueCause(le.getValueCause());
		e.setNextException(loadException);
		loadException = e;

		// get the chained exception
		le = le.getNextException();
	    }
	}

	VRFileWarning w = olddata.getWarning();
	if (w != null) {
	    logger.log(logger.WARNING,
			"possible loss of transaction data in old store", w);
	}

	try {
            useMemoryMappedFile = config.getBooleanProperty(
                TXN_USE_MEMORY_MAPPED_FILE_PROP,
                DEFAULT_TXN_USE_MEMORY_MAPPED_FILE);
            updateOptimization = useMemoryMappedFile &&
                config.getBooleanProperty(
                    TXN_UPDATE_OPTIMIZATION_PROP,
                    DEFAULT_TXN_UPDATE_OPTIMIZATION);

            // pass in safe=false; caller decide when to sync
            // safe=false; reset=false
            if (useMemoryMappedFile) {
	        tidMap = new PHashMapMMF(
                    backingFile, oldFile.length(), false, false);
                if (updateOptimization) {
                    ((PHashMapMMF)tidMap).intClientData(CLIENT_DATA_SIZE);
                }
            } else {
                tidMap = new PHashMap(
                    backingFile, oldFile.length(), false, false);
            }
	} catch (IOException e) {
	    logger.log(logger.ERROR, br.X_UPGRADE_TRANSACTIONS_FAILED,
			oldFile, backingFile, e);
	    throw new BrokerException(
			br.getString(br.X_UPGRADE_TRANSACTIONS_FAILED,
				oldFile, backingFile), e);
	}

	try {
	    tidMap.load();

            // Process client data
            loadClientData();
	} catch (ClassNotFoundException e) {
	    // should not happen so throw exception
	    logger.log(logger.ERROR, br.X_UPGRADE_TRANSACTIONS_FAILED,
			oldFile, backingFile, e);
	    throw new BrokerException(
			br.getString(br.X_UPGRADE_TRANSACTIONS_FAILED,
				oldFile, backingFile), e);
	} catch (IOException e) {
	    // should not happen so throw exception
	    logger.log(logger.ERROR, br.X_UPGRADE_TRANSACTIONS_FAILED,
			oldFile, backingFile, e);
	    throw new BrokerException(
			br.getString(br.X_UPGRADE_TRANSACTIONS_FAILED,
				oldFile, backingFile), e);
	} catch (PHashMapLoadException e) {
	    // should not happen so throw exception
	    logger.log(logger.ERROR, br.X_UPGRADE_TRANSACTIONS_FAILED,
			oldFile, backingFile, e);
	    throw new BrokerException(
			br.getString(br.X_UPGRADE_TRANSACTIONS_FAILED,
				oldFile, backingFile), e);
	}

	w = tidMap.getWarning();
	if (w != null) {
	    logger.log(logger.WARNING,
			"possible loss of transaction data", w);
	}

	// just copy old data to new store
	Iterator itr = olddata.entrySet().iterator();
	while (itr.hasNext()) {
	    Map.Entry entry = (Map.Entry)itr.next();
	    Object key = entry.getKey();
	    Object value = entry.getValue();
	    tidMap.put(key, value);
	}
	olddata.close();

	if (Store.DEBUG) {
	    logger.log(logger.DEBUG, "TidList: upgraded "+
					tidMap.size() + " transactions");
	}

	// if upgradeNoBackup, remove oldfile
	if (p.upgradeNoBackup()) {
	    if (!oldFile.delete()) {
		logger.log(logger.ERROR, br.I_DELETE_FILE_FAILED, oldFile);
	    }
	}

	// load transaction acknowledgements
	txnAckList = new TxnAckList(topDir, oldTop, p.upgradeNoBackup());
    }

    LoadException getLoadException() {
	return loadException;
    }

    LoadException getLoadTransactionAckException() {
	return txnAckList.getLoadException();
    }

    void close(boolean cleanup) {
	if (Store.DEBUG) {
	    logger.log(logger.DEBUGHIGH,
		"TidList: closing, "+tidMap.size()+" persisted transactions");
	}

	tidMap.close();
	txnAckList.close(cleanup);
    }

    /**
     * Store a transaction.
     *
     * @param id	the id of the transaction to be persisted
     * @param ts	the transaction's state to be persisted
     * @exception IOException if an error occurs while persisting
     *		the transaction
     * @exception BrokerException if the same transaction id exists
     *			the store already
     * @exception NullPointerException	if <code>id</code> is
     *			<code>null</code>
     */
    void storeTransaction(TransactionUID id, TransactionState ts, boolean sync)
	throws IOException, BrokerException {

	synchronized (tidMap) {

	    if (tidMap.containsKey(id)) {
		logger.log(logger.ERROR, br.E_TRANSACTIONID_EXISTS_IN_STORE,
				id.toString());
		throw new BrokerException(
				br.getString(br.E_TRANSACTIONID_EXISTS_IN_STORE,
					id.toString()));
	    }

	    try {
                // TransactionState is mutable, so we must store a copy
                // See bug 4989708
	    	tidMap.put(id, new TransactionState(ts));

		if (sync) {
		    sync();
		}
	    } catch (RuntimeException e) {
		logger.log(logger.ERROR, br.X_PERSIST_TRANSACTION_FAILED,
				id.toString(), e);
		throw new BrokerException(
				br.getString(br.X_PERSIST_TRANSACTION_FAILED,
					id.toString()), e);
	    }
	}
    }

    /**
     * Remove the transaction. The associated acknowledgements
     * will not be removed.
     *
     * @param id	the id of the transaction to be removed
     * @exception BrokerException if the transaction is not found
     *			in the store
     */
    void removeTransaction(TransactionUID id, boolean sync)
	throws BrokerException {

	synchronized (tidMap) {

            if (!tidMap.containsKey(id)) {
		logger.log(logger.ERROR,
			br.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
	    		id.toString());
		throw new BrokerException(
			br.getString(br.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
					id.toString()));
	    }

	    try {
	    	tidMap.remove(id);

		if (sync) {
		    sync();
		}
	    } catch (RuntimeException e) {
		logger.log(logger.ERROR, br.X_REMOVE_TRANSACTION_FAILED,
			id.toString(), e);
		throw new BrokerException(
			br.getString(br.X_REMOVE_TRANSACTION_FAILED,
					id.toString()), e);
	    }
	}
    }

    /**
     * Update the state of a transaction
     *
     * @param id	the transaction id to be updated
     * @param ts	the new transaction state
     * @exception IOException if an error occurs while persisting
     *		the transaction id
     * @exception BrokerException if the transaction id does NOT exists in
     *			the store already
     * @exception NullPointerException	if <code>id</code> is
     *			<code>null</code>
     */
    void updateTransactionState(TransactionUID id, int ts, boolean sync)
	throws IOException, BrokerException {

	synchronized (tidMap) {

            TransactionState stateobj = (TransactionState)tidMap.get(id);

            if (stateobj == null) {
		logger.log(logger.ERROR, br.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
				id.toString());
		throw new BrokerException(
			br.getString(br.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
					id.toString()));
            }

	    if (stateobj.getState() != ts) {
		try {
		    stateobj.setState(ts);

                    if (updateOptimization) {
                        // To improve I/O performance, just persist the new
                        // state as client data and not the whole record
                        PHashMapMMF tidMapMMF = (PHashMapMMF)tidMap;
                        tidMapMMF.putClientData(id, new byte[] {(byte)ts});
                    } else {
	    	        tidMap.put(id, stateobj);
                    }

		    if (sync) {
			sync();
		    }
		} catch (RuntimeException e) {
		    logger.log(logger.ERROR, br.X_UPDATE_TXNSTATE_FAILED,
				id.toString(), e);
		    throw new BrokerException(
				br.getString(br.X_UPDATE_TXNSTATE_FAILED,
					id.toString()), e);
		}
	    }
	}
    }

    /**
     * Retrieve all transaction ids with their state in the store.
     *
     * @return A HashMap. The key is a TransactionUID.
     * The value a TransactionState.
     * @exception IOException if an error occurs while getting the data
     */
    HashMap getAllTransactionStates() throws IOException {
	synchronized (tidMap) {
            return (HashMap)(tidMap.clone());
	}
    }

    /**
     * Clear all transaction ids and the associated ack lists
     */
    // clear the store; when this method returns, the store has a state
    // that is the same as an empty store (same as when TxnAckList is
    // instantiated with the clear argument set to true
    void clearAll(boolean sync) {

	if (Store.DEBUG) {
	    logger.log(logger.DEBUGHIGH, "TidList.clearAll() called");
	}

	synchronized (tidMap) {
	    try {
		tidMap.clear();

		if (sync) {
		    sync();
		}
	    } catch (BrokerException e) {
		// at least log it
		logger.log(logger.ERROR, br.getString(
				br.X_CLEAR_TRANSACTION_FILE_FAILED,
				backingFile), e);
	    } catch (RuntimeException e) {
		// at least log it
		logger.log(logger.ERROR, br.getString(
				br.X_CLEAR_TRANSACTION_FILE_FAILED,
				backingFile), e);
	    }
	}

	// clear ack lists
	txnAckList.clearAll(sync);
    }

    /**
     * Clear all transactions that are NOT in the specified state.
     *
     * @param state State of transactions to spare
     */
    void clear(int state, boolean sync) throws BrokerException {

        boolean error = false;
	Exception exception = null;

	synchronized (tidMap) {
            Iterator itr = tidMap.entrySet().iterator();
            while (itr.hasNext()) {
		try {
                    Map.Entry entry = (Map.Entry)itr.next();
                    TransactionUID tid = (TransactionUID)entry.getKey();
                    TransactionState ts = (TransactionState)entry.getValue();
                    // Remove if not in prepared state
                	if (ts.getState() != state) {
                	// XXX PERF 12/7/2001 This operation updates disk
                	// with every call.
                	itr.remove();
                	try {
                            txnAckList.removeAcks(tid, sync);
                	} catch (BrokerException e) {
                            // TxnAckList Logs error
                            error = true;
			    exception = e;
			    break;
                	}
                    }
		} catch (RuntimeException e) {
            	    error = true;
		    exception = e;
	    	    logger.log(logger.ERROR, br.X_CLEAR_TXN_NOTIN_STATE_FAILED,
			new Integer(state), e);
		}

            }

	    if (sync) {
		sync();
	    }
        }

        if (!error) {
            // We may have transactions left in the txnAckList that did not
            // have an entry in the tidMap (maybe it was removed by a commit
            // that never had a chance to complete processing the acks).
            // Check for those "orphaned" ack transactions here. We check
            // state here too just to be anal.
            TransactionUID[] tids = txnAckList.getAllTids();
            for (int i = 0; i < tids.length; i++) {
                TransactionState ts = (TransactionState)tidMap.get(tids[i]);
                if (ts == null || ts.getState() != state) {
                    // Orphan. Remove from txnList
                    try {
                        txnAckList.removeAcks(tids[i], sync);
                    } catch (BrokerException e) {
                        // TxnAckList Logs error
                        error = true;
			exception = e;
			break;
                    }
                }
            }
        }


        // If we got an error just clear all transactions. 
        if (error) {
            clearAll(sync);
	    throw new BrokerException(
			br.getString(br.X_CLEAR_TXN_NOTIN_STATE_FAILED,
			new Integer(state)), exception);
        }
    }

    /**
     * Store an acknowledgement for th specified transaction.
     */
    void storeTransactionAck(TransactionUID tid, TransactionAcknowledgement ack,
	boolean sync) throws BrokerException {

	synchronized (tidMap) {
	    if (!tidMap.containsKey(tid)) {
		logger.log(logger.ERROR, br.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
			tid.toString());
		throw new BrokerException(
			br.getString(br.E_TRANSACTIONID_NOT_FOUND_IN_STORE,
					tid.toString()));
	    }
	}

	txnAckList.storeAck(tid, ack, sync);
    }

    /**
     * Remove all acknowledgements associated with the specified
     * transaction from the persistent store.
     */
    void removeTransactionAck(TransactionUID id, boolean sync)
	throws BrokerException {
	txnAckList.removeAcks(id, sync);
    }

    /**
     * Retrieve all acknowledgements for the specified transaction.
     */
    TransactionAcknowledgement[] getTransactionAcks(TransactionUID tid)
	throws BrokerException {

	return txnAckList.getAcks(tid);
    }

    /**
     * Retrieve all acknowledgement list in the persistence store
     * together with their associated transaction id.
     */
    HashMap getAllTransactionAcks() {
	return txnAckList.getAllAcks();
    }

    /**
     * @return the total number of transaction acknowledgements in the store
     */
    public int getNumberOfTxnAcks() {
	return txnAckList.getNumberOfTxnAcks();
    }

    /**
     * Get debug information about the store.
     * @return A Hashtable of name value pair of information
     */  
    Hashtable getDebugState() {
	Hashtable t = new Hashtable();
	t.put("Transactions", String.valueOf(tidMap.size()));
	t.putAll(txnAckList.getDebugState());
	return t;
    }

    void printInfo(PrintStream out) {
	// transacation ids
	out.println("\nTransaction IDs");
	out.println("---------------");
	out.println("backing file: "+ backingFile);
	out.println("number of transaction ids: " + tidMap.size());

	// transaction acknowledgement list info
	txnAckList.printInfo(out);
    }

    /**
     * When update optimization is enabled, we'll only write out the modified
     * transaction state as client data instead of the whole record (key and
     * value) to improve I/O performance. So when the broker start up and the
     * persisted transaction states are loaded from file, we need to update the
     * TransactionState object with the modified value that is stored in the
     * client data section of the record.
     * @throws PHashMapLoadException if an error occurs while loading data
     */
    private void loadClientData() throws PHashMapLoadException {

        if (!updateOptimization) {
            return; // nothing to do
        }

        PHashMapLoadException loadException = null;

        Set entries = tidMap.entrySet();
        Iterator itr = entries.iterator();
        while (itr.hasNext()) {
            Throwable ex = null;
            Map.Entry entry = (Map.Entry)itr.next();
            Object key = entry.getKey();
            TransactionState value = (TransactionState)entry.getValue();
            byte[] cData = null;
            try {
                cData = ((PHashMapMMF)tidMap).getClientData(key);
                if (cData != null && cData.length > 0) {
                    int state = (int)cData[0]; // 1st byte is the modified state
                    value.setState(state);     // update txn state
                }
            } catch (Throwable e) {
                ex = e;
            }

            if (ex != null) {
                PHashMapLoadException le = new PHashMapLoadException(
                    "Failed to load client data [cData=" + cData + "]");
                le.setKey(key);
                le.setValue(value);
                le.setNextException(loadException);
                le.initCause(ex);
                loadException = le;
            }
        }

        if (loadException != null) {
            throw loadException;
        }
    }

    private void sync() throws BrokerException {
	try {
	    tidMap.force();
	} catch (IOException e) {
	    throw new BrokerException(
		"Failed to synchronize file: " + backingFile, e);
	}
    }
}


