/*
 * @(#)DBTool.java	1.92 02/13/06
 *
 * Copyright 2003-2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.Version;
import com.sun.messaging.jmq.io.Status;
import com.sun.messaging.jmq.jmsserver.persist.Store;
import com.sun.messaging.jmq.jmsserver.config.*;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.*;
import com.sun.messaging.jmq.jmsserver.cluster.BrokerState;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.util.StringUtil;

import com.sun.messaging.jmq.jmsserver.persist.StoreManager;
import com.sun.messaging.jmq.jmsserver.persist.HABrokerInfo;
import com.sun.messaging.jmq.jmsserver.persist.file.FileStore;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.jmsserver.data.TransactionState;
import com.sun.messaging.jmq.jmsserver.data.TransactionAcknowledgement;
import com.sun.messaging.jmq.io.Packet;
import com.sun.messaging.jmq.io.SysMessageID;

import java.io.*;
import java.sql.*;
import java.util.*;

/**
 * This class is used to create, delete, and recreate database tables.
 * It may also be used to create a database.
 * All database specific information is obtained from property file,
 * except for username and password which may also be specified as command
 * line arguments.
 *
 * @version     1.92
 */
public class DBTool implements DBConstants {

    static final String SQLFILEDIR_PROP =
		DBManager.JDBC_PROP_PREFIX + "sqlfile.dirpath";

    static final String SQLFILENAME_PROP =
		DBManager.JDBC_PROP_PREFIX + "sqlfile.name";

    // parser exception reasons
    private static int EXTRA_CMD_SPECIFIED	= 0;
    private static int BAD_CMD_ARG		= 1;
    private static int BAD_OPT			= 2;
    private static int BAD_OPT_ARG		= 3;
    private static int MISSING_OPT_ARG		= 4;
    private static int MISSING_CMD_ARG		= 5;

    private static String CMD_NAME		= "dbmgrcmd";
    private static String CREATE_ALL_CMD	= "createall";
    private static String CREATE_TBL_CMD	= "createtbl";
    private static String DELETE_TBL_CMD	= "deletetbl";
    private static String RECREATE_TBL_CMD	= "recreatetbl";
    private static String REMOVE_BKR_CMD	= "removebkr";
    private static String DUMP_CMD		= "dump";
    private static String DROPTBL_CMD	        = "droptbl";
    private static String RESET_CMD		= "reset";
    private static String BACKUP_CMD            = "backup";
    private static String RESTORE_CMD           = "restore";
    private static String UPGRADE_STORE_CMD     = "upgradestore";
    private static String UPGRADE_HASTORE_CMD   = "upgradehastore";
    private static String QUERY_CMD             = "query";

    private static String ARG_NAME		= "dbmgrarg";

    private static String CREATE_CMD_STR = "create";
    private static String DELETE_CMD_STR = "delete";
    private static String RECREATE_CMD_STR = "recreate";
    private static String REMOVE_CMD_STR = "remove";
    private static String UPGRADE_CMD_STR = "upgrade";
    private static String ARGU_ALL = "all";
    private static String ARGU_TBL = "tbl";
    private static String ARGU_BKR = "bkr";
    private static String ARGU_OLDTBL = "oldtbl";
    private static String ARGU_LCK = "lck";
    private static String ARGU_STORE = "store";
    private static String ARGU_HASTORE = "hastore";
    private static String OPT_H = "-h";
    private static String OPT_LH = "-help";
    private static String OPT_V = "-v";
    private static String OPT_LV = "-version";
    private static String OPT_B = "-b";
    private static String OPT_N = "-n";
    private static String OPT_U = "-u";
    private static String OPT_P = "-p";
    private static String OPT_PW = "-pw";
    private static String OPT_PASSFILE = "-passfile";
    private static String OPT_D = "-D";
    private static String OPT_VARHOME = "-varhome";
    private static String OPT_VERBOSE = "-verbose";
    private static String OPT_DEBUG = "-debug";
    private static String OPT_DIR = "-dir";
    private static String OPT_FORCE = "-f";

    private static BrokerResources br = Globals.getBrokerResources();
    private static BrokerConfig config;
    private static Logger logger;

    private Version version;
    private DBManager dbmgr = null;
    private boolean standalone = true;
    private boolean cliPasswdSpecified = false;
    private boolean debugSpecified = false;
    private boolean forceSpecified = false;

    DBTool(boolean standalone) {
	this.version = new Version();
	this.standalone = standalone;
    }

    private void doCreate(boolean createdb)
	throws BrokerException {

	Connection conn = null;
	try {
            if (createdb) {
                conn = dbmgr.connectToCreate();
                conn.setAutoCommit( true );
            } else {
                conn = dbmgr.newConnection( true ); // set autoCommit to true
            }

            // Check if store exist
            boolean continueOnError = false;
            int status = dbmgr.checkStoreExists( conn );
            if (status > 0) {
                // All tables have already been created
                throw new BrokerException(br.getKString(
                    BrokerResources.E_DATABASE_TABLE_ALREADY_CREATED));
            } else if (status < 0) {
                // Some tables are missings so try to create the tables
                // but ignore error if table already exists
                continueOnError = true;
            }

            createTables( conn, continueOnError );

	    if (standalone) {
                if ( Globals.getHAEnabled() ) {
                    System.out.println( br.getString(
                        BrokerResources.I_DATABASE_TABLE_HA_CREATED,
                        Globals.getClusterID() ) );
                } else {
		    System.out.println(
                        br.getString(BrokerResources.I_DATABASE_TABLE_CREATED));
                }
	    }
	} catch (Throwable t) {
	    String url;
	    if (createdb)
		url = dbmgr.getCreateDBURL();
	    else
		url = dbmgr.getOpenDBURL();

	    logger.logStack(Logger.ERROR, BrokerResources.E_CREATE_DATABASE_TABLE_FAILED, url, t);
	    throw new BrokerException(br.getKString(
                BrokerResources.E_CREATE_DATABASE_TABLE_FAILED, url, t));
	} finally {
            if ( !createdb && conn != null ) {
                // Since the connection is not from the pool; we've to close it
                try {
                    conn.close();
                } catch (SQLException e) {
                    throw new BrokerException(
                        Globals.getBrokerResources().getKString(
                            BrokerResources.E_INTERNAL_BROKER_ERROR,
                            "Unable to close JDBC resources", e ) );
                }
            }
	}
    }

    // create database tables used in the current version of persistent store
    static void createTables( Connection conn ) throws BrokerException {
        createTables( conn, false );
    }

    static void createTables( Connection conn, boolean continueOnError )
        throws BrokerException {

        DBManager dbMgr = DBManager.getDBManager();
        DAOFactory daoFactory = dbMgr.getDAOFactory();
        Iterator itr = daoFactory.getAllDAOs().iterator();
        while ( itr.hasNext() ) {
            BaseDAO dao = (BaseDAO)itr.next();
            try {
                dao.createTable( conn );
            } catch (BrokerException be) {
                // Verify if the store has already been created
                if ( dbMgr.checkStoreExists( conn ) > 0 ) {
                    Globals.getLogger().log(Logger.WARNING,
                        BrokerResources.E_CREATE_DATABASE_TABLE_FAILED,
                        Globals.getBrokerResources().getString(
                            BrokerResources.E_DATABASE_TABLE_ALREADY_CREATED));
                    continueOnError = true;
                    break;
                } else if (continueOnError) {
                    // Just log msg and continue
                    Globals.getLogger().log(Logger.WARNING, be.toString(), be.getCause());
                } else {
                    throw be;
                }
            }
        }

	// Insert version info in the version table
        VersionDAO versionDAO = daoFactory.getVersionDAO();
        if ( continueOnError ) {
            // Do this only if version is missing from version table
            int storeVersion = versionDAO.getStoreVersion( conn );
            if ( storeVersion != JDBCStore.STORE_VERSION ) {
                versionDAO.insert( conn, JDBCStore.STORE_VERSION );
            }
        } else {
            versionDAO.insert( conn, JDBCStore.STORE_VERSION );
        }
    }

    private void doReset() throws BrokerException {

	Connection conn = null;
	try {
            conn = dbmgr.getConnection(true);

            // unlock the tables
	    DBManager.lockTables(conn, false); // false = unlock
	} finally {
            Util.close( null, null, conn );
	}
    }

    private void doDelete(String arg) throws BrokerException {

	boolean deleted = false;
        Connection conn = null;
	try {
            conn = dbmgr.getConnection(true);

	    if (arg == null || arg.length() == 0) {
                // Check if store exist
                boolean continueOnError = false;
                int status = dbmgr.checkStoreExists( conn );
                if (status > 0) {
                    // Verify cluster is not active in HA mode
                    if ( Globals.getHAEnabled() && dbmgr.isHAClusterActive(conn) ) {
                        throw new BrokerException(br.getKString(
                            BrokerResources.E_HA_CLUSTER_STILL_ACTIVE, dbmgr.getClusterID()));
                    }

                    try {
                        // lock the tables first (implemented since version 350);
                        // note that we don't need to unlock the tables since
                        // the tables will be dropped when we are done
                        if (!forceSpecified) {
                            DBManager.lockTables(conn, true); // true = lock
                        }
                    } catch ( BrokerException e ) {
                        if ( e.getStatusCode() == Status.NOT_FOUND ) {
                            // For some reason if version table doesn't exist or
                            // version data is not found we can just ignore the error!
                            continueOnError = true;
                        } else {
                            throw e;
                        }
                    }
                } else if (status < 0) {
                    // Some tables are missings so try to delete the rest
                    // but ignore error if table does not exists
                    continueOnError = true;
                } else {
                    // All tables have already been deleted
                    throw new BrokerException(br.getKString(
                        BrokerResources.E_DATABASE_TABLE_ALREADY_DELETED));
                }

		deleted = dropTables(conn, null, continueOnError);
	    } else if (arg.equals(ARGU_OLDTBL)) {
                int oldStoreVersion = -1;
                if ( checkVersion( conn,
                    VERSION_TBL_37 + dbmgr.getBrokerID() ) ) {
                    oldStoreVersion = JDBCStore.OLD_STORE_VERSION;
                } else if ( checkVersion( conn,
                    VERSION_TBL_35 + dbmgr.getBrokerID() ) ) {
                    oldStoreVersion = JDBCStore.OLD_STORE_VERSION_350;
                } else {
                    throw new BrokerException("Old persistent store (version " +
                        JDBCStore.OLD_STORE_VERSION + ") not found");
                }

                deleted = dropTables(conn, dbmgr.getTableNames(oldStoreVersion));
	    } else {
		// not possible since argument is checked already
	    }

	    if (standalone && deleted) {
                if ( Globals.getHAEnabled() ) {
                    System.out.println( br.getString(
                        BrokerResources.I_DATABASE_TABLE_HA_DELETED,
                        Globals.getClusterID() ) );
                } else {
		    System.out.println(br.getString(BrokerResources.I_DATABASE_TABLE_DELETED));
                }
	    }
	} catch (BrokerException e) {
	    logger.log(Logger.ERROR, BrokerResources.E_DELETE_DATABASE_TABLE_FAILED,
                dbmgr.getOpenDBURL(), e);
	    throw new BrokerException(
                br.getKString(BrokerResources.E_DELETE_DATABASE_TABLE_FAILED,
                    dbmgr.getOpenDBURL()), e);
	} catch (SQLException e) {
	    logger.log(Logger.ERROR, BrokerResources.E_DELETE_DATABASE_TABLE_FAILED,
                dbmgr.getOpenDBURL(), e);
	    throw new BrokerException(
                br.getKString(BrokerResources.E_DELETE_DATABASE_TABLE_FAILED,
		    dbmgr.getOpenDBURL()), e);
	} finally {
            Util.close( null, null, conn );
	}
    }

    static boolean dropTables(Connection conn, String names[])
        throws SQLException, BrokerException {
        return dropTables(conn, names, false);
    }

    static boolean dropTables(Connection conn, String names[], boolean continueOnError)
        throws SQLException, BrokerException {

        boolean deleted = false;
        try {
            if ( names != null && names.length > 0 ) {
                // Removing old tables, i.e. 3.5
                Statement stmt = null;
                try {
                    stmt = conn.createStatement();
                    for ( int i = 0, len = names.length; i < len; i++ ) {
                        String oldTableName = names[i];
                        Globals.getLogger().logToAll( Logger.INFO,
                            br.getString( BrokerResources.I_DROP_TABLE,
                                oldTableName ) );
                        stmt.executeUpdate( "DROP TABLE " + oldTableName );
                    }
                } finally {
                    Util.close( null, stmt, null );
                }
            } else {
                DAOFactory daoFactory = DBManager.getDBManager().getDAOFactory();
                Iterator itr = daoFactory.getAllDAOs().iterator();
                while ( itr.hasNext() ) {
                    BaseDAO dao = (BaseDAO)itr.next();
                    try {
                        dao.dropTable( conn );
                    } catch (BrokerException be) {
                        if (continueOnError) {
                            // Just log msg and continue
                            Globals.getLogger().log(Logger.WARNING, be.toString(), be.getCause());
                        } else {
                            throw be;
                        }
                    }
                }
            }
            deleted = true;
        } catch ( SQLException e ) {
            Globals.getLogger().log(Logger.ERROR,
                BrokerResources.E_DELETE_DATABASE_TABLE_FAILED,
                e.toString(), e);
            throw e;
        } catch ( BrokerException e ) {
            Globals.getLogger().log(Logger.ERROR,
                BrokerResources.E_DELETE_DATABASE_TABLE_FAILED,
                e.toString(), e);
            throw e;
        }

	return deleted;
    }

    static boolean checkVersion(Connection conn, String vtable)
        throws BrokerException {

        // get version from table
        String selectSQL = "SELECT * FROM " + vtable;

        Statement stmt = null;
        ResultSet rs = null;
        try {
            stmt = conn.createStatement();
            rs = stmt.executeQuery(selectSQL);
        } catch (SQLException e) {
            // assume that the table does not exist
            return false;
        } finally {
            Util.close(rs, stmt, null);
        }

        return true;
    }

    private void doRecreate() throws BrokerException {

	Connection conn = null;
	try {
            conn = dbmgr.getConnection(true);

            boolean deleted = false;
            boolean continueOnError = false;
            int status = dbmgr.checkStoreExists( conn );
            if (status > 0) {
                // Verify cluster is not active in HA mode
                if ( Globals.getHAEnabled() && dbmgr.isHAClusterActive(conn) ) {
                    throw new BrokerException(br.getKString(
                        BrokerResources.E_HA_CLUSTER_STILL_ACTIVE, dbmgr.getClusterID()));
                }

                try {
                    // lock the tables first (implemented since version 350);
                    // note that we don't need to unlock the tables since
                    // the tables will be recreated when we are done
                    if (!forceSpecified) {
                        DBManager.lockTables(conn, true); // true = lock
                    }
                } catch (BrokerException e) {
                    if ( e.getStatusCode() == Status.NOT_FOUND ) {
                        // For some reason if version table doesn't exist or
                        // version data is not found we can just ignore the error!
                        continueOnError = true;
                    } else {
                        // unable to get the lock
                        throw e;
                    }
                }
            } else if (status < 0) {
                // Some tables are missings so try to delete the rest
                // but ignore error if table does not exists
                continueOnError = true;
            } else {
                deleted = true;
            }

            if ( !deleted ) {
                dropTables( conn, null, continueOnError );
            }

	    createTables( conn, false );

	    if (standalone) {
                if ( Globals.getHAEnabled() ) {
                    System.out.println( br.getString(
                        BrokerResources.I_DATABASE_TABLE_HA_CREATED,
                        Globals.getClusterID() ) );
                } else {
		    System.out.println(
                        br.getString(BrokerResources.I_DATABASE_TABLE_CREATED) );
                }
	    }
	} catch (Throwable t) {
	    logger.log(Logger.ERROR, BrokerResources.E_RECREATE_DATABASE_TABLE_FAILED,
                dbmgr.getOpenDBURL(), t);
	    throw new BrokerException(
                br.getKString(BrokerResources.E_RECREATE_DATABASE_TABLE_FAILED,
                    dbmgr.getOpenDBURL()), t);
	} finally {
            Util.close( null, null, conn );
	}
    }

    private void doRemoveBkr() throws BrokerException {

        String brokerID = Globals.getBrokerID();
	Connection conn = null;
        String errorMsg = null;
	try {
            conn = dbmgr.getConnection(true);

            // try to lock the tables first
            DBManager.lockTables(conn, true); // true = lock

            try {
                DAOFactory daoFactory = dbmgr.getDAOFactory();
                BrokerDAO brokerDAO = daoFactory.getBrokerDAO();
                BrokerState state = brokerDAO.getState(conn, brokerID);
                if ( state == null ) {
                    errorMsg = br.getString(
                        BrokerResources.E_BROKERINFO_NOT_FOUND_IN_STORE, brokerID );
                    errorMsg = br.getString(
                        BrokerResources.E_REMOVE_BROKER_FAILED,
                        brokerID, errorMsg );
                } else if ( !state.isActiveState()  ) {
                    try {
                        // Remove broker's and its data
                        conn.setAutoCommit(false); // do this in 1 txn
                        brokerDAO.delete(conn, brokerID);
                        daoFactory.getDestinationDAO().deleteAll(conn);
                        daoFactory.getMessageDAO().deleteAll(conn);
                        daoFactory.getConsumerStateDAO().deleteAll(conn);
                        daoFactory.getTransactionDAO().deleteAll(conn);
                        conn.commit(); // commit changes
                    } finally {
                        conn.setAutoCommit(true);
                    }
                } else {
                    errorMsg = br.getString(
                        BrokerResources.E_REMOVE_BROKER_FAILED,
                        brokerID, "broker is still active - " + state );
                }
            } finally {
                // remove the lock
                DBManager.lockTables(conn, false); // false = unlock
            }
	} catch (Exception e) {
	    logger.log(Logger.ERROR,
                BrokerResources.E_REMOVE_BROKER_2_FAILED, brokerID, e);
	    throw new BrokerException(
		br.getKString( BrokerResources.E_REMOVE_BROKER_2_FAILED,
                    brokerID ), e );
	} finally {
            Util.close( null, null, conn );
	}

        if ( standalone ) {
            if ( errorMsg == null ) {
                System.out.println(
                    br.getString( BrokerResources.I_BROKER_REMOVE, brokerID ) );
            } else {
                System.out.println(errorMsg);
            }
        }
    }

    private void doUpgrade(boolean haStore)
	throws BrokerException {

	Connection conn = null;
	try {
            conn = dbmgr.getConnection(true);

            if ( haStore ) {
                new UpgradeHAStore().upgradeStore( conn );
            } else {
                JDBCStore store = (JDBCStore)StoreManager.getStore();
                int oldStoreVersion = -1;
                if ( store.checkOldStoreVersion( conn,
                    VERSION_TBL_37 + dbmgr.getBrokerID(), TVERSION_CVERSION,
                    JDBCStore.OLD_STORE_VERSION ) ) {
                    oldStoreVersion = JDBCStore.OLD_STORE_VERSION;
                } else if ( store.checkOldStoreVersion( conn,
                    VERSION_TBL_35 + dbmgr.getBrokerID(), TVERSION_CVERSION,
                    JDBCStore.OLD_STORE_VERSION_350 ) ) {
                    oldStoreVersion = JDBCStore.OLD_STORE_VERSION_350;
                } else {
                    throw new BrokerException("Old persistent store (version " +
                        JDBCStore.OLD_STORE_VERSION + ") not found");
                }

                new UpgradeStore(store, oldStoreVersion).upgradeStore(conn);
            }
	} finally {
            Util.close( null, null, conn );
	}
    }

    /**
     * dump info about the database.
     * for debugging purpose
     */
    private void doDump(String arg) throws SQLException, BrokerException {
	int storeVersion = JDBCStore.STORE_VERSION;

	if (arg != null) {
	    try {
		storeVersion = Integer.parseInt(arg);
	    } catch (NumberFormatException e) {
		storeVersion = 0;
	    }
	}

        String names[] = null;
        if ( storeVersion != JDBCStore.STORE_VERSION ) {
            // Old store
            names = dbmgr.getTableNames(storeVersion);
            if (names == null || names.length == 0) {
                throw new BrokerException("version " + arg + " not supported");
            }
        }

        Connection conn = null;
        try {
            conn = dbmgr.getConnection(true);
            doDump(conn, names);
        } finally {
            Util.close( null, null, conn );
        }
    }

    static void doDump(Connection dbconn, String tables[])
	throws BrokerException, SQLException {

        if ( tables != null && tables.length > 0 ) {
	    Statement stmt = null;
            try {
                stmt = dbconn.createStatement();
                for (int i = 0; i < tables.length; i++) {
                    ResultSet r = null;
                    String tname = tables[i];
                    String sql = "SELECT COUNT(*) FROM " + tname;
                    try {
                        r = stmt.executeQuery(sql);
                        if (r.next()) {
                            System.out.println(tname + ": number of row="
                                                + r.getInt(1));
                        }
                        r.close();
                    } catch (SQLException e) {
                        SQLException ex = DBManager.wrapSQLException(
                                                        "[" + sql + "]", e);
                        logger.log(Logger.ERROR, "failed to dump tables", ex);
                    }
                }
            } finally {
                Util.close( null, stmt, null );
            }
        } else {
            // Starting in 4.0, use info from BaseDAO.getDebugInfo()
            DAOFactory daoFactory = DBManager.getDBManager().getDAOFactory();
            Iterator itr = daoFactory.getAllDAOs().iterator();
            while ( itr.hasNext() ) {
                BaseDAO dao = (BaseDAO)itr.next();
                System.out.println( dao.getDebugInfo( dbconn ).toString() );

            }
        }
    }

    /**
     * drop tables that start with mq and match the specified table name pattern.
     * for internal use!!!
     */
    private void doDropTablesByPattern(String arg) throws BrokerException {

        Connection conn = null;
        try {
            conn = dbmgr.getConnection(true);
            Map map = dbmgr.getTableNamesFromDB(conn, arg);
            String[] names = (String[])map.values().toArray(new String[0] );

            if ( names == null || names.length == 0 ) {
                System.out.println(
                    "There were no tables that match the name pattern 'MQ*" +
                    arg + "'.");
                return;
            }

            // List all tables that match the name pattern
            System.out.println(
                "Tables matching the name pattern 'MQ*" + arg + "':");

            for (int i = 0, len = names.length; i < len; i++) {
                System.out.println("\t" + names[i]);
            }

            // Ask for confirmation only if force option is not specified
            if (!forceSpecified) {
                System.out.println("Remove all tables that match 'MQ*" + arg +
                    "'.\nDo you wish to proceed? [y/n] ");
                System.out.flush();

                String val = (new BufferedReader(new InputStreamReader(System.in))).readLine();

                // if not positive confirmation, do nothing!
                if (!"y".equalsIgnoreCase(val) && !"yes".equalsIgnoreCase(val)) {
                    return;
                }
            }

            dropTables(conn, names, true);
        } catch (Exception e) {
            System.err.println(
                "Failed to drop tables by name pattern: " + e.getMessage());
            throw new BrokerException(
                br.getKString(BrokerResources.E_DELETE_DATABASE_TABLE_FAILED,
                    "table name pattern '" + arg + "'"), e);
        } finally {
            Util.close( null, null, conn );
        }
    }

    /**
     * Backup the JDBC store to filebased backup files.
     */
    private void doBackup() throws BrokerException {
        Destination[] destArray = null;
        Properties props = System.getProperties();
        Store jdbcStore = null;
        Store fileStore = null;
        
        // if HA enabled, get the JDBC store instance.
        if(Globals.getHAEnabled()) {
            try{
                jdbcStore = StoreManager.getStore();
            } catch(Exception ex){
                ex.printStackTrace();
            }
        }
        else{
            throw new BrokerException(br.getKString(BrokerResources.I_HA_NOT_ENABLE,
                dbmgr.getBrokerID()));
        }
        
        // instantiate the file store.
        fileStore = new FileStore((String) props.get(Globals.IMQ + ".backupdir"), false);
        
        // for backup, need to clear the store before storing anything.
        fileStore.clearAll(false);
        
        // Start retrieving information from JDBC store and fill up
        // the file store.
        
        try {
            // Brokers table.
            HashMap bkrMap = jdbcStore.getAllBrokerInfos();
            Iterator iterator = bkrMap.keySet().iterator();
            while(iterator.hasNext()) {
                HABrokerInfo bkrInfo = (HABrokerInfo) bkrMap.get(iterator.next());
                fileStore.addBrokerInfo(bkrInfo.getId(), bkrInfo.getTakeoverBrokerID(),
                bkrInfo.getUrl(), BrokerState.getState(bkrInfo.getState()),
                bkrInfo.getVersion(), bkrInfo.getSessionID(), bkrInfo.getHeartbeat());
            }
            
            // Destination table.
            destArray = jdbcStore.getAllDestinations();
            for(int i = 0; i < destArray.length; i++) {
                fileStore.storeDestination(destArray[i],  false);
            }
            
            // Storing messages for each destination.
            for(int i=0; i < destArray.length; i++) {
                for(Enumeration e = jdbcStore.messageEnumeration(destArray[i]);
                e.hasMoreElements();) {
                    fileStore.storeMessage(destArray[i].getDestinationUID(), (Packet)e.nextElement(), false);
                }
            }
            
            // Consumer table
            Consumer[] consumerArray = jdbcStore.getAllInterests();
            for(int i = 0; i < consumerArray.length; i++) {
                fileStore.storeInterest(consumerArray[i], false);
            }
            
            // Consumer State table
            for(int i = 0; i < destArray.length; i++) {
                for(Enumeration e = jdbcStore.messageEnumeration(destArray[i]); 
                        e.hasMoreElements();) {
                    Packet message = (Packet) e.nextElement();
                    SysMessageID mID = message.getSysMessageID();
                    DestinationUID dUID = destArray[i].getDestinationUID();

                    ConsumerUID[] iids = jdbcStore.getConsumerUIDs(dUID, mID);
                    int[] states = new int[iids.length];
                    for(int j = 0; j < iids.length; j++)
                    {
                        states[j] = jdbcStore.getInterestState(dUID, mID, iids[j]);
                    }
                    // store interest states
                    if(iids.length != 0)
                        fileStore.storeInterestStates(dUID, mID, iids, states, false);
                }
            }

            // Transaction table
            HashMap txnMap = jdbcStore.getAllTransactionStates();
            Iterator iter = txnMap.keySet().iterator();
            while(iter.hasNext())
            {
                Object key = iter.next();
                fileStore.storeTransaction((TransactionUID) key, (TransactionState) txnMap.get(key), false);
            }
            iter = txnMap.keySet().iterator();
            while(iter.hasNext())
            {
                Object key = iter.next();
                TransactionAcknowledgement txnAck[] = jdbcStore.getTransactionAcks((TransactionUID) key);
                for(int i = 0; i < txnAck.length; i++)
                {
                    fileStore.storeTransactionAck((TransactionUID) key, txnAck[i], false);
                }
            }

            // Configuration Change Record table.
            Object[] data = jdbcStore.getAllConfigRecords();
            for(int i = 0; i < ((ArrayList) data[0]).size(); i++)
            {
                fileStore.storeConfigChangeRecord(((Long) ((ArrayList) data[0]).get(i)).longValue(), 
                            (byte[])((ArrayList) data[1]).get(i), false);
            }
            
            // Properties table
            String[] names = jdbcStore.getPropertyNames();
            for(int i = 0; i < names.length; i++)
            {
                fileStore.updateProperty(names[i], jdbcStore.getProperty(names[i]), false);
            }
            
        }catch(Exception ex){
            throw new BrokerException(br.getString(BrokerResources.E_CAN_NOT_WRITE));
        }
    }
    
    /*
     * Restore the JDBC store from filebased backup files.
     */
    private void doRestore() throws BrokerException {
        Properties props = System.getProperties();
        Store jdbcStore = null;
        Store fileStore = null;
        Destination[] destArray = null;
        
        System.out.println("Backup dir : " + props.get(Globals.IMQ + ".backupdir"));
        
        // We will be creating the JDBC store from scratch.
        // We recreate the Database tables.
        if(Globals.getHAEnabled()) {
            try{
                // instantiate the file store first, so that if exception occurs
                // we will not end up recreating jdbc store.
                fileStore = new FileStore((String) props.get(Globals.IMQ + ".backupdir"), false);
                
                doRecreate();
                jdbcStore = StoreManager.getStore();
            } catch(Exception ex){
                ex.printStackTrace();
            }
        }
        else{
            throw new BrokerException(br.getKString(BrokerResources.I_HA_NOT_ENABLE,
                dbmgr.getBrokerID()));
        }
        // Start retrieving information from backup files and fill up
        // restore the JDBC store.
        
        // Brokers table.
        try {
            HashMap bkrMap = fileStore.getAllBrokerInfos();
            Iterator iterator = bkrMap.keySet().iterator();
            while(iterator.hasNext()) {
                HABrokerInfo bkrInfo = (HABrokerInfo) bkrMap.get(iterator.next());
                // add the entry.
                jdbcStore.addBrokerInfo(bkrInfo.getId(), bkrInfo.getUrl(),
                BrokerState.getState(bkrInfo.getState()), bkrInfo.getVersion(),
                bkrInfo.getSessionID(), bkrInfo.getHeartbeat());
                // update the entry
                jdbcStore.updateBrokerInfo(bkrInfo.getId(), bkrInfo.getTakeoverBrokerID(),
                bkrInfo.getUrl(), bkrInfo.getVersion(),
                BrokerState.getState(bkrInfo.getState()),
                bkrInfo.getSessionID());
            }
        
            // Destination table.
            destArray = fileStore.getAllDestinations();
            for(int i = 0; i < destArray.length; i++) {
                jdbcStore.storeDestination(destArray[i],  false);
            }
            // Storing messages for each destination.
            for(int i=0; i < destArray.length; i++) {
                for(Enumeration e = fileStore.messageEnumeration(destArray[i]);
                e.hasMoreElements();) {
                    jdbcStore.storeMessage(destArray[i].getDestinationUID(), (Packet)e.nextElement(), false);
                }
            }
        
            // Consumer table
            Consumer[] consumerArray = fileStore.getAllInterests();
            for(int i = 0; i < consumerArray.length; i++) {
                jdbcStore.storeInterest(consumerArray[i], false);
            }

            // Consumer State table
            for(int i = 0; i < destArray.length; i++) {
                for(Enumeration e = fileStore.messageEnumeration(destArray[i]); 
                        e.hasMoreElements();) {
                    Packet message = (Packet) e.nextElement();
                    SysMessageID mID = message.getSysMessageID();
                    DestinationUID dUID = destArray[i].getDestinationUID();

                    ConsumerUID[] iids = fileStore.getConsumerUIDs(dUID, mID);
                    int[] states = new int[iids.length];
                    for(int j = 0; j < iids.length; j++)
                    {
                        states[j] = fileStore.getInterestState(dUID, mID, iids[j]);
                    }
                    // store interest states
                    if(iids.length != 0)
                        jdbcStore.storeInterestStates(dUID, mID, iids, states, false);
                }
            }

            // Transaction table
            HashMap txnMap = fileStore.getAllTransactionStates();
            Iterator iter = txnMap.keySet().iterator();
            while(iter.hasNext())
            {
                Object key = iter.next();
                jdbcStore.storeTransaction((TransactionUID) key, (TransactionState) txnMap.get(key), false);
            }
            iter = txnMap.keySet().iterator();
            while(iter.hasNext())
            {
                Object key = iter.next();
                TransactionAcknowledgement txnAck[] = fileStore.getTransactionAcks((TransactionUID) key);
                for(int i = 0; i < txnAck.length; i++)
                {
                    jdbcStore.storeTransactionAck((TransactionUID) key, txnAck[i], false);
                }
            }

            // Configuration Change Record table.
            Object[] data = fileStore.getAllConfigRecords();
            for(int i = 0; i < ((ArrayList) data[0]).size(); i++)
            {
                jdbcStore.storeConfigChangeRecord(((Long)((ArrayList) data[0]).get(i)).longValue(), 
                            (byte[])((ArrayList) data[1]).get(i), false);
            }
            
            // Properties table
            String[] names = fileStore.getPropertyNames();
            for(int i = 0; i < names.length; i++)
            {
                jdbcStore.updateProperty(names[i], fileStore.getProperty(names[i]), false);
            }
        }catch(Exception ex){
            throw new BrokerException(br.getString(BrokerResources.E_CAN_NOT_WRITE));
        }
    }

    /*
     * Check persistence store.
     */
    private void doQuery() throws BrokerException {

        Connection conn = null;
        try {
            conn = dbmgr.getConnection(true);

            if ( Globals.getHAEnabled() ) {
                System.out.println( br.getString(
                    BrokerResources.I_RUNNING_IN_HA,
                    Globals.getBrokerID(), Globals.getClusterID() ) );
            } else {
                System.out.println( br.getString(
                    BrokerResources.I_STANDALONE_INITIALIZED ) );
            }

            int status = dbmgr.checkStoreExists( conn );
            if ( status == 0 ) {
                System.out.println( br.getString(
                    BrokerResources.E_NO_DATABASE_TABLES ) );
            } else if (status > 0) {
                // All tables have already been created
                System.out.println( br.getString(
                    BrokerResources.E_DATABASE_TABLE_ALREADY_CREATED ) );
            } else {
                // Some tables are missings
                System.out.println( br.getString(
                    BrokerResources.E_BAD_STORE_MISSING_TABLES ) );
            }
        } finally {
            Util.close( null, null, conn );
        }
    }

    private boolean printHelp(String[] args) {
	for (int i = 0; i < args.length; i++) {
	    if (args[i].equals(OPT_H) || args[i].equals(OPT_LH)) {
		return true;
	    }
	}
	return false;
    }

    private boolean printVersion(String[] args) {
	for (int i = 0; i < args.length; i++) {
	    if (args[i].equals(OPT_V) || args[i].equals(OPT_LV)) {
		return true;
	    }
	}
	return false;
    }

    // throw exception if argument is missing
    private void checkArg(String cmd, String opt, int next, int total)
	throws ParserException {

	if (next >= total) {
	    ParserException pe;
	    if (cmd != null) {
		pe = new ParserException(MISSING_CMD_ARG);
		pe.cmd = cmd;
	    } else {
		pe = new ParserException(MISSING_OPT_ARG);
		pe.opt = opt;
	    }

	    throw pe;
	}
    }

    private void throwParserException(int reason, String cmd, String cmdarg,
	String opt, String optarg) throws ParserException {

	ParserException pe = new ParserException(reason);
	pe.cmd = cmd;
	pe.cmdarg = cmdarg;
	pe.opt = opt;
	pe.optarg = optarg;
	throw pe;
    }

    private void handleParserException(ParserException e) {
	if (e.reason == MISSING_CMD_ARG) {
	    System.out.println(br.getString(
			BrokerResources.E_MISSING_DBMGR_CMD_ARG, e.cmd));
	} else if (e.reason == MISSING_OPT_ARG) {
	    System.out.println(br.getString(
			BrokerResources.E_MISSING_DBMGR_OPT_ARG, e.opt));
	} else if (e.reason == BAD_CMD_ARG) {
	    System.out.println(br.getString(
			BrokerResources.E_INVALID_DBMGR_CMD_ARG, e.cmd, e.cmdarg));
	} else if (e.reason == BAD_OPT) {
	    if (e.opt.equals(OPT_P)) {
	        System.out.println(br.getString(BrokerResources.E_PASSWD_OPTION_NOT_SUPPORTED, e.opt));
	    } else {
	        System.out.println(br.getString(BrokerResources.E_INVALID_DBMGR_OPT, e.opt));
	    }
	} else if (e.reason == BAD_OPT_ARG) {
	    System.out.println(br.getString(
			BrokerResources.E_INVALID_DBMGR_OPT_ARG, e.opt, e.optarg));
	} else if (e.reason == EXTRA_CMD_SPECIFIED) {
	    System.out.println(br.getString(
			BrokerResources.E_EXTRA_DBMGR_CMD, e.cmd));
	}
    }

    private Properties parseArgs(String[] args) throws ParserException {

	Properties props = System.getProperties();
	boolean gotcmd = false;

	// check cmd and options: -b, -p, -u (-h and -v checked already)
	for (int i = 0; i < args.length; i++) {
	    if (args[i].equals(CREATE_CMD_STR)) {
		if (gotcmd) {
		    throwParserException(EXTRA_CMD_SPECIFIED, args[i], null,
					null, null);
		} else {
		    gotcmd = true;
		}

		checkArg(args[i], null, ++i, args.length);

		// find argument
		if (args[i].equals(ARGU_ALL)) {
		    props.put(CMD_NAME, CREATE_ALL_CMD);
		} else if (args[i].equals(ARGU_TBL)) {
		    props.put(CMD_NAME, CREATE_TBL_CMD);
		} else {
		    throwParserException(BAD_CMD_ARG, CREATE_CMD_STR,
					args[i], null, null);
		}
	    } else if (args[i].equals(DELETE_CMD_STR)) {
		if (gotcmd) {
		    throwParserException(EXTRA_CMD_SPECIFIED, args[i],
					null, null, null);
		} else {
		    gotcmd = true;
		}

		checkArg(args[i], null, ++i, args.length);

		// find argument
		if (args[i].equals(ARGU_TBL)) {
		    props.put(CMD_NAME, DELETE_TBL_CMD);
		} else if (args[i].equals(ARGU_OLDTBL)) {
		    props.put(CMD_NAME, DELETE_TBL_CMD);
		    props.put(ARG_NAME, ARGU_OLDTBL);
		} else {
		    throwParserException(BAD_CMD_ARG, DELETE_CMD_STR,
					args[i], null, null);
		}
	    } else if (args[i].equals(RECREATE_CMD_STR)) {
		if (gotcmd) {
		    throwParserException(EXTRA_CMD_SPECIFIED, args[i],
					null, null, null);
		} else {
		    gotcmd = true;
		}

		checkArg(args[i], null, ++i, args.length);

		// find argument
		if (args[i].equals(ARGU_TBL)) {
		    props.put(CMD_NAME, RECREATE_TBL_CMD);
		} else {
		    throwParserException(BAD_CMD_ARG, RECREATE_CMD_STR,
					args[i], null, null);
		}
	    } else if (args[i].equals(REMOVE_CMD_STR)) {
		if (gotcmd) {
		    throwParserException(EXTRA_CMD_SPECIFIED, args[i],
					null, null, null);
		} else {
		    gotcmd = true;
		}

		checkArg(args[i], null, ++i, args.length);

		// find argument
		if (args[i].equals(ARGU_BKR)) {
		    props.put(CMD_NAME, REMOVE_BKR_CMD);
		} else {
		    throwParserException(BAD_CMD_ARG, REMOVE_CMD_STR,
					args[i], null, null);
		}
            } else if (args[i].equals(DUMP_CMD)) {
		// private command for debug purpose only
		if (gotcmd) {
		    throwParserException(EXTRA_CMD_SPECIFIED, args[i],
					null, null, null);
		} else {
		    gotcmd = true;
		}

		checkArg(args[i], null, ++i, args.length);

		props.put(CMD_NAME, DUMP_CMD);
		props.put(ARG_NAME, args[i]);
            } else if (args[i].equals(DROPTBL_CMD)) {
		// private command to drop tables that match the specified
                // table name pattern for internal use only!!!
		if (gotcmd) {
		    throwParserException(EXTRA_CMD_SPECIFIED, args[i],
					null, null, null);
		} else {
		    gotcmd = true;
		}

		checkArg(args[i], null, ++i, args.length);

		props.put(CMD_NAME, DROPTBL_CMD);
		props.put(ARG_NAME, args[i]);
	    } else if (args[i].equals(RESET_CMD)) {
		// command for resetting the table lock
		if (gotcmd) {
		    throwParserException(EXTRA_CMD_SPECIFIED, args[i],
					null, null, null);
		} else {
		    gotcmd = true;
		}

		checkArg(args[i], null, ++i, args.length);

		// find argument
		if (args[i].equals(ARGU_LCK)) {
		    props.put(CMD_NAME, RESET_CMD);
		} else {
		    throwParserException(BAD_CMD_ARG, RESET_CMD,
					args[i], null, null);
		}
            } else if (args[i].equals(BACKUP_CMD)) {
                // command for backing up the jdbc store
                if (gotcmd) {
                    throwParserException(EXTRA_CMD_SPECIFIED, args[i],
                    null, null, null);
                } else {
                    gotcmd = true;
                }

                checkArg(args[i], null, ++i, args.length);

                // find argument
                if (args[i].equals(OPT_DIR)) {
                    props.put(CMD_NAME, BACKUP_CMD);
                    checkArg(args[i], null, ++i, args.length);
                    props.put(Globals.IMQ + ".backupdir", args[i]);
                } else {
                    throwParserException(BAD_CMD_ARG, BACKUP_CMD,
                    args[i], null, null);
                }
            } else if (args[i].equals(RESTORE_CMD)) {
                // command for backing up the jdbc store
                if (gotcmd) {
                    throwParserException(EXTRA_CMD_SPECIFIED, args[i],
                    null, null, null);
                } else {
                    gotcmd = true;
                }

                checkArg(args[i], null, ++i, args.length);

                // find argument
                if (args[i].equals(OPT_DIR)) {
                    props.put(CMD_NAME, RESTORE_CMD);
                    checkArg(args[i], null, ++i, args.length);
                    props.put(Globals.IMQ + ".backupdir", args[i]);
                } else {
                    throwParserException(BAD_CMD_ARG, RESTORE_CMD,
                    args[i], null, null);
                }
            } else if (args[i].equals(UPGRADE_CMD_STR)) {
                if (gotcmd) {
                    throwParserException(EXTRA_CMD_SPECIFIED, args[i],
                                        null, null, null);
                } else {
                    gotcmd = true;
                }

                checkArg(args[i], null, ++i, args.length);

                // find argument
                if (args[i].equals(ARGU_STORE)) {
                    props.put(CMD_NAME, UPGRADE_STORE_CMD);
                } else if (args[i].equals(ARGU_HASTORE)) {
                    props.put(CMD_NAME, UPGRADE_HASTORE_CMD);
                } else {
                    throwParserException(BAD_CMD_ARG, UPGRADE_CMD_STR,
                                        args[i], null, null);
                }
            } else if (args[i].equals(QUERY_CMD)) {
                if (gotcmd) {
                    throwParserException(EXTRA_CMD_SPECIFIED, args[i],
                                        null, null, null);
                } else {
                    gotcmd = true;
                }
                props.put(CMD_NAME, QUERY_CMD);
            } else if (args[i].equals(OPT_B)) {

                checkArg(null, OPT_B, ++i, args.length);
                props.put(Globals.IMQ + ".instancename", args[i]);

            } else if (args[i].equals(OPT_N)) {

                checkArg(null, OPT_N, ++i, args.length);
                props.put(Globals.BROKERID_PROPERTY, args[i]);

	    } else if (args[i].equals(OPT_U)) {

		checkArg(null, OPT_U, ++i, args.length);
		props.put(DBManager.FALLBACK_USER_PROP, args[i]);

	    } else if (args[i].equals(OPT_PW)) {

		checkArg(null, OPT_PW, ++i, args.length);
		props.put(DBManager.FALLBACK_PWD_PROP, args[i]);
		cliPasswdSpecified = true;

	    } else if (args[i].equals(OPT_PASSFILE)) {

		checkArg(null, OPT_PASSFILE, ++i, args.length);

		File passfile = null;
		try {
		    passfile = (new File(args[i])).getCanonicalFile();
		} catch (Exception e) {
		    throwParserException(BAD_OPT, null, null, args[i], null);
		}
		props.put(Globals.KEYSTORE_USE_PASSFILE_PROP, "true");
		props.put(Globals.KEYSTORE_PASSDIR_PROP,
					passfile.getParent());
		props.put(Globals.KEYSTORE_PASSFILE_PROP,
					passfile.getName());

            } else if (args[i].equals(OPT_VERBOSE)) {
                // Handled by wrapper script
                ;
            } else if (args[i].equals(OPT_DEBUG)) {
                debugSpecified = true;
            } else if (args[i].equals(OPT_FORCE)) {
                forceSpecified = true;
            } else if (args[i].equals(OPT_VARHOME)) {
                // Handled by wrapper script
                i++;
	    } else if (args[i].startsWith(OPT_D)) {
		int value_index = 0;
		String prop_name = null, prop_value = "";

		value_index = args[i].indexOf('=');
		if (args[i].length() <= 2) {
		    // -D
		    continue;
		} else if (value_index < 0) {
		    // -Dfoo
		    prop_name = args[i].substring(2);
		} else if (value_index == args[i].length() - 1) {
		    // -Dfoo=
		    prop_name = args[i].substring(2, value_index);
                } else {
		    // -Dfoo=bar
		    prop_name = args[i].substring(2, value_index);
		    prop_value = args[i].substring(value_index + 1);
                }

		props.put(prop_name, prop_value);
		if (Store.DEBUG) {
		    logger.log(Logger.DEBUG, "added sys prop: " +
				prop_name + "=" + prop_value);
		}
	    } else {
		throwParserException(BAD_OPT, null, null, args[i], null);
	    }
	}

	return props;
    }

    /*
     * Read/parse the passfile if it is specified.
     *
     * The objective is to obtain the database password.
     * It can be specified directly via the -p option
     * or via a passfile, specified using the -passfile option.
     *
     * This method is called to to see if a passfile is specified.
     * If it was, it is read and parsed.
     *
     * Matters are complicated by the fact that imqdbmgr reads
     * the broker's configuration files. This means that
     * A password or passfile can be specified either via
     * the broker's config.properties file or imqdbmgr's
     * -p/-passfile command line options.
     *
     * A slightly less frequent use case to consider is the one
     * where the user (intentionally or not) specifies a password
     * as well as a passfile. In this case, the password takes
     * precedence over the passfile.
     *
     * Summary of behavior:
     *  - Values specified on the command line (-p, -passfile) override
     *    config.properties (imq.persist.jdbc.password, imq.passfile.*).
     *
     *  - The passwd obtained from a 'password' configuration (-p or
     *    imq.persist.jdbc.password in config.properties) will override
     *    passwords obtained from a 'passfile' configuration (-passfile
     *    or imq.passfile.* properties in config.properties). This means
     *    -p overrides -passfile.
     *
     *  - A password obtained from -passfile will override a value for
     *    imq.persist.jdbc.password in config.properties.
     */
    private void parsePassfile()
        throws FileNotFoundException
    {
        String	pf_value = null,
        	pf_dir = null,
		usePassfile = null;

	/*
	 * This method checks if a passfile was specified and merges
	 * it's contents (it should contain imq.persist.jdbc.password)
	 * with the broker configuration.
	 */

	/*
	 * Return if a passfile was not specified on the cmdline or via
	 * the broker configuration file.
	 */
        usePassfile = config.getProperty(Globals.KEYSTORE_USE_PASSFILE_PROP);
        if ((usePassfile == null) ||
	    !usePassfile.equalsIgnoreCase(Boolean.TRUE.toString())) {
	    return;
	}

	// get passfile location
        if ((pf_value = config.getProperty(Globals.KEYSTORE_PASSDIR_PROP))
                                                                != null) {
	    pf_value = StringUtil.expandVariables(pf_value, config);
	    pf_dir = pf_value;
	} else {
	    pf_dir = config.getProperty(Globals.IMQ + ".etchome")
					+File.separator + "security";
	}

	String passfile_location = pf_dir  +File.separator +
                    config.getProperty(Globals.KEYSTORE_PASSFILE_PROP);

	// Check if the passfile exists else throw exception
	File pf = new File(passfile_location);
	if (pf.exists()) {
	    Properties props = new Properties();
	    // read password from passfile
            //
	    try {
	        FileInputStream fis = new FileInputStream(
				        passfile_location);
                props.load(fis);
	    } catch (IOException ioex) {
	        logger.log(Logger.ERROR,
		    br.getKString(BrokerResources.X_READ_PASSFILE), ioex);
	    }
            config.putAll(props);
        } else {
	    logger.log(Logger.ERROR,
		                br.getKString(BrokerResources.E_GET_PASSFILE,
				passfile_location));

	    throw new FileNotFoundException(
				br.getKString(BrokerResources.E_GET_PASSFILE,
				passfile_location));
	}
    }


    private void exit(int status) {
	if (standalone) {
            DBConnectionPool.reset(); // Free connection from connection pool
	    System.exit(status);
	}
    }

    void doCommand(String[] args) throws SQLException,
			BrokerException, FileNotFoundException {

	// print all
	if (args.length == 0) {
	    System.out.println(version.getBanner(true));
	    System.out.println(br.getString(BrokerResources.M_DBMGR_USAGE));
	    exit(0);
	}

	// print help
	if (printHelp(args)) {
	    System.out.println(br.getString(BrokerResources.M_DBMGR_USAGE));
	    exit(0);
	}

	// print version
	if (printVersion(args)) {
	    System.out.println(version.getBanner(true));
	    System.out.println(br.getString(BrokerResources.I_JAVA_VERSION) +
		System.getProperty("java.version") + " " +
		System.getProperty("java.vendor") + " " +
		System.getProperty("java.home"));
	    System.out.println(br.getString(BrokerResources.I_JAVA_CLASSPATH) +
		System.getProperty("java.class.path"));
	    exit(0);
	}

	Properties props = null;

	try {
            props = parseArgs(args);
	} catch (ParserException e) {
	    handleParserException(e);
	    exit(1);
	}

        // Limit the # of conn in the pool to 2
        props.setProperty( DBConnectionPool.NUM_CONN_PROP, "2" );

	if (cliPasswdSpecified)  {
            System.err.println(br.getString(BrokerResources.W_PASSWD_OPTION_DEPRECATED, OPT_PW));
            System.err.println("");
	}

        // 1st check existence of broker instance because Globals.init()
        // method will create an instance if it does not exist!
        String configName = (String)props.getProperty(Globals.IMQ + ".instancename");
        if (configName != null && configName.length() > 0) {
            Globals.pathinit(null);
            String topname =
                Globals.JMQ_INSTANCES_HOME + File.separator + configName;

            if (!(new File(topname)).exists()) {
                System.err.println(br.getString(
                    BrokerResources.E_INSTANCE_NOT_EXIST, configName));
                System.exit(1);
            }
        }

	Globals.init(props, false);
        config = Globals.getConfig();
	logger = Globals.getLogger();

        // Password in passfile override broker's properties object
        parsePassfile();

        // User/Password passed in from command line option take precedence.
        // Inorder to override the User/Password we need to construct the
        // correct property based on the dbVendor value; we do this after
        // calling Globals.init() so we can determine the dbVendor value.
        String dbVendor = config.getProperty(DBManager.JDBC_PROP_PREFIX + ".dbVendor");
        String vendorPropPrefix = DBManager.JDBC_PROP_PREFIX + "." + dbVendor;
        String user = props.getProperty(DBManager.FALLBACK_USER_PROP);
        if (user != null && user.length() > 0) {
            config.put(vendorPropPrefix + ".user", user);
        }

        String pwd = props.getProperty(DBManager.FALLBACK_PWD_PROP);
        if (pwd != null && pwd.length() > 0) {
            config.put(vendorPropPrefix + ".password", pwd);
        }

	dbmgr = DBManager.getDBManager();

	String cmd = props.getProperty(CMD_NAME);

	// print out info messages
	String brokerid = dbmgr.getBrokerID();
        String url;
        if (CREATE_ALL_CMD.equals(cmd)) {
            url = dbmgr.getCreateDBURL();
        } else {
            url = dbmgr.getOpenDBURL();
        }
        if ( url == null ) {
            url = "not specified";
        }
        user = dbmgr.getUser();
        if ( user == null ) {
            user = "not specified";
        }

        String msgArgs[] = { String.valueOf(JDBCStore.STORE_VERSION), brokerid,
                             url, user };
        logger.logToAll(Logger.INFO,
            br.getString(BrokerResources.I_JDBC_STORE_INFO, msgArgs));

	if (CREATE_ALL_CMD.equals(cmd)) {
	    doCreate(true); // create database as well
	} else if (CREATE_TBL_CMD.equals(cmd)) {
	    doCreate(false); // just tables
	} else if (DELETE_TBL_CMD.equals(cmd)) {
	    doDelete(props.getProperty(ARG_NAME));
        } else if (RECREATE_TBL_CMD.equals(cmd)) {
            doRecreate();
        } else if (REMOVE_BKR_CMD.equals(cmd)) {
            doRemoveBkr();
        } else if (DUMP_CMD.equals(cmd)) {
            String arg = props.getProperty(ARG_NAME);
            doDump(arg);
        } else if (DROPTBL_CMD.equals(cmd)) {
            String arg = props.getProperty(ARG_NAME);
            doDropTablesByPattern(arg);
	} else if (RESET_CMD.equals(cmd)) {
	    doReset();
        } else if(BACKUP_CMD.equals(cmd)) {
            doBackup();
        } else if(RESTORE_CMD.equals(cmd)) {
            doRestore();
        } else if(UPGRADE_STORE_CMD.equals(cmd)) {
            doUpgrade(false);
        } else if(UPGRADE_HASTORE_CMD.equals(cmd)) {
            doUpgrade(true);
        } else if(QUERY_CMD.equals(cmd)) {
            doQuery();
	} else if (cmd == null) {
	    System.out.println(br.getString(BrokerResources.E_MISSING_DBMGR_CMD));
	} else {
	    System.out.println(br.getString(BrokerResources.E_INVALID_DBMGR_CMD, cmd));
	}
    }

    private static class ParserException extends Exception {
	String cmd;
	String cmdarg;
	String opt;
	String optarg;
	int reason;

	ParserException(int reason) {
	    this.reason = reason;
	}
    }

    public static void main(String[] args) {

        int exitCode = 0;
        DBTool tool = new DBTool(true); // standalone;
	try {
	    tool.doCommand(args);
	} catch (Exception e) {
            if (tool.debugSpecified) {
                e.printStackTrace();
            }
	    exitCode = 1;
	} finally {
            if (tool.dbmgr != null) {
                tool.dbmgr.close(); // Free connection from connection pool
            }
        }

        System.exit(exitCode);
    }
}
