/*
 * @(#)DestinationDAOImpl.java	1.16 09/28/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.jmsserver.util.*;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.service.ConnectionUID;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.io.Status;

import java.util.*;
import java.sql.*;
import java.io.ByteArrayInputStream;
import java.io.IOException;

/**
 * This class implement a generic DestinationDAO.
 *
 * @version	1.16
 */
class DestinationDAOImpl extends BaseDAOImpl implements DestinationDAO {

    protected String tableName;

    // SQLs
    protected String insertSQL;
    protected String updateSQL;
    protected String updateConnectedTimeSQL;
    protected String deleteSQL;
    protected String selectSQL;
    protected String selectConnectedTimeSQL;
    protected String selectCountSQL;
    protected String selectAllSQL;
    protected String selectAllNamesSQL;
    protected String selectExistSQL;
    protected String selectAutoCreateDestActiveSQL;
    protected String selectTakeoverSQL;
    protected String takeoverSQL;

    /**
     * Constructor
     * @throws BrokerException
     */
    DestinationDAOImpl() throws BrokerException {

        // Initialize all SQLs
        DBManager dbMgr = DBManager.getDBManager();

        tableName = dbMgr.getTableName( TABLE_NAME_PREFIX );

        insertSQL = new StringBuffer(128)
            .append( "INSERT INTO " ).append( tableName )
            .append( " ( " )
            .append( ID_COLUMN ).append( ", " )
            .append( DESTINATION_COLUMN ).append( ", " )
            .append( IS_LOCAL_COLUMN ).append( ", " )
            .append( CONNECTION_ID_COLUMN ).append( ", " )
            .append( CONNECTED_TS_COLUMN ).append( ", " )
            .append( BROKER_ID_COLUMN ).append( ", " )
            .append( CREATED_TS_COLUMN )
            .append( ") VALUES ( ?, ?, ?, ?, ?, ?, ? )" )
            .toString();

        updateSQL = new StringBuffer(128)
            .append( "UPDATE " ).append( tableName )
            .append( " SET " )
            .append( DESTINATION_COLUMN ).append( " = ?, " )
            .append( IS_LOCAL_COLUMN ).append( " = ?, " )
            .append( CONNECTION_ID_COLUMN ).append( " = ?, " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        updateConnectedTimeSQL = new StringBuffer(128)
            .append( "UPDATE " ).append( tableName )
            .append( " SET " )
            .append( CONNECTED_TS_COLUMN ).append( " = ?" )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        deleteSQL = new StringBuffer(128)
            .append( "DELETE FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        selectSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( DESTINATION_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        selectConnectedTimeSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( CONNECTED_TS_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        selectCountSQL = new StringBuffer(128)
            .append( "SELECT COUNT(*)" )
            .append( " FROM " ).append( tableName )
            .toString();

        selectAllSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( DESTINATION_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( IS_LOCAL_COLUMN ).append( " = 0" )
            .append( " OR " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .toString();

        selectAllNamesSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( ID_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( IS_LOCAL_COLUMN ).append( " = 0" )
            .append( " OR " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .toString();

        selectExistSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( ID_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( ID_COLUMN ).append( " = ?" )
            .toString();

        selectAutoCreateDestActiveSQL = new StringBuffer(128)
            .append( "SELECT COUNT(*) FROM " )
            .append( dbMgr.getTableName( MessageDAO.TABLE_NAME_PREFIX ) )
            .append( " WHERE " )
            .append( MessageDAO.DESTINATION_ID_COLUMN ).append( " = ?" )
            .append( " AND " )
            .append( MessageDAO.BROKER_ID_COLUMN ).append( " <> ?" )
            .toString();

        selectTakeoverSQL = new StringBuffer(128)
            .append( "SELECT " )
            .append( DESTINATION_COLUMN )
            .append( " FROM " ).append( tableName )
            .append( " WHERE " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .toString();

        takeoverSQL = new StringBuffer(128)
            .append( "UPDATE " ).append( tableName )
            .append( " SET " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .append( " WHERE " )
            .append( BROKER_ID_COLUMN ).append( " = ?" )
            .toString();
    }

    /**
     * Get the prefix name of the table.
     * @return table name
     */
    public final String getTableNamePrefix() {
        return TABLE_NAME_PREFIX;
    }

    /**
     * Get the name of the table.
     * @return table name
     */
    public final String getTableName() {
        return tableName;
    }

    /**
     * Insert a new entry.
     * @param conn database connection
     * @param destination the Destination
     * @param brokerID the broker ID
     * @param connectedTime timestamp
     * @param createdTime timestamp
     * @throws BrokerException if destination already exists in the store
     */
    public void insert( Connection conn, Destination destination,
        String brokerID, long connectedTime, long createdTime )
        throws BrokerException {

        String destName = destination.getUniqueName();

        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( true );
                myConn = true;
            }

            if ( hasDestination( conn, destName) ) {
                throw new BrokerException(
                    br.getKString( BrokerResources.E_DESTINATION_EXISTS_IN_STORE,
                    destName ) );
            }

            try {
                int isLocal = destination.getIsLocal() ? 1 : 0;
                long connectionID = -1;
                if ( isLocal > 0 ) {
                    // Store additional info for temp destination
                    ConnectionUID cUID = destination.getConnectionUID();
                    if ( cUID != null ) {
                        if ( connectedTime <= 0 ) {
                            connectedTime = System.currentTimeMillis();
                        }
                        connectionID = cUID.longValue();
                    }

                    if ( brokerID == null ) {
                        brokerID = dbMgr.getBrokerID();
                    }
                }

                pstmt = conn.prepareStatement( insertSQL );
                pstmt.setString( 1, destName );
                ByteArrayInputStream bais = Util.setObject( pstmt, 2, destination );
                pstmt.setInt( 3, isLocal );
                Util.setLong( pstmt, 4, connectionID );
                Util.setLong( pstmt, 5, connectedTime );
                Util.setString( pstmt, 6, brokerID );
                pstmt.setLong( 7, createdTime );
                pstmt.executeUpdate();
                bais.close();
            } catch ( Exception e ) {
                try {
                    if ( (conn != null) && !conn.getAutoCommit() ) {
                        conn.rollback();
                    }
                } catch ( SQLException rbe ) {
                    logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
                }

                Exception ex;
                if ( e instanceof BrokerException ) {
                    throw (BrokerException)e;
                } else if ( e instanceof IOException ) {
                    ex = DBManager.wrapIOException("[" + insertSQL + "]", (IOException)e);
                } else if ( e instanceof SQLException ) {
                    ex = DBManager.wrapSQLException("[" + insertSQL + "]", (SQLException)e);
                } else {
                    ex = e;
                }

                throw new BrokerException(
                    br.getKString( BrokerResources.X_PERSIST_DESTINATION_FAILED,
                    destName ), ex );
            }
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Update existing entry.
     * @param conn database connection
     * @param destination the Destination
     * @throws BrokerException if destination does not exists in the store
     */
    public void update( Connection conn, Destination destination )
        throws BrokerException {

        String destName = destination.getUniqueName();

        boolean updated = false;
        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( true );
                myConn = true;
            }

            int isLocal = destination.getIsLocal() ? 1 : 0;
            long connectionID = -1;
            String brokerID = null;
            if ( isLocal > 0 ) {
                // Store additional info for temp destination
                ConnectionUID cUID = destination.getConnectionUID();
                if ( cUID != null ) {
                    connectionID = cUID.longValue();
                }
                brokerID = dbMgr.getBrokerID();
            }

            pstmt = conn.prepareStatement( updateSQL );
            ByteArrayInputStream bais = Util.setObject( pstmt, 1, destination );
            pstmt.setInt( 2, isLocal );
            Util.setLong( pstmt, 3, connectionID );
            Util.setString( pstmt, 4, brokerID );
            pstmt.setString( 5, destName );
            if ( pstmt.executeUpdate() == 1 ) {
                updated = true;
            }
            bais.close();
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof IOException ) {
                ex = DBManager.wrapIOException("[" + updateSQL + "]", (IOException)e);
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + updateSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_PERSIST_DESTINATION_FAILED,
                destName ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }

        if ( !updated ) {
            // Otherwise we're assuming the entry does not exist
            throw new BrokerException(
                br.getKString( BrokerResources.E_DESTINATION_NOT_FOUND_IN_STORE,
                destName ), Status.NOT_FOUND );
        }
    }

    /**
     * Update existing entry.
     * @param conn database connection
     * @param destination the Destination
     * @throws BrokerException if destination does not exists in the store
     */
    public void updateConnectedTime( Connection conn, Destination destination,
        long connectedTime ) throws BrokerException {

        String destName = destination.getUniqueName();

        if ( !destination.getIsLocal() ) {
            // We've a problem, trying to update a non-local destination!
            throw new BrokerException(
                br.getKString( BrokerResources.E_UPDATE_NONLOCAL_DST_CONNECTED_TIME,
                    destName ) );
        }

        boolean updated = false;
        boolean myConn = false;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( updateConnectedTimeSQL );
            pstmt.setLong( 1, connectedTime );
            pstmt.setString( 2, destName );
            if ( pstmt.executeUpdate() == 1 ) {
                updated = true;
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + updateConnectedTimeSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_PERSIST_DESTINATION_FAILED,
                destName ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }

        if ( !updated ) {
            // Otherwise we're assuming the entry does not exist
            throw new BrokerException(
                br.getKString( BrokerResources.E_DESTINATION_NOT_FOUND_IN_STORE,
                destName ), Status.NOT_FOUND );
        }
    }

    /**
     * Delete an existing entry.
     * @param conn Database Connection
     * @param destination the Destination
     * @throws BrokerException
     */
    public void delete( Connection conn, Destination destination )
        throws BrokerException {

        String destName = destination.getUniqueName();

        boolean myConn = false;
        String sql = null;
        PreparedStatement pstmt = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( false );
                myConn = true;
            }

            boolean deleteDestination = true;

            if ( destination.isAutoCreated() && Globals.getHAEnabled() ) {
                // Since auto-create destination are shared in HA mode,
                // delete only if there are no msgs for this destination, i.e.
                // check the # of brokers still using this destination
                // excluding the current broker.
                sql = selectAutoCreateDestActiveSQL;
                pstmt = conn.prepareStatement( sql );
                pstmt.setString( 1, destName );
                pstmt.setString( 2, dbMgr.getBrokerID() );
                ResultSet rs = pstmt.executeQuery();
                if ( rs.next() && ( rs.getInt( 1 ) > 0 ) ) {
                    // Other brokers still have msgs for this destination
                    deleteDestination = false;
                }
                rs.close();
            }

            // First remove all messages from this destination for current broker
            dbMgr.getDAOFactory().getMessageDAO().deleteByDestination( conn, destination );

            if ( deleteDestination ) {
                sql = deleteSQL;
                pstmt = conn.prepareStatement( sql );
                pstmt.setString( 1, destName );
                pstmt.executeUpdate();
            }

            if ( myConn ) {
                conn.commit();
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + sql + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_REMOVE_DESTINATION_FAILED,
                destName ), ex );
        } finally {
            if ( myConn ) {
                Util.close( null, pstmt, conn );
            } else {
                Util.close( null, pstmt, null );
            }
        }
    }

    /**
     * Delete all entries.
     * @param conn Database Connection
     * @throws BrokerException
     */
    public void deleteAll( Connection conn )
        throws BrokerException {

        String whereClause = null;
        if ( Globals.getHAEnabled() ) {
            // Only delete destinations that belong to the running broker,
            // construct the where clause for the delete statement:
            //   DELETE FROM mqdst40cmycluster WHERE broker_id = 'mybroker'            
            whereClause = new StringBuffer(128)
                .append( BROKER_ID_COLUMN ).append( " = '")
                .append( DBManager.getDBManager().getBrokerID() ).append( "'" )
                .toString();
        }

        deleteAll( conn, whereClause, null, 0 );
    }

    /**
     * Take over the local destinations.
     * @param conn database connection
     * @param id the current or local broker ID
     * @param targetBrokerID the broker ID of the store being taken over
     * @return a List of all local destinations the target broker owns
     * @throws BrokerException
     */
    public List takeover( Connection conn, String id, String targetBrokerID )
        throws BrokerException {

        List list = Collections.EMPTY_LIST;

        String sql = null;
        PreparedStatement pstmt = null;
        try {
            // First retrieve all local destinations for the target broker
            sql = selectTakeoverSQL;
            pstmt = conn.prepareStatement( sql );
            pstmt.setString( 1, targetBrokerID );
            ResultSet rs = pstmt.executeQuery();
            list = (List)loadData( rs, false );
            rs.close();
            pstmt.close();

            // Now takeover those destinations
            sql = takeoverSQL;
            pstmt = conn.prepareStatement( sql );
            pstmt.setString( 1, id );
            pstmt.setString( 2, targetBrokerID );
            int count = pstmt.executeUpdate();

            // Verify that we are able to takeover all destinations
            if ( count != list.size() ) {
                // This shouldn't occur but just being safe
                String[] args = { targetBrokerID,
                                  String.valueOf( list.size() ),
                                  String.valueOf( count ) };
                throw new BrokerException(
                    br.getKString( BrokerResources.E_TAKEOVER_DST_FAILED, args ) );
            }
        } catch ( Exception e ) {
            try {
                if ( (conn != null) && !conn.getAutoCommit() ) {
                    conn.rollback();
                }
            } catch ( SQLException rbe ) {
                logger.log( Logger.ERROR, BrokerResources.X_DB_ROLLBACK_FAILED, rbe );
            }

            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + sql + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.E_TAKEOVER_DST_2_FAILED,
                    targetBrokerID ), ex );
        } finally {
            Util.close( null, pstmt, null );
        }

        return list;
    }

    /**
     * Get a destination.
     * @param conn database connection
     * @param destName destination's name
     * @return Destination the Destination object
     * @throws BrokerException
     */
    public Destination getDestination( Connection conn, String destName )
        throws BrokerException {

        Destination dest = null;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectSQL );
            pstmt.setString( 1, destName );
            rs = pstmt.executeQuery();
            dest = (Destination)loadData( rs, true );
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_DESTINATION_FAILED,
                    destName ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return dest;
    }

    /**
     * Get connected timestamp for a local destination.
     * @param conn database connection
     * @param destName destination's name
     * @return connected timestamp
     * @throws BrokerException
     */
    public long getDestinationConnectedTime( Connection conn, String destName )
        throws BrokerException {

        long connectedTime = -1;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectConnectedTimeSQL );
            pstmt.setString( 1, destName );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                connectedTime = rs.getLong( 1 );
            } else {
                throw new BrokerException(
                    br.getKString( BrokerResources.E_DESTINATION_NOT_FOUND_IN_STORE,
                    destName ), Status.NOT_FOUND );
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectConnectedTimeSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_DESTINATION_FAILED,
                    destName ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return connectedTime;
    }

    /**
     * Retrieve all non-local destinations and local destination for the
     * specified broker.
     * @param conn database connection
     * @param brokerID the broker ID
     * @return a List of Destination objects; an empty List is returned
     * if no destinations exist in the store
     */
    public List getAllDestinations( Connection conn, String brokerID )
        throws BrokerException {

        List list = Collections.EMPTY_LIST;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            DBManager dbMgr = DBManager.getDBManager();
            if ( conn == null ) {
                conn = dbMgr.getConnection( true );
                myConn = true;
            }

            if ( brokerID == null ) {
                brokerID = dbMgr.getBrokerID();
            }

            pstmt = conn.prepareStatement( selectAllSQL );
            pstmt.setString( 1, brokerID );
            rs = pstmt.executeQuery();
            list = (List)loadData( rs, false );
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectAllSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_LOAD_DESTINATIONS_FAILED ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return list;
    }

    /**
     * Check whether the specified destination exists.
     * @param conn database connection
     * @param destName name of destination
     * @return return true if the specified destination exists
     */
    public boolean hasDestination( Connection conn, String destName ) throws BrokerException {

        boolean found = false;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectExistSQL );
            pstmt.setString( 1, destName );
            rs = pstmt.executeQuery();
            if ( rs.next() ) {
                found = true;
            }
        } catch ( Exception e ) {
            Exception ex;
            if ( e instanceof BrokerException ) {
                throw (BrokerException)e;
            } else if ( e instanceof SQLException ) {
                ex = DBManager.wrapSQLException("[" + selectExistSQL + "]", (SQLException)e);
            } else {
                ex = e;
            }

            throw new BrokerException(
                br.getKString( BrokerResources.X_JDBC_QUERY_FAILED,
                selectExistSQL ), ex );
        } finally {
            if ( myConn ) {
                Util.close( rs, pstmt, conn );
            } else {
                Util.close( rs, pstmt, null );
            }
        }

        return found;
    }

    /**
     * Check whether the specified destination exists.
     * @param conn database connection
     * @param destName name of destination
     * @throws BrokerException if the destination does not exists in the store
     */
    public void checkDestination( Connection conn, String destName )
        throws BrokerException {

        if ( !hasDestination( conn, destName ) ) {
            throw new BrokerException(
                br.getKString( BrokerResources.E_DESTINATION_NOT_FOUND_IN_STORE,
                destName ), Status.NOT_FOUND );
        }
    }

    /**
     * Get debug information about the store.
     * @return a HashMap of name value pair of information
     * @param conn
     */
    public HashMap getDebugInfo( Connection conn ) {

        HashMap map = new HashMap();
        int size = -1;

        boolean myConn = false;
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        try {
            // Get a connection
            if ( conn == null ) {
                conn = DBManager.getDBManager().getConnection( true );
                myConn = true;
            }

            pstmt = conn.prepareStatement( selectCountSQL );
            rs = pstmt.executeQuery();

            if ( rs.next() ) {
                size = rs.getInt( 1 );
            }
        } catch ( Exception e ) {
            logger.log( Logger.ERROR, BrokerResources.X_JDBC_QUERY_FAILED,
                selectCountSQL, e );
        } finally {
            try {
                if ( myConn ) {
                    Util.close( rs, pstmt, conn );
                } else {
                    Util.close( rs, pstmt, null );
                }
            } catch ( BrokerException be ) {
                logger.log( Logger.ERROR, be.getMessage(), be.getCause() );
            }
        }

        map.put( "Destinations(" + tableName + ")", String.valueOf( size ) );
        return map;
    }

    /**
     * Load a single destination or destinations from a ResultSet.
     * @param rs the ResultSet
     * @param isSingleRow specify interesed in only the 1st row of the ResultSet
     * @return a single Destination or List of Destinations
     * @throws SQLException
     */
    protected Object loadData( ResultSet rs, boolean isSingleRow )
        throws SQLException {

        ArrayList list = null;
        if ( !isSingleRow ) {
            list = new ArrayList( 100 );
        }

        while ( rs.next() ) {
            try {
                Destination dest = (Destination)Util.readObject( rs, 1 );
                if ( isSingleRow ) {
                    return dest;
                } else {
                    list.add( dest );
                }
            } catch ( Exception e ) {
                // fail to parse destination object; just log it
                logger.logStack( Logger.ERROR,
                    BrokerResources.X_PARSE_DESTINATION_FAILED, e );
            }
        }

        return list;
    }
}
