/*
 * @(#)TransactionDAO.java	1.9 08/25/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.jmsserver.persist.jdbc;

import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.jmsserver.data.TransactionState;

import java.sql.SQLException;
import java.sql.Connection;
import java.util.HashMap;
import java.util.List;

/**
 * This class is an interface for the Transaction table which will be implemented
 * by database specific code.
 *
 * @version	1.9
 */
public interface TransactionDAO extends BaseDAO {

    /**
     * Destination table:
     * Holds all the destination in the system.
     *
     * CREATE TABLE MQTXN<schemaVersion>[C<clusterID>|S<brokerID>] (
     *      ID              NUMBER NOT NULL,
     *      STATE           INTEGER NOT NULL,
     *      TXN_STATE       LONGVARBINARY NOT NULL,
     *      AUTO_ROLLBACK   INTEGER NOT NULL,
     *      XID             VARCHAR(100),
     *      BROKER_ID       VARCHAR(100) NOT NULL,
     *      EXPIRED_TS_COLUMN      BIGINT NOT NULL,
     *      ACCESSED_TS_COLUMN     BIGINT NOT NULL
     *      PRIMARY KEY(ID)
     * );
     *
     * ID - Long value of local broker transaction
     * STATE - State of the transaction
     * TXN_STATE - Serialized Transaction state information
     * AUTO_ROLLBACK - Transaction rollback mode
     * XID - XID (if any) associated with this transaction
     * BROKER_ID - Broker ID that owns or responsible for reaping the transaction
     * EXPIRED_TS_COLUMN - Timestamp when the transaction expired
     * ACCESSED_TS_COLUMN - Timestamp when the transaction was last accessed
     */
    public static final String TABLE_NAME_PREFIX = "MQTXN" + DBConstants.SCHEMA_VERSION;
    public static final String ID_COLUMN = "ID";
    public static final String STATE_COLUMN = "STATE";
    public static final String TXN_STATE_COLUMN = "TXN_STATE";
    public static final String AUTO_ROLLBACK_COLUMN = "AUTO_ROLLBACK";
    public static final String XID_COLUMN = "XID";
    public static final String BROKER_ID_COLUMN = "BROKER_ID";
    public static final String EXPIRED_TS_COLUMN = "EXPIRED_TS";
    public static final String ACCESSED_TS_COLUMN = "ACCESSED_TS";

    void insert( Connection conn, TransactionUID txnUID, TransactionState txnState )
        throws BrokerException;

    void updateTransactionState( Connection conn, TransactionUID txnUID,
        TransactionState state ) throws BrokerException;

    void updateAccessedTime( Connection conn, TransactionUID txnUID,
        long timeStamp ) throws BrokerException;

    void updateBroker( Connection conn, TransactionUID txnUID, String brokerID )
        throws BrokerException;

    void delete( Connection conn, TransactionUID txnUID ) throws BrokerException;

    List takeover( Connection conn, String brokerID, String targetBrokerID )
        throws BrokerException;

    TransactionState getTransactionState( Connection conn, TransactionUID txnUID )
        throws BrokerException;

    long getAccessedTime( Connection conn, TransactionUID txnUID ) throws BrokerException;

    String getBroker( Connection conn, TransactionUID txnUID ) throws BrokerException;

    List getTransactionsByBrokerID( Connection conn, String brokerID ) throws BrokerException;

    HashMap getAllTransactionStates( Connection conn ) throws BrokerException;

    int[] getTransactionUsageInfo( Connection conn, TransactionUID txnUID )
        throws BrokerException;

    void checkTransaction( Connection conn, long txnUID ) throws BrokerException;

    boolean hasTransaction( Connection conn, long txnUID ) throws BrokerException;
}
