/*
 * @(#)HAMonitorService.java	1.40 11/02/05
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved
 *
 */

package com.sun.messaging.jmq.jmsserver.service;

import java.util.*;
import java.io.*;
import java.net.*;

import com.sun.messaging.jmq.util.JMQXid;
import com.sun.messaging.jmq.io.MQAddress;
import com.sun.messaging.jmq.io.PortMapperTable;
import com.sun.messaging.jmq.io.PortMapperEntry;
import com.sun.messaging.jmq.io.Status;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.util.UID;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.Broker;
import com.sun.messaging.jmq.jmsserver.BrokerStateHandler;
import com.sun.messaging.jmq.jmsserver.BrokerNotification;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.BrokerAddress;
import com.sun.messaging.jmq.jmsserver.cluster.*;
import com.sun.messaging.jmq.jmsserver.resources.*;
import com.sun.messaging.jmq.jmsserver.cluster.ha.*;
import com.sun.messaging.jmq.jmsserver.data.TransactionUID;
import com.sun.messaging.jmq.io.PacketType;
import com.sun.messaging.jmq.jmsserver.data.PacketRouter;
import com.sun.messaging.jmq.jmsserver.data.handlers.TransactionHandler;
import com.sun.messaging.jmq.jmsserver.data.TransactionList;
import com.sun.messaging.jmq.jmsserver.data.AutoRollbackType;
import com.sun.messaging.jmq.jmsserver.data.TransactionState;
import com.sun.messaging.jmq.util.timer.JMQTimerTask;
import com.sun.messaging.jmq.util.timer.JMQTimer;
import com.sun.messaging.jmq.jmsserver.persist.TakeoverStoreInfo;
import com.sun.messaging.jmq.jmsserver.persist.TakeoverLockException;
import com.sun.messaging.jmq.jmsserver.persist.HABrokerInfo;


/**
 * This class handles general setup of an HA broker, monitoring the state of
 * an HA broker (for split brain) and HA Takeover.
 * <P>
 * The logic to determine when a failover should occur is not handled by
 * this class
 */
public class HAMonitorService implements ClusterListener
{

   private Logger logger = Globals.getLogger();


   /**
    * Value to use for the monitoring threshold.
    */
   private int MAX_MONITOR = Globals.getConfig().
               getIntProperty(Globals.IMQ + 
                  ".cluster.monitor.threshold", 2);

   /**
    * Value to use for the monitoring timeout.
    */
    private int monitorTimeout = Globals.getConfig().getIntProperty(Globals.IMQ +
                    ".cluster.monitor.interval", 30)* 1000;

   /**
    * Value to use for the reaper timeout.
    */
    private int reaperTimeout = Globals.getConfig().getIntProperty(
                    Globals.IMQ +
                    ".cluster.reaptime", 300)* 1000;

    /**
     * Configuration information
     */
    ClusterManager clusterconfig = null;

    /**
     * ID associated with the local broker
     */
    String mybrokerid = null;

    /**
     * Monitor task which periodically updated timestamps
     */
    private HAMonitorTask haMonitor = null;


    /**
     * Indoubt brokers which are being monitored
     */
    Map indoubtBrokers = Collections.synchronizedMap(new LinkedHashMap());

    /**
     * private class to hold information associated with an
     * indoubt broker
     */
    class indoubtData
    {
        String brokerid;
        long lastts = 0;
        int monitorCnt = 0;
    }


    /**
     * timer task which is called periodically to monitor
     * any indoubt brokers and update heartbeat timestamps
     */
    class HAMonitorTask implements Runnable {

        boolean valid = true;

        public HAMonitorTask() {
        }

        public void cancel() {
            valid = false;
        }

        public void run() {

            if (!valid) return;

            monitor(); 
        }
    }    

    /**
     * use instead of timer thread so that we have control of
     * thread priority
     */
    class HATimerThread implements Runnable
    {

        long nexttime = 0;
        long repeatItr = 0;
        Thread thr = null;
        Runnable child = null;

        public HATimerThread(String name, Runnable runner, int initTO, int repeatItr) {
            nexttime = initTO + System.currentTimeMillis();
            this.repeatItr = repeatItr;
            this.child = runner;
            thr = new Thread(this, name);
            thr.setPriority(Thread.MAX_PRIORITY);
            thr.start();
        }


        public void run() {
            while (true) {
                long time = System.currentTimeMillis();
                synchronized(this) {
                    if (time < nexttime) {
                        try {
                            this.wait(nexttime  - time);
                        } catch (InterruptedException ex) {}
                        continue;
                     } else {
                         child.run();
                     }
                 }
                 if (repeatItr == 0) break;
                 nexttime = nexttime + repeatItr;
            }      
                       
        }
    }


    /**
     * timer task which is to reap old temp destinations
     * and transactions
     */
    class TakeoverReaper extends JMQTimerTask {

        List txns = null;
        List dsts = null;

        Map dstttl = null;
        String id = null;
        long time = 0;
        public TakeoverReaper(String broker, List txns, List dsts)
        {
            this.txns = txns;
            this.dsts = dsts;
            this.id = broker;

            dstttl = new WeakHashMap();
            try {
                logger.log(logger.DEBUG,"monitoring " + txns.size() +
                     " transactions");
            
                logger.log(logger.DEBUG,"monitoring " + dsts.size() +
                     " destinations");
                dstttl = new WeakHashMap();
                Iterator itr = dsts.iterator();
                while (itr.hasNext()) {
                    Destination dst = (Destination)itr.next();
                    long time=Globals.getStore().getDestinationConnectedTime(dst);
                    dstttl.put(dst,new Long(time));
                }
            } catch (Exception ex) {
                logger.logStack(Logger.WARNING,
                    BrokerResources.E_INTERNAL_BROKER_ERROR,
                    "reaping destinations", ex);
            }
            // reap any destinations immediately @ creation
            // after that reaping will happen on a periodic
            // basis
            
 
        }

        public void run() {
            boolean done = processTxns();
            if (done)
                cancel();
        }

        public boolean processTxns() {
            logger.log(logger.INFO,
                      BrokerResources.I_REAP_DST,
                      id);
            try {
                // see if txn still exists
                
                // if it does and there are no msgs or acks, clean it up
                Iterator itr = txns.iterator();
                while (itr.hasNext()) {
                    TransactionUID tid = (TransactionUID) itr.next();

                    // ok, make sure it really exists
                    TransactionState state = null;
                    try {
                        state = Globals.getStore().getTransactionState(tid);
                    } catch (BrokerException ex) { // nope, gone
                        Globals.getTransactionList().removeTransactionAck(tid);
                        Globals.getTransactionList().removeTransactionID(tid);
                        itr.remove();
                        continue;
                    }

                    int realstate = state.getState();
                    int msgs = 0;
                    int acks = 0;
                    try {
                        int[] retval = Globals.getStore().getTransactionUsageInfo(tid);
                        msgs = retval[0];
                        acks = retval[1];
                    } catch (Exception ex) {
                    }

                    // OK .. if not in one of the "valid" states
                    //  immediately rollback
                    // else  count the # of acks, messages
                    if ((realstate != TransactionState.ROLLEDBACK &&
                        realstate != TransactionState.PREPARED &&
                        realstate != TransactionState.COMMITTED) ||
                        msgs == 0 && acks == 0) {
                        logger.log(Logger.DEBUG,"Removing finished transaction " + tid);
                         TransactionHandler thandler = (TransactionHandler)
                                Globals.getPacketRouter(0).getHandler(PacketType.ROLLBACK_TRANSACTION);

                          thandler.doRollback(tid, null, null, state, null, null);
                        itr.remove();
                    } else if (realstate == TransactionState.ROLLEDBACK) {
                    }
                    
                }


                // takeover here
                itr = dsts.iterator();
                while (itr.hasNext()) {
                    Destination dst = (Destination)itr.next();
                    long age = Globals.getStore().getDestinationConnectedTime(dst);
                    Long oldAge = (Long)dstttl.get(dst);
                    if (oldAge != null && oldAge.longValue() == age) {
                        // no one is using it, reap
                        Globals.getStore().removeDestination(dst, true);
                    }
                    itr.remove();
                }
                logger.log(logger.INFO,
                     BrokerResources.I_REAP_DST_DONE,
                     id);
            } catch (Exception ex) {
                logger.logStack(Logger.WARNING,
                    BrokerResources.E_INTERNAL_BROKER_ERROR,
                    "removing reaped destinations", ex);
            }
            if (dsts.size() == 0 && txns.size() == 0 ) {
               return true;
            }
            return false;
                
        }
    }    


    /**
     * Start the monitoring service.
     * @param clusterid the id of this cluster
     * @param brokerURL the address to use for the broker (which may be
     *            different than the one currently associated with
     *            this broker in the persistent store.
     *
     * @throws IllegalStateException if the clusterid is invalid
     * @throws IllegalAccessException if another broker is using the 
     *              same brokerid
     * @throws BrokerException if there are issues accessing the store.
     */

    public HAMonitorService(String clusterid, MQAddress brokerURL) 
        throws IllegalStateException, BrokerException, IllegalAccessException
    {

        clusterconfig = Globals.getClusterManager();
        clusterconfig.addEventListener(this);
        mybrokerid = clusterconfig.getLocalBroker().getBrokerName();

        logger.log(logger.INFO, BrokerResources.I_MONITOR_INFO,
               mybrokerid, brokerURL);

        // validate id is valid
        if (!clusterid.equals(clusterconfig.getClusterId())) {
            logger.log(Logger.ERROR,
                  BrokerResources.E_ERROR_MONITOR_BAD_CID,
                      clusterid, clusterconfig.getClusterId());
            throw new IllegalStateException("Bad Cluster ID " + clusterid);
        }
       
        // retrieve broker state info
        HAClusteredBroker cb = (HAClusteredBroker)clusterconfig.getLocalBroker();

        // make sure we are valid

        MQAddress mqa = cb.getBrokerURL();
        if (!mqa.equals(brokerURL)) {

            logger.log(Logger.INFO,
                 BrokerResources.I_UPD_STORED_PORT,
                 mybrokerid, brokerURL);

                // we need to vaidate that there isnt a misconfiguration
                // somewhere
            try {
                String version = String.valueOf(PortMapperTable.PORTMAPPER_VERSION) 
                         + "\n";
                PortMapperTable pt = new PortMapperTable();

                Socket s = new Socket(mqa.getHostName(), mqa.getPort());
                InputStream is = s.getInputStream();
                OutputStream os = s.getOutputStream();
        
                try {
                        os.write(version.getBytes());
                        os.flush();
                } catch (IOException e) {
                        // This can sometimes fail if the server already wrote
                        // the port table and closed the connection
                        // Ignore...
                }
                pt.read(is);
    
                is.close();
                os.close();
                s.close();
                PortMapperEntry pme = pt.get(PortMapper.SERVICE_NAME);

                String remotebrokerid = pme.getProperty("brokerid");

                if (mybrokerid.equals(remotebrokerid)) {
                        logger.log(Logger.ERROR, BrokerResources.E_BID_CONFLICT,
                           mybrokerid);
                        Broker.getBroker().exit(1,
                           Globals.getBrokerResources().getKString(
                               BrokerResources.E_BID_CONFLICT,
                               mybrokerid),
                            BrokerNotification.REASON_FATAL);
                }
            } catch (IOException ex ) {
                    logger.log(Logger.DEBUG,"Unable to reach old remote broker " 
                           + "associated with " + mybrokerid);
            }

            // OK, if we made it here, we know that we dont have a previous
            // broker running on the same system with a different brokerid

            // XXX-HA replace with real core
            cb.setBrokerURL(brokerURL);
        }
        // OK now check the state
        BrokerState state = cb.getState();
        String tkovrbkr = cb.getTakeoverBroker();

        if ((state == BrokerState.FAILOVER_STARTED ||
             state == BrokerState.FAILOVER_PENDING)
                 && tkovrbkr != null && tkovrbkr.trim().length() != 0) {
            logger.log(Logger.WARNING,
                 BrokerResources.W_TAKEOVER_IN_PROGRESS,
                 mybrokerid, tkovrbkr);
            state = cb.getState();
            do  {
                logger.log(Logger.INFO,
                    BrokerResources.I_STARTUP_PAUSE);
                try {
                    Thread.currentThread().sleep(10*1000);
                } catch (InterruptedException ex) {
                    break;
                }
                state = cb.getState();
            } while (state == BrokerState.FAILOVER_STARTED ||
                     state == BrokerState.FAILOVER_PENDING);
        }

        // make sure we werent taking over other stores
        // 
        Iterator itr = clusterconfig.getConfigBrokers();
        while (itr.hasNext()) {
            HAClusteredBroker nextb = (HAClusteredBroker)itr.next();
            String bkr = nextb.getTakeoverBroker();
            if (bkr != null && bkr.equals(mybrokerid) &&
                nextb.getState() == BrokerState.FAILOVER_STARTED ) {
                logger.log(logger.INFO,
                    BrokerResources.I_TAKEOVER_RESET, nextb.getBrokerName());

                // XXX-HA
                nextb.setState(BrokerState.FAILOVER_COMPLETE);
            }
        }


        cb.setState(BrokerState.OPERATING);
 
        haMonitor = new HAMonitorTask();
        try {
            new HATimerThread("HAMonitor", haMonitor, monitorTimeout, monitorTimeout);
        } catch (Exception ex) {
            logger.log(Logger.WARNING,
                 BrokerResources.E_INTERNAL_BROKER_ERROR,
                 "Unable to start monitor Timeout", ex);
        }

    }


   /**
    * @return the monitor interval in seconds
    */
    public int getMonitorInterval() {
        return monitorTimeout/1000;
    }
    /**
     * Retrieves the current Session for this broker.
     * @return the session uid
     */
    public UID getStoreSession() {
        return ((HAClusteredBroker)clusterconfig.getLocalBroker())
                   .getStoreSessionUID();
    }

    /**
     * a string representation of the object
     */
    public String toString() {
        return "HAMonitorService[" + clusterconfig.getLocalBroker() +"]";
    }


    /**
     * Called when the timer calls the HAMonitorTask, this method is
     * used to update the local broker's heartbeat and check the
     * state of any indbout brokers.
     */
    public void monitor() {
        HAClusteredBroker cb = (HAClusteredBroker)
                 clusterconfig.getLocalBroker();

        logger.log(Logger.DEBUG,"HAMonitor is updating timestamp state of ["+ mybrokerid + "]");
        try {
            cb.updateHeartbeat();
        } catch (BrokerException ex) {
            logger.log(logger.ERROR,
                  BrokerResources.E_SPLIT_BRAIN,
                  ex);
            Broker.getBroker().exit(Globals.getBrokerStateHandler().getRestartCode(),
                 Globals.getBrokerResources().getKString(
                     BrokerResources.E_SPLIT_BRAIN),
                 BrokerNotification.REASON_RESTART);
        }

        try {
            BrokerState state = cb.getState();
            if (state == BrokerState.QUIESCE_STARTED ||
                state == BrokerState.QUIESCE_COMPLETED ||
                state == BrokerState.FAILOVER_PENDING ||
                state == BrokerState.FAILOVER_FAILED ||
                state == BrokerState.SHUTDOWN_STARTED ||
                state == BrokerState.SHUTDOWN_FAILOVER ||
                state == BrokerState.SHUTDOWN_COMPLETE) {
                // shutting down, nothing to do
                return;
            }
        } catch (BrokerException ex) {
            logger.logStack(Logger.WARNING,
              BrokerResources.E_INTERNAL_BROKER_ERROR,
              "unable to access state", ex);
        }

        if (indoubtBrokers.size() > 0) {
            logger.log(Logger.INFO,
               BrokerResources.I_INDOUBT_COUNT,
                 String.valueOf(indoubtBrokers.size()));
        }

        ArrayList takeover = null;

        // make a copy of the indoubt Brokers (so we dont have to worry about
        // changes)

        Set s = new HashSet(indoubtBrokers.keySet());
        Iterator itr = s.iterator();
        while (itr.hasNext()) {
            Object key = itr.next();
            indoubtData wbd = (indoubtData)indoubtBrokers.get(key);
            try {
                HAClusteredBroker idbcb = (HAClusteredBroker)
                          clusterconfig.getBroker(wbd.brokerid);
                long ts = idbcb.getHeartbeat();
                if (idbcb.getState() == BrokerState.SHUTDOWN_COMPLETE)
                {
                    // dont takeover
                    logger.log(logger.INFO,
                         BrokerResources.I_NO_TAKEOVER_SHUTDOWN,
                         wbd.brokerid);
                    idbcb.setBrokerIsUp(false, null);
                    indoubtBrokers.remove(key);
                    itr.remove();
                }else if (idbcb.getState() == BrokerState.FAILOVER_STARTED
                   || idbcb.getState() == BrokerState.FAILOVER_COMPLETE)
                {
                    // dont takeover
                    logger.log(logger.INFO,
                        BrokerResources.I_OTHER_TAKEOVER,
                        wbd.brokerid);
                    idbcb.setBrokerIsUp(false, null);
                    indoubtBrokers.remove(key);
                    itr.remove();
                } else if (ts > wbd.lastts && 
                           idbcb.getState() != BrokerState.FAILOVER_PENDING) {
                    logger.log(logger.INFO,
                         BrokerResources.I_BROKER_OK,
                            idbcb.getBrokerName());
                    // OK .. we are NOT indoubt or down
                    idbcb.setBrokerInDoubt(false, null);
                    indoubtBrokers.remove(key);
                    itr.remove();
                } else {
                   wbd.monitorCnt ++;
                   if (wbd.monitorCnt > MAX_MONITOR 
                         || idbcb.getState() == BrokerState.SHUTDOWN_FAILOVER) {
                       logger.log(logger.INFO,
                            BrokerResources.I_BROKER_NOT_OK,
                            idbcb.getBrokerName());
                        // we are dead -> takeover
                       if (takeover == null)
                           takeover = new ArrayList();
                       takeover.add(idbcb);
                       idbcb.setBrokerIsUp(false, null);
                       itr.remove();
                       indoubtBrokers.remove(key);
                   } else {
                       logger.log(logger.INFO,
                           BrokerResources.I_BROKER_INDOUBT,
                           idbcb.getBrokerName());
                   }
                }
            } catch (Exception ex) {
                logger.logStack(Logger.INFO,
                   BrokerResources.E_INTERNAL_BROKER_ERROR,
                       "Unable to monitor broker " 
                        + wbd.brokerid, ex);
            }

        }

        if (takeover != null && takeoverRunnable == null) {
            takeoverRunnable = new TakeoverThread(takeover);
            Thread thr = new Thread(takeoverRunnable, "Takeover");
            thr.start();  
        } else if (takeoverRunnable != null) {
            logger.log(Logger.DEBUG,"Pausing to takeover broker until "+
                     " previous broker takeover completes");
        }
    }

    /**
     * called when a broker goes down to process
     * any in-process transactions
     */
    public void brokerDown(BrokerAddress addr)
        throws BrokerException
    {
        logger.log(Logger.DEBUG,"Processing brokerDown for " + addr);
        List l = Globals.getTransactionList().getRemoteBrokers(addr);

        if (l == null) {
            logger.log(Logger.DEBUG,"XXX - No txns to process " + addr);
            return;
        }

        // ok, when we crash (brokers are down) everything needs
        // to be cleaned up
        //
        // those transactions are no longer viable

        // what we want to do here is:
        //    - see if any transactions need to finish rolling back
        //    - see if any transactions need to finish committing
        //

        // what we are handling is messages CONSUMED by the remote
        // broker which are handled here

        Iterator itr = l.iterator();
        while (itr.hasNext()) {
            TransactionUID tuid = (TransactionUID) itr.next();
            logger.log(Logger.DEBUG,"Processing tuid " + tuid + " from " + addr);

            // ok we need to retrieve the PERSISTENT state

            TransactionState ts = Globals.getStore().getTransactionState(tuid);
            
            // ok, look @ each transaction
            switch (ts.getState()) {
                case TransactionState.CREATED:
                case TransactionState.STARTED:
                case TransactionState.FAILED:
                case TransactionState.INCOMPLETE:
                case TransactionState.COMPLETE:
                case TransactionState.ROLLEDBACK:
                    // ok in all these cases, we rollback
                    // get CONSUMED messages
                    Globals.getTransactionList().rollbackRemoteTxn(tuid);
                    break;
                case TransactionState.PREPARED:
                    // ok in this case we do NOTHING
                    break;
                case TransactionState.COMMITTED:
                    // ok in this case we commit
                    Globals.getTransactionList().commitRemoteTxn(tuid);
                    break;
            }
           
        }



        // ok now we've processed all transactions ... get rid of it
    }

    Runnable takeoverRunnable = null;


    /**
     * Class used to handle takeover of indoubt brokers
     * after the monitor thread has determined that the
     * broker should be considered dead.
     * The takeover runs as a seperate thread to prevent
     * this processing from slowing down normal heartbeat
     * monitoring.
     */
    class TakeoverThread implements Runnable
    {
        /**
         * list of down brokers.
         */
        ArrayList downBkrs = null;

        /**
         * create an instance of TakeoverThread.
         * @param downBkrs the list of brokers to takeover
         */
        public TakeoverThread(ArrayList downBkrs)
        {
            this.downBkrs = downBkrs;
        }


        /**
         * Processes each of the brokers which need to be
         * taken over.
         */
        public void run() {
            Iterator itr = downBkrs.iterator();
            while (itr.hasNext()) {
                HAClusteredBroker cb = (HAClusteredBroker)itr.next();
                logger.log(Logger.INFO,
                        BrokerResources.I_START_TAKEOVER,
                         cb.getBrokerName() );
                try {
                    TakeoverStoreInfo info = cb.takeover(false);
                    logger.log(Logger.INFO,
                         BrokerResources.I_TAKEOVER_OK,
                         cb.getBrokerName());

                    // handle takeover logic
                    List dsts = info.getDestinationList();
                    List msgs = info.getMessageList();
                    List txn = info.getTransactionList();
                    logger.log(Logger.INFO,
                          BrokerResources.I_TAKEOVER_TXN,
                          cb.getBrokerName(),
                           String.valueOf(txn == null ? 0 : txn.size()));

                    // OK .. we dont have to worry about this brokers
                    // remote transactions .. they have already been handled
                    // yeah

                    // process rolling back transactions

                    List openTxns = new ArrayList();
                    Iterator titr = txn.iterator();
                    while (titr.hasNext()) {
                        // 
                        PacketRouter pr = Globals.getPacketRouter(0); // jms
                        TransactionHandler thandler = (TransactionHandler) 
                           pr.getHandler(PacketType.ROLLBACK_TRANSACTION);
                        boolean rollbackTxn = false;
                        TransactionUID tid = (TransactionUID) titr.next();
                        JMQXid xid = Globals.getTransactionList().
                                      UIDToXid(tid);
/*
                        TransactionState ts = Globals.getTransactionList()
                                      .retrieveState(tid);
                        if (ts == null) {
                            ts = Globals.getStore().getTransactionState(tid);
                            Globals.getTransactionList().addTransactionID(tid,
                                  ts, false);
                            Globals.getTransactionList().setTransactionIDStored(
                                  tid);

                        }
*/
                        TransactionState ts = Globals.getStore().getTransactionState(tid);
                        if (ts == null) {
                            titr.remove();
                            continue;
                        }
                        AutoRollbackType type = ts.getType();
                        int state = ts.getState();

                        // OK .. first handle loading messages because
                        // that means we have all the refs in order


                        // ok first handle database changes for ANY
                        // committed or rolledback message
                        if (ts.getState() == TransactionState.ROLLEDBACK)
                        {
                            logger.log(Logger.INFO,"XXX - DEBUG Rolling back "
                                   + " transaction " + tid);

                             // ok we may not be done with things yet
                             // add to opentxn list
                             openTxns.add(tid);
                        
                        } else if (ts.getState() == TransactionState.COMMITTED)
                        {
                             logger.log(Logger.INFO,"XXX - DEBUG Committing "
                                   + " transaction " + tid);
                             // ok we may not be done with things yet
                             // add to opentxn list
                             openTxns.add(tid);

                        } else if (ts.getType() == AutoRollbackType.ALL) {
                            // rollback
                            String args[] = {
                                 cb.getBrokerName(),
                                 String.valueOf(tid.longValue()),
                                 ts.toString(ts.getState()) };
                            logger.log(Logger.INFO,
                                BrokerResources.I_TAKEOVER_TXN_A_ROLLBACK,
                                args);
                            ts.setState(TransactionState.ROLLEDBACK);
                            try {
                                Globals.getStore().updateTransactionState(tid, ts,
                                    Destination.PERSIST_SYNC);
                            } catch (IOException e) {
                                throw new BrokerException(null, e);
                            }

                             // ok we may not be done with things yet
                             // add to opentxn list
                             openTxns.add(tid);
                        } else if (ts.getType() == AutoRollbackType.NOT_PREPARED 
                            &&  ts.getState() < TransactionState.PREPARED) {
                            String args[] = {
                                 cb.getBrokerName(),
                                 String.valueOf(tid.longValue()),
                                 ts.toString(ts.getState()) };
                            logger.log(Logger.INFO,
                                BrokerResources.I_TAKEOVER_TXN_P_ROLLBACK,
                                args);
                            ts.setState(TransactionState.ROLLEDBACK);
                            try {
                                Globals.getStore().updateTransactionState(tid, ts,
                                    Destination.PERSIST_SYNC);
                            } catch (IOException e) {
                                throw new BrokerException(null, e);
                            }
                             // ok we may not be done with things yet
                             // add to opentxn list
                             openTxns.add(tid);
                        }
                    } 

                    TakeoverReaper reaper = new TakeoverReaper(
                                     cb.getBrokerName(),
                                     openTxns, dsts);

                    Map m = Globals.getTransactionList().loadTakeoverTxns(txn);
                    logger.log(Logger.INFO,
                          BrokerResources.I_TAKEOVER_MSGS,
                          cb.getBrokerName(),
                         String.valueOf(msgs == null ? 0 : msgs.size()));

                    Destination.loadTakeoverMsgs(msgs, txn, m);

                    // OK rollback Open txns - we want to do this
                    boolean done = reaper.processTxns();
                    
                    logger.log(Logger.INFO,
                           BrokerResources.I_TAKEOVER_DSTS,
                           cb.getBrokerName(),
                           String.valueOf(dsts == null ? 0 : dsts.size()));

                    JMQTimer timer = Globals.getTimer();
                    try {
                        if (!done)
                            timer.schedule(reaper, reaperTimeout, reaperTimeout);

                    } catch (IllegalStateException ex) {
ex.printStackTrace();
                        logger.log(Logger.WARNING,
                            BrokerResources.E_INTERNAL_BROKER_ERROR,
                            "Unable to start reaper", ex);
                    }

                    // we have done processing data, set state to 
                    // complete
                    cb.setBrokerIsUp(false, null);
                    cb.setState(BrokerState.FAILOVER_COMPLETE);

                    logger.log(logger.INFO, 
                          BrokerResources.I_TAKEOVER_COMPLETE,
                          cb.getBrokerName());


                    itr.remove();
                } catch (Exception ex) {
                    if ( ex instanceof TakeoverLockException ) {
                        BrokerState state = null;
                        String takeoverBy = null;
                        HABrokerInfo bkrInfo = ((TakeoverLockException)ex).getBrokerInfo();
                        if (bkrInfo == null) {
                            // This shouldn't happens but just in case
                            try {
                                state = cb.getState();
                                takeoverBy = cb.getTakeoverBroker();
                            } catch ( BrokerException e ) {}
                        } else {
                            state = BrokerState.getState(bkrInfo.getState());
                            takeoverBy = bkrInfo.getTakeoverBrokerID();
                        }

                        if ( state == BrokerState.FAILOVER_STARTED ||
                             state == BrokerState.FAILOVER_PENDING ||
                             state == BrokerState.FAILOVER_COMPLETE ) {
                            logger.log(Logger.INFO,"XXX Unable to takeover broker "
                                 + cb.getBrokerName() + ", broker is being taken over or was taken over by "
                                 + takeoverBy);
                        } else {
                            logger.log(Logger.INFO,"XXX Unable to takeover broker "
                                 + cb.getBrokerName() + " due to takeover lock error (state=" +
                                 state + ", takeoverBroker=" + takeoverBy + ")");
                        }
                        try {
                            cb.setState(BrokerState.FAILOVER_FAILED);
                        } catch (Exception ex2) {
                            logger.logStack(Logger.ERROR,"XXX Internal Error:",
                                 ex2);
                        }
                                  
                    } else if (ex instanceof BrokerException && 
                               ((BrokerException)ex).getStatusCode() == Status.CONFLICT){
                        logger.log(Logger.INFO, ex.getMessage());
                    } else {
                        logger.logStack(Logger.ERROR,"XXX Internal Error: Unable to takeover broker "
                             + cb.getBrokerName(), ex);
                    }
                    cb.setBrokerIsUp(false, null);
                }
            }
            takeoverRunnable = null;
       
        }
    }


    /**
     * Places a broker on the "indoubt" list.
     *
     * @param brokerid broker who is indoubt
     */
    private void watchBroker(String brokerid)
        throws BrokerException
    {
        synchronized (indoubtBrokers)
        {
            if (indoubtBrokers.get(brokerid) != null) {
                   return;
            }
            indoubtData wbd = new indoubtData();
            wbd.brokerid = brokerid;
            wbd.lastts = ((HAClusteredBroker)clusterconfig.getBroker(
                          brokerid)).getHeartbeat();
            wbd.monitorCnt = 0;
            indoubtBrokers.put(brokerid, wbd);
        }
    }

    /**
     * Removed broker from the "indoubt" list.
     *
     * @param brokerid broker who is indoubt
     */
    private void stopWatchingBroker(String brokerid)
    {
        synchronized (indoubtBrokers)
        {
            indoubtBrokers.remove(brokerid);
        }
    }


    //-------------------------------------------------------------
    //  CLUSTER LISTENER
    //-------------------------------------------------------------

    /**
     * Notification that the cluster configuration has changed.
     * The monitoring service ignores any configuration changes.
     * @see ClusterListener
     * @param name property changed
     * @param value new setting for the property 
     */
    public void clusterPropertyChanged(String name, String value)
    {
        // do nothing, we dont care
    }

   /**
    * Called when a new broker has been added.
    * @param broker the new broker added.
    */
    public void brokerAdded(ClusteredBroker broker, UID brokerSession)
    {
        // do nothing, we dont care
    }

   /**
    * Called when a broker has been removed.
    * @param broker the broker removed.
    */
    public void brokerRemoved(ClusteredBroker broker, UID brokerSession)
    {
        // do nothing, we dont care
    }


   /**
    * Called when the broker who is the master broker changes
    * (because of a reload properties).
    * @param oldMaster the previous master broker.
    * @param newMaster the new master broker.
    */
    public void masterBrokerChanged(ClusteredBroker oldMaster,
                    ClusteredBroker newMaster)
    {
        // do nothing, we dont care
    }


   /**
    * Called when the status of a broker has changed. The
    * status may not be accurate if a previous listener updated
    * the status for this specific broker.
    * @param brokerid the name of the broker updated.
    * @param userData optional userData
    * @param oldStatus the previous status.
    * @param newStatus the new status.
    */
    public void brokerStatusChanged(String brokerid,
                  int oldStatus, int newStatus, UID brokerSession,
                  Object userData)
    {
        logger.log(Logger.DEBUG,"brokerStatusChanged " + brokerid + ":" + "\n\t"
             + BrokerStatus.toString(oldStatus) + "\n\t"
             + BrokerStatus.toString(newStatus) + "\n\t" + userData );
        // do nothing, we dont care
        if (BrokerStatus.getBrokerInDoubt(newStatus) && 
                 BrokerStatus.getBrokerIsUp(newStatus)) {
            ClusteredBroker cb = clusterconfig.getBroker(brokerid); 
            if (cb.isLocalBroker()) return;
            try {
                if (cb.getState() == BrokerState.SHUTDOWN_COMPLETE
                   || cb.getState() == BrokerState.FAILOVER_COMPLETE) {
                    logger.log(logger.INFO,
                         BrokerResources.I_BROKER_OK,
                            cb.getBrokerName());
                    cb.setBrokerIsUp(false, null);
                    return;
                }
                logger.log(Logger.INFO,
                         BrokerResources.I_BROKER_NOT_OK,
                         brokerid);
                watchBroker(brokerid);
            } catch (Exception ex) {
                    logger.logStack(Logger.INFO,
                        BrokerResources.E_INTERNAL_BROKER_ERROR,
                       "Unable to monitor broker " + brokerid, ex);
            }
        } else if (BrokerStatus.getBrokerInDoubt(oldStatus) 
              && BrokerStatus.getBrokerIsUp(newStatus)) {
            // XXX - stop watching
            stopWatchingBroker(brokerid);
        }
    }


   /**
    * Called when the state of a broker has changed. The
    * state may not be accurate if a previous listener updated
    * the state for this specific broker.
    * @param brokerid the name of the broker updated.
    * @param oldState the previous state.
    * @param newState the new state.
    */
    public void brokerStateChanged(String brokerid,
                  BrokerState oldState, BrokerState newState)
    {
        // do nothing, we dont care
        ClusteredBroker cb = clusterconfig.getBroker(brokerid);

        if (cb.isLocalBroker() && 
           ( newState == BrokerState.SHUTDOWN_COMPLETE ||
            newState == BrokerState.SHUTDOWN_FAILOVER)) {
                haMonitor.cancel();
                clusterconfig.removeEventListener(this);

        }
    }


   /**
    * Called when the version of a broker has changed. The
    * state may not be accurate if a previous listener updated
    * the version for this specific broker.
    * @param brokerid the name of the broker updated.
    * @param oldVersion the previous version.
    * @param newVersion the new version.
    */
    public void brokerVersionChanged(String brokerid,
                  int oldVersion, int newVersion)
    {
        // do nothing, we dont care
    }


   /**
    * Called when the url address of a broker has changed. The
    * address may not be accurate if a previous listener updated
    * the address for this specific broker.
    * @param brokerid the name of the broker updated.
    * @param newAddress the previous address.
    * @param oldAddress the new address.
    */
    public void brokerURLChanged(String brokerid,
                  MQAddress oldAddress, MQAddress newAddress)
    {
        // do nothing, we dont care
    }



}



