/*
 * @(#)NFLPriorityFifoSet.java	1.32 04/27/05
 *
 * Copyright 2002 Sun Microsystems, Inc. All Rights Reserved
 * SUN PROPRIETARY/CONFIDENTIAL
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jmq.util.lists;

import java.util.*;


/**
 * This is an Priority Fifo set which implements the
 * SortedSet interface.
 */

public class NFLPriorityFifoSet extends PriorityFifoSet 
    implements FilterableSet,  EventBroadcaster, Limitable
{

    public static boolean DEBUG = false;

    Set gni = new HashSet();

    // filter stuff
    Map filterSets = null;
    Map comparatorSets = null;

    // event stuff
    EventBroadcastHelper ebh = new EventBroadcastHelper();

    // limit stuff
    private boolean enforceLimits = true;
    private int highWaterCnt = 0;
    private long highWaterBytes = 0;
    private long largestMessageHighWater = 0;

    private float averageCount = 0.0F;
    private double averageBytes = 0.0D;
    private double messageAverage = 0.0D;
    private long numberSamples = 0;

    protected int maxCapacity = UNLIMITED_CAPACITY;
    protected long maxByteCapacity = UNLIMITED_BYTES;
    protected long bytes = 0;
    protected long maxBytePerObject = UNLIMITED_BYTES;
    protected boolean orderMaintained = true;

    public NFLPriorityFifoSet() {
        this(10, false);
    }

    public NFLPriorityFifoSet(int levels) {
        this(levels, false);
    }

    public NFLPriorityFifoSet(int levels, boolean maintainOrder) {
        super(levels);
        orderMaintained = maintainOrder;
        ebh.setOrderMaintained(maintainOrder);
        setLock(this);
        lookup = Collections.synchronizedMap(lookup);
    }

    public void enforceLimits(boolean enforce) {
        this.enforceLimits = enforce;
    }

    public boolean getEnforceLimits() {
        return enforceLimits;
    }

    protected boolean cleanupEntry(SetEntry e) {
        synchronized(lock) {
            return super.cleanupEntry(e);
        }
    }

    public boolean addAll(Collection c) {
        boolean ok = false;
        Iterator itr = c.iterator();
        while (itr.hasNext()) {
            ok |= add(itr.next());
        }
        return ok;
    }

    public void clear() {
        synchronized(lock) {
            Iterator itr = iterator();
            while (itr.hasNext()) {
                itr.next();
                itr.remove();
            }
            bytes = 0;
        }
    }
          
    
    public boolean add(Object o) {
        return add(defaultPriority, o, null);
    }

    public int size() {
        synchronized(lock) {
            return super.size();
        }
    }


    class ComparatorSet extends TreeSet implements SubSet
    {
        EventBroadcastHelper ebh = new EventBroadcastHelper();

        Object uid;

        public ComparatorSet(Object uid, Comparator c) {
            super(c);
            this.uid = uid;
        }
        public String toDebugString() {
            return "ComparatorSet [" +
                  comparator() + "]" 
                + NFLPriorityFifoSet.this.toDebugString();
        }

        public void destroy() {
            NFLPriorityFifoSet.this.destroyComparatorSet(this.uid);
        }

        void addItem(Object o) {
            super.add(o);
        }

        void removeItem(Object o) {
            super.remove(o);
        }

        public boolean add(Object o) {
            return add(o, null);
        }

        public boolean add(Object o, Reason r) {
            boolean ok = super.add(o);
            NFLPriorityFifoSet.this.add(o, r);
            return ok;
        }

        public boolean remove(Object o) {
            return remove(o, null);
        }
        public boolean remove(Object o, Reason r) {
            boolean ok = super.remove(o);
            NFLPriorityFifoSet.this.remove(o, r);
            return ok;
        }

         public Object removeNext() {
             boolean ok = false;
             Object o = null;
 
             o = first();
             if (o != null) {
                 NFLPriorityFifoSet.this.remove(o);
             }
             return o;
         }

         public Object getUID() {
             return uid;
         }

         public Object addEventListener(EventListener listener,
                        EventType type, Object userData)
             throws UnsupportedOperationException 
         {
             if (type != EventType.EMPTY) {
                 throw new UnsupportedOperationException(
                   "Event " + type + " not supported");
             }
             return ebh.addEventListener(listener, type, userData);
         }
         public Object addEventListener(EventListener listener,
                         EventType type, Reason r, Object userData)
             throws UnsupportedOperationException 
         {
             if (type != EventType.EMPTY) {
                 throw new UnsupportedOperationException(
                   "Event " + type + " not supported");
             }
             return ebh.addEventListener(listener, type, r, userData);
        }
 
        public Object removeEventListener(Object id)
        {
            return ebh.removeEventListener(id);
        }

        
        public void notifyEmptyChanged(boolean empty, Reason r)
        {
           if (ebh.hasListeners(EventType.EMPTY))
                ebh.notifyChange(EventType.EMPTY, r, this,
                    (empty ? Boolean.TRUE : Boolean.FALSE),
                    (empty ? Boolean.FALSE : Boolean.TRUE));
        }
    }



    class FilterSet extends AbstractSet implements SubSet, Prioritized
    {
        EventBroadcastHelper ebh = new EventBroadcastHelper();

        Object uid;
        Filter f = null;
        long lastMatchedIds[] = new long[NFLPriorityFifoSet.this.getLevels()];
        int currentPriority;

        // NOTE: either currentEntry is set OR
        //       nextEntry is set NEVER both
        nSetEntry nextEntry = null;
        nSetEntry currentEntry = null;

        public String toString() {
            return "FilterSet["+f+"]" + super.toString();
        }

        public void resetFilterSet(nSetEntry top)
        {
            nextEntry = top;
            currentEntry = null;
            lastMatchedIds = new long[NFLPriorityFifoSet.this.getLevels()];
        }

        public String toDebugString() {
            String str = "FilterSet[" + f + "]\n";
            str += "\tDumping FilterSet\n";
            Iterator itr = iterator();
            while(itr.hasNext()) {
                str += "\t\t"+itr.next() + "\n";
            }
            str += "\tcurrentPriority " + currentPriority + "\n";
            str += "\tnextEntry " + nextEntry + "\n";
            str += "\tcurrentEntry " + currentEntry + "\n";
            str += NFLPriorityFifoSet.this.toDebugString();
            return str;
        }

        public void addAllToFront(Collection c, int pri) {
            NFLPriorityFifoSet.this.addAllToFront(c, pri);
        }
        
        class filterIterator implements Iterator 
        {
            nSetEntry current = null;

            public filterIterator() {
                synchronized(lock) {
                current = currentEntry;
                    if (current == null) {
                        current = nextEntry;
                        findNext();
                    }
                }
            }

            void findNext() {
                synchronized(lock) {
                    while (current != null) {
                        if (current != null && current.isValid() &&
                            (f == null || f.matches(current.getData()))){
                            break;
                        }
                        current = (nSetEntry)current.getNext();
                    }
                }
            }

            public boolean hasNext() {
                synchronized(lock) {
                    return current != null;
                }
            }

            public Object next() {
                synchronized(lock) {
                    Object n = current.getData();
                    current = (nSetEntry)current.getNext();
                    findNext();
                    return n;
                }
            }

            public void remove() {
                throw new UnsupportedOperationException(
                    "remove is not supported on this iterator");
            }

        }
        public FilterSet(Object uid, Filter f) {
            ebh.setOrderMaintained(orderMaintained);
            synchronized (lock) {
                this.uid = uid;
                this.f = f;

                this.nextEntry = (nSetEntry)(NFLPriorityFifoSet.this.start == null
                    ? NFLPriorityFifoSet.this.head
                      : NFLPriorityFifoSet.this.start);
            }
           
        }

        public Object getUID() {
            return uid;
        }

        private boolean skipToNext() {
            assert Thread.holdsLock(lock);
            if (currentEntry != null) {
                if (!currentEntry.isValid()) {
                    nextEntry = (nSetEntry)currentEntry.getNext();
                    currentEntry = null;
                } else {
                    return true;
                }
            }
            if (nextEntry == null) {
                return false;
            }
            nSetEntry se = nextEntry;

            while (se != null && !se.isValid()) {
                se = (nSetEntry)se.getNext();
            }

            // OK .. first skip already viewed items
            // IF we are using a selector
            // 
            while (se != null && f != null && (se.getUID() <=
                   lastMatchedIds[se.getPriority()] ||
                   !f.matches(se.getData()))) {

                if (se.getUID() > lastMatchedIds[se.getPriority()]) {
                    lastMatchedIds[se.getPriority()] = se.getUID();
                }
                currentPriority = se.getPriority();
                se =(nSetEntry) se.getNext();
            }
            // OK .. at this point nextEntry is a valid item
            currentEntry = se;
            nextEntry = null;

            if (currentEntry != null) {
                currentPriority = currentEntry.getPriority();
            }
            return true;

        }

        void removeItem(Object o)
        {
            assert Thread.holdsLock(lock);
            assert o != null;

            if (nextEntry != null && nextEntry.getData() == o) {
               nextEntry = (nSetEntry)nextEntry.getNext();
            }
            if (currentEntry != null && currentEntry.getData() == o) {
                nextEntry = (nSetEntry)currentEntry.getNext();
                currentEntry = null;
            }
        }

        void addItem(Object o) {
            assert Thread.holdsLock(lock);
            

            nSetEntry pe = (nSetEntry)lookup.get(o);

            if (currentEntry == null && nextEntry == null) {
                nextEntry = pe;
                currentPriority = pe.priority;
            } else if (pe.getPriority() == currentPriority
                && ((currentEntry != null && !currentEntry.isValid()) ||
                   (currentEntry != null && !currentEntry.isValid()))) {
                nextEntry = pe;
                currentEntry = null;
            } else if (pe.getPriority() < currentPriority) {
                currentPriority = pe.getPriority();
                nextEntry = pe;
                currentEntry = null;
            }
        }
        void addItem(Object o, boolean toFront) {
            assert lock == null || Thread.holdsLock(lock);
            nSetEntry pe = (nSetEntry)lookup.get(o);
            if (toFront) {
                nextEntry = pe;
                currentEntry = null;
                currentPriority = 0;
            } else {
                addItem(o);
            }
           
        }


        public boolean add(Object o) 
        {
            return add(o, null);
        }

        public boolean add(int p, Object o) {
            return add(p, o, null);
        }

        public boolean add(Object o, Reason r) 
        {
            if (f != null && !f.matches(o)) {
                throw new IllegalArgumentException("not part of set");
            }
            return NFLPriorityFifoSet.this.add(o, r);
        }

        public boolean add(int p, Object o, Reason r) 
        {
            if (f != null && !f.matches(o)) {
                throw new IllegalArgumentException("not part of set");
            }
            return NFLPriorityFifoSet.this.add(p, o, r);
        }

        public void clear() {

            // OK .. we only want matching items
            // AND this will also clear parent list
            Set s = new HashSet();
            synchronized(lock) {
                Iterator itr = iterator();
                while (itr.hasNext()) {
                    s.add(itr.next());
                }
            }
            removeAll(s);
        }


        public boolean remove(Object o) {
            return remove(o, null);

        }
        public boolean remove(Object o, Reason r) {
            if (f != null && !f.matches(o)) {
                return false;
            }
            return NFLPriorityFifoSet.this.remove(o, r);
        }

        public boolean contains(Object o) {

            synchronized(lock) {
                nSetEntry pse = (nSetEntry)
                    lookup.get(o);
                if (pse == null) return false;

                if (f == null) return true;

                long lastUID = lastMatchedIds[pse.getPriority()];
    
                if (pse.getUID() < lastUID) {
                    return false;
                }
                if (pse.getUID() == lastUID) {
                    return f.matches(o);
                }
                return f.matches(o);
            }
        }

        public int size() {
            // this is SLOW (we have to check each item
            synchronized(lock) {
                int cnt = 0;
                Iterator itr = iterator();
                while (itr.hasNext()) {
                    itr.next();
                    cnt ++;
                }
                return cnt;
            }
        }

        public boolean retainAll(Collection c) {
            Set s = new HashSet();
            synchronized(lock) {
                Iterator itr = NFLPriorityFifoSet.this.iterator();
                while (itr.hasNext()) {
                    Object o = itr.next();
                    if (!c.contains(o)) {
                        s.add(o);
                    }
                 }
            }
            return NFLPriorityFifoSet.this.removeAll(s);
        }

        public boolean isEmpty() {
            synchronized(lock) {
                boolean state = !skipToNext();
                return state;
            }
        }

        public boolean removeAll(Collection c) {
            return NFLPriorityFifoSet.this.removeAll(c);
        }

        public void destroy() {
            NFLPriorityFifoSet.this.destroyFilterSet(this.uid);
        }

        public Iterator iterator() {
            return new filterIterator();
        }

        public Object removeNext() {

            Object o =  null;
            NotifyInfo ni = null;
            synchronized(lock) {

                if (!skipToNext()) {
                    // removeNext failed
                    return null;
                }
                if (currentEntry == null) {
                    return null;
                }
                lastMatchedIds[currentEntry.getPriority()] =
                    currentEntry.getUID();

                o = currentEntry.getData();

                nextEntry = (nSetEntry)currentEntry.getNext();
                currentEntry = null;
                ni = internalRemove(o,null, null, hasListeners());
            }
            // yes .. this is the wrong order .. bummer
            preRemoveNotify(o, null);
            if (ni != null) {
                postRemoveNotify(o,ni, null);
            }
            
            if (DEBUG && f == null && currentEntry == null && nextEntry == null 
                && lookup.size() != 0)

            {
                System.err.println("CORRUPTION");
                System.err.println(this.toDebugString());
            }

            return o;
        }
         public Object addEventListener(EventListener listener,
                        EventType type, Object userData)
             throws UnsupportedOperationException 
         {
             if (type != EventType.EMPTY) {
                 throw new UnsupportedOperationException(
                   "Event " + type + " not supported");
             }
             return ebh.addEventListener(listener, type, userData);
         }
         public Object addEventListener(EventListener listener,
                         EventType type, Reason r, Object userData)
             throws UnsupportedOperationException 
         {
             if (type != EventType.EMPTY) {
                 throw new UnsupportedOperationException(
                   "Event " + type + " not supported");
             }
             return ebh.addEventListener(listener, type, r, userData);
        }
 
        public Object removeEventListener(Object id)
        {
            return ebh.removeEventListener(id);
        }

        
        public void notifyEmptyChanged(boolean empty, Reason r)
        {
           if (ebh.hasListeners(EventType.EMPTY))
                ebh.notifyChange(EventType.EMPTY, r, this,
                    (empty ? Boolean.TRUE : Boolean.FALSE),
                    (empty ? Boolean.FALSE : Boolean.TRUE));
        }
    }

    // XXX - only generate empty notification for now

    public void addAllToFront(Collection c, int pri) {
        addAllToFront(c, pri, null);
    }
    public boolean isEmpty() {
        return super.isEmpty();
    }

    public void addAllToFront(Collection c, int pri, Reason reason)
    {
        if (c.isEmpty()) 
            return;

        Set notify = null;
        boolean wasEmpty = false;
        
        synchronized(lock) {

            wasEmpty = isEmpty();

            SetEntry startOfList = null;
            if (priorities[pri] == null) {
                // hey .. we just put it in the real place
                Iterator itr = c.iterator();
                while (itr.hasNext()) {
                    Object o = itr.next();
                    super.add(pri, o); 
                    if (startOfList == null) {
                        startOfList = (SetEntry)lookup.get(o);
                    }
                }
            } else {
                SetEntry endEntry = priorities[pri];
                Iterator itr = c.iterator();
                while (itr.hasNext()) {
                    Object o = itr.next();

                    // make sure we dont have a dup entry
                    // if it is, remove it so we replace it
                    if (lookup.get(o) != null) {
                        remove(o);
                    }

                    // add the message @ the right priority
                    SetEntry e = createSetEntry(o, pri);
                    Object obj = lookup.put(o,e);
                    endEntry.insertEntryBefore(e);
                    if (startOfList == null)
                        startOfList = e;
                }
                priorities[pri] = startOfList;
                if (endEntry == head)
                    head = startOfList;
                
            }
            if (wasEmpty != isEmpty())  {
                notify = new HashSet();
            }

            // update any iterators             
                      
           if (filterSets != null) {
               Iterator fitr = filterSets.values().iterator();
               while (fitr.hasNext()) {
                        FilterSet s = (FilterSet)fitr.next();
                        s.addItem(startOfList.getData(), true);
                        if (notify != null)
                            notify.add(s);
                }
                if (comparatorSets != null) {
                    //LKS - XXX 
                    // not dealing w/ comparator sets yet
                    
                }                   
 
            }
        }

        if (notify != null) {

            if (hasListeners(EventType.EMPTY))
                notifyChange(EventType.EMPTY, Boolean.TRUE,
                    Boolean.FALSE, reason);

            Iterator nitr = notify.iterator();
            while (nitr.hasNext()) {
                FilterSet fs = (FilterSet)nitr.next();
                fs.notifyEmptyChanged(!wasEmpty, reason);
            }
        }

    }

    public boolean add(Object o, Reason r) {
        return add(defaultPriority, o, r);
    }

    public boolean add(int pri, Object o) {
        return add(pri, o, null);
    }

    private void preAdd(Object o, Reason reason) {
        // OK notify of changeRequest
        if (hasListeners(EventType.SET_CHANGED_REQUEST))
            notifyChange(EventType.SET_CHANGED_REQUEST, null,
                    o, reason);

        if (o == null ) {
            throw new NullPointerException("Unable to support null "
                      + " values");
        }
    }

    NotifyInfo internalAdd(int pri, Object o) {
        assert Thread.holdsLock(this);
        NotifyInfo ni = null;
        int oldsize =0;
        long oldbytes = 0;
        long objsize = 0;
        boolean added = false;
            if (maxByteCapacity != UNLIMITED_BYTES &&
                  !(o instanceof Sized)) 
            {
                throw new ClassCastException(
                   "Unable to add object not of"
                   + " type Sized when byteCapacity has been set");
            }
            if (maxBytePerObject != UNLIMITED_BYTES &&
                  !(o instanceof Sized)) 
            {
                throw new ClassCastException(
                   "Unable to add object not of"
                   + " type Sized when maxByteSize has been set");
            }

            if (enforceLimits && maxCapacity != UNLIMITED_CAPACITY &&
                ((maxCapacity -size()) <= 0)) {
                throw new OutOfLimitsException(
                      OutOfLimitsException.CAPACITY_EXCEEDED,
                      new Integer(size()),
                      new Integer(maxCapacity));
            }
    
            if (enforceLimits && maxByteCapacity != UNLIMITED_BYTES &&
                ((maxByteCapacity -bytes) <= 0)) {
                throw new OutOfLimitsException(
                      OutOfLimitsException.BYTE_CAPACITY_EXCEEDED,
                      new Long(bytes),
                      new Long(maxByteCapacity));
            }
    
            if (o instanceof Sized) {
                objsize = ((Sized)o).byteSize();
            }
    
            if (maxBytePerObject != UNLIMITED_BYTES && 
                objsize > maxBytePerObject) {
                throw new OutOfLimitsException(
                      OutOfLimitsException.ITEM_SIZE_EXCEEDED,
                      new Long(objsize),
                      new Long(maxByteCapacity));
            }
    
            oldsize = size();
            oldbytes = bytes;

            // OK -- add the actual data

            added = super.add(pri, o);
            bytes +=objsize;

            averageCount = (((float)numberSamples*averageCount 
                      + (float)size())/((float)numberSamples+1.0F)); 
            averageBytes = ((double)numberSamples*averageBytes 
                      + (double)bytes)/((double)numberSamples+1.0D); 
            messageAverage = ((double)numberSamples*messageAverage 
                      + (double)objsize)/((double)numberSamples+1.0D); 
            numberSamples ++;

            if (added) {
                if (size() > highWaterCnt) {
                    highWaterCnt = size();
                }
                if (objsize > largestMessageHighWater) {
                    largestMessageHighWater = objsize;
                }
                if (bytes > highWaterBytes) {
                    highWaterBytes = bytes;
                }

                if (hasListeners() || (filterSets != null &&
                    !filterSets.isEmpty()) ||
                    (comparatorSets != null && !comparatorSets.isEmpty()) ) {
                    ni = getNI();
                    ni.oldsize = oldsize;
                    ni.oldbytes = oldbytes;
                    ni.objsize = objsize;
                    ni.newbytes = oldbytes + objsize;
                    ni.newsize = size();
                    ni.curMaxCapacity = maxCapacity;
                    ni.curMaxBytesCapacity = maxByteCapacity;

                    int cnt = 0;

                    synchronized(lock) {
    
                        if (filterSets != null) {
                            Iterator itr = filterSets.values().iterator();
                            while (itr.hasNext()) {
                                FilterSet s = (FilterSet)itr.next();
                                boolean wasEmpty = s.isEmpty();
                                s.addItem(o);
                                if (wasEmpty != s.isEmpty() ) {
                                    if (ni.filters[cnt] == null) {
                                        ni.filters[cnt] = new EmptyChanged();
                                    }
                                    ni.filters[cnt].f = s;
                                    ni.filters[cnt].isEmpty = !wasEmpty;
                                    cnt ++;
                                }
                            }
                        }
                        if (comparatorSets != null) {
                            Iterator itr = comparatorSets.values().iterator();
                            while (itr.hasNext()) {
                                ComparatorSet s = (ComparatorSet)itr.next();
                                boolean wasEmpty = s.isEmpty();
                                s.addItem(o);
                                if (wasEmpty != s.isEmpty() ) {
                                    if (ni.filters[cnt] == null) {
                                        ni.filters[cnt] = new EmptyChanged();
                                    }
                                    ni.filters[cnt].f = s;
                                    ni.filters[cnt].isEmpty = !wasEmpty;
                                    cnt ++;
                                }
                            }
                        }
                    }
                    if (cnt < ni.filters.length && ni.filters[cnt] != null) {
                        ni.filters[cnt].f = null;
                    }
    
                }
            }
        return ni;
    }

    private void postAdd(Object o, NotifyInfo ni, Reason reason) {
        // send out any notifications !!!
        if (hasListeners(EventType.SIZE_CHANGED) &&
                 ni.oldsize != ni.newsize) {
            notifyChange(EventType.SIZE_CHANGED, 
                    new Integer(ni.oldsize),
                    new Integer(ni.newsize),
                    reason);
        }
        if (hasListeners(EventType.BYTES_CHANGED) &&
                ni.oldbytes != ni.newbytes) {
            notifyChange(EventType.BYTES_CHANGED, 
                new Long(ni.oldbytes),
                new Long(ni.newbytes),
                reason);
        }
        if (hasListeners(EventType.SET_CHANGED)) {
            notifyChange(EventType.SET_CHANGED, null,
                   o, reason);
        }
 

        if  (ni.oldsize == 0 && ni.newsize != 0 && 
                 hasListeners(EventType.EMPTY)) {
            notifyChange(EventType.EMPTY, Boolean.TRUE,
                    Boolean.FALSE, reason);
        }

        int curMaxCapacity = 0;
        long curMaxBytesCapacity = 0;
        if ( hasListeners(EventType.FULL) &&
             (ni.curMaxBytesCapacity != UNLIMITED_BYTES &&
              ((ni.curMaxBytesCapacity -ni.newbytes) <= 0)) 
             || (ni.curMaxCapacity != UNLIMITED_BYTES &&
            ((ni.curMaxCapacity -ni.newsize) <= 0)))
        {
            notifyChange(EventType.FULL, Boolean.FALSE,
                    Boolean.TRUE, reason);
        }
        for (int i=0; i < ni.filters.length; i ++) {
            if (ni.filters[i] == null || ni.filters[i].f  == null) break;
            SubSet s = ni.filters[i].f;
            if (s instanceof FilterSet) {
                ((FilterSet)s).notifyEmptyChanged(ni.filters[i].isEmpty, reason);
            } else {
                assert s instanceof ComparatorSet;
                ((ComparatorSet)s).notifyEmptyChanged(ni.filters[i].isEmpty,reason);
            }
        }
        putNI(ni);
    }


    public boolean add(int pri, Object o, Reason reason) 
    {
        NotifyInfo ni = null;
        preAdd(o, reason);
        synchronized(lock) {
            ni = internalAdd(pri, o);
        }
        if (ni != null) {
            postAdd(o, ni, reason);
        }
        return ni != null;
    }

    public boolean removeAll(Collection c, Reason r) {
        boolean removed = false;
        Iterator itr = c.iterator();
        while (itr.hasNext()) {
            removed |= remove(itr.next(), r);
        }
        return removed;
    }

    public boolean remove(Object o) {
        return remove(o, (Reason)null);
    }

    class NotifyInfo
    {
        long oldbytes = 0;
        long newbytes = 0;
        int oldsize =0;
        int newsize = 0;
        long objsize = 0;
        int curMaxCapacity = 0;
        long curMaxBytesCapacity = 0;
        EmptyChanged filters[] = null;

        public NotifyInfo() {
            filters = new EmptyChanged[0];
        }
    }        

    class EmptyChanged
    {
        SubSet f;
        boolean isEmpty = false;
    }

    NotifyInfo getNI() {
        NotifyInfo ni = null;
        synchronized(this) {
            
            if (gni.isEmpty()) {
                ni= new NotifyInfo();
            } else {
                Iterator itr = gni.iterator();
                ni = (NotifyInfo)itr.next();
                itr.remove();
            }
            int size = (filterSets == null ? 0 : filterSets.size());
            if (ni.filters == null || ni.filters.length < size ) {
                ni.filters = new EmptyChanged[size];
            }
        }
        return ni;
    }

    void putNI(NotifyInfo ni) {
        if (ni == null) return;
        synchronized (this) {
            gni.add(ni);
        }
    }

    NotifyInfo internalRemove(Object o, Reason r, Iterator pitr, boolean hasListeners) {
        synchronized(lock) {
            assert Thread.holdsLock(lock);
            long objsize = 0;
            int oldsize = size();
            long oldbytes = bytes;
            if (o instanceof Sized) {
                objsize = ((Sized)o).byteSize();
            }
            nSetEntry nse = (nSetEntry)lookup.get(o);
            if (nse == null) { // already removed
                return null;
            }
            boolean removed = true;
            if (pitr != null) {
                pitr.remove();
            } else {
                assert Thread.holdsLock(lock);
                removed = super.remove(o);
            }
            if (!removed)  {
                return null;
            }
            if (!hasListeners() && (filterSets == null ||
                 filterSets.isEmpty()) &&
                 (comparatorSets == null || comparatorSets.isEmpty()))  {
                return null;
            }
            NotifyInfo ni = getNI();
            ni.oldbytes = oldbytes;
            ni.oldsize = oldsize;
            ni.objsize = objsize;
            ni.newsize =size();
            ni.newbytes = bytes;
            ni.curMaxCapacity = maxCapacity;
            ni.curMaxBytesCapacity = maxByteCapacity;

            int cnt =0;
            if (filterSets != null) {
                Iterator itr = filterSets.values().iterator();
                while (itr.hasNext()) {
                    FilterSet s = (FilterSet)itr.next();
                    boolean empty = s.isEmpty();
                    s.removeItem(o);
                    if (empty != s.isEmpty()) {
                        if (ni.filters[cnt] == null) {
                            ni.filters[cnt] = new EmptyChanged();
                        }
                        ni.filters[cnt].f = s;
                        ni.filters[cnt].isEmpty = empty;
                        cnt ++;
                    }
                }
            }
            if (comparatorSets != null) {
                Iterator itr = comparatorSets.values().iterator();
                while (itr.hasNext()) {
                    ComparatorSet s = (ComparatorSet)itr.next();
                    boolean empty = s.isEmpty();
                    s.removeItem(o);
                    if (empty != s.isEmpty()) {
                        if (ni.filters[cnt] == null) {
                            ni.filters[cnt] = new EmptyChanged();
                        }
                        ni.filters[cnt].f = s;
                        ni.filters[cnt].isEmpty = !empty;
                        cnt ++;
                    }
                }
            }
            if (cnt < ni.filters.length && ni.filters[cnt] != null) {
                ni.filters[cnt].f = null;
            }
            if (pitr == null && nse != null)
                nse.clear();
            return ni;
        }
    }
     

    public boolean remove(Object o, Reason r) {
        if (o == null) {
            return false;
        }
        synchronized (lock) {
            if (!contains(o))
                return false;
        }
        preRemoveNotify(o, r);
        NotifyInfo ni = internalRemove(o,r, null, hasListeners());
        if (ni != null) {
            postRemoveNotify(o,ni, r);
        }
        synchronized (lock) {
            return !contains(o);
        }
    }

    public Object removeNext() {
        return removeNext(null);
    }
    public Object removeNext(Reason r) {
        Object o = null;
        NotifyInfo ni = null;
        synchronized(lock) {
            o = first();
            if (o == null) {
                return null;
            }
            ni = internalRemove(o,r, null, hasListeners());
        }
        preRemoveNotify(o, r);
        if (ni != null) {
            postRemoveNotify(o,ni, r);
        }
        return o;
    }

    class wrapIterator implements Iterator
    {
        Iterator parentIterator;
        Object next = null;

        public wrapIterator(Iterator itr) {
            synchronized(lock) {
                parentIterator = itr;
            }
        }
        public boolean hasNext() {
            synchronized(lock) {
                return parentIterator.hasNext();
            }
        }
        public Object next() {
            synchronized(lock) {
                next = parentIterator.next();
                return next;
            }
        }
        public void remove() {
            preRemoveNotify(next, null);
            NotifyInfo ni = internalRemove(next,null, parentIterator, hasListeners());
            if (ni != null) {
                postRemoveNotify(next,ni, null);
            }
            next = null;
        }
    }

    public Iterator iterator() {
        synchronized (lock) {
            return new wrapIterator(super.iterator());
        }
    }



    public SubSet subSet(Filter f) {
        synchronized(lock) {
            Object uid = new Object();
            FilterSet fs = new FilterSet(uid, f);
            if (filterSets == null) {
                filterSets = new WeakValueHashMap("FilterSet");
            }
            filterSets.put(uid, fs);
            return fs;
        }
    }

    public SubSet subSet(Comparator c) {
        synchronized(lock) {
            Object uid = new Object();
            ComparatorSet cs = new ComparatorSet(uid, c);
            if (comparatorSets == null) {
                comparatorSets = new WeakValueHashMap("ComparatorSet");
            }
            comparatorSets.put(uid, cs);
            return cs;
        }
    }

    public Set getAll(Filter f) {
        synchronized(lock) {
            Set s = new LinkedHashSet();
            Iterator itr = iterator();
            while (itr.hasNext()) {
                Object o = itr.next();
                if (f == null || f.matches(o)) {
                    s.add(o);
                }
            }
            return s;
        }
    }


    private void destroyFilterSet(Object uid)
    {
        assert filterSets != null;
        synchronized(lock) {
            if (filterSets != null) {
                filterSets.remove(uid);
            }
        }

    }

    private void destroyComparatorSet(Object uid)
    {
        assert comparatorSets != null;
        synchronized(lock) {
            if (comparatorSets != null) {
                comparatorSets.remove(uid);
            }
        }
    }


    public void destroy() {
        // clean up for gc
        ebh.clear();
        super.clear();
        synchronized(lock) {
            if (filterSets != null)
                filterSets.clear();
            if (comparatorSets != null)
                comparatorSets.clear();
        }
        gni.clear();
    }

    protected SetEntry createSetEntry(Object o, int p) {
        return new nSetEntry(o,p);
    }

    long currentID = 1;

    class nSetEntry extends PrioritySetEntry
    {
        long uid = 0;

        public nSetEntry(Object o, int p) {
            super(o,p);
            synchronized(lock) {
                uid = currentID ++;
            }
        }
        public long getUID() {
            return uid;
        }

        public boolean remove() {
            assert Thread.holdsLock(lock);
            // we are always locked (in lock mode)
            // when this is called
            Object o = getData();
            assert isValid();
            valid = false;
            boolean ok = super.remove();

            // update bytes 
            if (o instanceof Sized) {
                long removedBytes = ((Sized)o).byteSize();
                bytes -= removedBytes;
            }
            // update averages
            averageCount = (((float)numberSamples*averageCount 
                      + (float)size())/((float)numberSamples+1.0F)); 
            averageBytes = ((double)numberSamples*averageBytes 
                      + (double)bytes)/((double)numberSamples+1.0D); 
            numberSamples ++;

            return ok;
        }
    }
        


    public synchronized String toDebugString() {
        String str = "NFLPriorityFifoSet: "
            + "\n";
        if (filterSets != null) {
            str += "\tfilterSets: " + filterSets.size() + "\n";
            Iterator fitr = filterSets.values().iterator();
            while (fitr.hasNext()) {
                FilterSet fs = (FilterSet)fitr.next();
                str += "\t\tFilterSet " + fs.hashCode() + " filter["
                         + fs.f + "]\n";
            }
        }
        if (comparatorSets != null) {
            str += "\tComparatorSets: " + comparatorSets.size() + "\n";
            Iterator fitr = comparatorSets.values().iterator();
            while (fitr.hasNext()) {
                ComparatorSet fs = (ComparatorSet)fitr.next();
                str += "\t\tComparatorSet " + fs.hashCode() + " filter["
                         + fs.comparator() + "]\n";
            }
        }
        str += "\n\nSUBCLASS INFO\n";
        str += super.toDebugString();
        return str;
    }

    protected void preRemoveNotify(Object o, Reason reason) {
        if (hasListeners(EventType.SET_CHANGED_REQUEST))
            notifyChange(EventType.SET_CHANGED_REQUEST, o,
                       null, reason);
    }

    protected void postRemoveNotify(Object o, NotifyInfo ni, Reason reason) 
    {
        if (!hasListeners()) {
            return;
        }

        // first  notify SIZE changed
        if (ni.oldsize != ni.newsize &&
               hasListeners(EventType.SIZE_CHANGED)) {
            notifyChange(EventType.SIZE_CHANGED, 
                new Integer(ni.oldsize),
                new Integer(ni.newsize),
                reason);
        }
        if (ni.newbytes != ni.oldbytes &&
               hasListeners(EventType.BYTES_CHANGED)) {
            notifyChange(EventType.BYTES_CHANGED, 
                new Long(ni.oldbytes),
                new Long(ni.newbytes),
                reason);
        }
        if (hasListeners(EventType.SET_CHANGED))
            notifyChange(EventType.SET_CHANGED, o,
                   null, reason);

        if (ni.oldsize != 0 && ni.newsize == 0 &&
              hasListeners(EventType.EMPTY)) {
            notifyChange(EventType.EMPTY, Boolean.FALSE,
                    Boolean.TRUE, reason);
        }
        if (hasListeners(EventType.FULL) &&
            (ni.curMaxBytesCapacity != UNLIMITED_BYTES 
              && ((ni.curMaxBytesCapacity - ni.oldbytes) <= 0)
              && ((ni.curMaxBytesCapacity - ni.newbytes) > 0)) 
           || (ni.curMaxCapacity != UNLIMITED_CAPACITY 
              && ((ni.curMaxCapacity -ni.oldsize) <= 0)
              && ((ni.curMaxCapacity -ni.newsize) > 0)))
        {
              // not full
            notifyChange(EventType.FULL, Boolean.TRUE,
                    Boolean.FALSE, reason);
        }
        for (int i=0; i < ni.filters.length; i ++) {
            if (ni.filters[i] == null || ni.filters[i].f  == null) break;
            SubSet s = ni.filters[i].f;
            if (s instanceof FilterSet) {
                ((FilterSet)s).notifyEmptyChanged(ni.filters[i].isEmpty,reason);
            } else {
                assert s instanceof ComparatorSet;
                ((ComparatorSet)s).notifyEmptyChanged(
                       ni.filters[i].isEmpty,reason);
            }
        }
        putNI(ni);
    }

    /**
     * Maximum number of messages stored in this
     * list at any time since its creation.
     *
     * @return the highest number of messages this set
     * has held since it was created.
     */
    public int highWaterCount() {
        return highWaterCnt;
    }

    /**
     * Maximum number of bytes stored in this
     * list at any time since its creation.
     *
     * @return the largest size (in bytes) of
     *  the objects in this list since it was
     *  created.
     */
    public long highWaterBytes() {
        synchronized(this) {
            return highWaterBytes;
        }
    }

    /**
     * The largest message (which implements Sized)
     * which has ever been stored in this list.
     *
     * @return the number of bytes of the largest
     *  message ever stored on this list.
     */
    public long highWaterLargestMessageBytes() {
        synchronized(this) {
            return largestMessageHighWater;
        }
    }

    /**
     * Average number of messages stored in this
     * list at any time since its creation.
     *
     * @return the average number of messages this set
     * has held since it was created.
     */
    public float averageCount() {
        return averageCount;
    }

    /**
     * Average number of bytes stored in this
     * list at any time since its creation.
     *
     * @return the largest size (in bytes) of
     *  the objects in this list since it was
     *  created.
     */
    public double averageBytes() {
        synchronized(this) {
            return averageBytes;
        }
    }

    /**
     * The average message size (which implements Sized)
     * of messages which has been stored in this list.
     *
     * @return the number of bytes of the average
     *  message stored on this list.
     */
    public double averageMessageBytes() {
        synchronized(this) {
            return messageAverage;
        }
    }



    /** 
     * sets the maximum size of an entry allowed
     * to be added to the collection
     * @param bytes maximum number of bytes for
     *        an object added to the list or
     *        UNLIMITED_BYTES if there is no limit
     */   
    public void setMaxByteSize(long bytes) {
        if (bytes < UNLIMITED_BYTES) {
            bytes = UNLIMITED_BYTES;
        }
        synchronized(this) {
            maxBytePerObject = bytes;
        }
    }
 
    /** 
     * returns the maximum size of an entry allowed
     * to be added to the collection
     * @return maximum number of bytes for an object
     *        added to the list  or
     *        UNLIMITED_BYTES if there is no limit
     */   
    public long maxByteSize() {
        synchronized(this) {
            return maxBytePerObject;
        }
    }
 

    /**
     * Sets the capacity (size limit).
     *
     * @param cnt the capacity for this set (or
     *         UNLIMITED_CAPACITY if unlimited).
     */
    public void setCapacity(int cnt) {
        if (cnt < UNLIMITED_CAPACITY) {
            cnt = UNLIMITED_CAPACITY;
        }
        boolean nowFull = false;
        boolean nowNotFull = false;
        synchronized(this) {
            nowFull = (!isFull() 
                   && cnt != UNLIMITED_CAPACITY 
                   && cnt <= size());
            nowNotFull = isFull() &&
                 (cnt == UNLIMITED_CAPACITY ||
                  cnt > size());

            maxCapacity = cnt;
        }
        if (nowFull) {
            notifyChange(EventType.FULL, Boolean.FALSE,
                Boolean.TRUE, null);
        } else if (nowNotFull) {
            notifyChange(EventType.FULL, Boolean.TRUE,
                Boolean.FALSE, null);
        }
    }

    /**
     * Sets the byte capacity. Once the byte capacity
     * is set, only objects which implement Sized
     * can be added to the class
     *
     * @param size the byte capacity for this set (or
     *         UNLIMITED_BYTES if unlimited).
     */
    public void setByteCapacity(long size) {
        boolean nowFull = false;
        boolean nowNotFull = false;
        if (size < UNLIMITED_BYTES) {
            size = UNLIMITED_BYTES;
        } 

        synchronized(this) {
            nowFull = (!isFull() 
               && size != UNLIMITED_BYTES 
               && size <= byteSize());
            nowNotFull = isFull() &&
                 (size == UNLIMITED_CAPACITY ||
                  size > byteSize());
            maxByteCapacity = size;
        }
 
        if (nowFull) {
            notifyChange(EventType.FULL, Boolean.FALSE,
                    Boolean.TRUE, null);
        } else if (nowNotFull) {
            notifyChange(EventType.FULL, Boolean.TRUE,
                    Boolean.FALSE, null);
        }

    }

    /**
     * Returns the capacity (count limit) or UNLIMITED_CAPACITY
     * if its not set.
     *
     * @return the capacity of the list
     */
    public int capacity() {
        return maxCapacity;
    }

    /**
     * Returns the byte capacity or UNLIMITED_CAPACITY
     * if its not set.
     *
     * @return the capacity of the list
     */
    public long byteCapacity() {
        synchronized(this) {
            return maxByteCapacity;
        }
    }



    /**
     * Returns <tt>true</tt> if either the bytes limit
     *         or the count limit is set and
     *         has been reached or exceeded.
     *
     * @return <tt>true</tt> if the count limit is set and
     *         has been reached or exceeded.
     * @see #freeSpace
     * @see #freeBytes
     */
    public boolean isFull() {
        synchronized(this) {
            return freeSpace()==0 || freeBytes() == 0;
        }
    }


    /**
     * Returns number of entries remaining in the
     *         lists to reach full capacity or
     *         UNLIMITED_CAPACITY if the capacity
     *         has not been set
     *
     * @return the amount of free space
     */
    public int freeSpace() {
        synchronized(this) {
            if (maxCapacity == UNLIMITED_CAPACITY)
                return UNLIMITED_CAPACITY;
                
            int sz = maxCapacity - size();
            if (sz < 0) {
                return 0;
            }
            return sz;
        }
    }

    /**
     * Returns the number of bytesremaining in the
     *         lists to reach full capacity, 0
     *         if the list is greater than the 
     *         capacity  or UNLIMITED_BYTES if 
     *         the capacity has not been set
     *
     * @return the amount of free space
     */
    public long freeBytes() {
        synchronized(this) {
            if (maxByteCapacity == UNLIMITED_BYTES)
                return UNLIMITED_BYTES;
               
            long retval = maxByteCapacity - bytes;
            if (retval < 0) {
                return 0;
            }
            return retval;
        }
    }

    /**
     * Returns the number of bytes used by all entries in this 
     * set which implement Sized.  If this
     * set contains more than <tt>Long.MAX_VALUE</tt> elements, returns
     * <tt>Long.MAX_VALUE</tt>.
     *
     * @return the total bytes of data from all objects implementing
     *         Sized in this set.
     * @see Sized
     * @see #size
     */
    public long byteSize() {
        synchronized(this) {
            return bytes;
        }
    }

   // ----------------------------------------------------
   //   Notification Events
   // ----------------------------------------------------

    /**
     * Request notification when the specific event occurs.
     * @param listener object to notify when the event occurs
     * @param type event which must occur for notification
     * @param userData optional data sent with any notifications
     * @return an id associated with this notification
     */
    public Object addEventListener(EventListener listener, 
                        EventType type, Object user_data) 
    {
        return ebh.addEventListener(listener, type, 
                       user_data);
    }

    /**
     * Request notification when the specific event occurs AND
     * the reason matched the passed in reason.
     * @param listener object to notify when the event occurs
     * @param type event which must occur for notification
     * @param userData optional data sent with any notifications
     * @param reason reason which must be associated with the
     *               event (or null for all events)
     * @return an id associated with this notification
     */
    public Object addEventListener(EventListener listener, 
                        EventType type, Reason reason,
                        Object userData)
    {
        return ebh.addEventListener(listener, type, 
                       reason, userData);
    }

    /**
     * remove the listener registered with the passed in
     * id.
     * @return the listener which was removed
     */
    public Object removeEventListener(Object id)
    {
        return ebh.removeEventListener(id);
    }


    protected boolean hasListeners(EventType e) {
        return ebh.hasListeners(e);
    }

    protected boolean hasListeners() {
        return ebh.hasListeners();
    }

    protected void notifyChange(EventType e, Object oldval, Object newval, Reason r) {
        if (!hasListeners()) {
            return;
        }
        ebh.notifyChange(e, r, this, oldval, newval);
    }


    public void sort(Comparator c) {
        super.sort(c);
        // reset subsets
        if (filterSets != null) {
            Iterator fitr = filterSets.values().iterator();
            while (fitr.hasNext()) {
                FilterSet s = (FilterSet)fitr.next();
                s.resetFilterSet((nSetEntry)head);
            }
        }
    }

}

