/*
 * @(#)EndpointConsumer.java	1.29 06/02/09
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jms.ra;


import javax.jms.*;
import javax.resource.NotSupportedException;
import javax.resource.ResourceException;
import javax.resource.spi.endpoint.MessageEndpointFactory;

import com.sun.messaging.ConnectionConfiguration;
import com.sun.messaging.jmq.ClientConstants;
import com.sun.messaging.jmq.DestinationName;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.logging.Logger;

/**
 * Encapsulates a message consumer for the Sun Java System Message Queue J2EE Resource Adapter.
 * 
 * @author George Tharakan
 */

public class EndpointConsumer implements
    javax.jms.ExceptionListener,
    com.sun.messaging.jms.notification.EventListener
{

    /** Resource Adapter holding this epConsumer */
    protected com.sun.messaging.jms.ra.ResourceAdapter ra = null;

    /** XAConnectionFactory for this epConsumer */
    private com.sun.messaging.XAConnectionFactory xacf = null;

    /** The Consumer ID associated with this epConsumer */
    private int cID = 0;

    /** The MessageEndpointFactory ID associated with this epConsumer */
    private int fID = 0;

    /** The destination type associated with this epConsumer */
    private int destinationType = ClientConstants.DESTINATION_TYPE_UNKNOWN;

    /** The subscription durability associated with this epConsumer */
    private boolean isDurable = false;

    /** Flags whether this epConsumer is transcated */
    private boolean transactedDelivery = false;

    /** Flags Deactivation */
    protected boolean deactivated = false;

    /** Connection and Session held by this epConsumer */
    protected com.sun.messaging.jmq.jmsclient.XAConnectionImpl xac = null;
    protected javax.jms.XASession xas = null;

    /** MessageListener for this epConsumer */
    protected MessageListener msgListener = null;

    /** MessageConsumer for this epConsumer */
    protected MessageConsumer msgConsumer = null;
    protected MessageConsumer msgConsumer2 = null;

    /** Destination for this epConsumer */
    protected Destination destination = null;


    private javax.resource.spi.ActivationSpec aSpec = null;
    private MessageEndpointFactory mepFactory = null;
    private boolean stopping = false;
    private int reconnectAttempts = 6;
    private int reconnectInterval = 30000;
    private int addressListIterations = 3;
    private boolean logRCFailures = true;
    private int maxLoopDelay = 120000;

    /* Loggers */
    private static transient final String _className = "com.sun.messaging.jms.ra.EndpointConsumer";
    protected static transient Logger _loggerIM;
    protected static transient final String _lgrNameInboundMessage = "javax.resourceadapter.mqjmsra.inbound.message";
    protected static transient final String _lgrMIDPrefix = "MQJMSRA_EC";
    protected static transient final String _lgrMID_EET = _lgrMIDPrefix + "1001: ";
    protected static transient final String _lgrMID_INF = _lgrMIDPrefix + "1101: ";
    protected static transient final String _lgrMID_WRN = _lgrMIDPrefix + "2001: ";
    protected static transient final String _lgrMID_ERR = _lgrMIDPrefix + "3001: ";
    protected static transient final String _lgrMID_EXC = _lgrMIDPrefix + "4001: ";

    static { 
        _loggerIM = Logger.getLogger(_lgrNameInboundMessage); 
    }   

    /** Constructs an Endpoint Consumer */
    public EndpointConsumer(com.sun.messaging.jms.ra.ResourceAdapter ra)
    {
        _loggerIM.entering(_className, "constructor()", ra);

        this.ra = ra;
        this.xacf = ra._getXACF();
        if (!ra.getInAppClientContainer()) {
            AccessController.doPrivileged(
                new PrivilegedAction()
                {
                    public Object run() {
                        System.setProperty("imq.DaemonThreads", "true");
                        return null;
                    }
                }
            );
            //System.setProperty("imq.DaemonThreads", "true");
        }
        reconnectAttempts = ra.getReconnectAttempts();
        reconnectInterval = ra.getReconnectInterval();
        addressListIterations = ra.getAddressListIterations();
        try {
            //Disable Client runtime reconnection; RA controls the iterations,
            this.xacf.setProperty(ConnectionConfiguration.imqReconnectEnabled, Boolean.toString(false));
            this.xacf.setProperty(ConnectionConfiguration.imqReconnectAttempts, Integer.toString(reconnectAttempts));
            this.xacf.setProperty(ConnectionConfiguration.imqReconnectInterval, Integer.toString(reconnectInterval));
            this.xacf.setProperty(ConnectionConfiguration.imqAddressListIterations, Integer.toString(1));
        } catch (JMSException jmse) {
            System.err.println("MQRA:EC:constr:Exception setting cf reconnect params-"+jmse.getMessage());
        }
    }

    //javax.jms.ExceptionListener interface method
    public void onException(JMSException exception)
    {
        _loggerIM.severe(_lgrMID_EXC+"onException:"+exception.getMessage());
        //System.err.println("MQRA:EC:EL:Got Connection Exception:"+exception.getMessage());
        logRCFailures = true;
        if (msgListener != null) {
            //System.err.println("MQRA:EC:EL:invalidatingOMRs");
            msgListener.invalidateOnMessageRunners();
        }
        int loopDelay = reconnectInterval;
        int loopCount = 0;
        while (!stopping) {
            //wait till initial interval expires
            try {
                Thread.sleep(loopDelay);
            } catch (Exception ie) {
            }
            try {
                loopCount += 1;
                if (logRCFailures) {
                    _loggerIM.severe(_lgrMID_EXC+"onException:"+aSpec.toString());
                    //System.err.println("MQRA:EC:EL:"+aSpec.toString());
                }
                _loggerIM.severe(_lgrMID_EXC+"onException:reconnect attempt loop# "+loopCount+" :Delayed "+loopDelay+" milliseconds.");
                //System.err.println("MQRA:EC:EL:addressList reconnect attempt_loop#"+loopCount+":Delayed "+loopDelay+" milliseconds.");
                synchronized (this) {
                    if (!stopping) {
                        createMessageConsumer(mepFactory, aSpec);
                        _loggerIM.severe(_lgrMID_EXC+"onException:reconnect success on loop# "+loopCount+" for "+aSpec.toString());
                        //System.err.println("MQRA:EC:EL:RE-CONNECTED consumer:on loop#"+loopCount+" for "+aSpec.toString());
                    }
                }
                break;
            } catch (Exception e) {
                if (logRCFailures) {
                    _loggerIM.severe(_lgrMID_EXC+"onException:Unable to re-establish connection for "+aSpec.toString()+"\nin "+ra.toString());
                    //System.err.println("MQRA:EC:EL:Exception SEVERE:Unable to re-establish connection for "+aSpec.toString()+"\n"+ra.toString());
                } else {
                    logRCFailures = false;
                }
                if (loopDelay < maxLoopDelay) {
                    loopDelay *= 3;
                    if (loopDelay > maxLoopDelay) loopDelay = maxLoopDelay;
                }
            }
        }
    }

    //com.sun.messaging.jms.notification.EventListener interface method
    public void
    onEvent(com.sun.messaging.jms.notification.Event evnt)
    {
        _loggerIM.entering(_className, "onEvent()", evnt);
        _loggerIM.info(_lgrMID_INF+"onEvent:Connection Event:"+evnt.toString());
    }


    // Public Methods
    //

    /** Returns the ResourceAdapter associated with this
     *  EndpointConsumer
     *
     *  @return The ResourceAdapter
     */
    public com.sun.messaging.jms.ra.ResourceAdapter
    getResourceAdapter()
    {
        return ra;
    }

    /** Returns the consumerID associated with this
     *  EndpointConsumer
     *
     *  @return The consumerID
     */
    public int
    getConsumerID()
    {
        return cID;
    }

    /* Returns the factoryID for the MessageFactory
     * associated with this EndpointConsumer
     *
     *  @return The factoryID
     */  
    public int
    getFactoryID()
    {
        return fID;
    }

    /** Returns the MessageEndpointFactory associated with this
     *  EndpointConsumer
     *
     *  @return The MessageEndpointFactory
     */
    public MessageEndpointFactory
    getMessageEndpointFactory()
    {
        //System.out.println("MQRA:EC:getMsgEpFctry:fID="+fID);
        MessageEndpointFactory epf = ra._getMessageFactory(fID);
        return epf;
    }

    /** Returns the XASession associated with this
     *  EndpointConsumer
     *
     *  @return The XASession
     */
    public javax.jms.XASession
    getXASession()
    {
        return xas;
    }

    /** Sets this EndpointConsumer to Deactivated
     *
     */
    public void
    setDeactivated()
    {
        deactivated = true;
    }

    /** Creates a MessageConsumer associated with this EndpointConsumer
     *  after validating the passed in ActivationSpec parameter.
     *  The MessageConsumer delivers the messages to a MessageEndpoint
     *  manufactured  using the MessageEndpointFactory passed in.
     * 
     *  @param endpointFactory The MessageEndpointFactory used to create a
     *         MessageEndpoint to which messages will be delivered.
     *  @param spec The ActivationSpec instance. This must be an instance
     *         of the MQ RA Activation spec implementation class.
     */  
    public void
    createMessageConsumer(MessageEndpointFactory endpointFactory,
        javax.resource.spi.ActivationSpec spec)
    throws NotSupportedException
    {
        //XXX:tharakan:guard against re-creation
        //System.out.println("MQRA:EC:createMessageConsumer():"+spec.toString());

        if (!(spec instanceof com.sun.messaging.jms.ra.ActivationSpec)) {
            throw new NotSupportedException("MQRA:EC:createMsgConsumer:Cannot use non-MQ ActivationSpec -"+spec.getClass());
        }
        aSpec = spec;
        mepFactory = endpointFactory;

        try {
            if (ra != null) {
                transactedDelivery = endpointFactory.isDeliveryTransacted(ra._getOnMessageMethod());
            }
        } catch (java.lang.NoSuchMethodException nsme) {
            //Ignore - assume delivery is non-transacted
        }

        ActivationSpec mqSpec = (com.sun.messaging.jms.ra.ActivationSpec)spec;

        try {
            this.xacf.setProperty(ConnectionConfiguration.imqDefaultUsername, mqSpec.getUserName());
            this.xacf.setProperty(ConnectionConfiguration.imqDefaultPassword, mqSpec.getPassword());
        } catch (JMSException jmse) {
            System.err.println("MQRA:EC:createMsgConsumer:Exception setting cf username/password-"+jmse.getMessage());
        }

        setDestinationType(mqSpec);

        if (destinationType == ClientConstants.DESTINATION_TYPE_TOPIC) {
            setIsDurable(mqSpec);
        }

        if (isDurable) {

            createDurableMessageConsumer(mqSpec, endpointFactory);

        } else {

            createNonDurableMessageConsumer(mqSpec, endpointFactory);
        }

    }


    /** Stops a consumer and connections associated with this EndpointConsumer
     */  
    public void stopMessageConsumer(javax.resource.spi.ActivationSpec spec)
    throws Exception
    {
        //System.out.println("MQRA:EC:stopMessageConsumer():"+spec.toString());
        stopMessageConsumer();
    }
    
    /** Stops a consumer and connections associated with this EndpointConsumer
     */  
    public void stopMessageConsumer()
    throws Exception
    {
        stopping = true;
        synchronized (this) {
        com.sun.messaging.jmq.jmsclient.SessionImpl mqsess;
        //System.out.println("MQRA:EC:stopMessageConsumer()");
        if (msgConsumer != null) {
            try {
                if (msgListener != null) {
                    mqsess = ((com.sun.messaging.jmq.jmsclient.SessionImpl)xas);
                    //System.out.println("MQRA:EC:stopMessageConsumer:_stopFromRA:sessionId="+mqsess.getBrokerSessionID());
                    mqsess._stopFromRA();
                    //System.out.println("MQRA:EC:stopMessageConsumer:_stopFromRA-done/wfOMRs");
                    msgListener.waitForAllOnMessageRunners();
                    //System.out.println("MQRA:EC:stopMessageConsumer:wfOMRs-done/releaseOMRs");
                    msgListener.releaseOnMessageRunners();
                    //System.out.println("MQRA:EC:stopMessageConsumer:Done releasing OMRs/session.close()");
                    xas.close();
                    //System.out.println("MQRA:EC:stopMessageConsumer:Done session.close()");
                }
                //System.out.println("MQRA:EC:stopMessageConsumer:_stopFromRA");
                ////////<---((com.sun.messaging.jmq.jmsclient.SessionImpl)xas)._stopFromRA();
                //System.out.println("MQRA:EC:stopMessageConsumer:done _stopFromRA");
                //System.out.println("MQRA:EC:stopMessageConsumer:closing msgConsumer");
                ///////////////msgConsumer.close();
                //System.out.println("MQRA:EC:stopMessageConsumer:closed msgConsumer...........................");
            } catch (JMSException jmse) {
                ResourceException re = new ResourceException("MQRA:EC:Error on closing MessageConsumer");
                re.initCause(jmse);
                throw re;
            }
        }
        if (xac != null) {
            try {
                xac.close();
            } catch (JMSException jmse) {
                ResourceException re = new ResourceException("MQRA:EC:Error closing JMS Connection");
                re.initCause(jmse);
                throw re;
            }
        }
        }
    }

    /** Updates the factory and consumer tables held by the resource adapter
     *
     */
    private void
    updateFactoryConsumerTables(MessageEndpointFactory endpointFactory, ActivationSpec spec)
    {
        cID = ra.addEndpointConsumer(this);
        fID = ra.addMessageFactory(endpointFactory);
        ra.addFactorytoConsumerLink(fID, cID);
        //System.out.println("MQRA:EC:updateFactoryConsumerTables:fID="+fID+" cID="+cID+":"+spec.toString());
    }

    /** Sets destinationType from the ActivationSpec
     *  instance passed in and validates related configs
     */
    private void
    setDestinationType(ActivationSpec spec)
    throws NotSupportedException
    {

        String destName = spec.getDestination();

        try {
            if (spec._isDestTypeQueueSet()) {
                destination = new com.sun.messaging.Queue(destName);
                destinationType = ClientConstants.DESTINATION_TYPE_QUEUE;
            } else {
                if (spec._isDestTypeTopicSet()) {
                    destination = new com.sun.messaging.Topic(destName);
                    destinationType = ClientConstants.DESTINATION_TYPE_TOPIC;
                }
            }
        } catch (JMSException jmse) {
            NotSupportedException nse = new NotSupportedException(
                "MQRA:EC:Invalid Destination-"+destName);
            nse.initCause(jmse);
            throw nse;

        }

        //XXX:Does MDB Deployment need physical dest to exist?
        //If so need Admin API to check


        //XXX:Can MDB depoyment handle destname as resource-ref vs. physical dest name?
        //If so, need to handle in ActivationSpec (how will AS handle this?)
    }

    /** Sets isDurable from the ActivationSpec
     *  instance passed in and validates related configs
     */
    private void
    setIsDurable(ActivationSpec spec)
    throws NotSupportedException
    {
        //If it's a durable subscription, then validat subscriptionName and clientID
        if (spec._isDurableSet()) {
            String sName = spec.getSubscriptionName();
            if ((sName == null) || ((sName != null) && (sName.length() <= 0))) {
                throw new NotSupportedException("MQRA:EC:Need Valid SubscriptionName-"+sName);
            }
            String cID = spec.getClientId();
            if ((cID == null) || ((cID != null) && (cID.length() <= 0))) {
                throw new NotSupportedException("MQRA:EC:Need Valid ClientID-"+cID);
            }
            //Setting this indicates everything is valid
            isDurable = true;
        }
    }

    /** Creates a Durable Message Consumer from the ActivationSpec
     *  and MessageEndpointFactory instances passed in
     */
    private void
    createDurableMessageConsumer(ActivationSpec spec, MessageEndpointFactory epFactory)
    throws NotSupportedException
    {
        boolean isDeliveryTransacted = false;
        try {
            xacf.setProperty(ConnectionConfiguration.imqAddressList, spec._AddressList());
            //ClientId sharing is driven by RA.inClusteredContainer
            //xacf.setProperty(ConnectionConfiguration.imqEnableSharedClientID,
                    //Boolean.toString(spec.getEnableSharedClientID()));
            xacf.setProperty(ConnectionConfiguration.imqReconnectEnabled,
                    Boolean.toString(false));

        } catch (JMSException jmse) {
            System.err.println("MQRA:EC:Exception setting cf properties:addressList="+spec._AddressList());
        }
        for (int i= 1; i <= addressListIterations; i++) {
            try {
                xac = (com.sun.messaging.jmq.jmsclient.XAConnectionImpl)xacf.createXAConnection();
                if (spec._isInClusteredContainerSet()) {
                    xac.setRANamespaceUID(spec._getRAUID());
                }
                String cId = spec.getClientId();
                if (cId != null && !("".equals(cId))) {
                    xac.setClientID(cId);
                } else {
                    if (spec._isInClusteredContainerSet()) {
                        if (spec._getGroupName() != null) {
                            xac.setClientID(spec._getGroupName()+"{m:"+spec.getMdbName()+"}");
                        } else {
                            xac.setClientID("{m:"+spec.getMdbName()+"}");
                        }
                    }
                }
                xac.setExceptionListener(this);
                ((com.sun.messaging.jms.Connection)xac).setEventListener(this);
                break;
            } catch (JMSException jmse) {
                System.err.println("MQRA:EC:cDMC():createConnection failed on addressListIteration # "+i+" of "+addressListIterations+":Exception="+jmse.getMessage());
                if (xac != null) {
                    try {
                        //System.out.println("MQRA:EC:cDMC:closing xac on conn");
                        xac.close();
                    } catch (JMSException jmsecc) {
                        //System.out.println("MQRA:EC:cDMC:closed xac on conn creation exception-"+jmsecc.getMessage());
                    }
                    xac = null;
                }
                if (i >= addressListIterations || stopping) {
                    System.err.println("MQRA:EC:cDMC():createConnction failed:aborting after "+addressListIterations+" addressListIterations");
                    if (logRCFailures) jmse.printStackTrace();
                    NotSupportedException nse =
                        new NotSupportedException(
                        "MQRA:EC:Error:createDurableConsumer:createConnection failed:aborting due to:"+jmse.getMessage());
                    nse.initCause(jmse);
                    throw nse;
                }
                try {
                    Thread.sleep(reconnectInterval);
                } catch (Exception e) {}
            }
        }
        try {
            //System.err.println("MQRA:EC:cDMC:getDebugState="+xac.getDebugState(true).toString());
            xas = (XASession)xac.createSession(isDeliveryTransacted, Session.CLIENT_ACKNOWLEDGE);
            ((com.sun.messaging.jmq.jmsclient.SessionImpl)xas)._setRAEndpointSession();
            msgConsumer = xas.createDurableSubscriber((Topic)destination,
                                spec.getSubscriptionName(),
                                spec.getMessageSelector(), false);
            msgListener = new MessageListener(this, epFactory, spec);
            //System.out.println("MQRA:EC:Created msgListener");
            //msgConsumer.setMessageListener(new MessageListener(this, epFactory, spec));
            msgConsumer.setMessageListener(msgListener);
            //System.out.println("MQRA:EC:Set msgListener");
            //System.out.println("MQRA:EC:Starting Connection");
            xac.start();
            updateFactoryConsumerTables(epFactory, spec);
        } catch (JMSException jmse) {
            if (xac != null) {
                try {
                    xac.close();
                } catch (JMSException jmsecc) {
                    //System.out.println("MQRA:EC:closed xac on conn creation exception-"+jmsecc.getMessage());
                }
                xac = null;
            }
            NotSupportedException nse = new NotSupportedException(
                    "MQRA:EC:Error creating Durable Message Consumer:\n"+jmse.getMessage());
            nse.initCause(jmse);
            throw nse;
        }
    }

    /** Creates a Non-Durable Message Consumer from the ActivationSpec
     *  and MessageEndpointFactory instances passed in
     */
    private void
    createNonDurableMessageConsumer(ActivationSpec spec, MessageEndpointFactory epFactory)
    throws NotSupportedException
    {
        boolean isDeliveryTransacted = false;
        boolean noAckDelivery = false;

        String cId = spec.getClientId();
        String mName = spec.getMdbName();
        if (spec._isInClusteredContainerSet()) {
            if ((cId == null) || ("".equals(cId))) {
                if ((mName == null) || ("".equals(mName))) {
                    throw new NotSupportedException(
                        "MQRA:EC:Error:Clustered Non-Durable Message Consumer requires non-null clientID OR mdbName:" +
                        "clientID="+cId+":mdbName="+mName);
                }
            }
        }
        try {
            xacf.setProperty(ConnectionConfiguration.imqAddressList, spec._AddressList());
            //ClientId sharing is driven by RA.inClusteredContainer
            //xacf.setProperty(ConnectionConfiguration.imqEnableSharedClientID,
                    //Boolean.toString(spec.getEnableSharedClientID()));
            xacf.setProperty(ConnectionConfiguration.imqReconnectEnabled,
                    Boolean.toString(false));

        } catch (JMSException jmse) {
            System.err.println("MQRA:EC:Exception setting cf properties:addressList="+spec._AddressList());
        }
        for (int i= 1; i <= addressListIterations; i++) {
            try {
                xac = (com.sun.messaging.jmq.jmsclient.XAConnectionImpl)xacf.createXAConnection();
                if (spec._isInClusteredContainerSet()) {
                    xac.setRANamespaceUID(spec._getRAUID());
                }
                if (cId != null && !("".equals(cId))) {
                        xac.setClientID(cId);
                } else {
                    if (spec._isInClusteredContainerSet()) {
                        if (spec._getGroupName() != null) {
                            xac.setClientID(spec._getGroupName()+"{m:"+spec.getMdbName()+"}");
                        } else {
                            xac.setClientID("{m:"+spec.getMdbName()+"}");
                        }
                    }
                }
                xac.setExceptionListener(this);
                ((com.sun.messaging.jms.Connection)xac).setEventListener(this);
                break;
            } catch (JMSException jmse) {
                System.err.println("MQRA:EC:cNDMC():createConnection failed on addressListIteration # "+i+" of "+addressListIterations+":Exception="+jmse.getMessage());
                if (xac != null) {
                    try {
                        xac.close();
                    } catch (JMSException jmsecc) {
                        //System.out.println("MQRA:EC:cNDMC:closed xac on conn creation exception-"+jmsecc.getMessage());
                    }
                    xac = null;
                }
                if (i >= addressListIterations || stopping) {
                    System.err.println("MQRA:EC:cNDMC():createConnction failed:aborting after "+addressListIterations+" addressListIterations");
                    if (logRCFailures) jmse.printStackTrace();
                    NotSupportedException nse =
                        new NotSupportedException(
                        "MQRA:EC:Error:createNonDurableConsumer:createConnection failed:aborting due to:"+jmse.getMessage());
                    nse.initCause(jmse);
                    throw nse;
                }
                try {
                    Thread.sleep(reconnectInterval);
                } catch (Exception e) {}
            }
        }
        try {
            if ((spec._isNoAckDeliverySet()) && 
                (destination instanceof com.sun.messaging.Topic) &&
                (!transactedDelivery)) {
                noAckDelivery = true;
            }
            //System.err.println("MQRA:EC:cDMC:getDebugState="+xac.getDebugState(true).toString());
            if (noAckDelivery) {
                xas = (XASession)xac.createSession(com.sun.messaging.jms.Session.NO_ACKNOWLEDGE);
            } else {
                xas = (XASession)xac.createSession(isDeliveryTransacted, Session.CLIENT_ACKNOWLEDGE);
            }
            ((com.sun.messaging.jmq.jmsclient.SessionImpl)xas)._setRAEndpointSession();
            msgConsumer = xas.createConsumer(destination, spec.getMessageSelector());
            //test to see if Queue is enabled for more than one consumer when InClustered true
            if (destination instanceof javax.jms.Queue && spec._isInClusteredContainerSet()) {
                //Fail activation if it is not
                try {
                    msgConsumer2 = xas.createConsumer(destination, spec.getMessageSelector());
                    msgConsumer2.close();
                    msgConsumer2 = null;
                } catch (JMSException jmse) {
                    if (xac != null) {
                        try {
                            xac.close();
                        } catch (JMSException jmsecc) {
                            //System.out.println("MQRA:EC:closed xac on conn creation exception-"+jmsecc.getMessage());
                        }
                        xac = null;
                    }
                    NotSupportedException nse = new NotSupportedException(
                           "MQRA:EC:Error clustering multiple consumers on Queue:\n"+jmse.getMessage());
                    nse.initCause(jmse);
                    throw nse;
                }
            }
            msgListener = new MessageListener(this, epFactory, spec, noAckDelivery);
            //System.out.println("MQRA:EC:Created msgListener");
            //msgConsumer.setMessageListener(new MessageListener(this, epFactory, spec));
            msgConsumer.setMessageListener(msgListener);
            //System.out.println("MQRA:EC:Set msgListener");
            //System.out.println("MQRA:EC:Starting Connection");
            xac.start();
            //System.out.println("MQRA:EC:Started connection");
            updateFactoryConsumerTables(epFactory, spec);
        } catch (JMSException jmse) {
            if (xac != null) {
                try {
                    xac.close();
                } catch (JMSException jmsecc) {
                    //System.out.println("MQRA:EC:closed xac on conn creation exception-"+jmsecc.getMessage());
                }
                xac = null;
            }
            NotSupportedException nse = new NotSupportedException(
                    "MQRA:EC:Error creating Non-Durable Message Consumer:\n"+jmse.getMessage());
            nse.initCause(jmse);
            throw nse;
        }
    }
}

