/*
 * @(#)LocalBrokerRunner.java	1.12 06/02/09
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jms.ra;

import java.util.logging.Logger;
import java.util.Vector;
import java.util.Properties;
import java.util.Enumeration;
import java.util.StringTokenizer;

import javax.jms.JMSSecurityException;

import com.sun.messaging.jmq.jmsspi.JMSAdmin;
import com.sun.messaging.jmq.jmsspi.JMSAdminFactory;


/**
 * Runs a local broker instance through
 * the JMSSPI interfaces exposed for out-of-process
 * broker lifecycle control
 * 
 * @author George Tharakan
 */

public class LocalBrokerRunner {

    /* Properties to be used by the embedded broker */
    private Properties brokerProps = null;

    /* The JMS Admin SPI related objects */
    private JMSAdmin jmsadmin = null;
    private JMSAdmin tjmsadmin = null;

    /* The admin username that was configured when starting the broker */
    private String adminUsername;

    /* The admin password file that was configured when starting the broker */
    private String adminPassFile;

    /* The cmd line args for starting the broker */
    String[] brokerArgs;

    /* The bin directory for starting the broker */
    String brokerBinDir;

    /* The instance name to be used for starting the broker */
    String brokerInstanceName;

    /* The full path to the broker log file to be checked for broker start failures */
    String brokerLogFilename;

    /* The timeout allowed when starting the broker */
    int brokerStartTimeout;

    /* The main broker thread */
    //private Thread bt = null;

    /* Loggers */
    private static transient final String _className = "com.sun.messaging.jms.ra.LocalBrokerRunner";
    protected static transient Logger _loggerL;
    protected static transient final String _lgrNameLifecycle = "javax.resourceadapter.mqjmsra.lifecycle";
    protected static transient final String _lgrMIDPrefix = "MQJMSRA_LB";
    protected static transient final String _lgrMID_EET = _lgrMIDPrefix + "1001: ";
    protected static transient final String _lgrMID_INF = _lgrMIDPrefix + "1101: ";
    protected static transient final String _lgrMID_WRN = _lgrMIDPrefix + "2001: ";
    protected static transient final String _lgrMID_ERR = _lgrMIDPrefix + "3001: ";
    protected static transient final String _lgrMID_EXC = _lgrMIDPrefix + "4001: ";

    static {
        _loggerL = Logger.getLogger(_lgrNameLifecycle);
    }

    public
    LocalBrokerRunner(
        String brokerInstanceName, String brokerBindAddress, int brokerPort,
        String brokerHomeDir, String brokerLibDir, String brokerVarDir, String brokerJavaDir,
        String brokerExtraArgs,
        boolean useJNDIRMIServiceURL, int rmiRegistryPort, boolean startRMIRegistry,
        boolean useSSLJMXConnector, int brokerStartTimeout,
        String adminUsername, String adminPassword, String adminPassFile, Properties sysProps)
    throws Exception
    {
        Object params[] = new Object[16];
        params[0] = brokerInstanceName;
        params[1] = brokerBindAddress;
        params[2] = Integer.toString(brokerPort);
        params[3] = brokerHomeDir;
        params[4] = brokerLibDir;
        params[5] = brokerVarDir;
        params[6] = brokerJavaDir;
        params[7] = brokerExtraArgs;
        params[8] = new Boolean(useJNDIRMIServiceURL);
        params[9] = new Integer(rmiRegistryPort);
        params[10] = new Boolean(startRMIRegistry);
        params[11] = new Boolean(useSSLJMXConnector);
        params[12] = new Integer(brokerStartTimeout);
        params[13] = adminUsername;
        params[14] = adminPassFile;
        params[15] = sysProps;

        this.adminUsername = adminUsername;

        _loggerL.entering(_className, "constructor()", params);

        String brokerURL = ((brokerBindAddress == null) ? "localhost" : brokerBindAddress ) + ":" + brokerPort;

        jmsadmin = ((JMSAdminFactory)(new com.sun.messaging.jmq.admin.jmsspi.JMSAdminFactoryImpl())).getJMSAdmin(
                                brokerURL, adminUsername, adminPassword);
        tjmsadmin = ((JMSAdminFactory)(new com.sun.messaging.jmq.admin.jmsspi.JMSAdminFactoryImpl())).getJMSAdmin(
                                brokerURL, "admin", "admin");

        checkVersion(jmsadmin);

        this.brokerStartTimeout = brokerStartTimeout;

        Vector v = new Vector();

        if (brokerJavaDir != null) {
            v.add("-javahome");
            v.add(brokerJavaDir);
        }

        //Add extra args first; explicit config will override args 
        if (brokerExtraArgs != null && !("".equals(brokerExtraArgs)) ) {
            StringTokenizer st = new StringTokenizer(brokerExtraArgs, " ");
            while (st.hasMoreTokens()) {
                String t = st.nextToken();
                v.add(t);
            }
        }

        String key;
        String val;
        //System.out.println("SJSMQRA_EB:DebugENM_BEGIN>>>>>>>>>>>>>>>>");
        for (Enumeration e = sysProps.keys() ; e.hasMoreElements() ;) {
            key = (String)e.nextElement();
            //System.out.println("SJSMQRA_EB:DebugENM_Key="+key);
            if (key != null) {
                val = (String)sysProps.getProperty(key);
                if (val != null)
                    v.add("-D"+key+"="+val);
                //System.out.println("SJSMQRA_EB:DebugENM_Val="+val);
                //v.add("-D\""+key+"="+val+"\"");
            } else {
                //System.out.println("SJSMQRA_EB:DebugENM_Val=null");
            }
        }

        this.brokerInstanceName = brokerInstanceName;
        this.brokerBinDir = brokerHomeDir + java.io.File.separator + "bin";
        this.brokerLogFilename = brokerVarDir
                                     + java.io.File.separator + "instances"
                                     + java.io.File.separator + brokerInstanceName
                                     + java.io.File.separator + "log"
                                     + java.io.File.separator + "log.txt" ;

        if (brokerVarDir != null) {
            v.add("-varhome");
            v.add(brokerVarDir);
        }

        if (useJNDIRMIServiceURL == true) {
            if (startRMIRegistry == true) {
                v.add("-startRmiRegistry");
            } else {
                v.add("-useRmiRegistry");
            }
            v.add("-rmiRegistryPort");
            v.add(Integer.toString(rmiRegistryPort));
        }

        if ((adminPassFile != null) &&
            !("".equals(adminPassFile)) &&
            (adminUsername != null) && 
            !("".equals(adminUsername)) ) {

            v.add("-Dimq.imqcmd.user="+adminUsername);
            v.add("-passfile");
            v.add(adminPassFile);

            //Set adminPassFile to non-null
            this.adminPassFile = adminPassFile;
        }
        if (brokerBindAddress != null && !("localhost".equals(brokerBindAddress)) ) {
            v.add("-Dimq.hostname="+brokerBindAddress);
        }
 
        brokerArgs = (String []) v.toArray(new String[0]);

        //System.out.println("SJSMQRA_LB:DebugDump_BEGIN>>>>>>>>>>>>>");
        //System.out.println("SJSMQRA_LB:brokerArgs="+brokerArgs);
        //System.out.println("SJSMQRA_LB:LocalBroker:SystemProps="+System.getProperties().toString());
        //System.out.println("SJSMQRA_LB:DebugDump___END>>>>>>>>>>>>>");

    }

    protected synchronized void
    start()
    throws Exception
    {
        _loggerL.entering(_className, "start()");

        jmsadmin.startProvider(brokerBinDir, brokerArgs, brokerInstanceName);

        boolean ready = false;
        long firstPingTime = java.lang.System.currentTimeMillis();

        while (true) {
            try {
                //Wait a sec bit before attempting to ping
                try {
                    Thread.sleep(1000);
                } catch (Exception e) {}
                jmsadmin.pingProvider();
                ready = true;
                break;
            } catch (JMSSecurityException jmsse) {
                //Here because the auth info is incorrect  and the broker was able to be started.
                //i.e. they did not use adminPassFile to prevent broker startup with wrong auth
                //Attempt to shut down broker with the default admin/admin username/password
                //And then trow an Exception back to ra.start()

                Exception startEx = new Exception(_lgrMID_EXC+"start:Unable to ping Broker due to authentication error for user " + adminUsername + " : shutting down broker.");
                startEx.initCause(jmsse);
                try {
                    tjmsadmin.connectToProvider();
                    tjmsadmin.shutdownProvider();
                } catch (Exception spe) {
                    _loggerL.warning(_lgrMID_EXC+"start:Exception on LOCAL broker shutdown after connect auth failure:msg="+spe.getMessage());
                    //spe.printStackTrace();
                }
                throw startEx;
            } catch (Exception e) {
                //If here, then broker cannot be running on the given port
                //If adminPassFile was given, then likely that auth is wrong
                //If adminPassFile was NOT given, then broker wasn't able to start for some other reason

                if (java.lang.System.currentTimeMillis() - firstPingTime >= brokerStartTimeout) {
                    _loggerL.severe(_lgrMID_EXC+"start:Ping broker failed " + (java.lang.System.currentTimeMillis() - firstPingTime) + " millis after broker start performed. Failing ra.start()");
                    _loggerL.warning(_lgrMID_EXC+"start:Aborting:Check Broker Log File at:"+brokerLogFilename);

                    if (adminPassFile != null) {
                        _loggerL.warning(_lgrMID_EXC+"start:Aborting:Check Broker Password File at:"+adminPassFile);
                    }
                    break;
                }
            }
        }
        if (!ready) {
            throw new RuntimeException(_lgrMID_EXC+"start:Aborted:Unable to ping Broker within " + brokerStartTimeout + " millis (startTimeOut)");
        }
    }

    protected synchronized void
    stop()
    {
        _loggerL.entering(_className, "stop()");
        if (jmsadmin != null) {
           try {
               jmsadmin.connectToProvider();
               jmsadmin.shutdownProvider();
           } catch (Exception bse) {
               _loggerL.severe(_lgrMID_EXC+"stop:Exception on LOCAL broker shutdown:msg="+bse.getMessage());
               bse.printStackTrace();
           }
        }
    }

    private static void
    checkVersion(JMSAdmin jmsadm)
    {
        float ver_f = 0.0f;
        String ver_s = "?.?";
        try {
            ver_s = jmsadm.getVersion();
            ver_f = Float.parseFloat(ver_s);
        } catch (Exception e) {
            throw new RuntimeException("Error while parsing SJSMQ SPI version string (" + ver_s + ").");
        }
        if (ver_f < 2.0 || ver_f >= 3.0) {
            throw new RuntimeException("Incorrect SJSMQ SPI version detected (" + ver_s + ").");
        }
    }
}
