/*
 * @(#)MessageListener.java	1.19 06/02/09
 *
 * Copyright 2005 Sun Microsystems, Inc. All Rights Reserved.
 * SUN PROPRIETARY/CONFIDENTIAL.
 * Use is subject to license terms.
 *
 */

package com.sun.messaging.jms.ra;

import javax.jms.JMSException;

import javax.resource.*;
import javax.resource.spi.*;
import javax.resource.spi.endpoint.*;

import javax.transaction.xa.XAResource;

import java.lang.reflect.Method;

import java.util.logging.Logger;

/**
 * Implements the JMS MessageListener interface for the S1 MQ RA
 * and forwards the messages to the MessageEndpoint created by the
 * MessageEndpointFactory.
 *
 * @author George Tharakan
 */

public class MessageListener
implements javax.jms.MessageListener
{
    /** The EndpointConsumer for this MessageListener instance */
    private EndpointConsumer epConsumer = null;

    /** The MessageEndpointFactory for this MessageListener instance */
    private MessageEndpointFactory epFactory = null;

    /** The ActivationSpec for this MessageListener instance */
    private com.sun.messaging.jms.ra.ActivationSpec spec = null;

    /** The MessageEndpoint for this MessageListener instance */
    private MessageEndpoint msgEndpoint = null;

    /** The XAResource for this MessageListener instance */
    private XAResource xar = null;

    /** The onMessage Method that this MessageListener instance will call */
    private Method onMessage = null;

    /** The onMessageRunnerPool that this MessageListener instance will use */
    private OnMessageRunnerPool omrPool = null;

    private boolean transactedDelivery = false;
    private boolean noAckDelivery = false;

    /* Loggers */
    private static transient final String _className = "com.sun.messaging.jms.ra.ConnectionFactoryAdapter";
    protected static transient Logger _loggerIM;
    protected static transient final String _lgrNameInboundMessage = "javax.resourceadapter.mqjmsra.inbound.message";
    protected static transient final String _lgrMIDPrefix = "MQJMSRA_ML";
    protected static transient final String _lgrMID_EET = _lgrMIDPrefix + "1001: ";
    protected static transient final String _lgrMID_INF = _lgrMIDPrefix + "1101: ";
    protected static transient final String _lgrMID_WRN = _lgrMIDPrefix + "2001: ";
    protected static transient final String _lgrMID_ERR = _lgrMIDPrefix + "3001: ";
    protected static transient final String _lgrMID_EXC = _lgrMIDPrefix + "4001: ";

    static {
        _loggerIM = Logger.getLogger(_lgrNameInboundMessage);
    }

    /** Constructor */
    public MessageListener(
        EndpointConsumer epConsumer,
        MessageEndpointFactory epFactory,
        ActivationSpec spec)
    {
        this(epConsumer, epFactory, spec, false);
    }

    /** Constructor */
    public MessageListener(
        EndpointConsumer epConsumer,
        MessageEndpointFactory epFactory,
        ActivationSpec spec,
        boolean noAckDelivery)
    {
        Object params[] = new Object[4];
        params[0] = epConsumer;
        params[1] = epFactory;
        params[2] = spec;
        params[3] = new Boolean(noAckDelivery);

        _loggerIM.entering(_className, "constructor()", params);

        //System.out.println("MQRA:ML:Constructor()-omrp:min,max="+spec.getEndpointPoolSteadySize()+","+spec.getEndpointPoolMaxSize());
        this.epConsumer = epConsumer;
        this.epFactory = epFactory;
        this.noAckDelivery = noAckDelivery;
        this.spec = (com.sun.messaging.jms.ra.ActivationSpec)spec;

        //XXX: This really should be allowed here - but as8b47 blocks until activation is complete
        //omrPool = new OnMessageRunnerPool(epFactory, epConsumer, spec);

        onMessage = epConsumer.getResourceAdapter()._getOnMessageMethod();
        xar = epConsumer.getXASession().getXAResource();
        try {
            transactedDelivery = epFactory.isDeliveryTransacted(onMessage);
        } catch (java.lang.NoSuchMethodException nsme) {
            //Ignore - assume delivery is non-transacted
        }
    }

    public void waitForAllOnMessageRunners() throws JMSException {
        if (omrPool != null) {
            omrPool.waitForAllOnMessageRunners();
        }
    }

    public void releaseOnMessageRunners() {
        if (omrPool != null) {
            omrPool.releaseOnMessageRunners();
        }
    }

    public void invalidateOnMessageRunners() {
        if (omrPool != null) {
            omrPool.invalidateOnMessageRunners();
        }
    }

    // JMS MessageListener interface methods //
    // 

    public void
    onMessage(javax.jms.Message message)
    {
        //System.out.println("MQRA:ML:onMessage():Msg="+message.toString());
        //Set message being consumed in RA
        ((com.sun.messaging.jmq.jmsclient.MessageImpl)message)._setConsumerInRA();

        if (spec._deliverySerial()) {
            _onMessage(message);
            return;
        }

        //XXX:Remove once as8 allows createEndpoint in endpointActivation
        if (omrPool == null) {
            omrPool = new OnMessageRunnerPool(epFactory, epConsumer, spec);
        }
        //XXX:remove till here

        OnMessageRunner omr;

        try {
            omr = omrPool.getOnMessageRunner();
            omr.onMessage(message);
        } catch (JMSException jmse) {
            System.err.println("MQRA:ML:onMessage:JMSException on getOnMessageRunner");
            jmse.printStackTrace();
        }
    }

    /** Upon receipt of a JMS 'onMessage'  method call,
     *  this method delivers the JMS Message to the MessageEndpoint
     *  that is associated with the EndpointConsumer that is
     *  associated with this MessageListener
     */
    public void
    _onMessage(javax.jms.Message message)
    {
        //System.out.println("MQRA:ML:onMessage()");
        com.sun.messaging.jmq.jmsclient.MessageImpl mqmsg =
            (com.sun.messaging.jmq.jmsclient.MessageImpl)message;
        com.sun.messaging.jmq.jmsclient.SessionImpl mqsess =
            (com.sun.messaging.jmq.jmsclient.SessionImpl)epConsumer.getXASession();

        try {
            transactedDelivery = epFactory.isDeliveryTransacted(onMessage);
        } catch (java.lang.NoSuchMethodException nsme) {
            //Ignore - what should be assumed for the
            //value of transactedDelivery ???
        }

        //XXX:Message Logging?

        Object[] msgArg = { message };

        msgEndpoint = null;
        //This is not configurable since ideally we'd want
        //createEndpoint to simply block or Unavailable mean
        //that it's never going to be available
        //otherwise we'll never resolve why we couldn't get
        //the endpoint
        for (int i = 1; i < 6; i++ ) {
            try {
                //If it's not deactivated
                if (epConsumer.deactivated != true) {
                    msgEndpoint = epFactory.createEndpoint(xar);
                    break;
                }
            } catch (UnavailableException ue) {
                try {
                    //System.err.println("MQRA:ML:Unavailable:Sleeping for:"+i*200);
                    Thread.sleep(i * 200);
                } catch (InterruptedException ie) {
                }
            }
        }
        if (msgEndpoint == null) {
            //Could not acquire - shut down delivery of messages in this session
            System.err.println("MQRA:ML:Endpoint Unavailable:Shutting down delivery for "+spec.toString());
            //_logger.log(Level.SEVERE, "MQRA:ML:Endpoint Unavailable:Shutting down delivery for "+spec.toString());
            mqmsg._getSession().closeFromRA();
            //endpoint should be shutdown normally by AS via RA.endpointDeactivation()
            return;
        }

        ClassLoader cl = spec.getContextClassLoader();
        int exRedeliveryAttempts = spec.getEndpointExceptionRedeliveryAttempts();
        int exRedeliveryInterval = spec.getEndpointExceptionRedeliveryInterval();
        //Deliver message to msgEndpoint
        boolean redeliver = true;
        while (redeliver == true) {
            try {
                if (transactedDelivery) {
                    msgEndpoint.beforeDelivery(onMessage);
                }
                try {
                    //System.out.println("MQRA:ML:Delivering to onMessage()");
                    if (cl != null) {
                        //System.out.println("MQRA:ML:Setting ContextClassLoader:"+cl.toString());
                        try {
                            Thread.currentThread().setContextClassLoader(cl);
                        } catch (Exception sccle) {
                            System.err.println("MQRA:ML:Exception setting ContextClassLoader:"+sccle.getMessage());
                        }
                    }
                    ((javax.jms.MessageListener)msgEndpoint).onMessage(message);
                    redeliver = false;
                    //System.out.println("MQRA:ML:Delivered successfully");
                    try {
                        mqsess.acknowledgeFromRAEndpoint(mqmsg);
                    } catch (JMSException jmse) {
                        System.err.println("MQRA:ML:JMSException on acknowledge");
                    }
                } catch (Exception rte) {
                    System.err.println("MQRA:ML:Caught Exception from onMessage():"+rte.getMessage());
                    if (exRedeliveryAttempts > 0) {
                        try {
                            //System.out.println("MQRA:ML:RedeliverInterval-start");
                            Thread.sleep(exRedeliveryInterval);
                            //System.out.println("MQRA:ML:RedeliverInterval-stop");
                        } catch (InterruptedException ie) {
                            //System.err.println("MQRA:ML:RedeliverInterval-interrupted");
                        }
                        exRedeliveryAttempts -= 1;
                    } else {
                        System.err.println("MQRA:ML:Exhausted redeliveryAttempts-shutting down delivery for "+spec.toString());
                        redeliver = false;
                        mqmsg._getSession().closeFromRA();
                    }
                }
                if (transactedDelivery) {
                    msgEndpoint.afterDelivery();
                }
            } catch (Throwable t) {
                System.err.println("MQRA:ML:onMessage caught Throwable-before/on/afterDelivery:Class="+t.getClass().getName()+
                        "Msg="+t.getMessage());
                redeliver = false;
                mqmsg._getSession().closeFromRA();
            } finally {
                if (msgEndpoint != null) {
                    try {
                        msgEndpoint.release();
                    } catch (Exception e) {
                        //System.err.println("MQRA:ML:onMessage Exception-msgEp.release");
                    }
                }
            }
        }
    }

}
