/*
 * @(#)util.c	1.13 06/01/11
 * 
 * Copyright (c) 2002 Sun Microsystems, Inc.
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 */

#include <windows.h>
#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <jni.h>
#include "mqapp.h"

int SourceFile(const char *file);
int myisspace(const char c);

#define IMQ_JAVAHOME "IMQ_JAVAHOME"
#define IMQ_DEFAULT_JAVAHOME "IMQ_DEFAULT_JAVAHOME"
#define IMQENV1  "\\etc\\imqenv.conf"
#define IMQENV2  "\\lib\\imqenv.conf"
#define SET "set "
#define REM "REM "
#define SKIP_WHITE(p) while (myisspace(*p)) p++

/*
 * VMs to use for JNI java clients. Order is most desireable to 
 * least desireable.
 * Relative paths are assumed to be relative to jrehome/bin
 */
char *client_vm_libs[] = {
    "client\\jvm.dll",
    "server\\jvm.dll",
    "classic\\jvm.dll",
    "hotspot\\jvm.dll"
};
int nclient_vm_libs = sizeof (client_vm_libs) / sizeof(char *);



/**
 * Find a suitable Java runtime. Here is where we look.
 *
 * 1. IMQ_JAVAHOME
 * 2. What's set for IMQ_JAVAHOME in %IMQ_HOME%\var\jdk-env.bat
 * 3. IMQ_DEFAULT_JAVAHOME (may be set via imqenv.conf)
 * 4. %IMQ_HOME%\jre (bundled)
 *
 * home     Value of IMQ_HOME
 * varhome  Value of IMQ_VARHOME
 * path     Memory location to copy path to runtime into. It must be
 *          at least size MAX_PATH. If it is a JDK then the path
 *          will point to the "jre" directory.
 *
 * returns 0 if we find a runtime, else < 0
 */
int FindJavaRuntime(const char *home, const char *varhome, char *path) {

    char tmpbuf[MAX_PATH];
    char pathbuf[MAX_PATH];
    char *p = NULL;
    HANDLE hfile = 0;

    *path = '\0';

    memset(tmpbuf,    '\0', sizeof(tmpbuf));
    memset(pathbuf, '\0', sizeof(pathbuf));

    if ((p = getenv(IMQ_JAVAHOME)) == NULL) {
        /* IMQ_JAVAHOME not set. Try and get it from jdk-env.bat file */
        strcpy(tmpbuf, varhome);
        strcat(tmpbuf, "\\jdk-env.bat");
        if (_access(tmpbuf, 00) == 0) {
            /* This should set IMQ_JAVAHOME */
            SourceFile(tmpbuf);
            p = getenv(IMQ_JAVAHOME);
        }
    }


    if (p == NULL) {
        /* Still no IMQ_JAVAHOME. Check for IMQ_DEFAULT_JAVAHOME */
        p = getenv(IMQ_DEFAULT_JAVAHOME);
    }

    if (p == NULL) {
        /* Still no IMQ_JAVAHOME. Look for bundled */
        strcpy(tmpbuf, home);
        strcat(tmpbuf, "\\jre");
        if (_access(tmpbuf, 00) == 0) {
            p = tmpbuf;
        }
    }

    if (p == NULL) {
        return -1;
    }

    if (_access(p, 00) == 0) {
        /*
         * Valid IMQ_JAVAHOME. either it is pointing to a JDK or a JRE.
         * We always return a path to the JRE directory.
         */
        strcpy(pathbuf, p);
        strcat(pathbuf, "\\jre");
        if (_access(pathbuf, 00) == 0) {
            /* IMQ_JAVAHOME was pointing to a JDK */
            strcpy(path, pathbuf);
        } else {
            /* IMQ_JAVAHOME was pointing to a JRE */
            strcpy(path, p);
        }
        /* printf("===== Found java runtime %s\n", path); */
        return 0;
    } else {
        /* Return invalid path so we can give a good error message */
        strcpy(path, p);
        return -1;
    }
}

/*
 * Our version of isspace(). Why? Because I'm getting link errors
 * on windows if I use isspace() and I don't have time to figure out
 * why.
 */
int myisspace(const char c) {
    return (c != '\0' && c <= ' ');
}

/*
 * Set environment variables based on "set" commands in a file.
 * For example if the file contains
 *     set COLOR=blue
 *     set SIZE=large
 * The COLOR and SIZE would be set in the current process' environment.
 * All other lines in the file are ignored.
 */
int SourceFile(const char *file) {

    char buf[MAX_PATH];
    char env[MAX_PATH];
    FILE *fp;
    char *p;
    char *start = NULL;

    if ((fp = fopen(file, "r")) == NULL) {
        sprintf(buf, "MQ service couldn't read configuration file %s", file);
	perror(buf);
	return -1;
    }

    memset(buf, '\0', sizeof(buf));

    while (fgets(buf, sizeof(buf), fp) != NULL) {
	p = buf;
        *env = '\0';
	/* Skip leading white space */
        SKIP_WHITE(p);

        if (strncmp(REM, p, strlen(REM)) == 0) {
            // Skip comments
            ;
        } else if (strncmp(SET, p, strlen(SET)) == 0) {
            /* Skip "set "*/
            p += strlen(SET);

            SKIP_WHITE(p);

            /* Start of variable */
            start = p;

            /* Find end of varible */
            while (!myisspace(*p) && *p != '=') {
                p++;
            }

            if (*p == '\0') {
                continue;
            }

            *p = '\0';
            strcpy(env, start);
            strcat(env, "=");
            p++;
            SKIP_WHITE(p);

            if (*p == '=') {
                p++;
                SKIP_WHITE(p);
            }

            /* Get value */
            start = p;

            while (*p != '\0' && *p != '\n') {
                p++;
            }
            *p = '\0';
            strcat(env, start);

            if (*env != '\0') {
                /* printf("===== Setting %s", env); */
                _putenv(_strdup(env));
            }

        }
    }

    fclose(fp);

    return 0;
}


/**
 * Add all the jar files in the $IMQ_VARHOME/lib/ext directory to classpath.
 * These are user defined jar files needed for things like JDBC connectors
 */
VOID MqAppAddLibJars(const char *libhome, char *classpath) {

    WIN32_FIND_DATA dir;
    char lib[MAX_PATH];
    HANDLE handle;

    /* Added $IMQ_HOME/lib/ext directory */
    strcat(classpath, ";");
    strcat(classpath, libhome);
    strcat(classpath, "\\ext");

    sprintf(lib, "%s\\ext\\*", libhome);

    /* Add jars if any */
    handle = FindFirstFile(lib, &dir);
    if (handle == INVALID_HANDLE_VALUE) {
	return;
    }
    do {
	int len = strlen(dir.cFileName);

	if (len <= 4) {
            /* not a .jar file */
	} else if (strcmp(dir.cFileName + len - 4, ".jar") != 0 &&
                   strcmp(dir.cFileName + len - 4, ".zip") != 0) {
            /* File doesn't end in .jar or .zip */
	} else {
            strcat(classpath, ";");
            strcat(classpath, libhome);
            strcat(classpath, "\\ext\\");
            strcat(classpath, dir.cFileName);
	}
    } while (FindNextFile(handle, &dir));
}

/*
 * Do only MQ related initialization - no java initialization. This basically locates IMQ
 * and sets the relevant pieces of info in the MqEnv structure.
 *
 * The following fields in the MqEnv are updated.
 *  imqhome
 *  imqlibhome
 *  imqvarhome
 *
 * All the fields above need to point to pre-allocated space.
 *
 * Params:
 *  me			ptr to MqEnv structure, must not be NULL.
 */
int MqAppInitializeNoJava(MqEnv *me)
{
    char *slash_loc;
    char *p;
    char tmpbuf[MAX_PATH];

    /* Determine IMQ_HOME */
    GetModuleFileName(0, me->imqhome, sizeof(me->imqhome));

    /* Take off file name and bin directory */
    slash_loc = strrchr(me->imqhome, '\\');
    *slash_loc = '\0';
    slash_loc = strrchr(me->imqhome, '\\');
    *slash_loc = '\0';

    if (me->imqlibhome[0] == '\0') {
        strcpy(me->imqlibhome, me->imqhome);
        strcat(me->imqlibhome, "\\lib");
    }

    /* Source imqenv.conf file to set default values
     * For example this may set IMQ_DEFAULT_JAVAHOME and
     * IMQ_DEFAULT_VARHOME
     */
    memset(tmpbuf, 0, MAX_PATH);
    strncpy(tmpbuf, me->imqhome, MAX_PATH - strlen(IMQENV1) - 2 );
    strcat(tmpbuf, IMQENV1);
    if (_access(tmpbuf, 00) == 0) {
        SourceFile(tmpbuf);
    }

    /* We considered this in 3.6, but decided against it since in Solaris
     * /usr/share/lib/imq should be read-only.
    memset(tmpbuf, 0, MAX_PATH);
    strncpy(tmpbuf, me->imqhome, MAX_PATH - strlen(IMQENV2) - 2 );
    strcat(tmpbuf, IMQENV2);
    if (_access(tmpbuf, 00) == 0) {
        SourceFile(tmpbuf);
    }
    */

    /* Determine imqvarhome. This can be set via an environment variable */
    if (me->imqvarhome[0] == '\0') {
        if ((p = getenv("IMQ_VARHOME")) == NULL) {
            if ((p = getenv("IMQ_DEFAULT_VARHOME")) != NULL) {
                /* IMQ_DEFAULT_VARHOME may be set by imqenv.conf */
                strcpy(me->imqvarhome, p);
            } else {
                strcpy(me->imqvarhome, me->imqhome);
                strcat(me->imqvarhome, "\\var");
            }
        } else {
            strcpy(me->imqvarhome, p);
        }
    }

    /*
    printf("IMQ_HOME: %s\n", me->imqhome);
    printf("IMQ_VARHOME: %s\n", me->imqvarhome);
    */

    return 0;
}


/*
 * Do some initialization. This basically locates IMQ, the JDK, determines
 * the run classpath (for -cp later).
 *
 * The following fields in the MqEnv are updated.
 *  imqhome
 *  imqlibhome
 *  imqvarhome
 *  jrehome
 *  classpath
 *
 * All the fields above need to point to pre-allocated space.
 *
 * Params:
 *  me			ptr to MqEnv structure, must not be NULL.
 *  classpath_entries	Array containing entries relative to IMQ_HOME/lib that
 *			need to be added to the run classpath. This must not
 *			be NULL.
 *  nclasspath_entries  Number of classpath entries in classpath_entries.
 *  append_libjars	Boolean flag - if true the jars/zips in IMQ_HOME/lib/ext
 *			will be appended to the run classpath.
 *  append_classpath	Boolean flag - if true the value of CLASSPATH will
 *			be appended to the run classpath.
 */
int MqAppInitialize(MqEnv *me, char *classpath_entries[], int nclasspath_entries,
			BOOL append_libjars, BOOL append_classpath)
{
    char *p;
    int n;

    /*
     * Initialize the MQ related fields
     */
    MqAppInitializeNoJava(me);

    if (*(me->jrehome) == '\0') {
        /* Locate a java runtime. */
        FindJavaRuntime(me->imqhome, me->imqvarhome, me->jrehome);
    }

    if (*(me->jrehome) == '\0') {
        fprintf(stderr, "Please specify a Java runtime using the IMQ_JAVAHOME\nenvironment variable, or -javahome command line option\n");
        return -1;
    }

    if (_access(me->jrehome, 00) < 0) {
        fprintf(stderr, "Invalid Java Runtime '%s'", me->jrehome);
        return -1;
    }

    /* Initialize classpath */
    *(me->classpath) = '\0';
    for (n = 0; n < nclasspath_entries; n++) {
        if (n != 0) strcat(me->classpath, ";");
        if (*classpath_entries[n] != '\\') {
            strcat(me->classpath, me->imqhome);
            strcat(me->classpath, "\\lib\\");
        }
        strcat(me->classpath, classpath_entries[n]);
    }

    if (append_libjars)  {
        MqAppAddLibJars(me->imqlibhome, me->classpath);
    }

    if (append_classpath)  {
        /*
         * If CLASSPATH is set, append it to the value of 'classpath'
         */
        if ((p = getenv("CLASSPATH")) != NULL) {
            strcat(me->classpath, ";");
            strcat(me->classpath, p);
        }
    }

    /*
    printf("IMQ_HOME: %s\n", me->imqhome);
    printf("IMQ_VARHOME: %s\n", me->imqvarhome);
    printf("jrehome: %s\n", me->jrehome);
    */

    return 0;
}

/*
 * Parse generic application arguments, setting the relevant
 * fields in the MqEnv structure.
 *
 * The following fields in the MqEnv are updated.
 *  jrehome
 *  application_argv
 *  application_argc
 *
 * The application_argv field needs to point to pre-allocated
 * array of (char *) pointers.
 *
 * Params:
 *  me	ptr to MqEnv structure, must not be NULL.
 *  argv argument vector
 *  argc number of elements in argv
 */
void MqAppParseArgs (MqEnv *me, char *argv[], int argc)
{
    argv++; argc--;
    while (argc > 0) {
        if (strcmp(*argv, "-javahome") == 0) {
            argv++; argc--;
            if (argc > 0) {
                strncpy(me->jrehome, *argv, sizeof(me->jrehome) - 32);
                strcat(me->jrehome, "\\jre");
            }
        } else if (strcmp(*argv, "-jrehome") == 0) {
            argv++; argc--;
            if (argc > 0) {
                strncpy(me->jrehome, *argv, sizeof(me->jrehome));
            }
        } else if (strcmp(*argv, "-varhome") == 0) {
            argv++; argc--;
            if (argc > 0) {
                strncpy(me->imqvarhome, *argv, sizeof(me->imqvarhome));
            }
        } else {
            /* We don't recognize the option, pass it on to application */
            me->application_argv[me->application_argc] = _strdup(*argv);
            me->application_argc++;
        }
        argv++; argc--;
    }
}

/*
 * Create the java command line based on information
 * in the MqEnv structure.
 *
 * No fields in the MqEnv are updated.
 *
 * Params:
 *  me		ptr to MqEnv structure, must not be NULL.
 *  set_varhome	Boolean flag that indicates whether "-Dimq.varhome"
 *		should be appended to the command line or not.
 *  cmdLine	buffer for storing command line. This
 *		points to pre allocated space.
 */
void MqAppCreateJavaCmdLine(MqEnv *me, BOOL set_varhome, char *cmdLine)
{
    char javaCmd[512];
    int   jvm_argc = 0, i;
    char *jvm_argv_buffer[128];
    char **jvm_argv = &(jvm_argv_buffer[1]);

    jvm_argv[jvm_argc] = malloc(sizeof(me->classpath) + 80);
    sprintf(jvm_argv[jvm_argc], "-cp");
    jvm_argc++;

    jvm_argv[jvm_argc] = malloc(sizeof(me->classpath) + 80);
    sprintf(jvm_argv[jvm_argc], "%s", me->classpath);
    jvm_argc++;

    jvm_argv[jvm_argc] = malloc(sizeof(me->imqhome) + 80);
    sprintf(jvm_argv[jvm_argc], "-Dimq.home=%s", me->imqhome);
    jvm_argc++;

    if (set_varhome)  {
        jvm_argv[jvm_argc] = malloc(sizeof(me->imqvarhome) + 80);
        sprintf(jvm_argv[jvm_argc], "-Dimq.varhome=%s", me->imqvarhome);
        jvm_argc++;
    }

    sprintf(javaCmd, "\"%s\\bin\\java\"", me->jrehome);

    /* Copy Java command and command line arguments into command line */
    strcpy(cmdLine, javaCmd);
    for (i = 0; i < jvm_argc; i++) {
        strcat(cmdLine, " ");
        strcat(cmdLine, "\"");
        strcat(cmdLine, jvm_argv[i]);
        strcat(cmdLine, "\"");
    }

    /* Copy main class into command line */
    strcat(cmdLine, " ");
    strcat(cmdLine, me->main_class);
    strcat(cmdLine, " ");

    /* Copy main class arguments into command line */
    for (i = 0; i < me->application_argc; i++) {
        strcat(cmdLine, " ");
        strcat(cmdLine, "\"");
        strcat(cmdLine, me->application_argv[i]);
        strcat(cmdLine, "\"");
    }
}

/*
 * Run the command, wait for it to exit,
 * and return the exit code.
 * A process is forked to run the command via
 * CreateProcess()
 *
 * Params:
 *  cmdLine	buffer for storing command line. This
 *		points to pre allocated space.
 */
DWORD MqAppRunCmd(char *cmdLine)
{
    /* Information about the forked child process */
    STARTUPINFO jsi;
    PROCESS_INFORMATION jpi;
    /* exit code */
    DWORD exitCode = 0;

    ZeroMemory(&jsi, sizeof(jsi));
    jsi.cb = sizeof(jsi);
    ZeroMemory(&jpi, sizeof(jpi));

    /*
     * Run a Java application and block until it exits.
     */
    /* Execute the child process */
    if ( !CreateProcess(
        NULL,               /* Use cmdLine */
        cmdLine,            /* Command line */
        NULL,               /* Prcess handle not inheritable. */
        NULL,               /* Thread handle not inheritable. */
        TRUE,               /* Set handle inheritance to TRUE. */
        0,                  /* No creation flags, use parent's console. */
        NULL,               /* Use parent's environment block */
        NULL,               /* Use parent's current directory */
        &jsi,                /* Pointer to STARTUPINFO struct [in] */
        &jpi                 /* Pointer to PROCESS_INFORMATION struct [out] */
        )
       ) {

        printf("Starting process failed. Could not execute %s", cmdLine);
    }

    /* Wait for child to exit */
    WaitForSingleObject (jpi.hProcess, INFINITE);

    /* Get exit code from child process */
    if (!GetExitCodeProcess(jpi.hProcess, &exitCode)) {
        fprintf(stderr, "Unable to get child's exit code.");
    }

    /* Close process and thread handles */
    CloseHandle(jpi.hProcess);
    CloseHandle(jpi.hThread);

    return (exitCode);
}

/*
 * Run the java command using the information in the
 * MqEnv structure passed in.
 * This function uses the VM invocation APIs in JNI.
 *
 * Params:
 *  me	ptr to MqEnv structure, must not be NULL.
 */
DWORD MqAppJNIRunJavaCmd(MqEnv *me)
{
    FARPROC		MqApp_JNI_CreateJavaVM = NULL;
    HINSTANCE		hinstLib;
    JNIEnv		*env;
    JavaVM		*jvm;
    JavaVMInitArgs	vm_args;
    JavaVMOption	options[4];
    jint		res;
    jclass		cls;
    jmethodID		mid;
    jstring		jstr;
    jobjectArray	args;
    DWORD 		exitCode = 0;
    char 		dllpath[MAX_PATH];
    char		*main_class, *tmp;
    int			i;

    /*
     * Try to load the VM's in the client_vm_libs array.
     * Stop at the first successful load.
     */
    for (i = 0; i < nclient_vm_libs; i++)
    {
        /*
         * Construct path to one jvm.dll in client_vm_libs array
         */
        dllpath[0] = '\0';
        strcpy(dllpath, me->jrehome);
        strcat(dllpath, "\\bin\\");
        strcat(dllpath, client_vm_libs[i]);

	if ((hinstLib = LoadLibrary(dllpath)) == NULL) {
	    continue;
	} else {
	    break;
	}
    }

    /*
     * If can't load any VM libs, print error msg and return.
     */
    if (hinstLib == NULL)
    {
	/*
	 * Couldn't load a VM
	 */
        fprintf(stderr, "Couldn't load any of the following Java VMs from %s:\n",
	        me->jrehome);
        for (i = 0; i < nclient_vm_libs; i++)
	{
            fprintf(stderr, "\t%s\n", client_vm_libs[i]);
        }

        return (1);
    }

    /*
     * Get function ptr to JNI_CreateJavaVM()
     */
    MqApp_JNI_CreateJavaVM = GetProcAddress(hinstLib, "JNI_CreateJavaVM");
    if (NULL == MqApp_JNI_CreateJavaVM)
    {
	fprintf(stderr, "Failed to obtain func ptr to JNI_CreateJavaVM() !\n");
        return (1);
    }


    /*
     * Setup for creating VM.
     */

    /*
     * Construct VM args
     */
    i = 0;
    options[i].optionString = malloc(sizeof(me->classpath) + 80);
    sprintf(options[i].optionString, "-Djava.class.path=%s", me->classpath);
    i++;

    options[i].optionString = malloc(sizeof(me->imqhome) + 80);
    sprintf(options[i].optionString, "-Dimq.home=%s", me->imqhome);
    i++;

    vm_args.version = JNI_VERSION_1_4;
    vm_args.options = options;
    vm_args.nOptions = i;
    vm_args.ignoreUnrecognized = JNI_TRUE;

    res = (*MqApp_JNI_CreateJavaVM)(&jvm, (void **)&env, &vm_args);
    if (res < 0) {
        fprintf(stderr, "Can't create Java VM\n");

        return (1);
    }

    /*
     * Convert main_class string in MqEnv to class string
     * with slashes instead of periods.
     */
    main_class = _strdup(me->main_class);
    tmp = main_class;
    while ((tmp = strchr(tmp, '.')) != NULL)  {
        *tmp = '/';
    }

    /*
     * Load main class.
     */
    cls = (*env)->FindClass(env, main_class);
    if (cls == NULL)
    {
	fprintf(stderr, "Failed to load class: %s\n", main_class);

	return (1);
    }

    /*
     * Get method ID for main method.
     */
    mid = (*env)->GetStaticMethodID(env, cls, "main", "([Ljava/lang/String;)V");
    if (mid == NULL)
    {
	fprintf(stderr, "Cannot find main method.\n");

	return (1);
	
    }

    /*
     * Construct main method (application) args.
     */
    if (me->application_argc > 0)  {
	args = (*env)->NewObjectArray(env, me->application_argc,
		    (*env)->FindClass(env, "java/lang/String"), NULL);
	if (args == NULL)  
	{
	    fprintf(stderr, "Failed to allocate args array.\n");
	    return (1);
	}

        for (i = 0; i < me->application_argc; i++) {
	    jstr = (*env)->NewStringUTF(env, me->application_argv[i]);
	    if (jstr == NULL)
	    {
	        fprintf(stderr, 
		    "Failed to allocate arg: %s\n", me->application_argv[i]);
	        return (1);
	    }

	    (*env)->SetObjectArrayElement(env, args, i, jstr);
        }
    } else  {
	args = (*env)->NewObjectArray(env, 0,
		    (*env)->FindClass(env, "java/lang/String"), NULL);
    }

    /*
     * Invoke main method.
     */
    (*env)->CallStaticVoidMethod(env, cls, mid, args);

    if ((*env)->ExceptionOccurred(env))
    {
	(*env)->ExceptionDescribe(env);
    }

    /*
     * Cleanup:
     *  - Detach the current thread so that it appears to have exited when
     *    the application's main method exits.
     *  - Destroy VM.
     */
    if ((*jvm)->DetachCurrentThread(jvm) != 0) {
        fprintf(stderr, "Could not detach main thread.\n");
	exitCode = 1;
    }
    (*jvm)->DestroyJavaVM(jvm);

    return (exitCode);
}

/*
 * Initialize MqEnv structure
 *
 * The following fields in the MqEnv are updated.
 *  imqhome
 *  imqlibhome
 *  imqvarhome
 *  jrehome
 *  classpath
 *  application_argc
 *  main_class
 *
 * All the string fields above need to point to pre-allocated space.
 *
 * Params:
 *  me		ptr to MqEnv structure, must not be NULL.
 *  main_class	string containing main class of application
 *	e.g. 
 *	"com.sun.messaging.jmq.admin.apps.console.AdminConsole"
 */
void MqAppInitMqEnv(MqEnv *me, char *main_class)
{
    me->imqhome[0] = '\0';
    me->imqlibhome[0] = '\0';
    me->imqvarhome[0] = '\0';
    me->jrehome[0] = '\0';
    me->classpath[0] = '\0';
    me->application_argc = 0;
    me->main_class = main_class;
}

