/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2000-2007 Sun Microsystems, Inc. All rights reserved. 
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License ("CDDL") (collectively, the "License").  You may
 * not use this file except in compliance with the License.  You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or mq/legal/LICENSE.txt.  See the License for the specific language
 * governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at mq/legal/LICENSE.txt.  Sun designates
 * this particular file as subject to the "Classpath" exception as provided by
 * Sun in the GPL Version 2 section of the License file that accompanied this
 * code.  If applicable, add the following below the License Header, with the
 * fields enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or  to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright holder. 
 */

/*
 * @(#)BrokerProcess.java	1.16 06/28/07
 */ 

package com.sun.messaging.jmq.jmsserver;

import java.util.*;
import com.sun.messaging.jmq.jmsserver.service.Service;
import com.sun.messaging.jmq.jmsserver.service.ServiceManager;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQService;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQDirectService;
import com.sun.messaging.jmq.jmsservice.JMSBroker;
import com.sun.messaging.jmq.jmsservice.JMSService;
import com.sun.messaging.jmq.jmsservice.BrokerEventListener;

/**
 * Wrapper used to start the broker. It wraps a singleton class
 * (only one broker can be running in any process).<P>
 *
 * <u>Example</u><P>
 * <code><PRE>
 *      BrokerProcess bp = BrokerProcess.getBrokerProcess();
 *      try {
 *      
 *          Properties ht = BrokerProcess.convertArgs(args);
 *          int exitcode = bp.start(true, ht, null);
 *          System.out.println("Broker exited with " + exitcode);
 *
 *      } catch (IllegalArgumentException ex) {
 *          System.err.println("Bad Argument " + ex.getMessage());
 *          System.out.println(BrokerProcess.usage());
 *      }
 * </PRE></code>
 */
public class BrokerProcess implements JMSBroker
{
    private static final String	DEFAULT_DIRECTMODE_SERVICE_NAME = "jmsdirect";

    Broker b = null;
    private static BrokerProcess bp = null;

    /**
     * Constructor
     */
    public BrokerProcess() {
        b = Broker.getBroker();
    }

    /**
     * Retrieve the singleton BrokerProcess object
     */
    public static BrokerProcess getBrokerProcess()
    {
        synchronized (BrokerProcess.class) {
            if (bp == null)
                bp = new BrokerProcess();
        }

        return bp;
    }

    /**
     * Change command line args into a hashtable format
     *<P>
     * Additional arguments are:
     *    <UL>
     *       <LI> -varhome</LI>
     *       <LI> -imqhome</LI>
     *    </UL>
     *
     * @param args arguments in broker format
     */
    public static Properties convertArgs(String[] args)
        throws IllegalArgumentException
    {
        Properties props = new Properties();

        // first look for var home and the like
        for (int i =0; i < args.length; i ++) {
            String arg = args[i];
            if (arg.equals("-varhome")) {
                props.setProperty("imq.varhome",
                        args[i+1]);
                i ++;
            } else if (arg.equals("-imqhome")) {
                props.setProperty("imq.home",
                        args[i+1]);
                i ++;
            } else if (arg.equals("-libhome")) {
                props.setProperty("imq.libhome",
                        args[i+1]);
                i ++;
            }
        }
        Globals.pathinit(props);
        return getBrokerProcess().b.convertArgs(args);
    }

    public Properties parseArgs(String[] args)
        throws IllegalArgumentException
    {
        return (convertArgs(args));
    }

    /**
     * Usage string
     *
     */
    public static String usage()
    {
        return getBrokerProcess().b.usage();
    }

    /**
     * Checks the state of the Broker
     *
     * @return the state of the broker
     */
    public boolean isRunning() {
        return true;
    }

    /**
     * Start the broker (only one broker can be running in a given
     * vm).<p>This call returns as soon as the broker sucessfully starts.
     * @param inProcess - indicates that the broker is running inprocess
     *                    and the shutdown hook and memory management
     *                    code should not be used.
     * @param properties - configuration setttings for the broker
     *
     * @param bn - optional class to notify when a broker has completed
     *             starting or has been shutdown.
     *
     * @return the exit code what would be returned by the broker if it
     *       was running as a standalone process. (or 0 if it sucessfully
     *       started).
     *
     * @throws OutOfMemoryError - if the broker can not allocate enough 
     *          memory to continue running
     * @throws IllegalStateException - the broker is already running.  
     * @throws IllegalArgumentException - an invalid value for a property
     *                was passed on the command line
     */
    public int start(boolean inProcess, Properties properties, BrokerNotification bn)
        throws OutOfMemoryError, IllegalStateException, IllegalArgumentException
    {
        return getBrokerProcess().b.start(inProcess, properties, bn);
    }

    public int start(boolean inProcess, Properties properties, BrokerEventListener bel)
        throws OutOfMemoryError, IllegalStateException, IllegalArgumentException
    {
        return getBrokerProcess().b.start(inProcess, properties, bel);
    }
    /**
     * init the broker (only one broker can be running in a given
     * vm).<p>This call returns as soon as the broker sucessfully initializes.
     * Equivalent to starting the broker with -init.
     * @param inProcess - indicates that the broker is running inprocess
     *                    and the shutdown hook and memory management
     *                    code should not be used.
     * @param properties - configuration setttings for the broker
     *
     * @param bn - optional class to notify when a broker has completed
     *             starting or has been shutdown.
     *
     * @return the exit code what would be returned by the broker if it
     *       was running as a standalone process. (or 0 if it sucessfully
     *       started).
     *
     * @throws OutOfMemoryError - if the broker can not allocate enough 
     *          memory to continue running
     * @throws IllegalStateException - the broker is already running.  
     * @throws IllegalArgumentException - an invalid value for a property
     *                was passed on the command line
     */
    public int init(boolean inProcess, Properties properties, BrokerNotification bn)
        throws OutOfMemoryError, IllegalStateException, IllegalArgumentException
    {
        return getBrokerProcess().b.init(inProcess, properties, bn);
    }

    public int init(boolean inProcess, Properties properties, BrokerEventListener bel)
        throws OutOfMemoryError, IllegalStateException, IllegalArgumentException
    {
        return getBrokerProcess().b.init(inProcess, properties, bel);
    }
    /**
     * Stop the broker (only one broker can be running in a given
     * vm).<p>
     * @param cleanup - if false, the code does not need to worry about freeing
     *                  unused resources. (broker is about to exit)
     * @throws IllegalStateException - the broker is already stopped.  
     */
    public void stop(boolean cleanup)
        throws IllegalStateException
    {

        b.destroyBroker(cleanup);
        b = null;
        bp = null;
    }

    /**
     *  Return the default JMS Service that supports 'DIRECT' in-JVM Java EE JMS
     *  clients.
     *
     *  @throws IllegalStateException if the broker is already stopped
     * 
     */
    public JMSService getJMSService() 
			throws IllegalStateException  {
	ServiceManager sm = Globals.getServiceManager();
	JMSService jmsService = getJMSService(DEFAULT_DIRECTMODE_SERVICE_NAME);

	if (jmsService != null)  {
	    return (jmsService);
	}

	/*
	 * If "jmsdirect" is not available, loop through all services
	 */
	List serviceNames = sm.getAllServiceNames();
	Iterator iter = serviceNames.iterator();

	while (iter.hasNext())  {
	    jmsService = getJMSService((String)iter.next());

	    if (jmsService != null)  {
	        return (jmsService);
	    }
	}

	return (null);
    }

    /**
     *  Return the named JMS Service that supports 'DIRECT' in-JVM Java EEJMS
     *  clients.
     *
     *  @param  serviceName The name of the service to return
     *
     *  @throws IllegalStateException if the broker is already stopped
     */
    public JMSService getJMSService(String serviceName) 
				throws IllegalStateException  {
	ServiceManager sm = Globals.getServiceManager();
	Service svc;
	IMQService imqSvc;
	IMQDirectService imqDirectSvc;

	if (sm == null)  {
	    return (null);
	}

	svc = sm.getService(serviceName);

	if (svc == null)  {
	    return (null);
	}

	if (!(svc instanceof IMQService))  {
	    return (null);
	}

	imqSvc = (IMQService)svc;

	if (!imqSvc.isDirect())  {
	    return (null);
	}

	if (!(imqSvc instanceof IMQDirectService))  {
	    return (null);
	}

	imqDirectSvc = (IMQDirectService)imqSvc;

	return ((JMSService)imqDirectSvc);
    }
    

    public static void main(String args[]) {
try {

        BrokerProcess bp = getBrokerProcess();

        BrokerNotification bn = new BrokerNotification()
            {
                public boolean exitRequested(int reason,
                   String info, Throwable thr) {
System.out.println("EXIT REQUESTED " + reason);
                      return false;
                }

                public void brokerShutdown() {
                   System.err.println("XXX should never be called");
                }

                public void brokerStarted() {
                   System.err.println("XXX hey broker started");
                }
            };

        String newargs[] = new String[args.length + 4];
        int i =0;
        for (i =0; i < args.length; i ++)
            newargs[i] = args[i];
        newargs[i++] = "-varhome";
        newargs[i++] = "../../../solaris/opt/var";
        newargs[i++] = "-imqhome";
        newargs[i++] = "../../../solaris/opt/";
       
        Properties p =bp.convertArgs(newargs);

System.err.println("HEY INIT");
        int exit = bp.init(true, p, bn);
System.err.println("HEY START");
        exit = bp.start(true, p, bn);
System.err.println("HEY STARTED");
        Thread.currentThread().sleep(10*1000);
        System.out.println("HEY SHUTTING DOWN");
        bp.stop(true);
        System.out.println("Sleeping");
        Thread.currentThread().sleep(1000*10);
        System.out.println("Restarting");
        exit = bp.start(true, p, (BrokerNotification)null);
        System.out.println("Sleeping");
        Thread.currentThread().sleep(10*1000);
        bp.stop(true);
        System.out.println("HEY SHUTTING DOWN");
        System.out.println("Sleeping");
        Thread.currentThread().sleep(1000*60);
        System.exit(0);
} catch (Exception ex) {
ex.printStackTrace();
}

    }
}


