/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2000-2007 Sun Microsystems, Inc. All rights reserved. 
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License ("CDDL") (collectively, the "License").  You may
 * not use this file except in compliance with the License.  You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or mq/legal/LICENSE.txt.  See the License for the specific language
 * governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at mq/legal/LICENSE.txt.  Sun designates
 * this particular file as subject to the "Classpath" exception as provided by
 * Sun in the GPL Version 2 section of the License file that accompanied this
 * code.  If applicable, add the following below the License Header, with the
 * fields enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or  to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright holder. 
 */

/*
 * @(#)AdminDataHandler.java	1.50 06/28/07
 */ 

package com.sun.messaging.jmq.jmsserver.data.handlers.admin;

import java.util.Properties;
import java.util.Hashtable;
import java.io.IOException;
import java.net.InetAddress;

import com.sun.messaging.jmq.jmsserver.Broker;
import com.sun.messaging.jmq.jmsserver.BrokerStateHandler;
import com.sun.messaging.jmq.jmsserver.data.PacketRouter;
import com.sun.messaging.jmq.jmsserver.data.TransactionList;
import com.sun.messaging.jmq.jmsserver.net.ProtocolStreams;
import com.sun.messaging.jmq.jmsserver.data.handlers.DataHandler;
import com.sun.messaging.jmq.jmsserver.util.BrokerException;
import com.sun.messaging.jmq.io.Packet;
import com.sun.messaging.jmq.io.PacketType;
import com.sun.messaging.jmq.io.Status;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQConnection;
import com.sun.messaging.jmq.jmsserver.service.imq.IMQIPConnection;
import com.sun.messaging.jmq.jmsserver.resources.BrokerResources;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.util.log.Logger;
import com.sun.messaging.jmq.util.admin.MessageType;


/**
 * Handler class that deals with JMS administration messages.
 * We first check if a message is an administartion message. If it is
 * we intercept it and process it via the AdminCmdHandlers. If it is not
 * then we pass it to the DataHandler for standard handling.
 */
public class AdminDataHandler extends DataHandler
{

    private static boolean DEBUG = getDEBUG();

    private AdminCmdHandler[] handlers = null;

    private BrokerResources rb = Globals.getBrokerResources();

    protected TransactionList tl = null;

    // Need this so that admin message handlers can locate standard JMS
    // message handlers.
    public PacketRouter adminPktRtr = null;

    private int activeHandlers = 0;

    public AdminDataHandler(
                 TransactionList list) 
    {
	super(list);
        tl = list;
	initHandlers();
    }

    public void setPacketRouter(PacketRouter pktrtr) {
        adminPktRtr = pktrtr;
    }

    /**
     * Method to handle administration messages
     */
    public boolean handle(IMQConnection con, Packet msg) 
        throws BrokerException
    {
        if (DEBUG) {
            logger.log(Logger.DEBUGMED, "AdminDataHandler: handle() [ Received JMS Admin Message] {0} ", msg.toString());
	    if (logger.level >= Logger.DEBUGHIGH) {
	        msg.dump(System.out);
	    }
        }


	String  dest = msg.getDestination();

	// Administration messages are sent to a well known 
	// administration Queue
	if (!msg.getIsQueue() || !dest.equals(MessageType.JMQ_ADMIN_DEST)) {
	    // Normal message. Let standard JMS data handler deal with it.
	    return super.handle(con, msg);
	}

        // send the reply (if necessary)
        if (msg.getSendAcknowledge()) {
             Packet pkt = new Packet(con.useDirectBuffers());
             pkt.setPacketType(PacketType.SEND_REPLY);
             pkt.setConsumerID(msg.getConsumerID());
             Hashtable hash = new Hashtable();
             hash.put("JMQStatus", new Integer(Status.OK));
             pkt.setProperties(hash);
             con.sendControlMessage(pkt);
        }

	// Administrative message. Process it.
	// Get message type property
	Hashtable props = null;
	Integer msgType = null;
	try {
	    props = msg.getProperties();
	    msgType = (Integer)props.get(MessageType.JMQ_MESSAGE_TYPE);
	} catch (IOException e) {
	    // Programming error. No need to I18N
	    logger.log(Logger.WARNING, rb.E_INTERNAL_BROKER_ERROR,
                "Admin: Could not extract properties from pkt", e);
	} catch (ClassNotFoundException e) {
	    logger.log(Logger.WARNING, rb.E_INTERNAL_BROKER_ERROR,
                "Admin: Could not extract properties from pkt", e);
	}

	if (props == null || msgType == null) { 
	    // Programming error. No need to I18N
	    logger.log(Logger.WARNING, rb.E_INTERNAL_BROKER_ERROR,
	        "Message received on administration destination " +
		MessageType.JMQ_ADMIN_DEST +
                " has no " +
		MessageType.JMQ_MESSAGE_TYPE +
                " property ignoring it.");
	    return true;
	}

	// Get AdminCmdHandler for this message type
	int t = msgType.intValue();
	AdminCmdHandler ach = null;

        /* If the connection is authenticated using admin key authentication
         * then it is considered "restricted" and can only perform minimal
         * operations. Anything else is forbidden.
         */
        if (con.getAccessController().isRestrictedAdmin() &&
            t != MessageType.SHUTDOWN &&
	    t != MessageType.HELLO &&
	    t != MessageType.RESTART) {

            logger.log(Logger.WARNING, BrokerResources.W_FORBIDDEN_ADMIN_OP,
				       MessageType.getString(t));

	    Packet reply = new Packet(con.useDirectBuffers());
	    reply.setPacketType(PacketType.OBJECT_MESSAGE);

	    // By convention reply message is the message type + 1
	    AdminCmdHandler.setProperties(reply,
                     t + 1, Status.FORBIDDEN, null);

	    sendReply(con, msg, reply);

            return true; // done
        }

        // if we arent shutdown .. track our handler cnt
        if ( t != MessageType.SHUTDOWN)
            incrementActiveHandlers();

        try {
            if (BrokerStateHandler.shuttingDown) {
                 
                String message = Globals.getBrokerResources().getKString(
                            BrokerResources.I_ADMIN_BKR_SHUTTINGDOWN,
                     MessageType.getString(t));

                logger.log(Logger.WARNING, 
                     message);
    
	        Packet reply = new Packet(con.useDirectBuffers());
	        reply.setPacketType(PacketType.OBJECT_MESSAGE);
    
	        // By convention reply message is the message type + 1
	        AdminCmdHandler.setProperties(reply,
                         t + 1, Status.UNAVAILABLE, message);
    
	        sendReply(con, msg, reply);
    
                return true; // done
            }
            if (!Broker.getBroker().startupComplete) {
                String message = Globals.getBrokerResources().getKString(
                            BrokerResources.I_ADMIN_BKR_NOT_READY,
                     MessageType.getString(t));

                logger.log(Logger.WARNING, 
                     message);

	        Packet reply = new Packet(con.useDirectBuffers());
	        reply.setPacketType(PacketType.OBJECT_MESSAGE);
	        // By convention reply message is the message type + 1
	        AdminCmdHandler.setProperties(reply,
                         t + 1, Status.UNAVAILABLE, message);
    
	        sendReply(con, msg, reply);
    
                return true; // done
            }
    
	    try {
	        ach = handlers[t];
	    } catch (IndexOutOfBoundsException e) {
	        logger.log(Logger.ERROR, BrokerResources.E_INTERNAL_BROKER_ERROR,
                     "Bad " + MessageType.JMQ_MESSAGE_TYPE +
		       ": " + t);
	        return true;
            }
	    if (ach == null) {
	        logger.log(Logger.ERROR,BrokerResources.E_INTERNAL_BROKER_ERROR,
			"No administration handler found for message type " +
			    msgType + ". Ignoring.");
	        return true;
	    } else {
	        // Call command handler  to handle message
	        return ach.handle(con, msg, props);
	    }
	} finally {
            if ( t != MessageType.SHUTDOWN)
                decrementActiveHandlers();
        }

    }

    /**
     * Initialize the array of AdminCmdHandlers. Each administration 
     * message has its own handler.
     */
    private void initHandlers() {

	// This may be a sparse array 
	handlers = new AdminCmdHandler[MessageType.LAST];

	// Initialize. Base class (AdminCmdHandler) does a default no-op
	// implementation of a handler.
        handlers[MessageType.CREATE_DESTINATION] =
                                new CreateDestinationHandler(this);
        handlers[MessageType.DESTROY_CONNECTION] =
                                new DestroyConnectionsHandler(this);
        handlers[MessageType.DESTROY_DESTINATION] =
                                new DestroyDestinationHandler(this);
        handlers[MessageType.DESTROY_DURABLE] =
                                new DestroyDurableHandler(this);
        handlers[MessageType.GET_CONNECTIONS] =
                                new GetConnectionsHandler(this);
        handlers[MessageType.GET_CONSUMERS] =
                                new GetConsumersHandler(this);
        handlers[MessageType.GET_DESTINATIONS] =
                                new GetDestinationsHandler(this);
        handlers[MessageType.GET_DURABLES] =
                                new GetDurablesHandler(this);
        handlers[MessageType.GET_LOGS] =
                                new GetLogsHandler(this);
        handlers[MessageType.GET_SERVICES] =
                                new GetServicesHandler(this);
        handlers[MessageType.HELLO] =
                                new HelloHandler(this);
        handlers[MessageType.PAUSE] =
                                new PauseHandler(this);
        handlers[MessageType.PURGE_DESTINATION] =
                                new PurgeDestinationHandler(this);
        handlers[MessageType.RESUME] =
                                new ResumeHandler(this);
        handlers[MessageType.SHUTDOWN] =
                                new ShutdownHandler(this);
        handlers[MessageType.UPDATE_DESTINATION] =
                                new UpdateDestinationHandler(this);
        handlers[MessageType.UPDATE_SERVICE] =
                                new UpdateServiceHandler(this);
        handlers[MessageType.VIEW_LOG] =
                                new ViewLogHandler(this);
        handlers[MessageType.GET_METRICS] =
                                new GetMetricsHandler(this);
        handlers[MessageType.GET_BROKER_PROPS] =
                                new GetBrokerPropsHandler(this);
        handlers[MessageType.UPDATE_BROKER_PROPS] =
                                new UpdateBrokerPropsHandler(this);
        handlers[MessageType.RELOAD_CLUSTER] =
                                new ReloadClusterHandler(this);
        handlers[MessageType.GET_TRANSACTIONS] =
                                new GetTransactionsHandler(this);
        handlers[MessageType.ROLLBACK_TRANSACTION] =
                                new RollbackCommitHandler(this);
        handlers[MessageType.COMMIT_TRANSACTION] =
                                new RollbackCommitHandler(this);
        handlers[MessageType.PURGE_DURABLE] =
                                new PurgeDurableHandler(this);
        handlers[MessageType.COMPACT_DESTINATION] =
                                new CompactDestinationHandler(this);
        handlers[MessageType.DEBUG] =
                                new DebugHandler(this);
        handlers[MessageType.QUIESCE_BROKER] =
                                new QuiesceHandler(this);
        handlers[MessageType.TAKEOVER_BROKER] =
                                new TakeoverHandler(this);
        handlers[MessageType.GET_CLUSTER] =
                                new GetClusterHandler(this);
        handlers[MessageType.GET_JMX] =
                                new GetJMXConnectorsHandler(this);
        handlers[MessageType.UNQUIESCE_BROKER] =
                                new UnquiesceHandler(this);
        handlers[MessageType.RESET_BROKER] =
                                new ResetMetricsHandler(this);
        handlers[MessageType.GET_MESSAGES] =
                                new GetMessagesHandler(this);
        handlers[MessageType.DELETE_MESSAGE] =
                                new DeleteMessageHandler(this);
        handlers[MessageType.REPLACE_MESSAGE] =
                                new ReplaceMessageHandler(this);
    }


    /**
     * Send a reply to an administration command message.
     *
     * @param con	Connection cmd_msg came in on
     * @param cmd_msg	Administrative command message from client
     * @param reply_msg Broker's reply to cmd_msg.
     */
    public void sendReply(IMQConnection con, Packet cmd_msg,
	Packet reply_msg) 

    {
	// We send message to the ReplyTo destination
	String destination = cmd_msg.getReplyTo();
	if (destination == null) {
	    // Programming error. No need to I18N
	    logger.log(Logger.ERROR, rb.E_INTERNAL_BROKER_ERROR,
	    "Administration message has no ReplyTo destination. Not replying.");
	    return;
        }

	reply_msg.setDestination(destination);
	reply_msg.setIsQueue(true);

	// Make sure message is not persistent, in a transaction, or ACK'd
	reply_msg.setPersistent(false);
	reply_msg.setTransactionID(0);
	reply_msg.setSendAcknowledge(false);

	// Set port number and IP of packet
	if (con instanceof IMQIPConnection) {
	    reply_msg.setPort(((IMQIPConnection)con).getLocalPort());
	}

	reply_msg.setIP(Globals.getBrokerInetAddress().getAddress());

	// Set sequence number and timestamp on packet. This is typically
	// done by writePacket(), but since we're simulating an incomming
	// packet we must to it now.
	reply_msg.updateTimestamp();
	reply_msg.updateSequenceNumber();

	// Turn off auto-generation of timestamp and sequence for this packet
	// This ensures the values we just set are never overwritten.
	reply_msg.generateTimestamp(false);
	reply_msg.generateSequenceNumber(false);

	if ( DEBUG ) {
	    try {
	        logger.log(Logger.DEBUG, "AdminDataHandler: REPLY: " +
		reply_msg + ": " + reply_msg.getProperties());
	    } catch (IOException e) {
	        // Programming error. No need to I18N
	        logger.log(Logger.WARNING, rb.E_INTERNAL_BROKER_ERROR,
                    "Admin: Could not extract properties from pkt", e);
	    } catch (ClassNotFoundException e) {
	        logger.log(Logger.WARNING, rb.E_INTERNAL_BROKER_ERROR,
                    "Admin: Could not extract properties from pkt", e);
	    }
	}

	// DataHandler's handle method will treat this as a message being
	// sent by a client.
	// XXX REVISIT 08/30/00 dipol: can 'con' be null to indicate
	// message is originating from an internal source?
	try {
	    super.handle(con, reply_msg, true /* we are admin */);
	} catch (BrokerException e) {
	    logger.log(Logger.ERROR, rb.E_INTERNAL_BROKER_ERROR,
	        "Could not reply to administrative message.", e);
	}
    }


    private void incrementActiveHandlers()
    {
        synchronized (this) {
            activeHandlers ++;
        }
    }
    public void decrementActiveHandlers() {
        synchronized (this) {
            activeHandlers --;
            if (activeHandlers == 0) {
                this.notifyAll();
            }
        }
    }
    public void waitForHandlersToComplete(int secs)
    {
        // if we have more than 0 active handlers wait
        // the seconds passed in
        synchronized (this) {
            try {
                if (activeHandlers > 0)
                     wait(secs*1000); 
            } catch (Exception ex) {}
        }
    }


}
