/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2000-2007 Sun Microsystems, Inc. All rights reserved. 
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License ("CDDL") (collectively, the "License").  You may
 * not use this file except in compliance with the License.  You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or mq/legal/LICENSE.txt.  See the License for the specific language
 * governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at mq/legal/LICENSE.txt.  Sun designates
 * this particular file as subject to the "Classpath" exception as provided by
 * Sun in the GPL Version 2 section of the License file that accompanied this
 * code.  If applicable, add the following below the License Header, with the
 * fields enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or  to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright holder. 
 */

/*
 * @(#)CallbackDispatcher.java	1.33 07/23/07
 */ 

package com.sun.messaging.jmq.jmsserver.multibroker;

import java.util.*;
import com.sun.messaging.jmq.io.*;
import com.sun.messaging.jmq.jmsserver.core.PacketReference;
import com.sun.messaging.jmq.jmsserver.core.ConsumerUID;
import com.sun.messaging.jmq.jmsserver.core.Consumer;
import com.sun.messaging.jmq.jmsserver.core.Subscription;
import com.sun.messaging.jmq.jmsserver.core.Destination;
import com.sun.messaging.jmq.jmsserver.core.DestinationUID;
import com.sun.messaging.jmq.jmsserver.core.BrokerAddress;
import com.sun.messaging.jmq.jmsserver.service.ConnectionUID;
import com.sun.messaging.jmq.jmsserver.Globals;
import com.sun.messaging.jmq.jmsserver.multibroker.raptor.ClusterGoodbyeInfo;
import com.sun.messaging.jmq.jmsserver.multibroker.raptor.ProtocolGlobals;
import com.sun.messaging.jmq.util.log.*;
import com.sun.messaging.jmq.jmsserver.resources.*;

/**
 * This class schedules the MessageBusCallback notification
 * invokations.
 */
public class CallbackDispatcher extends Thread {

    private static boolean DEBUG = false; 
    protected static final Logger logger = Globals.getLogger();
    private MessageBusCallback cb = null;
    private LinkedList eventQ = null;
    private boolean stopThread = false;

    public CallbackDispatcher(MessageBusCallback cb) {
        this.cb = cb;
        eventQ = new LinkedList();
        setName("MessageBusCallbackDispatcher");
        setDaemon(true);
        start();
    }

    /**
     * Initial sync with the config server is complete.
     * We are now ready to accept connections from clients.
     */
    public void configSyncComplete() {
        CallbackEvent cbe = new ConfigSyncCompleteCallbackEvent();

        synchronized (eventQ) {
            eventQ.add(cbe);
            eventQ.notify();
        }
    }

    public void processGPacket(BrokerAddress sender, GPacket pkt, Protocol p) {
        CallbackEvent cbe =
            new GPacketCallbackEvent(sender, pkt, p);

        synchronized (eventQ) {
            eventQ.add(cbe);
            eventQ.notify();
        }
    }

    public void processGoodbye(BrokerAddress sender, ClusterGoodbyeInfo cgi) {
        CallbackEvent ev = null;
        synchronized (eventQ) {
            Iterator itr = eventQ.iterator();
            while (itr.hasNext()) {
                ev = (CallbackEvent)itr.next();
                if (!(ev instanceof GPacketCallbackEvent)) continue;
                GPacketCallbackEvent gpev = (GPacketCallbackEvent)ev;
                if (!gpev.getSender().equals(sender)) continue; //XXXbrokerSessionUID
                if (gpev.getEventType() != ProtocolGlobals.G_MESSAGE_ACK) continue;
                logger.log(logger.WARNING, "Discard unprocessed G_MESSAGE_ACK "
                                           + " because received GOODBYE from " +sender);
                itr.remove();
            }
        }
    }

    public void processGoodbyeReply(BrokerAddress sender) {
        CallbackEvent ev = null;
        synchronized (eventQ) {
            Iterator itr = eventQ.iterator();
            while (itr.hasNext()) {
                ev = (CallbackEvent)itr.next();
                if (!(ev instanceof GPacketCallbackEvent)) continue;
                GPacketCallbackEvent gpev = (GPacketCallbackEvent)ev;
                if (!gpev.getSender().equals(sender)) continue; //XXXbrokerSessionUID
                if (gpev.getEventType() != ProtocolGlobals.G_MESSAGE_ACK_REPLY) continue;
                if (DEBUG) {
                logger.log(logger.INFO, "Processed G_MESSAGE_ACK_REPLY from " +sender);
                }
                gpev.dispatch(cb);
                itr.remove();
            }
        }
    }

    /**
     * Interest creation notification. This method is called when
     * any local / remote interest is created.
     */
    public void interestCreated(Consumer intr) {
        CallbackEvent cbe = new InterestCreatedCallbackEvent(intr);

        synchronized (eventQ) {
            eventQ.add(cbe);
            eventQ.notify();
        }
    }

    /**
     * Interest removal notification. This method is called when
     * any local / remote interest is removed. USED by Falcon only
     */
    public void interestRemoved(Consumer intr, Set pendingMsgs, boolean cleanup) {
        CallbackEvent cbe = new InterestRemovedCallbackEvent(intr, pendingMsgs, cleanup);

        synchronized (eventQ) {
            eventQ.add(cbe);
            eventQ.notify();
        }
    }

    public void activeStateChanged(ConsumerUID intid) {
        CallbackEvent cbe = new PrimaryInterestChangedCallbackEvent(intid);

        synchronized (eventQ) {
            eventQ.add(cbe);
            eventQ.notify();
        }
    }

    /**
     * Primary interest change notification. This method is called when
     * a new interest is chosen as primary interest for a failover queue.
     * USED by Falcon only
     */
    public void activeStateChanged(Consumer intr) {
        CallbackEvent cbe = new PrimaryInterestChangedCallbackEvent(intr);

        synchronized (eventQ) {
            eventQ.add(cbe);
            eventQ.notify();
        }
    }

    /**
     * Client down notification. This method is called when a local
     * or remote client connection is closed.
     */
    public void clientDown(ConnectionUID conid) {
        CallbackEvent cbe = new ClientDownCallbackEvent(conid);

        synchronized (eventQ) {
            eventQ.add(cbe);
            eventQ.notify();
        }
    }

    /**
     * Broker down notification. This method is called when any broker
     * in this cluster goes down.
     */
    public void brokerDown(BrokerAddress broker) {
        CallbackEvent e = null;
        synchronized (eventQ) {
            Iterator itr = eventQ.iterator();
            while (itr.hasNext()) {
                e = (CallbackEvent)itr.next();
                if (!(e instanceof GPacketCallbackEvent)) continue;
                GPacketCallbackEvent ge = (GPacketCallbackEvent)e;
                if (!ge.getSender().equals(broker)) continue; 
                if (!ge.getSender().getBrokerSessionUID().equals(
                                                  broker.getBrokerSessionUID())) continue; 
                if (ge.getEventType() == ProtocolGlobals.G_MESSAGE_ACK_REPLY ||
                    ge.getEventType() == ProtocolGlobals.G_NEW_INTEREST ||
                    ge.getEventType() == ProtocolGlobals.G_REM_DURABLE_INTEREST ||
                    ge.getEventType() == ProtocolGlobals.G_CLIENT_CLOSED) {

                    ge.dispatch(cb);
                }
                itr.remove();
            }
        }
        cb.brokerDown(broker);
    }

    /**
     * A new destination was created by the administrator on a remote
     * broker.  This broker should also add the destination if it is
     * not already present.
     */
    public void notifyCreateDestination(Destination d) {
        CallbackEvent cbe = new
            ClusterCreateDestinationCallbackEvent(d, new CallbackEventListener());

        synchronized (eventQ) {
            if (stopThread)  {
                logger.log(logger.DEBUG, 
                  "Cluster shutdown, ignore create destination event on " + d);
                return;
            }
            eventQ.add(cbe);
            eventQ.notify();
        }
        cbe.getEventListener().waitEventProcessed();
    }

    /**
     * A destination was removed by the administrator on a remote
     * broker. This broker should also remove the destination, if it
     * is present.
     */
    public void notifyDestroyDestination(DestinationUID uid) {
        CallbackEvent cbe = new
            ClusterDestroyDestinationCallbackEvent(uid, new CallbackEventListener());

        synchronized (eventQ) {
            if (stopThread)  {
                logger.log(logger.DEBUG, 
                  "Cluster shutdown, ignore destroy destination event on " + uid);
                return;
            }
            eventQ.add(cbe);
            eventQ.notify();
        }
        cbe.getEventListener().waitEventProcessed();
    }

    /**
     * A destination was removed by the administrator on a remote
     * broker. This broker should also remove the destination, if it
     * is present.
     */
    public void notifyUpdateDestination(DestinationUID uid, Map changes) {
        CallbackEvent cbe = new
            ClusterUpdateDestinationCallbackEvent(uid, changes, 
                                        new CallbackEventListener());
        synchronized (eventQ) {
            if (stopThread)  {
                logger.log(logger.DEBUG, 
                  "Cluster shutdown, ignore update destination event on " + uid);
                return;
            }
            eventQ.add(cbe);
            eventQ.notify();
        }
        cbe.getEventListener().waitEventProcessed();
    }

    /**
     * Switch to HA_ACTIVE state.
     *
     * Falcon HA: Complete the initialization process, start all the
     * ServiceType.NORMAL services and start processing client work.
     */
    public void goHAActive() {
        CallbackEvent cbe = new
            GoHAActiveCallbackEvent();

        synchronized (eventQ) {
            eventQ.add(cbe);
            eventQ.notify();
        }
    }

    public void shutdown() {
        synchronized (eventQ) {
            stopThread = true;
            eventQ.notify();
        }

        try {
            join(30000);
        }
        catch (InterruptedException e) {
            // unable to complete join
        }
    }

    public void run() {
        CallbackEvent cbe = null;

        try {

        while (true) {

            try {
            synchronized (eventQ) {
                while (! stopThread && eventQ.isEmpty()) {
                    try {
                        eventQ.wait();
                    }
                    catch (Exception e) {}
                }

                if (stopThread)
                    break; // Ignore the pending events.

                cbe = (CallbackEvent) eventQ.removeFirst();
            }
            try { 
            cbe.dispatch(cb);
            } finally {

            CallbackEventListener l = cbe.getEventListener(); 
            if (l != null) l.eventProcessed();

            }

            } catch (Exception e) {
            logger.logStack(Logger.WARNING, 
                   "Cluster dispatcher thread encountered exception: "+e.getMessage(), e);
            }
        }

        } finally {

        if (!stopThread) {
        logger.log(Logger.WARNING, "Cluster dispatcher thread exiting");
        }

        synchronized(eventQ) {
            try {

            cbe = (CallbackEvent) eventQ.removeFirst();
            while (cbe != null) {
                CallbackEventListener l = cbe.getEventListener(); 
                if (l != null) l.eventProcessed();
                cbe = (CallbackEvent) eventQ.removeFirst();
            }

            } catch (NoSuchElementException e) { }
            stopThread = true;
        }

        }
    }
}

abstract class CallbackEvent {
    public abstract void dispatch(MessageBusCallback cb);
    public CallbackEventListener getEventListener() {
        return null;
    }
}

class CallbackEventListener {
    Object lock = new Object();
    boolean processed = false;

    public void waitEventProcessed()  {
        synchronized(lock) {
            if (processed) return;
            try {
            lock.wait();
            } catch (InterruptedException e) {}
        }
    }
    public void eventProcessed() {
        synchronized(lock) {
            processed = true;
            lock.notify();
        }
    }
}


class ConfigSyncCompleteCallbackEvent extends CallbackEvent {
    public ConfigSyncCompleteCallbackEvent() {
    }

    public void dispatch(MessageBusCallback cb) {
        cb.configSyncComplete();
    }
}

class GPacketCallbackEvent extends CallbackEvent {
    private BrokerAddress sender;
    private GPacket pkt;
    private Protocol p;

    public GPacketCallbackEvent(BrokerAddress sender, GPacket pkt, Protocol p) {
        this.sender = sender;
        this.pkt = pkt;
        this.p = p;
    }

    public int getEventType() {
        return pkt.getType();
    }

    public BrokerAddress getSender() {
        return sender;
    }

    public void dispatch(MessageBusCallback cb) {
        p.handleGPacket(cb, sender, pkt);
    }
}


class InterestCreatedCallbackEvent extends CallbackEvent {
    private Consumer intr;

    public InterestCreatedCallbackEvent(Consumer intr) {
        this.intr = intr;
    }

    public void dispatch(MessageBusCallback cb) {
        cb.interestCreated(intr);
    }
}

class InterestRemovedCallbackEvent extends CallbackEvent {
    private Consumer intr;
    private Set pendingMsgs = null;
    private boolean cleanup = false;

    public InterestRemovedCallbackEvent(Consumer intr, Set pendingMsgs, boolean cleanup) {
        this.intr = intr;
        this.pendingMsgs = pendingMsgs;
        this.cleanup = cleanup;
    }

    public void dispatch(MessageBusCallback cb) {
        cb.interestRemoved(intr, pendingMsgs,  cleanup);
    }
}

class PrimaryInterestChangedCallbackEvent extends CallbackEvent {
    private static final Logger logger = Globals.getLogger();
    private static BrokerResources br = Globals.getBrokerResources();
    private Consumer intr;
    private ConsumerUID intid;

    public PrimaryInterestChangedCallbackEvent(ConsumerUID intid) {
        this.intid = intid;
        this.intr = null;
    }

    public PrimaryInterestChangedCallbackEvent(Consumer intr) {
        this.intr = intr;
        this.intid = null;
    }

    public void dispatch(MessageBusCallback cb) {
        if (intr == null && intid != null) {
            intr = Consumer.getConsumer(intid);

            if (intr == null) {
                logger.log(logger.WARNING,
                    br.W_MBUS_BAD_PRIMARY_INT, intid);

                return;
            }
        }

        cb.activeStateChanged(intr);
    }
}

class ClientDownCallbackEvent extends CallbackEvent {
    private ConnectionUID conid;

    public ClientDownCallbackEvent(ConnectionUID conid) {
        this.conid = conid;
    }

    public void dispatch(MessageBusCallback cb) {
        cb.clientDown(conid);
    }
}

class BrokerDownCallbackEvent extends CallbackEvent {
    private BrokerAddress broker;

    public BrokerDownCallbackEvent(BrokerAddress broker) {
        this.broker = broker;
    }

    public void dispatch(MessageBusCallback cb) {
        cb.brokerDown(broker);
    }
}

class ClusterCreateDestinationCallbackEvent extends CallbackEvent {
    private Destination d;
    private CallbackEventListener l;

    public ClusterCreateDestinationCallbackEvent(
        Destination d, CallbackEventListener listener) {
        this.d = d;
        this.l = listener;
    }

    public void dispatch(MessageBusCallback cb) {
        cb.notifyCreateDestination(d);
    }
   
    public CallbackEventListener getEventListener() {
        return l;
    }

}
class ClusterUpdateDestinationCallbackEvent extends CallbackEvent {
    private DestinationUID duid;
    private Map changes;
    private CallbackEventListener l;

    public ClusterUpdateDestinationCallbackEvent(
        DestinationUID duid, Map changes, CallbackEventListener listener) {
        this.duid = duid;
        this.changes = changes;
        this.l = listener;
    }

    public void dispatch(MessageBusCallback cb) {
        cb.notifyUpdateDestination(duid, changes);
    }

    public CallbackEventListener getEventListener() {
        return l;
    }    
}

class ClusterDestroyDestinationCallbackEvent extends CallbackEvent {
    private DestinationUID d;
    private CallbackEventListener l;

    public ClusterDestroyDestinationCallbackEvent(
        DestinationUID d, CallbackEventListener listener) {
        this.d = d;
        this.l = listener;
    }

    public void dispatch(MessageBusCallback cb) {
        cb.notifyDestroyDestination(d);
    }

    public CallbackEventListener getEventListener() {
        return l;
    }    
}

class GoHAActiveCallbackEvent extends CallbackEvent {
    public GoHAActiveCallbackEvent() {
    }

    public void dispatch(MessageBusCallback cb) {
        cb.goHAActive();
    }
}


/*
 * EOF
 */
