/*
 * Apple built-in iSight firmware extractor
 *
 * Copyright © 2006 Ronald S. Bultje <rbultje@ronald.bitfreak.net>
 * Copyright © 2007 Étienne Bersac <bersace@gmail.com>
 *
 *
 * This software extract firmware from any Mac OS X iSight driver
 * (named AppleUSBVideoSupport).
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <fcntl.h>
#include <unistd.h>
#include <gcrypt.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>

/*
 * Foreach file identified by their sha1sum, we store the offset where
 * the firmware begin in the file.
 *
 * If a file does not work, you may need to add an entry here. To find
 * the offset in the driver file, use a hexadecimal editor (e.g. ghex)
 * and search for "USBSend". You will find data like this :
 *
 * "start USBSend #4 returned 0x%x.....X.....4.~.........%\"
 *                                                  ^
 *
 * Each period means 0x00. The firmware start at the ^ sign
 * position. Read the offset and store in in this array. The tool must
 * output the sha1sum of the file, use this to fill the .sha1sum
 * field.
 *
 */

struct offset {
	gchar *desc;
	unsigned char sha1sum[20];
	long offset;
};

static const struct offset offsets[] = {
	/* Mac OS X.10 PPC64 (iMac G5 iSight) */
	{ "Mac OS X.4.10 PPC64",
	  .sha1sum	= { 0xde, 0xf7, 0xf4, 0xf0, 0x08, 0xd7, 0x8f, 0x39,
			    0x3f, 0x95, 0x94, 0x6f, 0xbf, 0x85, 0xd7, 0xb1,
			    0xbd, 0xdc, 0x41, 0x11 },
	  .offset	= 0x1318
	},
	/* ? */
	{ "",
	  .sha1sum	= { 0x86, 0x43, 0x0c, 0x04, 0xf9, 0xb6, 0x7c, 0x5c,
			    0x3d, 0x84, 0x40, 0x91, 0x38, 0xa7, 0x67, 0x98,
			    0x27, 0x02, 0x5e, 0xc2 },
	  .offset	= 0x1434
	},
	/* ? */
	{ "",
	  .sha1sum	= { 0xa1, 0x4c, 0x15, 0x9b, 0x17, 0x6d, 0x27, 0xa6,
			    0xe9, 0x8d, 0xcb, 0x5d, 0xea, 0x5d, 0x78, 0xb8,
			    0x1e, 0x15, 0xad, 0x41 },
	  .offset	= 0x23D8
	},
	/* ? */
	{ "",
	  .sha1sum	= { 0xc6, 0xc9, 0x4d, 0xd7, 0x7b, 0x86, 0x4f, 0x8b,
			    0x2d, 0x31, 0xab, 0xf3, 0xcb, 0x2d, 0xe4, 0xc9,
			    0xd1, 0x39, 0xe1, 0xbf },
	  .offset	= 0x1434
	},
	{ "Mac OS X.4 intel",
	  .sha1sum	= { 0x01, 0xe2, 0x91, 0xd5, 0x29, 0xe7, 0xc1, 0x8d,
			    0xee, 0xa2, 0xeb, 0xa2, 0x52, 0xd1, 0x81, 0x14,
			    0xe0, 0x96, 0x27, 0x6e },
	  .offset	= 0x2060
	},
	{ "Mac OS X.5.1",
	  .sha1sum	= { 0xb6, 0x9f, 0x49, 0xd3, 0xfa, 0x68, 0x58, 0x41,
			    0x63, 0x24, 0xc3, 0x90, 0xef, 0xfe, 0x14, 0x33,
			    0x6a, 0x1d, 0xdb, 0x0b },
	  .offset	= 0xC404
	},
	{ "Mac OS X.5.1 late 2007",
	  .sha1sum	= { 0x1c, 0x60, 0xef, 0x27, 0xd5, 0x72, 0x21, 0xcf,
			    0x3d, 0x76, 0x68, 0x7a, 0x49, 0x73, 0xec, 0x72,
			    0xff, 0x6f, 0xa1, 0x03 },
	  .offset	= 0x20D8
	},
	{ "Mac OS X.5.1 driver 1.0.9",
	  .sha1sum	= { 0xa1, 0x7b, 0x71, 0xc0, 0xe6, 0x3b, 0xd3, 0x87,
			    0x82, 0x10, 0x88, 0xdb, 0x3a, 0x66, 0x5b, 0xb8,
			    0x7a, 0xdc, 0xa2, 0x08 },
	  .offset	= 0xC404
	},
	{ "Mac OS X.5.2",
	  .sha1sum	= { 0x89, 0x84, 0x60, 0x94, 0x2e, 0x12, 0x65, 0xa3, 
			    0x57, 0xb5, 0xac, 0x86, 0x82, 0x4a, 0xf2, 0xd4,
			    0xa4, 0xf7, 0x3b, 0x98 },
	  .offset	= 0x20D8
	}
};

/*
 * The vanilla firmware have bugs in USB Interface descriptor
 * values. We override buggy value with correct one. Those value are
 * triplet, we replace 0xFF 0xFF 0xFF by specific value.
 */
struct patch {
	const char* desc;
	long offset;
	unsigned char value[3];
};

/* From linux-2.6:linux/usb/ch9.J */
#define	USB_CLASS_VIDEO			0x0e

/* From linux-uvc: uvcvideo.h */
#define SC_VIDEOCONTROL                 0x01
#define SC_VIDEOSTREAMING               0x02
#define SC_VIDEO_INTERFACE_COLLECTION   0x03
#define PC_PROTOCOL_UNDEFINED           0x00

static struct patch patches[] = {
	{ N_("Fix video control interface descriptor"),
	  .offset	= 0x2380,
	  .value	= { USB_CLASS_VIDEO,
			    SC_VIDEOCONTROL,
			    PC_PROTOCOL_UNDEFINED }
	},
	{ N_("Fix video streaming interface descriptor"),
	  .offset	= 0x23C6,
	  .value	= { USB_CLASS_VIDEO,
			    SC_VIDEOSTREAMING,
			    PC_PROTOCOL_UNDEFINED }
	},
	/* needs to be checked */
	{ N_("Fix video streaming device qualifier"),
	  .offset	= 0x2364,
	  .value	= { USB_CLASS_VIDEO,
			    SC_VIDEOSTREAMING,
			    PC_PROTOCOL_UNDEFINED }
	},
};

/* OPTIONS */
static gchar *driver_filename	= NULL;
static gchar *firmware_dir	= "/lib/firmware";
static gchar *firmware_filename = "isight.fw";

static GOptionEntry entries[] = {
	{ "apple-driver", 'a',
	  0,
	  G_OPTION_ARG_FILENAME, &driver_filename,
	  N_("Path to the AppleUSBVideoSupport driver file."),
	  NULL
	},
	{ "output-directory", 'd',
	  0,
	  G_OPTION_ARG_FILENAME, &firmware_dir,
	  N_("Output firmware directory"),
	  "/lib/firmware"
	},
	{ "output-filename", 'f',
	  0,
	  G_OPTION_ARG_FILENAME, &firmware_filename,
	  N_("Name of the output firmware file"),
	  "isight.fw"
	},
	{ NULL }
};


/* IMPLEMENTATION */

static unsigned char*
get_sha1sum(char*filename)
{
	unsigned char *digest;
	GMappedFile *file;
	GError *gerr = NULL;

	digest = g_malloc0 (gcry_md_get_algo_dlen (GCRY_MD_SHA1));
	if (!(file = g_mapped_file_new (filename, FALSE, &gerr))) {
		g_error(_("Unable to open driver: %s"),
			gerr ? gerr->message : "?");
		return NULL;
	}
	gcry_md_hash_buffer (GCRY_MD_SHA1, digest,
			     g_mapped_file_get_contents (file),
			     g_mapped_file_get_length (file));
	g_mapped_file_free (file);

	return digest;
}

static gchar*
sha1sum_string(unsigned char*digest)
{
	gchar *sha1sum = "";
	gint i;

	for (i = 0; i < 20; i++) {
		sha1sum = g_strdup_printf("%s %02x", sha1sum, digest[i]);
	}
	return sha1sum;
}

static const struct offset*
find_offset(unsigned char*digest)
{
	gint n;

	for (n = 0; n < G_N_ELEMENTS (offsets); n++)
		if (!memcmp (offsets[n].sha1sum, digest, 20))
			break;

	if (n == G_N_ELEMENTS (offsets)) {
		g_warning(_("Unknown driver. Please report it to %s with "
			    "machine description and Mac OS X version."),
			    PACKAGE_BUGREPORT);
		return NULL;
	} else {
		/* translators : %s is the known origin of the driver */
		g_message(_("Found %s driver"),
			  offsets[n].desc);
		return &offsets[n];
	}
}

/* extract firmware from filename at offset to output, checking the
   format. */
static int
extract(char *filename,
	long offset,
	char*output)
{
	int fd, fdo, len, req, ret = -1;
	unsigned char data[4], rdata[1024];

	if ((fd = g_open (filename, O_RDONLY)) == -1) {
		g_error(_("Unable to open %s."), filename);
		return -1;
	} else if (lseek (fd, offset, SEEK_SET) != offset) {
		g_error(_("Failed to seek %s at position %x."),
			filename, (guint) offset);
		close (fd);
		return -1;
	}

	if ((fdo = g_open(output, O_WRONLY | O_CREAT | O_TRUNC,
			  S_IRUSR | S_IWUSR)) == -1) {
		g_error(_("Unable to open %s for writing."), output);
		close (fdo);
		return -1;
	}

	while (1) {
		if ((len = read (fd, data, 4)) != 4) {
			if (len == 0) {
				g_error(_("Unexpected EOS - corrupt driver?"));
				goto end;
			} else {
				g_error(_("Reading firmware header chunk "
					  "failed"));
				goto end;
			}
		}
		len = (data[0] << 8) | data[1];
		req = (data[2] << 8) | data[3];
		if (len == 0x8001)
			break; /* success */
		else if (len == 0)
			continue;
		else if (len < 0 || len >= 1024) {
			g_error(_("Invalid firmware data_length %d, "
				  "load aborted"), len);
			goto end;
		} else if (read (fd, rdata, len) != len) {
			g_error(_("Error reading firmware data"));
			goto end;
		}

		/* write to file */
		write(fdo, data, 4);
		write(fdo, rdata, len);
	}

	ret = 0;
 end:
	close(fd);
	fchmod(fdo, 0644);
	close(fdo);

	return 0;
}

static int
patch(char *filename)
{
	int fd, i, n, ret = 0;

	if ((fd = g_open(filename, O_RDWR)) == -1) {
		g_error(_("Error while opening firmware file for patching."));
		return -1;
	}

	n = G_N_ELEMENTS(patches);
	for (i = 0; i < n; i++) {
		g_message(_("Apply patch %i : %s"),
			  i, gettext(patches[i].desc));
		if (lseek(fd, patches[i].offset, SEEK_SET) == -1) {
			g_error(_("Unable to patch the firmware."));
			ret = -1;
			goto end;
		}

		if (write(fd, &patches[i].value, 3) ==0) {
			g_error(_("Failed to write patched value !"));
			ret = -1;
			goto end;
		}
	}

 end:
	close(fd);
	return ret;
}


int main (int argc, char *argv[])
{
	GOptionContext *context;
	GError *error = NULL;
	int status = 0;
	gchar *sha1sum;
	gchar *pathname;
	unsigned char*digest;
	const struct offset *offset;
	
#if ENABLE_NLS
	/* ??????? */
	setlocale(LC_ALL, "");
	bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);
#endif

	gchar *appname = _("built-in iSight firmware extractor "
			   "and patcher");

	g_set_application_name(appname);
	g_set_prgname("ift-extract");

	/* translators: this is the first --help output line */
	context = g_option_context_new(g_strdup_printf("- %s",
						       appname));
	/* translators: this is shown at the end of --help output */
	gchar *desc = g_strdup_printf(_("AppleUSBVideoSupport driver is usualy "					"found in /System/Library/Extensions/"
					"IOUSBFamily.kext/Contents/PlugIns/"
					"AppleUSBVideoSupport.kext/Contents/"
					"MacOS/ in your Mac OS X root volume."
					"\n\n"
					"If you have a non-working Apple "
					"driver file, please send it to %s with"
					"machine description and OS X "
					"version."),
				      PACKAGE_BUGREPORT);
	g_option_context_set_description(context, desc);
	g_option_context_add_main_entries(context, entries, GETTEXT_PACKAGE);
	if (!g_option_context_parse(context, &argc, &argv, &error)) {
		fprintf(stderr, _("Error: %s\n"),
			error->message); /* comment traduire ? */
		fprintf(stderr, g_option_context_get_help(context,
							  FALSE, NULL));
	}

	if (g_access(driver_filename, R_OK)) {
		g_error(_("Unable to read driver %s."), driver_filename);
	}

	pathname = g_build_filename(firmware_dir, firmware_filename, NULL);
	g_option_context_free(context);

	/* check sha1sum on firmware, to prevent loading crap into the
	 * iSight and thus possibly damaging it. */
	digest = get_sha1sum(driver_filename);
	if (!digest) {
		return -1;
	}

	sha1sum = sha1sum_string(digest);
	offset = find_offset(digest);

	if (!offset)
		g_error(_("Unable to find firmware in the file."));

	/* extract */
	status = extract(driver_filename, offset->offset, pathname);
	if (!status)
		g_message(_("Firmware extracted successfully in %s"), pathname);

	/* patch */
	status = patch(pathname);
	if (!status)
		g_message(_("Firmware patched successfully"));
	else
		g_error(_("Failed to apply patches to %s"), pathname);

	g_free(digest);
	g_free(sha1sum);

 	return status;
}
