/*
 *  Jajuk
 *  Copyright (C) 2007 The Jajuk Team
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *  $Revision$
 */

package org.jajuk.ui.thumbnails;

import java.awt.Container;
import java.awt.Image;
import java.awt.MediaTracker;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.Properties;
import java.util.StringTokenizer;

import javax.imageio.ImageIO;
import javax.swing.ImageIcon;

import org.jajuk.base.Album;
import org.jajuk.base.AlbumManager;
import org.jajuk.events.JajukEvent;
import org.jajuk.events.JajukEvents;
import org.jajuk.events.ObservationManager;
import org.jajuk.util.Const;
import org.jajuk.util.IconLoader;
import org.jajuk.util.JajukIcons;
import org.jajuk.util.UtilGUI;
import org.jajuk.util.UtilSystem;
import org.jajuk.util.log.Log;

/**
 * Manage thumbnails
 */
public final class ThumbnailManager {

  /** No instances */
  private ThumbnailManager() {
  }

  /**
   * Delete all thumbs for a given size
   * 
   * @param size
   *          size, eg: Const.THUMBNAIL_SIZE_150x150
   */
  public static void cleanThumbs(String size) {
    File fThumb = UtilSystem.getConfFileByPath(Const.FILE_THUMBS + '/' + size);
    if (fThumb.exists()) {
      File[] files = fThumb.listFiles();
      for (File file : files) {
        if (!file.getAbsolutePath().matches(".*" + Const.FILE_THUMB_NO_COVER)) {
          if (!file.delete()) {
            Log.warn("Could not delete " + file.toString());
          }
        }
      }
      // Refresh default cover
      File fDefault = UtilSystem.getConfFileByPath(Const.FILE_THUMBS + "/" + size + "/"
          + Const.FILE_THUMB_NO_COVER);
      fDefault.delete();
      try {
        int iSize = Integer.parseInt(new StringTokenizer(size, "x").nextToken());
        createThumbnail(IconLoader.getIcon(JajukIcons.NO_COVER), fDefault, iSize);
      } catch (Exception e) {
        Log.error(e);
      }
    }
    // Reset all thumbs cache
    for (Album album : AlbumManager.getInstance().getAlbums()) {
      cleanThumbs(album);
    }
  }

  /**
   * Delete all thumbs for a given album
   * 
   * @param album
   */
  public static void cleanThumbs(Album album) {
    // Now delete thumb files
    for (int size = 50; size <= 300; size += 50) {
      File fThumb = ThumbnailManager.getThumbBySize(album, size);
      if (fThumb.exists()) {
        boolean out = fThumb.delete();
        if (!out) {
          Log.warn("Cannot delete thumb for album: " + album);
        }
      }
      album.setAvailableThumb(size, false);
    }
  }

  /**
   * Reads an image in a file and creates a thumbnail in another file. Will be
   * created if necessary. the thumbnail must be maxDim pixels or less. Thanks
   * Marco Schmidt
   * http://schmidt.devlib.org/java/save-jpeg-thumbnail.html#source
   * 
   * @param orig
   *          source image
   * @param thumb
   *          destination file (jpg)
   * @param maxDim
   *          required size
   * @throws Exception
   */
  public static void createThumbnail(final File orig, final File thumb, final int maxDim)
      throws Exception {
    // do not use URL object has it can corrupt special paths
    createThumbnail(new ImageIcon(orig.getAbsolutePath()), thumb, maxDim);
  }

  /**
   * Reads an image in a file and creates a thumbnail in another file. Use this
   * method to get thumbs from images inside jar files, some bugs in URL
   * encoding makes impossible to create the image from a file. Will be created
   * if necessary. the thumbnail must be maxDim pixels or less. Thanks Marco
   * Schmidt http://schmidt.devlib.org/java/save-jpeg-thumbnail.html#source
   * 
   * @param orig
   *          source image
   * @param thumb
   *          destination file (jpg)
   * @param maxDim
   *          required size
   * @throws Exception
   */
  public static void createThumbnail(final ImageIcon ii, final File thumb, final int maxDim)
      throws Exception {
    final Image image = ii.getImage();
    // Wait for full image loading
    final MediaTracker mediaTracker = new MediaTracker(new Container());
    mediaTracker.addImage(image, 0);
    mediaTracker.waitForID(0); // wait for image loading
    // Check for image loading errors (like missing cover file for ie)
    Object[] loadingErrors = mediaTracker.getErrorsAny();
    if (loadingErrors != null) {
      throw new IOException("Problem reading the cover");
    }
    // determine thumbnail size from WIDTH and HEIGHT
    int thumbWidth = maxDim;
    int thumbHeight = maxDim;
    final double thumbRatio = (double) thumbWidth / (double) thumbHeight;
    final int imageWidth = image.getWidth(null);
    final int imageHeight = image.getHeight(null);
    final double imageRatio = (double) imageWidth / (double) imageHeight;
    if (thumbRatio < imageRatio) {
      thumbHeight = (int) (thumbWidth / imageRatio);
    } else {
      thumbWidth = (int) (thumbHeight * imageRatio);
    }
    // draw original image to thumbnail image object and
    // scale it to the new size on-the-fly
    final BufferedImage thumbImage = UtilGUI.toBufferedImage(image, !(UtilSystem
        .getExtension(thumb).equalsIgnoreCase("jpg")), thumbWidth, thumbHeight);
    // Need alpha only for png and gif files
    // save thumbnail image to OUTFILE
    ImageIO.write(thumbImage, UtilSystem.getExtension(thumb), thumb);
    // Free thumb memory
    thumbImage.flush();
  }

  /**
   * Make thumbnail file exists (album id.jpg or.gif or .png) in thumbs
   * directory if it doesn't exist yet
   * 
   * @param album
   * @return whether a new cover has been created
   */
  public static boolean refreshThumbnail(final Album album, final int size) {
    // Check if the thumb is known in cache
    if (album.isThumbAvailable(size)) {
      return false;
    }
    final File fThumb = getThumbBySize(album, size);
    final File fCover = album.getCoverFile();
    if (fCover != null) {
      try {
        createThumbnail(fCover, fThumb, size);
        // Update thumb availability
        album.setAvailableThumb(size, true);
        // Notify the thumb creation
        Properties details = new Properties();
        details.put(Const.DETAIL_CONTENT, album);
        ObservationManager.notify(new JajukEvent(JajukEvents.THUMB_CREATED, details));
        return true;
      } catch (final Exception e) {
        Log.error(e);
      }
    }
    return false; // thumb already exists or cannot be created (missing cover
    // for ie)
  }

  /**
   * Return thumb file by album and size
   * 
   * @param album
   *          the album
   * @param size
   *          the size (like 50)
   * @return thumb file by album and size
   */
  public static File getThumbBySize(Album album, int size) {
    StringBuilder thumb = new StringBuilder(Const.FILE_THUMBS).append('/').append(size).append('x')
        .append(size).append('/').append(album.getID()).append('.').append(Const.EXT_THUMB);
    File fThumb = UtilSystem.getConfFileByPath(thumb.toString());
    return fThumb;
  }

}
