/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * GXMame
 *
 * Copyright 2002-2005 Stephane Pontier <shadow_walker@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <klocale.h>

#include "gxmame.h"
#include "io.h"
#include "gui.h"

extern "C" {
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
}

#ifndef XMAME_ROOT
#define XMAME_ROOT "/usr/lib/games/xmame"
#endif

/*
converts a float to a string with precission 5.
this is locale independent.
buf must be of size G_ASCII_DTOSTR_BUF_SIZE
*/
const gchar *
my_dtostr (char * buf, gdouble d)
{
	char *p;
	g_ascii_dtostr (buf, FLOAT_BUF_SIZE, d);

	for (p = buf; *p && *p != '.'; p++); /* point p to the decimal point */

	if (*p)
	{
		p += 6;	/* precision + 1 */
		*p = '\0';
	}
	return buf;
}

#define LINE_BUF 512

gboolean
load_games_ini (void)
{
	FILE *gameini_list;
	gchar *filename;
	gchar line[LINE_BUF];
	gchar **option_value = NULL;
	GList *listpointer;
	RomEntry *tmprom = NULL;
	gboolean game_found = FALSE;
	
	GXMAME_DEBUG ("Loading games.ini");

	if(!strcmp(game_list.name,"xmame"))
		filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "games.ini", NULL);
	else
		filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "systems.ini", NULL);
		
	if (!filename)
		return FALSE;

	gameini_list = fopen (filename, "r");
	g_free (filename);
	if (!gameini_list)
	{
		GXMAME_DEBUG ("games.ini not found");
		game_list.not_checked_list = g_list_copy (game_list.roms);
		return FALSE;
	}

	while (fgets (line, LINE_BUF, gameini_list))
	{
		if (*line == '[')
		{	
			g_strstrip (g_strdelimit (line, "[]\r\n", ' '));
			/* ok only if mame32k games,ini doesn't change */
			/* find the the rom in the glist */
			for (listpointer = g_list_first (game_list.roms), game_found = FALSE;
			     (listpointer != NULL);
			     listpointer = g_list_next (listpointer))
			{
				tmprom = (RomEntry *) listpointer->data;
				if (!strcmp (tmprom->romname, line))
				{
					game_found = TRUE;
					break;
				}
			}
			if (game_found)
			{
				while (fgets (line, LINE_BUF, gameini_list))
				{/* need to test the carriage return if the file come from Mame32K under windows*/
					if (*line =='\r' || *line =='\n')
						break;
					option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
					if (option_value[1] != NULL)
					{	if (!strcmp (option_value[0], "PlayCount"))
						{	tmprom->timesplayed = atoi (option_value[1]);
						} else if (!strcmp (option_value[0], "HasRoms"))
						{	tmprom->has_roms = (RomStatus)atoi (option_value[1]);
						} else if (!strcmp (option_value[0], "HasSamples"))
						{	tmprom->has_samples = (RomStatus)atoi (option_value[1]);
						} else if (!strcmp (option_value[0], "Favorite"))
						{	tmprom->favourite = atoi (option_value[1]) ? TRUE : FALSE;
						}else if (!strcmp (option_value[0], "AutofireDelay"))
						{	/*store this somewhere to be able to put it again
							  when we will save the game.ini*/
						}
					g_strfreev (option_value);
					}
				}
				if ((tmprom->has_roms == 2) || ( (tmprom->has_roms == 0) && gui_prefs.GameCheck))
				{
					game_list.not_checked_list = g_list_append (game_list.not_checked_list, tmprom);
				}
			}
		}
		
	}
	fclose (gameini_list);
	return TRUE;
}


gboolean save_games_ini (XmameExecutable *exec)
{
	FILE *game_ini_file;
	gchar *filename;
	RomEntry *rom;
	GList *list_pointer;
	
	GXMAME_DEBUG ("Saving games.ini");
	
	if(!exec) return FALSE;
	
	if(!strcmp(exec->name,"xmame"))
		filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "games.ini", NULL);
	else
		filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "systems.ini", NULL);
	
	game_ini_file = fopen (filename, "w");
	g_free (filename);
	if (!game_ini_file)
	{
		GXMAME_DEBUG ("unable to write games.ini");
		return FALSE;
	}
		
	for (list_pointer = g_list_first (game_list.roms);
	     (list_pointer != NULL);
	     list_pointer=g_list_next (list_pointer))
	{
		rom = (RomEntry *)list_pointer->data;
		fprintf (game_ini_file, "[%s]\r\n", rom->romname);
		fprintf (game_ini_file, "PlayCount=%i\r\n", rom->timesplayed);
		fprintf (game_ini_file, "HasRoms=%i\r\n", rom->has_roms);
		fprintf (game_ini_file, "HasSamples=%i\r\n", rom->has_samples);
		fprintf (game_ini_file, "Favorite=%s\r\n", rom->favourite ? "1" : "0");
		fputs ("\r\n", game_ini_file);
	}
	fclose (game_ini_file);
	
	return TRUE;
}


gboolean
load_catver_ini (void)
{
	FILE *catver_file;
	gchar *filename;
	gchar line[2000];
	gchar **game_value = NULL;
	GList *listpointer;
	RomEntry *tmprom = NULL;
	gboolean game_found = FALSE;
	gchar *category;
	gchar *version;
	
	GXMAME_DEBUG ("Loading catver.ini");

	/* Initialize categories */
	if (!game_list.categories) {
		g_list_foreach (game_list.categories, (GFunc) g_free, NULL);
		g_list_free (game_list.categories);
	}
	/* Initialize versions */
	if (!game_list.versions) {
		g_list_foreach (game_list.versions, (GFunc) g_free, NULL);
		g_list_free (game_list.versions);
	}

	category = g_strdup (i18n("Unknown"));
	game_list.categories = g_list_append (NULL, category);

	version = g_strdup (i18n("Unknown"));
	game_list.versions = g_list_append (NULL, version);

	/* Set all roms to unknown */
	for (listpointer = g_list_first (game_list.roms); (listpointer != NULL);
		listpointer = g_list_next (listpointer))
	{
		tmprom = (RomEntry *) listpointer->data;
		tmprom->category = category;
		tmprom->mame_ver_added = version;
	}

	filename = g_build_filename (gui_prefs.catverDirectory, "catver.ini", NULL);	
	if (!filename)
		return FALSE;

	catver_file = fopen (filename, "r");
	if (!catver_file)
	{
		GXMAME_DEBUG ("catver.ini not found in %s", filename);
		g_free (filename);
		filename = g_build_filename (gui_prefs.catverDirectory, "catver", NULL);	
		catver_file = fopen (filename, "r");
		if (!catver_file)
		{
			GXMAME_DEBUG ("catver not found in %s", filename);
			g_free (filename);
			return FALSE;
		}
	}
	g_free (filename);

	fgets (line, 500, catver_file);
	if (*line!=';')
	{
		fclose (catver_file);
		return FALSE;
	}

	/* Jump initial comments and empty lines */
	while (*line == '\r' || *line == '\n' || *line == ';')
	{	g_strstrip (g_strdelimit (line, "\r\n", ' '));
		GXMAME_DEBUG ("Loading catver: %s", line);
		fgets (line, 500, catver_file);
	}
	
	if (strncmp (line, "[Category]", 10))
	{
		fclose (catver_file);
		return FALSE;
	}
		
	while (fgets (line, 500, catver_file))
	{
		/*end of categories */
		if (*line == '[')
			break;
		if (*line == '\r' || *line == '\n' || *line == ';')
			continue;
		
		g_strstrip (g_strdelimit (line, "\r\n", ' '));
		game_value = g_strsplit (line, "=", 2);
		for (listpointer = g_list_first (game_list.roms), game_found = FALSE;
		     (listpointer != NULL);
		     listpointer = g_list_next (listpointer))
		{
			tmprom = (RomEntry *) listpointer->data;
			if (!strcmp (tmprom->romname, game_value[0]))
			{
				game_found = TRUE;
				break;
			}
		}
		if (game_found)
		{
			tmprom->category = glist_insert_unique (&game_list.categories, game_value[1]);
		}
		g_strfreev (game_value);
	}

	/* and now come the version*/
	if (strncmp (line, "[VerAdded]", 10))
	{
		GXMAME_DEBUG ("[VerAdded] expected but found:%s", line);
		fclose (catver_file);
		return FALSE;
	}
		
	while (fgets (line, 500, catver_file))
	{
		/* Jump initials comment and empty line */
		if (*line == '\r' || *line == '\n' || *line == ';')
			continue;

		g_strstrip (g_strdelimit (line, "\r\n", ' '));
		game_value = g_strsplit (line, "=", 2);
		for (listpointer = g_list_first (game_list.roms), game_found = FALSE;
		     (listpointer != NULL);
		     listpointer = g_list_next (listpointer))
		{
			tmprom = (RomEntry *) listpointer->data;
			if (!strcmp (tmprom->romname, game_value[0]))
			{
				game_found = TRUE;
				break;
			}
		}
		if (game_found)
		{
			tmprom->mame_ver_added = glist_insert_unique (&game_list.versions, game_value[1]);
		}
		g_strfreev (game_value);
	}
	fclose (catver_file);
	return TRUE;
}


/* preferences for the gui */
gboolean
load_gxmame_ini (void)
{
	FILE *gameini_list;
	gchar *filename;
	gchar line[2000];
	//gchar *default_game = NULL;
	gchar **option_value = NULL;
	gchar **option_value2 = NULL;
	gint i;
	
	GXMAME_DEBUG ("Loading gxmame.ini");

	gui_prefs.FolderID = ALL;
	gui_prefs.ShowScreenShot = 1;
	gui_prefs.ShowScreenShotTab = 0;
	gui_prefs.ShowFlyer = SNAPSHOTS;
	gui_prefs.ShowToolBar = 1;
	gui_prefs.ShowStatusBar = 1;
	gui_prefs.ShowFolderList = 1;
	gui_prefs.SortColumn = 0;
	gui_prefs.SortReverse = 0;
	gui_prefs.GUIPosX = 0;
	gui_prefs.GUIPosY = 0;
	gui_prefs.GUIWidth = 640;
	gui_prefs.GUIHeight = 400;
	gui_prefs.previous_mode = DETAILS_TREE;
	gui_prefs.current_mode = DETAILS_TREE;
	gui_prefs.Splitters[0] = 150;
	gui_prefs.Splitters[1] = 300;
	gui_prefs.ModifyThe = TRUE;
	gui_prefs.GameCheck = FALSE;
	gui_prefs.VersionCheck = TRUE;
	gui_prefs.gui_joy = TRUE;
	gui_prefs.use_xmame_options = FALSE;
	gui_prefs.currentMame = gui_prefs.currentMess = NULL;
	for (i = 0; i < NUMBER_COLUMN; i++)
	{
		gui_prefs.ColumnWidth[i] = 0;  /* 0 --> Auto Size */
		gui_prefs.ColumnOrder[i] = i;
		gui_prefs.ColumnShown[i] = 0;
	}
	gui_prefs.ColumnShown[GAMENAME] = 1;
	gui_prefs.ColumnShown[HAS_SAMPLES] = 1;
	gui_prefs.ColumnShown[ROMNAME] = 1;
	gui_prefs.ColumnShown[VECTOR] = 1;
	gui_prefs.ColumnShown[CONTROL] = 1;
	gui_prefs.ColumnShown[TIMESPLAYED] = 1;
	gui_prefs.ColumnShown[MANU] = 1;
	gui_prefs.ColumnShown[YEAR] = 1;
	gui_prefs.ColumnShown[CLONE] = 1;
	gui_prefs.ColumnShown[DRIVER] = 1;
	gui_prefs.ColumnShown[MAMEVER] = 1;
	gui_prefs.ColumnShown[CATEGORY] = 1;

#ifdef KXMAME
	gui_prefs.iconSize = 16;
	gui_prefs.scaledScreenshot = TRUE;
	gui_prefs.kxmameGeometry[0] = 160;
	gui_prefs.kxmameGeometry[1] = 320;
	gui_prefs.kxmameGeometry[2] = 320;
	gui_prefs.kxmameGeometry[3] = 240;
	gui_prefs.kxmameGeometry[4] = 320;
	gui_prefs.showVideoPreview = TRUE;
#endif

	filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "gxmame.ini", NULL);
	gameini_list = fopen (filename, "r");
	g_free (filename);
	if (!gameini_list)
	{
		GXMAME_DEBUG ("gxmame.ini not found");
		return FALSE;
	}
	/* Be careful of the \r\n terminal due to windows */
	while (fgets (line, 500, gameini_list))
	{
		if (!strncmp (line, "[default]", 9))
		{	
			while (fgets (line, 500, gameini_list))
			{/* need to test the carriage return if the file come from Mame32K under windows*/
				if (*line == '\r' || *line == '\n')
					break;
				/* FIXME: leaks here */
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL)
				{	if (!strcmp (option_value[0], "DefaultGame"))
					{	gui_prefs.currentMame = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "DefaultSystem"))
					{	gui_prefs.currentMess = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "FolderID"))
					{	gui_prefs.FolderID = (folder_filters_list)atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "ShowScreenShot"))
					{	gui_prefs.ShowScreenShot = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "ShowScreenShotTab"))
					{	gui_prefs.ShowScreenShotTab = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "ShowFlyer"))
					{	gui_prefs.ShowFlyer = (screenshot_type)atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "ShowToolBar"))
					{	gui_prefs.ShowToolBar = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "ShowStatusBar"))
					{	gui_prefs.ShowStatusBar = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "ModifyThe"))
					{	gui_prefs.ModifyThe = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "ShowFolderList"))
					{	gui_prefs.ShowFolderList = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "GameCheck"))
					{	gui_prefs.GameCheck = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "VersionCheck"))
					{	gui_prefs.VersionCheck = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "UseXmameOptions"))
					{	gui_prefs.use_xmame_options = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "gui_joy"))
					{	gui_prefs.gui_joy = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "SortColumn"))
					{	gui_prefs.SortColumn = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "SortReverse"))
					{	gui_prefs.SortReverse = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "GUIPosX"))
					{	gui_prefs.GUIPosX = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "GUIPosY"))
					{	gui_prefs.GUIPosY = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "GUIWidth"))
					{	gui_prefs.GUIWidth = atoi (option_value[1]);
					}else if (!strcmp (option_value[0], "GUIHeight"))
					{	gui_prefs.GUIHeight = atoi (option_value[1]);
					}else if (!strcmp (option_value[0], "ListMode"))
					{	gui_prefs.current_mode = gui_prefs.previous_mode = (ListMode)atoi (option_value[1]);
					}else if (!strcmp (option_value[0], "Splitters"))
					{
						option_value2 = g_strsplit (option_value[1], ",", 2);
						gui_prefs.Splitters[0] = atoi (option_value2[0]);
						gui_prefs.Splitters[1] = atoi (option_value2[1]);
						g_strfreev (option_value2);
					} else if (!strcmp (option_value[0], "ListFont"))
					{	gui_prefs.ListFont = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "ColumnWidth"))
					{
						option_value2 = g_strsplit (option_value[1], ",", NUMBER_COLUMN);
						for (i = 0; i < NUMBER_COLUMN; i++)
							gui_prefs.ColumnWidth[i] = atoi (option_value2[i]);
						g_strfreev (option_value2);
					} else if (!strcmp (option_value[0], "ColumnOrder"))
					{
						option_value2 = g_strsplit (option_value[1], ",", NUMBER_COLUMN);
						for (i = 0; i < NUMBER_COLUMN; i++)
							gui_prefs.ColumnOrder[i] = atoi (option_value2[i]);
						g_strfreev (option_value2);
					} else if (!strcmp (option_value[0], "ColumnShown"))
					{
						option_value2 = g_strsplit (option_value[1], ",", NUMBER_COLUMN);
						for (i = 0; i < NUMBER_COLUMN; i++)
							gui_prefs.ColumnShown[i] = atoi (option_value2[i]);
						g_strfreev (option_value2);
					}
#ifdef KXMAME
					else if (!strcmp (option_value[0], "kxmameIconSize"))
					{	gui_prefs.iconSize = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "kxmameScaledScreenshot"))
					{	gui_prefs.scaledScreenshot = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "showVideoPreview"))
					{	gui_prefs.showVideoPreview = atoi (option_value[1]);
					} else if (!strcmp (option_value[0], "kxmameGeometry"))
					{
						option_value2 = g_strsplit (option_value[1], ",", 5);
						gui_prefs.kxmameGeometry[0] = atoi (option_value2[0]);
						gui_prefs.kxmameGeometry[1] = atoi (option_value2[1]);
						gui_prefs.kxmameGeometry[2] = atoi (option_value2[2]);
						gui_prefs.kxmameGeometry[3] = atoi (option_value2[3]);
						gui_prefs.kxmameGeometry[4] = atoi (option_value2[4]);
						g_strfreev (option_value2);
					}
#endif
				g_strfreev (option_value);
				}
			}
		}
	}
	fclose (gameini_list);

// 	if (game_list.roms && default_game) {
// 		RomEntry *rom;
// 		GList *listpointer = g_list_first (game_list.roms);
// 		
// 		while (listpointer) {
// 			rom = (RomEntry*)listpointer->data;
// 
// 			if (!strcmp (rom->romname, default_game)) {
// 				gui_prefs.current_game = rom;
// 				break;
// 			}
// 
// 			listpointer = g_list_next (listpointer);
// 		}
// 	}
// 
// 	g_free (default_game);

	return TRUE;
}


gboolean
save_gxmame_ini (void)
{
	FILE *gxmame_ini_file = NULL;
	gchar *filename;
	gint i;
	
	GXMAME_DEBUG ("Saving gxmame.ini");

	filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "gxmame.ini", NULL);
	gxmame_ini_file = fopen (filename, "w");
	g_free (filename);

	if (!gxmame_ini_file)
	{
		GXMAME_DEBUG ("unable to write gxmame.ini");
		return FALSE;
	}

	fputs ("[default]\r\n", gxmame_ini_file);

	if (gui_prefs.currentMame) {
		fprintf (gxmame_ini_file, "DefaultGame=%s\r\n", gui_prefs.currentMame);
	}
	if (gui_prefs.currentMess) {
		fprintf (gxmame_ini_file, "DefaultSystem=%s\r\n", gui_prefs.currentMess);
	}
	fprintf (gxmame_ini_file, "FolderID=%i\r\n", gui_prefs.FolderID);
	fprintf (gxmame_ini_file, "ShowScreenShot=%i\r\n", gui_prefs.ShowScreenShot);
	fprintf (gxmame_ini_file, "ShowScreenShotTab=%i\r\n", gui_prefs.ShowScreenShotTab);
	fprintf (gxmame_ini_file, "ShowFlyer=%i\r\n", gui_prefs.ShowFlyer);
	fprintf (gxmame_ini_file, "ShowToolBar=%i\r\n", gui_prefs.ShowToolBar);
	fprintf (gxmame_ini_file, "ModifyThe=%i\r\n", gui_prefs.ModifyThe);
	fprintf (gxmame_ini_file, "ShowStatusBar=%i\r\n", gui_prefs.ShowStatusBar);
	fprintf (gxmame_ini_file, "ShowFolderList=%i\r\n", gui_prefs.ShowFolderList);
	fprintf (gxmame_ini_file, "GameCheck=%i\r\n", gui_prefs.GameCheck);
	fprintf (gxmame_ini_file, "VersionCheck=%i\r\n", gui_prefs.VersionCheck);
	fprintf (gxmame_ini_file, "UseXmameOptions=%i\r\n", gui_prefs.use_xmame_options);
	fprintf (gxmame_ini_file, "gui_joy=%i\r\n", gui_prefs.gui_joy);
	fprintf (gxmame_ini_file, "SortColumn=%i\r\n", gui_prefs.SortColumn);
	fprintf (gxmame_ini_file, "SortReverse=%i\r\n", gui_prefs.SortReverse);

/*Can be a problem for gxmame if all coordinates are zero ?*/
#ifndef KXMAME
	gtk_window_get_position (GTK_WINDOW (MainWindow), &gui_prefs.GUIPosX, &gui_prefs.GUIPosY);
#endif
	fprintf (gxmame_ini_file, "GUIPosX=%i\r\n", gui_prefs.GUIPosX);
	fprintf (gxmame_ini_file, "GUIPosY=%i\r\n", gui_prefs.GUIPosY);

#ifndef KXMAME	
	gdk_drawable_get_size (MainWindow->window, &gui_prefs.GUIWidth, &gui_prefs.GUIHeight);
#endif
	fprintf (gxmame_ini_file, "GUIWidth=%i\r\n", gui_prefs.GUIWidth);
	fprintf (gxmame_ini_file, "GUIHeight=%i\r\n", gui_prefs.GUIHeight);

	fprintf (gxmame_ini_file, "ListMode=%i\r\n", gui_prefs.current_mode);

	fprintf (gxmame_ini_file, "Splitters=%i,%i",
#ifdef KXMAME
		gui_prefs.Splitters[0], gui_prefs.Splitters[1]);
#else
		gui_prefs.ShowFolderList ? main_gui.scrolled_window_filters->allocation.width : gui_prefs.Splitters[0],
		gui_prefs.ShowScreenShot ? main_gui.scrolled_window_games->allocation.width : gui_prefs.Splitters[1]);

	fputs ("\r\nListFont=", gxmame_ini_file);
	if (gui_prefs.ListFont)
		fputs (gui_prefs.ListFont, gxmame_ini_file);
#endif

	fputs ("\r\nColumnWidth=", gxmame_ini_file);
	for (i = 0; i < NUMBER_COLUMN; i++)
	{
		fprintf (gxmame_ini_file, "%i", gui_prefs.ColumnWidth[i]);

		if (i != NUMBER_COLUMN - 1)
			fputs (",", gxmame_ini_file);
	}

	fputs ("\r\nColumnOrder=", gxmame_ini_file);
	for (i = 0; i < NUMBER_COLUMN; i++)
	{
		fprintf (gxmame_ini_file, "%i", gui_prefs.ColumnOrder[i]);

		if (i != NUMBER_COLUMN - 1)
			fputs (",", gxmame_ini_file);
	}

	fputs ("\r\nColumnShown=", gxmame_ini_file);
	for (i = 0; i < NUMBER_COLUMN; i++)
	{
		fprintf (gxmame_ini_file, "%i", gui_prefs.ColumnShown[i]);
		if (i != NUMBER_COLUMN - 1)
			fputs (",", gxmame_ini_file);
	}
#ifdef KXMAME
	fprintf (gxmame_ini_file, "\r\nkxmameIconSize=%i\r\n", gui_prefs.iconSize);
	fprintf (gxmame_ini_file, "kxmameScaledScreenshot=%i\r\n", gui_prefs.scaledScreenshot);
	fprintf (gxmame_ini_file, "showVideoPreview=%i\r\n", gui_prefs.showVideoPreview);
	fprintf (gxmame_ini_file, "kxmameGeometry=%i,%i,%i,%i,%i\r\n",
		gui_prefs.kxmameGeometry[0],gui_prefs.kxmameGeometry[1],gui_prefs.kxmameGeometry[2],
		gui_prefs.kxmameGeometry[3],gui_prefs.kxmameGeometry[4]);
#endif
	fputs ("\r\n\r\n", gxmame_ini_file);

	fclose (gxmame_ini_file);
	return TRUE;
}


/* this the master config file, it will contain the path to every other config file
   when we will share these config files with another mame32 application */
gboolean
load_gxmamerc (void)
{
	FILE *gamerc_file;
	gchar *filename;
	gchar line[2000];
	gchar **option_value = NULL;
	gchar *mame_executable = NULL;
	gchar **xmame_executables;

	GXMAME_DEBUG ("Loading gxmamerc");

	gui_prefs.catverDirectory = g_build_filename (g_get_home_dir (), DOT_GXMAME, NULL);
	gui_prefs.CheatFile = g_build_filename (XMAME_ROOT, "cheat.dat", NULL);
	gui_prefs.HiscoreFile = g_build_filename (XMAME_ROOT, "hiscore.dat", NULL);
	gui_prefs.HistoryFile = g_build_filename (XMAME_ROOT, "history.dat", NULL);
	gui_prefs.MameInfoFile = g_build_filename (XMAME_ROOT, "mameinfo.dat", NULL);
	gui_prefs.sysinfoFile = g_build_filename (XMAME_ROOT, "sysinfo.dat", NULL);
	gui_prefs.clone_color.red = 47031;
	gui_prefs.clone_color.green = 47031;
	gui_prefs.clone_color.blue = 47031;
	
#ifdef KXMAME
	gui_prefs.Joystick_in_GUI = g_strdup ("/dev/js0");
#else
	gui_prefs.Joystick_in_GUI = g_strdup (get_joy_dev ());
#endif
	
	filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "gxmamerc", NULL);	
	gamerc_file = fopen (filename, "r");
	g_free (filename);
	if (!gamerc_file)
	{
		xmame_table_add ("xmame");
		GXMAME_DEBUG ("gxmamerc not found");
		return FALSE;
	}

	while (fgets (line, 500, gamerc_file))
	{
		/* skip comments */
		if (*line != '#')
		{	
			if (*line == '\n')
				break;
			/* FIXME: leaks here */
			option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\n", ' ')), "=", 0);
			if (option_value[1] != NULL)
			{
				if (!strcmp (option_value[0], "mame_executable"))
				{
					if (mame_executable)
						g_free (mame_executable);
					mame_executable = g_strdup (option_value[1]);
				} else if (!strcmp (option_value[0], "xmame_executables_array"))
				{
					gint i;
					xmame_executables = g_strsplit (option_value[1], ";", 0);
					if (xmame_executables)
					{
						for (i = 0; xmame_executables[i]; i++)
							xmame_table_add (xmame_executables[i]);
						g_strfreev (xmame_executables);
					}
					
				} else if (!strcmp (option_value[0], "catverDirectory"))
				{
					g_free (gui_prefs.catverDirectory);
					gui_prefs.catverDirectory = g_strdup (option_value[1]);
				} else if (!strcmp (option_value[0], "CheatFile"))
				{	
					g_free (gui_prefs.CheatFile);
					gui_prefs.CheatFile = g_strdup (option_value[1]);
				} else if (!strcmp (option_value[0], "HiscoreFile"))
				{	
					g_free (gui_prefs.HiscoreFile);
					gui_prefs.HiscoreFile = g_strdup (option_value[1]);
				} else if (!strcmp (option_value[0], "HistoryFile"))
				{	
					g_free (gui_prefs.HistoryFile);
					gui_prefs.HistoryFile = g_strdup (option_value[1]);
				} else if (!strcmp (option_value[0], "MameInfoFile"))
				{	
					g_free (gui_prefs.MameInfoFile);
					gui_prefs.MameInfoFile = g_strdup (option_value[1]);
				} else if (!strcmp (option_value[0], "Joystick_in_GUI"))
				{	
					g_free (gui_prefs.Joystick_in_GUI);
					gui_prefs.Joystick_in_GUI = g_strdup (option_value[1]);
				} else if (!strcmp (option_value[0], "clone_color.red"))
				{	
					gui_prefs.clone_color.red = atoi (option_value[1]);
				} else if (!strcmp (option_value[0], "clone_color.green"))
				{	
					gui_prefs.clone_color.green = atoi (option_value[1]);
				} else if (!strcmp (option_value[0], "clone_color.blue"))
				{	
					gui_prefs.clone_color.blue = atoi (option_value[1]);
				} else if (!strcmp (option_value[0], "SysInfoFile"))
				{	
					g_free (gui_prefs.sysinfoFile);
					gui_prefs.sysinfoFile = g_strdup (option_value[1]);
				} 
			}
			g_strfreev (option_value);
		}
	}
	fclose (gamerc_file);

	/* set the current_executable */
	if (mame_executable)
	{
		current_exec = xmame_table_get (mame_executable);
		g_free (mame_executable);
	}
	
	if (!current_exec)
		current_exec = xmame_table_get_by_index (0);

	return TRUE;
}


gboolean
save_gxmamerc (void)
{
	FILE *gamerc_file;
	gchar *filename;
	guint i;
	
	filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "gxmamerc", NULL);	
	gamerc_file = fopen (filename, "w");
	g_free (filename);
	if (!gamerc_file)
	{
		GXMAME_DEBUG ("unable to write gxmamerc");
		return (FALSE);
	}
	fputs ("# master config file of GXMame\n", gamerc_file);
	if (current_exec)
		fprintf (gamerc_file, "mame_executable=%s\n", current_exec->path);

	if (xmame_table_size () > 0)
	{
		gchar **exec_paths;
		exec_paths = xmame_table_get_all ();
		fputs ("xmame_executables_array=", gamerc_file);
		for (i = 0; exec_paths[i] != NULL; i++)
		{	
			fputs (exec_paths[i], gamerc_file);
			if (exec_paths[i + 1] != NULL)
				fputs (";", gamerc_file);
		}
		g_free (exec_paths);
	}

	fprintf (gamerc_file, "\ncatverDirectory=%s", gui_prefs.catverDirectory);
	fprintf (gamerc_file, "\nCheatFile=%s", gui_prefs.CheatFile);
	fprintf (gamerc_file, "\nHiscoreFile=%s", gui_prefs.HiscoreFile);
	fprintf (gamerc_file, "\nHistoryFile=%s", gui_prefs.HistoryFile);
	fprintf (gamerc_file, "\nMameInfoFile=%s", gui_prefs.MameInfoFile);
	fprintf (gamerc_file, "\nSysInfoFile=%s", gui_prefs.sysinfoFile);
	fprintf (gamerc_file, "\nJoystick_in_GUI=%s", gui_prefs.Joystick_in_GUI);
	fprintf (gamerc_file, "\nclone_color.red=%i", gui_prefs.clone_color.red);
	fprintf (gamerc_file, "\nclone_color.green=%i", gui_prefs.clone_color.green);
	fprintf (gamerc_file, "\nclone_color.blue=%i\n", gui_prefs.clone_color.blue);
	fclose (gamerc_file);
	return TRUE;
}


/* this is where directory paths are set (common with mame32k) */
gboolean
load_dirs_ini (void)
{
	FILE *dirs_ini_file;
	gchar *filename;
	gchar line[2000];
	gchar **option_value = NULL;
	
	gui_prefs.RomPath = g_strsplit (XMAME_ROOT "/roms/", ";", 2);
	gui_prefs.messBiosDirectory = g_build_filename (XMAME_ROOT, "bios", NULL);
	gui_prefs.SamplePath = g_strsplit (XMAME_ROOT "/samples/", ";", 2);
	gui_prefs.SnapshotDirectory = g_build_filename (XMAME_ROOT, "snap", NULL);
	gui_prefs.messSnapDirectory = g_build_filename (XMAME_ROOT, "messSnap", NULL);
	gui_prefs.VideoDirectory = g_build_filename (XMAME_ROOT, "video", NULL);
	gui_prefs.ArtworkDirectory = g_build_filename (XMAME_ROOT, "artwork", NULL);
	//gui_prefs.HiscoreDirectory = g_build_filename (g_get_home_dir () , ".xmame", "hi", NULL);
	gui_prefs.FlyerDirectory = g_build_filename (XMAME_ROOT, "flyers", NULL);
	gui_prefs.CabinetDirectory = g_build_filename (XMAME_ROOT, "cabinets", NULL);
	gui_prefs.MarqueeDirectory = g_build_filename (XMAME_ROOT, "marquees", NULL);
	gui_prefs.TitleDirectory = g_build_filename (XMAME_ROOT, "titles", NULL);
	gui_prefs.IconDirectory = g_build_filename (XMAME_ROOT, "icons", NULL);
	
	//gui_prefs.DiffDirectory = g_build_filename (g_get_home_dir (), ".xmame", "diff", NULL);
	gui_prefs.CtrlrDirectory = g_build_filename (XMAME_ROOT, "ctrlr", NULL);
	
	//gui_prefs.InputDirectory = g_build_filename (g_get_home_dir (), ".xmame", "inp", NULL);
	//gui_prefs.NVRamDirectory = g_build_filename (g_get_home_dir (), ".xmame", "nvram", NULL);
	//gui_prefs.MemCardDirectory = g_build_filename (g_get_home_dir (), ".xmame", "memcard", NULL);
	//gui_prefs.ConfigDirectory = g_build_filename (g_get_home_dir (), ".xmame" , "cfg", NULL);
	//gui_prefs.StateDirectory = g_build_filename (g_get_home_dir (), ".xmame", "sta", NULL);
	//gui_prefs.inipath = g_build_filename (g_get_home_dir (), ".xmame" , "ini", NULL);
	
	filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "dirs.ini", NULL);	
	dirs_ini_file = fopen (filename, "r");
	g_free (filename);
	if (!dirs_ini_file)
	{
		GXMAME_DEBUG ("dirs.ini not found");
		return FALSE;
	}
	/* Be careful of the \r\n terminal due to windows */
	while (fgets (line, 500, dirs_ini_file))
	{
		if (!strncmp (line, "[directories]", 13))
		{	
			/* FIXME : this is to prevent loss of directory preference when 
			           the directories preference is mor than 500 character long
				   we will have the same problem if it longer then 2000 */
			while (fgets (line, 2000, dirs_ini_file))
			{/* need to test the carriage return if the file come from Mame32K under windows*/
				if (*line == '\r' || *line == '\n')
					break;
				option_value = g_strsplit (g_strstrip (g_strdelimit (line, "[]\r\n", ' ')), "=", 0);
				if (option_value[1] != NULL)
				{	if (!strcmp (option_value[0], "RomPath"))
					{	
						g_strfreev (gui_prefs.RomPath);
						gui_prefs.RomPath = g_strsplit (option_value[1], ";", 0);
					} else if (!strcmp (option_value[0], "SamplePath"))
					{	
						g_strfreev (gui_prefs.SamplePath);
						gui_prefs.SamplePath = g_strsplit (option_value[1], ";", 0);
/*					} else if (!strcmp (option_value[0], "ConfigDirectory"))
					{	
						g_free (gui_prefs.ConfigDirectory);
						gui_prefs.ConfigDirectory = g_strdup (option_value[1]);*/
/*					} else if (!strcmp (option_value[0], "FolderDirectory"))
					{	
						g_free (gui_prefs.FolderDirectory);
						gui_prefs.FolderDirectory = g_strdup (option_value[1]);
*//*					} else if (!strcmp (option_value[0], "NVRamDirectory"))
					{	
						g_free (gui_prefs.NVRamDirectory);
						gui_prefs.NVRamDirectory = g_strdup (option_value[1]);*/
					} else if (!strcmp (option_value[0], "CtrlrDirectory"))
					{	
						g_free (gui_prefs.CtrlrDirectory);
						gui_prefs.CtrlrDirectory = g_strdup (option_value[1]);
/*					} else if (!strcmp (option_value[0], "MemCardDirectory"))
					{	
						g_free (gui_prefs.MemCardDirectory);
						gui_prefs.MemCardDirectory = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "InputDirectory"))
					{	
						g_free (gui_prefs.InputDirectory);
						gui_prefs.InputDirectory = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "HiscoreDirectory"))
					{	
						g_free (gui_prefs.HiscoreDirectory);
						gui_prefs.HiscoreDirectory = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "StateDirectory"))
					{	
						g_free (gui_prefs.StateDirectory);
						gui_prefs.StateDirectory = g_strdup (option_value[1]);*/
					} else if (!strcmp (option_value[0], "ArtworkDirectory"))
					{
						g_free (gui_prefs.ArtworkDirectory);
						gui_prefs.ArtworkDirectory = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "SnapshotDirectory"))
					{	
						g_free (gui_prefs.SnapshotDirectory);
						gui_prefs.SnapshotDirectory = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "VideoDirectory"))
					{	
						g_free (gui_prefs.VideoDirectory);
						gui_prefs.VideoDirectory = g_strdup (option_value[1]);
/*					} else if (!strcmp (option_value[0], "WavDirectory"))
					{
						g_free (gui_prefs.WavDirectory);
						gui_prefs.WavDirectory = g_strdup (option_value[1]);
*/					} else if (!strcmp (option_value[0], "FlyerDirectory"))
					{
						g_free (gui_prefs.FlyerDirectory);
						gui_prefs.FlyerDirectory = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "CabinetDirectory"))
					{
						g_free (gui_prefs.CabinetDirectory);
						gui_prefs.CabinetDirectory = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "MarqueeDirectory"))
					{	
						g_free (gui_prefs.MarqueeDirectory);
						gui_prefs.MarqueeDirectory = g_strdup (option_value[1]);
					} else if (!strcmp (option_value[0], "TitleDirectory"))
					{	
						g_free (gui_prefs.TitleDirectory);
						gui_prefs.TitleDirectory = g_strdup (option_value[1]);
/*					} else if (!strcmp (option_value[0], "DiffDirectory"))
					{	
						g_free (gui_prefs.DiffDirectory);
						gui_prefs.DiffDirectory = g_strdup (option_value[1]);*/
					} else if (!strcmp (option_value[0], "IconDirectory"))
					{	
						g_free (gui_prefs.IconDirectory);
						gui_prefs.IconDirectory = g_strdup (option_value[1]);
/*					} else if (!strcmp (option_value[0], "inipath"))
					{	
						g_free (gui_prefs.inipath);
						gui_prefs.inipath = g_strdup (option_value[1]);*/
					} else if (!strcmp (option_value[0], "MessBIOSDirectory"))
					{	
						g_free (gui_prefs.messBiosDirectory);
						gui_prefs.messBiosDirectory = g_strdup (option_value[1]);
					}  else if (!strcmp (option_value[0], "MessSnapshotDirectory"))
					{	
						g_free (gui_prefs.messSnapDirectory);
						gui_prefs.messSnapDirectory = g_strdup (option_value[1]);
					} 
				g_strfreev (option_value);
				}
			}
		}
	}
	fclose (dirs_ini_file);
	return TRUE;
}


gboolean
save_dirs_ini (void)
{
	FILE *dir_ini_file;
	gchar *filename;
	int i;
	
	filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "dirs.ini", NULL);	
	dir_ini_file = fopen (filename, "w");
	g_free (filename);
	if (!dir_ini_file)
	{
		GXMAME_DEBUG ("unable to write dirs.ini");
		return (FALSE);
	}

	fputs ("[directories]", dir_ini_file);
	fputs ("\r\nRomPath=", dir_ini_file);
	for (i = 0; gui_prefs.RomPath[i] != NULL; i++)
	{	
		fputs (gui_prefs.RomPath[i], dir_ini_file);
		if (gui_prefs.RomPath[i+1] != NULL)
			fputs (";", dir_ini_file);
	}
	fputs ("\r\nSamplePath=", dir_ini_file);
	for (i = 0; gui_prefs.SamplePath[i] != NULL; i++)
	{
		fputs (gui_prefs.SamplePath[i], dir_ini_file);
		if (gui_prefs.SamplePath[i+1] != NULL)
			fputs (";", dir_ini_file);
	}
/*	fputs ("\r\nConfigDirectory=", dir_ini_file);
	fputs (gui_prefs.ConfigDirectory, dir_ini_file);
	fputs ("\r\nNVRamDirectory=", dir_ini_file);
	fputs (gui_prefs.NVRamDirectory, dir_ini_file);*/
	fputs ("\r\nCtrlrDirectory=", dir_ini_file);
	fputs (gui_prefs.CtrlrDirectory, dir_ini_file);
/*	fputs ("\r\nMemCardDirectory=", dir_ini_file);
	fputs (gui_prefs.MemCardDirectory, dir_ini_file);
	fputs ("\r\nInputDirectory=", dir_ini_file);
	fputs (gui_prefs.InputDirectory, dir_ini_file);
	fputs ("\r\nHiscoreDirectory=", dir_ini_file);
	fputs (gui_prefs.HiscoreDirectory, dir_ini_file);
	fputs ("\r\nStateDirectory=", dir_ini_file);
	fputs (gui_prefs.StateDirectory, dir_ini_file);*/
	fputs ("\r\nArtworkDirectory=", dir_ini_file);
	fputs (gui_prefs.ArtworkDirectory, dir_ini_file);
	fputs ("\r\nSnapshotDirectory=", dir_ini_file);
	fputs (gui_prefs.SnapshotDirectory, dir_ini_file);
	fputs ("\r\nVideoDirectory=", dir_ini_file);
	fputs (gui_prefs.VideoDirectory, dir_ini_file);
	fputs ("\r\nFlyerDirectory=", dir_ini_file);
	fputs (gui_prefs.FlyerDirectory, dir_ini_file);
	fputs ("\r\nCabinetDirectory=", dir_ini_file);
	fputs (gui_prefs.CabinetDirectory, dir_ini_file);
	fputs ("\r\nMarqueeDirectory=", dir_ini_file);
	fputs (gui_prefs.MarqueeDirectory, dir_ini_file);
	fputs ("\r\nTitleDirectory=", dir_ini_file);
	fputs (gui_prefs.TitleDirectory, dir_ini_file);
/*	fputs ("\r\nDiffDirectory=", dir_ini_file);
	fputs (gui_prefs.DiffDirectory, dir_ini_file);*/
	fputs ("\r\nIconDirectory=", dir_ini_file);
	fputs (gui_prefs.IconDirectory, dir_ini_file);
/*	fputs ("\r\ninipath=", dir_ini_file);
	fputs (gui_prefs.inipath, dir_ini_file);*/
	fputs ("\r\nMessBIOSDirectory=", dir_ini_file);
	fputs (gui_prefs.messBiosDirectory, dir_ini_file);
	fputs ("\r\nMessSnapshotDirectory=", dir_ini_file);
	fputs (gui_prefs.messSnapDirectory, dir_ini_file);
	
	fputs ("\r\n\r\n", dir_ini_file);
	fclose (dir_ini_file);
	return TRUE;
}


void
quick_check (void)
{
	static gboolean quick_check_running = FALSE;
	GList *list_pointer;
	gchar *filename;
	gint nb_rom_not_checked;
	gfloat done;
	int i;
	RomEntry *rom;


	GXMAME_DEBUG ("Running quick check.");
	if (game_list.num_games == 0)
		return;

	/*prevent user to launch several quick check at the same time */
	if (quick_check_running)
	{
		GXMAME_DEBUG ("Quick check already running");
		return;
	}
	quick_check_running = TRUE;

#ifndef KXMAME
	show_progress_bar ();
#endif
	filename = (gchar *)g_malloc (200);
	nb_rom_not_checked = g_list_length (game_list.not_checked_list);
	k_message (i18n("Performing quick check, please wait:"));
#ifndef KXMAME
	/* Disable the callback */
	g_signal_handlers_block_by_func (G_OBJECT (gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (main_gui.scrolled_window_games))),
					 (gpointer)adjustment_scrolled, NULL);
#endif
	/* need to use the last or I can find the list anymore*/
	for (list_pointer = g_list_last (game_list.not_checked_list);list_pointer ;list_pointer = g_list_last (game_list.not_checked_list))
	{
		rom = (RomEntry *)list_pointer->data;

		/* Looking for roms */
		rom->has_roms = (RomStatus)0;
		for (i=0;gui_prefs.RomPath[i]!=NULL;i++)
		{
			g_snprintf (filename, 200, "%s" G_DIR_SEPARATOR_S "%s.zip", gui_prefs.RomPath[i], rom->romname);
			if (g_file_test (filename, G_FILE_TEST_EXISTS))
			{
				rom->has_roms = (RomStatus)1;
			}
			else
			{	/*test of directories*/
					g_snprintf (filename, 200, "%s" G_DIR_SEPARATOR_S "%s", gui_prefs.RomPath[i], rom->romname);
					if (g_file_test (filename, G_FILE_TEST_IS_DIR))
					{
						rom->has_roms = (RomStatus)1;
					}
					else if (strcmp (rom->cloneof, "-"))
				{
					g_snprintf (filename, 200, "%s" G_DIR_SEPARATOR_S "%s.zip", gui_prefs.RomPath[i], rom->cloneof);
					if (g_file_test (filename, G_FILE_TEST_EXISTS))
					{
						rom->has_roms = (RomStatus)1;
					}
						else
						{
							g_snprintf (filename, 200, "%s" G_DIR_SEPARATOR_S "%s", gui_prefs.RomPath[i], rom->cloneof);
							if (g_file_test (filename, G_FILE_TEST_IS_DIR))
							{
								rom->has_roms = (RomStatus)1;
							}
						}
					}
			}
			/* prevent to look into other folders if we already find the rom*/
			if (rom->has_roms) break;
		}

		/* Looking for samples */
		rom->has_samples = (RomStatus)0;

		if (rom->nb_samples > 0)
		{
			for (i = 0; gui_prefs.SamplePath[i] != NULL; i++)
			{
				g_snprintf (filename, 200, "%s" G_DIR_SEPARATOR_S "%s.zip", gui_prefs.SamplePath[i], rom->romname);
				if (g_file_test (filename, G_FILE_TEST_EXISTS))
		        {
				rom->has_samples = (RomStatus)1;
		        }
		        else
		        {	/*test of directories*/
					g_snprintf (filename, 200, "%s" G_DIR_SEPARATOR_S "%s", gui_prefs.SamplePath[i], rom->romname);
					if (g_file_test (filename, G_FILE_TEST_IS_DIR))
					{
						rom->has_samples = (RomStatus)1;
					}
					else if (strcmp (rom->sampleof, "-"))
			       	{
			       		g_snprintf (filename, 200, "%s" G_DIR_SEPARATOR_S "%s.zip", gui_prefs.SamplePath[i], rom->sampleof);
			       		if (g_file_test (filename, G_FILE_TEST_EXISTS))
			       		{
						rom->has_samples = (RomStatus)1;
			       		}
						else
						{
							g_snprintf (filename, 200, "%s" G_DIR_SEPARATOR_S "%s", gui_prefs.SamplePath[i], rom->sampleof);
							if (g_file_test (filename, G_FILE_TEST_IS_DIR))
							{
								rom->has_samples = (RomStatus)1;
							}
						}
					}
				}
				/* prevent to look into other folders if we already find the sample*/
				if (rom->has_samples) break;
			}
		}

		if (g_list_length (game_list.not_checked_list))
			game_list.not_checked_list = g_list_remove_link (game_list.not_checked_list, list_pointer);
		else {
			list_pointer->data = NULL;
			g_list_free_1 (list_pointer);
		}

		nb_rom_not_checked--;

		done = (gfloat) (game_list.num_games - nb_rom_not_checked) / (gfloat)game_list.num_games;

#ifndef KXMAME
		if (done >= 0.0 && done <= 1.0)
			update_progress_bar (done);
#endif

#ifdef KXMAME
/* FIXME */
#else
		/* if the game is in the list, update it */
		if (rom->is_in_list)
			update_game_in_list (rom);
#endif
	}
	g_free (filename);

#ifndef KXMAME
	hide_progress_bar ();
#endif
	quick_check_running= FALSE;

#ifndef KXMAME
	
	/* Re-Enable the callback */
	g_signal_handlers_unblock_by_func (G_OBJECT (gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (main_gui.scrolled_window_games))),
					   (gpointer)adjustment_scrolled, NULL);		
#endif
}

GList *
get_ctrlr_list (void)
{
	GList *ctrlr_list = NULL;
	GDir *di;
	const gchar *dent;
	gchar *filename;
	gsize filename_len;

	GXMAME_DEBUG ("Getting the ctrlr list. %s", gui_prefs.CtrlrDirectory);

	di = g_dir_open (gui_prefs.CtrlrDirectory, 0, NULL);
	if (!di)
	{
		GXMAME_DEBUG ("ERROR - unable to open folder %s", gui_prefs.CtrlrDirectory);
	}
	else
	{
		while ( (dent = g_dir_read_name (di)))
		{
			filename = g_build_filename (gui_prefs.CtrlrDirectory, dent, NULL);
			if (g_file_test (filename, G_FILE_TEST_IS_DIR))
				ctrlr_list = g_list_append (ctrlr_list, g_strdup (dent));

			if (g_str_has_suffix (filename, ".cfg")) {
				filename_len = strlen (dent) - 4;
				ctrlr_list = g_list_append (ctrlr_list, g_strndup (dent, filename_len));
			}

			g_free (filename);
		}
		g_dir_close (di);
	}

	return ctrlr_list;
}

