package com.limegroup.gnutella.gui;

import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.MissingResourceException;
import java.util.Properties;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;

import com.limegroup.gnutella.util.CommonUtils;


/**
 * This class constructs an <tt>Initializer</tt> instance that constructs
 * all of the necessary classes for the application.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public class Main {
	
	/** 
	 * Creates an <tt>Initializer</tt> instance that constructs the 
	 * necessary classes for the application.
	 *
	 * @param args the array of command line arguments
	 */
	public static void main(String args[]) {
	    
	    try {
	        sanityCheck();
	        Initializer.initialize(args);
        } catch(Throwable err) {
            showCorruptionError(err);
            System.exit(1);
        }
	}
	
	/**
	 * Display a standardly formatted internal error message
	 * coming from the backend.
	 *
	 * @param message the message to display to the user
	 *
	 * @param err the <tt>Throwable</tt> object containing information 
	 *  about the error
	 */	
	private static final void showCorruptionError(Throwable err) {
		err.printStackTrace();
		final Properties PROPS = System.getProperties();
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.println("STARTUP ERROR");
		
		pw.print("LimeWire version " + CommonUtils.getLimeWireVersion());
		pw.print("Java version ");
		pw.print(prop("java.version"));
		pw.print(" from ");
		pw.println(prop("java.vendor"));
		pw.print(prop("os.name"));
		pw.print(" v. ");
		pw.print(prop("os.version"));
		pw.print(" on ");
		pw.println(prop("os.arch"));
		Runtime runtime = Runtime.getRuntime();
		pw.println("Free/total memory: "
				   +runtime.freeMemory()+"/"+runtime.totalMemory());
		pw.println();
		
        err.printStackTrace(pw);
        
		File propsFile = new File(CommonUtils.getUserSettingsDir(),
								  "limewire.props");
		Properties props = new Properties();
		try {
			FileInputStream fis = new FileInputStream(propsFile);
			props.load(fis);
			fis.close();
			// list the properties in the PrintWriter.
			props.list(pw);
		} catch(FileNotFoundException fnfe) {
		} catch(IOException ioe) {
		}
		
	    if( err instanceof ExceptionInInitializerError ) {
	        ExceptionInInitializerError t = (ExceptionInInitializerError)err;
	        if( t.getException() != null ) {
    	        pw.println("Underlying error: ");
    	        t.getException().printStackTrace(pw);
    	        pw.println();
            }
        }		   		

        pw.println("");
        pw.println("");
        pw.println("");
		pw.println("FILES IN CURRENT DIRECTORY:");
        File curDir = new File(PROPS.getProperty("user.dir"));
        String[] files = curDir.list();
        for(int i=0; i<files.length; i++) {
            File curFile = new File(curDir, files[i]);
            pw.println(curFile.toString());
            pw.println("LAST MODIFIED: "+curFile.lastModified());
            pw.println("SIZE: "+curFile.length());
            pw.println();
        }

		pw.flush();
		
        displayError(sw.toString());
	}

    /** 
	 * Returns the System property with the given name, or
     * "?" if it is unknown. 
	 */
    private static final String prop(String name) {
        String value = System.getProperty(name);
        if (value == null) return "?";
        else return value;
    }
    
	/**
	 * Displays an internal error with specialized formatting.
	 */
    private static final void displayError(String error) {
		final JDialog DIALOG = new JDialog();
		DIALOG.setModal(true);
		final Dimension DIALOG_DIMENSION = new Dimension(260, 120);
		final Dimension INNER_SIZE = new Dimension(220, 100);
		DIALOG.setSize(DIALOG_DIMENSION);

        JPanel mainPanel = new JPanel();
        mainPanel.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));
		mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.Y_AXIS));


		String instr0;
		String instr1;
		String instr2;
		String instr3;
		String instr4;
		String instr5;
		
        instr0 = "One or more necessary files appear to be invalid.";
        instr1 = "This is generally caused by a corrupted installation.";
        instr2 = "Please try downloading and installing LimeWire again.";
        instr3 = "If the problem persists, please copy and paste the";
        instr4 = "message below and send it to: ";
        instr5 = "bugs@limewire.com.  Thank you.";

		JLabel label0 = new JLabel(instr0);
		JLabel label1 = new JLabel(instr1);
		JLabel label2 = new JLabel(instr2);
		JLabel label3 = new JLabel(instr3);
		JLabel label4 = new JLabel(instr4);
		JLabel label5 = new JLabel(instr5);
		
		JPanel labelPanel = new JPanel();
		JPanel innerLabelPanel = new JPanel();
		labelPanel.setLayout(new BoxLayout(labelPanel, BoxLayout.X_AXIS));
		innerLabelPanel.setLayout(new BoxLayout(innerLabelPanel, BoxLayout.Y_AXIS));
		innerLabelPanel.add(label0);
		innerLabelPanel.add(label1);
		innerLabelPanel.add(label2);
		innerLabelPanel.add(label3);
		innerLabelPanel.add(label4);
		innerLabelPanel.add(label5);
		innerLabelPanel.add(Box.createVerticalStrut(6));
		labelPanel.add(innerLabelPanel);
		labelPanel.add(Box.createHorizontalGlue());


        final JTextArea textArea = new JTextArea(error);
        textArea.selectAll();
        textArea.copy();
        textArea.setColumns(50);
        textArea.setEditable(false);
        JScrollPane scroller = new JScrollPane(textArea);
        scroller.setBorder(BorderFactory.createEtchedBorder());
		scroller.setPreferredSize(INNER_SIZE);


        JPanel buttonPanel = new JPanel();
        JButton copyButton = new JButton("Copy Selection");
        copyButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				textArea.copy();
			}
		});
        JButton quitButton = new JButton("Ok");
        quitButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e){
				DIALOG.dispose();
			}
		});
        buttonPanel.add(copyButton);
        buttonPanel.add(quitButton);

        mainPanel.add(labelPanel);
        mainPanel.add(scroller);
        mainPanel.add(buttonPanel);

        DIALOG.getContentPane().add(mainPanel);

		Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
		DIALOG.setLocation((screenSize.width - DIALOG_DIMENSION.width)/2,
						   (screenSize.height - DIALOG_DIMENSION.height)/2);
        DIALOG.pack();
		DIALOG.show();
    }	
	
    /**
     * Determines whether or not specific files exist and are the correct size.
     * If they do not exist or are the incorrect size, 
     * throws MissingResourceException.
     */         
    private static void sanityCheck() throws MissingResourceException {
        File test = new File("gpl.txt");
        boolean isCVS = false;
        
        // If the gpl.txt exists, then we're running off of CVS.
        if( test.exists() && test.isFile() ) {
            isCVS = true;
        }
        // If it doesn't, we're a production version.
        else {
            isCVS = false;
        }
        
        File collectionsJar = new File( isCVS ? "lib" : ".", "collections.jar");
        //File jl011Jar = new File(isCVS ? "lib" : ".", "jl011.jar");
        File logicryptoJar = new File(isCVS ? "lib" : ".", "logicrypto.jar");
        File themesJar = new File(isCVS ? "lib" : ".", "themes.jar");
        File xercesJar = new File(isCVS ? "lib" : ".", "xerces.jar");        
        File updateVer = new File("update.ver");
        File xmlWar = new File("xml.war");
        File messagesBundlesJar = new File("MessagesBundles.jar");
        
        if( !collectionsJar.exists() || !collectionsJar.isFile() || 
         collectionsJar.length() != 259265 )
            throw new MissingResourceException("invalid collections.jar", null, null);
        
        //if( !jl011Jar.exists() || !jl011Jar.isFile() ||
        // jl011Jar.length() != 132309 )
        //    throw new MissingResourceException("invalid jl011.jar", null, null);
            
        if( !logicryptoJar.exists() || !logicryptoJar.isFile() ||
         logicryptoJar.length() != 130255 )
            throw new MissingResourceException("invalid logicrypto.jar", null, null);
        
        if( !themesJar.exists() || !themesJar.isFile() )
            throw new MissingResourceException("invalid themes.jar", null, null);
            
        if( !updateVer.exists() || !updateVer.isFile() )
            throw new MissingResourceException("invalid update.ver", null, null);
            
        if( !xercesJar.exists() || !xercesJar.isFile() ||
          xercesJar.length() != 1812019 )
            throw new MissingResourceException("invalid xerces.jar", null, null);
            
        if( !xmlWar.exists() || !xmlWar.isFile() )
            throw new MissingResourceException("invalid xml.war", null, null);
            
        if( !isCVS && (!messagesBundlesJar.exists() || !messagesBundlesJar.isFile()) )
            throw new MissingResourceException("invalid MessagesBundles.jar", null, null);
            
    }   	
}
