package com.limegroup.gnutella.gui.library;

import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.MouseEvent;

import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.MouseInputListener;

/** 
 * This class handles the generation of a drag event from the dragging source.  
 * It is specialized to handle dragging and dropping within the library. 
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public class DragSourceListener implements MouseInputListener {


	private final LibraryMediator MEDIATOR = LibraryMediator.instance();

	private int _lowerRow;
	private int _upperRow;

	private boolean _potentialDrop;
	private boolean _dragging;
	private boolean _outsideSelection;

	/**
	 * Constructs a new DragSourceListener, setting the necessary member
	 * variables.
	 *
	 * @param library handle to the library mediator class
	 */
	DragSourceListener() {
		_outsideSelection = false;
	}

	/** 
	 * Handles resetting the current upper and lower row bounding variables for 
	 * accurately handling row selection. 
	 *
	 * @param e the <tt>MouseEvent</tt> instance containing data about 
	 *          the mouse press
	 */
	public void mousePressed(MouseEvent e) {		
		ListSelectionModel selectionModel = MEDIATOR.getTableSelectionModel();
		if(selectionModel.getMinSelectionIndex() < _lowerRow ||
		   selectionModel.getMaxSelectionIndex() > _upperRow)
			_outsideSelection = true;
		else
			_outsideSelection = false;
		_lowerRow = selectionModel.getMinSelectionIndex();
		_upperRow = selectionModel.getMaxSelectionIndex();
		if(_lowerRow != _upperRow) {
			MEDIATOR.setMultiSelection(true);
		}
		else {
			MEDIATOR.setMultiSelection(false);
		}
	}

	/** 
	 * Called when the mouse is released.  Ttranslates the mouse release 
	 * event's location into the tree's coordinate space to see if there is  
	 * a potential drop event (if the mouse has left the table's screen space).  
	 *
	 * @param e the <tt>MouseEvent</tt> instance containing data about 
	 *          the mouse release
	 */
	public void mouseReleased(MouseEvent e) {
		ListSelectionModel selectionModel = MEDIATOR.getTableSelectionModel();
		if(MEDIATOR.getTableMultiSelection() && !_dragging) {
			if(!_outsideSelection && !SwingUtilities.isRightMouseButton(e)) {
				int ic = MEDIATOR.getTableIndexCandidate();
				if(ic != -1) {
					MEDIATOR.setMultiSelection(false);
					selectionModel.setSelectionInterval(ic, ic);
				}
			}
		}
		
		if(_dragging) {
			// the mouse is released, so set the dragging flag
			// to false.
			_dragging = false;
			
			// and reset the cursors back to normal.
			MEDIATOR.setCursors(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
						
			// translate the point to the tree coordinate
			// system if this could be a drop event.
			if(_potentialDrop) {
				Point tableLocation = MEDIATOR.getTableLocation();
				Point treeLocation  = MEDIATOR.getTreeLocation();
				Dimension treeDim   = MEDIATOR.getTreeSize();
				
				Point mousePoint = e.getPoint();
				Point screenMousePoint = new Point((tableLocation.x+mousePoint.x),
												   (tableLocation.y+mousePoint.y));
				
				int treeMaxX = treeLocation.x + treeDim.width;
				int treeMaxY = treeLocation.y + treeDim.height;
				if(screenMousePoint.x>treeLocation.x &&
				   screenMousePoint.x<treeMaxX) {
					if(screenMousePoint.y>treeLocation.y &&
					   screenMousePoint.y<treeMaxY) {
						Point mouseTreePoint = new Point
						(screenMousePoint.x-treeLocation.x,
						 screenMousePoint.y-treeLocation.y);
						MEDIATOR.handleDropToTreePoint(mouseTreePoint);
					}
				}
			}			
		}	
		_potentialDrop = false;
	}
	

	/**
	 * Not used.  Let other listeners do it.
	 *
	 * @param e the <tt>MouseEvent</tt> instance containing data about 
	 *          the mouse click
	 */
	public void mouseClicked(MouseEvent e) { ; }	
	

	/** 
	 * Called when the mouse enters whatever component that has added this class 
	 * as a listener.  Sets the flag for a potential drop to false, since in our 
	 * cases you cannot drag and drop something within the same component.
	 *
	 * @param e the <tt>MouseEvent</tt> instance containing data about 
	 *          the mouse entering the <tt>Component</tt>
	 */
	public void mouseEntered(MouseEvent e) {
		_potentialDrop = false;
	}

	/** 
	 * Called when the mouse has "exited" the component that this listener is attached 
	 * to.  sets the drop flag to true, as we might be trying to drop 
	 * data into another component. 
	 *
	 * @param e the <tt>MouseEvent</tt> instance containing data about 
	 *          the mouse exiting the <tt>Component</tt>
	 */
	public void mouseExited(MouseEvent e) {
		_potentialDrop = true;
	}

	/**
	 * Called when the mouse is dragged. Sets the cursor to the drag cursor and 
	 * sets the dragging flag to true. 
	 *
	 * @param e the <tt>MouseEvent</tt> instance containing data about 
	 *          the mouse dragging
	 */
	public void mouseDragged(MouseEvent e) {
		if(MEDIATOR.canDragAndDrop()) {
			MEDIATOR.setCursors(Cursor.getPredefinedCursor(Cursor.MOVE_CURSOR));
			_dragging = true;
		}				
	}

	/** 
	 * Implements the mouseInputListener interface. Does nothing in this case.
	 *
	 * @param e the <tt>MouseEvent</tt> instance containing data about 
	 *          the mouse moving
	 */
	public void mouseMoved(MouseEvent e) {}
}








