package com.limegroup.gnutella.gui.search;

import com.limegroup.gnutella.*;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.gui.tables.ChatRenderer;
import com.limegroup.gnutella.gui.tables.HeaderMouseObserver;
import com.limegroup.gnutella.gui.tables.HeaderMouseListener;
import com.limegroup.gnutella.xml.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.gui.tables.LimeTableColumn;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.border.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import com.sun.java.util.collections.*;
import java.io.*;


/**
 * A panel for displaying the results of a single search.  Contains the list of
 * results plus information about the search.  Provides per-search GUI controls.
 * Delegates to a ResultPanelModel to actually stores results.  Holds onto a
 * TableLineGrouper to actually maintain the groupings. <i>Not all methods of
 * this are synchronized.</i>
 */
public class ResultPanel extends JPanel implements ThemeObserver {

    /** The query string */
    private String query;
    /** The XML metadata string, or null.  (Used for repeat search.) */
    private String richQuery;
    /** The MediaType associated with this query */
    private MediaType type;
    /** The GUID of the last search. (Use this to match up results.)
     *  May be a DummyGUID for the empty result list hack. */
    private volatile GUID guid;

    /** The time (in milliseconds) that we last received a Query Result */
    private long timeLastResultReceived;
    /** How long since the last query result received until we allow a repeat.
     *  Currently disabled (repeats are always allowed). */
    private final long REPEAT_SEARCH_DELAY = 0;

    /** The underlying model of the search results. This is implemented
     *  with list. (Call fireTableRowsInserted when data comes in.) */
    private TableRowFilter dataModel;
    /** The table of results.  (Call getSelectedRow when mouse is
     *  clicked.) */
    private JMultilineToolTipTreeTable table;
    /** True ifff results are grouped. */
    private boolean grouped;
    /** True iff sorted in normal order. */
    private boolean ascending=true;
    /** True iff this is a Result Tab for a 'browse host'.
     */
    private boolean isBrowseHostTab=false;
    /** A list of all top-level groups.  Used to efficiently implement
      * grouping.  grouper is not really used by this, but it is carried
      * around for use by SearchView. */
    private final TableLineGrouper grouper=new TableLineGrouper();
    //private TableLineGrouper normalGrouper;

    private boolean debug = false;

    private JPanel _tablePanel;
    private JScrollPane scrollpane;

    /**
     * package access boolean which is set to true by the
     * ColumnSizeChangeListener. If this variable is set to true,
     */
    boolean columnsSizeChanged = false;

    private ColumnSizeChangeListener sizeListener;

	/**
	 * Constant instance of <tt>SearchButtons</tt> for this panel.
	 */
	private final SearchButtons SEARCH_BUTTONS = new SearchButtons();

    private static final String GROUP_STRING=
        GUIMediator.getStringResource("SEARCH_PUBLIC_GROUP_STRING");

	/**
	 * Constant for the check box that groups and ungroups results.
	 */
    private final JCheckBox GROUP_BOX =
		new JCheckBox(GROUP_STRING, true);

	/**
	 * Specialized constructor for creating a "dummy" result panel.
	 * This should only be called once at search window creation-time.
	 */
	ResultPanel() {
		this(new GUID(new byte[16]), "", null, null, false);
		SEARCH_BUTTONS.setButtonsEnabled(false);
	}

    /**
     * @param guid the guid of the query.  Used to match results.
     * @param query the query string sent out.  Used on the tab.
     * @param type the media filter to apply, or null if none
     * @param richQuery the metadata portion of the query, or null if a
     *  simple query
     */
    ResultPanel(GUID guid, String query, MediaType type, String richQuery,
                boolean isBrowseHostTab) {
        this.type = type;
        this.guid=guid;
        this.query=query;
        this.richQuery=richQuery;
        this.grouped=GROUP_BOX.isSelected();
        this.setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
        this.isBrowseHostTab=isBrowseHostTab;

        this.dataModel=new TableRowFilter(new TableLine(),grouped, this);
        
        if( dataModel.getColumnCount() == 0 ) {
            DisplayManager.instance().resetPropsFile();
            this.dataModel=new TableRowFilter(new TableLine(),grouped, this);
        }   
        
        timeLastResultReceived = -1;

        //Create actual table.  Add listener for sorting.
        createTreeTable();


		GROUP_BOX.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent ae)  {
				try {
					ResultPanel.this.setGrouped(GROUP_BOX.isSelected());
				} catch(Throwable e) {
					GUIMediator.showInternalError(e, "Search.Grouper");
				}
			}
		});
        scrollpane = new JScrollPane(table);
        createPanel();
		GUIMediator.addThemeObserver(this);
    }

    private void markTimeLastResultReceived() {
        timeLastResultReceived = System.currentTimeMillis();
    }

    /**
     * Determines whether or not repeat search is currently enabled.
     * Repeat search will be disabled if, for example, the original
     * search was performed too recently.
     *
     * @return <tt>true</tt> if the repeat search feature is currently
     *  enabled, otherwise <tt>false</tt>
     */
    private boolean isRepeatSearchEnabled() {
        if (timeLastResultReceived == -1)
            return true;

        long curr = System.currentTimeMillis();
        return ((curr - timeLastResultReceived) > REPEAT_SEARCH_DELAY);
    }

    void repeatSearch() {
        if (isRepeatSearchEnabled()) {
            synchronized(SearchMediator.REPEAT_SEARCH_LOCK){
                resetPanel();
                SearchMediator.setTabDisplayCount(this);
                if (isBrowseHostTab) {
                    debug("ResultPanel.repeatSearch(): browse host...");
                    java.util.StringTokenizer st =
                        new java.util.StringTokenizer(query, ":");
                    String host = null;
                    int port = 0;
                    if (st.hasMoreTokens())
                        host = st.nextToken();
                    if (st.hasMoreTokens())
                        port = Integer.parseInt(st.nextToken());
                    if ((host != null) && (port != 0))
                        SearchMediator.reBrowseHost(host, port, this);
                }
                else {
                    debug("ResultPanel.repeatSearch(): regular...");
                    SearchMediator.repeatSearch(type, query, richQuery);
                }
            }
        }
    }

    private void resetPanel() {
        grouper.clear();
        dataModel = new TableRowFilter(new TableLine(),grouped, this);
        
        createTreeTable();
        scrollpane = new JScrollPane(table);
        createPanel();
    }

    private void createPanel(){
		if(_tablePanel != null)
			this.remove(_tablePanel);

        // Handle a double click on the list - trigger download
        // Right click causes cancel/download menu.
        MouseInputAdapter mouseListener = new MouseInputAdapter() {
            public void mouseClicked(MouseEvent me) {
				try {
					if(me.isPopupTrigger() || SwingUtilities.isRightMouseButton(me)) {
						handleClick(me);
						SearchMediator.showMenu(table,me.getX(),me.getY());
					}
					else if (me.getClickCount() >= 2) {
					    handleClick(me);
						SearchMediator.doDownload();
					} else if( SwingUtilities.isLeftMouseButton(me) ) {
					    handleClick(me);
					}
				} catch(Throwable e) {
					GUIMediator.showInternalError(e, "Search.MouseClicked");
				}
            }
            public void mousePressed(MouseEvent me) {
				try {
					if(me.isPopupTrigger() || SwingUtilities.isRightMouseButton(me)) {
						handleClick(me);
						SearchMediator.showMenu(table,me.getX(),me.getY());
					}
				} catch(Throwable e) {
					GUIMediator.showInternalError(e, "Search.MousePressed");
				}
            }
            public void mouseMoved(MouseEvent me) { }
        };

        MouseInputAdapter resizeListener = new MouseInputAdapter() {
            public void mouseReleased(MouseEvent me){
                //System.out.println("Sumeet - mouse released");
				try {
					if(!columnsSizeChanged){
						return;
					}
					commitColumnSizeChange();
				} catch(Throwable e) {
					GUIMediator.showInternalError(e, "Search.MouseReleased");
				}
            }
        };
        table.addMouseListener(mouseListener);
        table.addMouseMotionListener(mouseListener);

        JTableHeader header = table.getTableHeader();
        header.addMouseListener(resizeListener);

		_tablePanel = new JPanel();
        _tablePanel.setLayout(new BorderLayout());
		Color tableColor = ThemeFileHandler.TABLE_BACKGROUND_COLOR.getValue();
        _tablePanel.setBackground(tableColor);

        //The corner above the scrollbar is white because of _tablePanel.
        //So we create a 3D-looking blue square there.  It would
        //be nice if we had a class that took care of this.
        JPanel corner = new JPanel();
        corner.setBackground(table.getTableHeader().getBackground());
        corner.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
        scrollpane.setCorner(JScrollPane.UPPER_RIGHT_CORNER, corner);
        scrollpane.getViewport().setBackground(tableColor);
        _tablePanel.add(scrollpane, BorderLayout.CENTER);


		SEARCH_BUTTONS.transformDowloadButton();
		SEARCH_BUTTONS.setButtonEnabled(SearchButtons.CHAT_BUTTON_INDEX, true);
		SEARCH_BUTTONS.setButtonEnabled(SearchButtons.CHAT_BUTTON_INDEX, false);
        SEARCH_BUTTONS.setButtonEnabled(SearchButtons.BROWSE_BUTTON_INDEX,
                                        false);

		JPanel buttonPanel = new BoxPanel(BoxPanel.X_AXIS);
		buttonPanel.add(SEARCH_BUTTONS.getComponent());
        buttonPanel.add(GROUP_BOX);
        this.add(_tablePanel);
		this.add(buttonPanel);
    }

	/**
	 * Changes the selection in the table in response to a right-mouse click.
	 * This was contributed by Chance Moore.
	 */
	void handleClick(MouseEvent me) {

		int row = table.rowAtPoint(me.getPoint());
		//check its valid, should always be but cheap to check
		if(row < 0) return;
		if (!table.getSelectionModel().isSelectedIndex(row)) {
			//if right clicked row is not selected, make it so
			table.getSelectionModel().setSelectionInterval(row, row);
		}
	}

    private void commitColumnSizeChange(){
        HashMap map = new HashMap();//for passing to dummyPannel
        DisplayManager man = DisplayManager.instance();
        TableColumnModel tcm = table.getColumnModel();
        int size = tcm.getColumnCount();
        for(int i=0; i<size; i++){
            TableColumn col = tcm.getColumn(i);
            int newWidth = col.getPreferredWidth();
            map.put(col,new Integer(newWidth));
            Pair p = dataModel.getIndexSchemaPairForColumn(i);
            int offset = p.getKey();
            LimeXMLSchema schema=(LimeXMLSchema)p.getElement();
            String rawName = "";
            if(schema == null){
                rawName = DisplayManager.getKeyString(offset);
            }
            else{
                SchemaFieldInfo sfi=(SchemaFieldInfo)
                schema.getCanonicalizedFields().get(offset);
                rawName = sfi.getCanonicalizedFieldName();
            }
            man.setValue(rawName, newWidth);
        }
        man.commit();
        if(this!=SearchMediator.getDummyResultPanel())
            SearchMediator.columnSizeChanged(map);
        columnsSizeChanged = false;//OK...reset it.
    }

    void dummyCommitColumnSizeChanged(HashMap map){
        Iterator iter = map.keySet().iterator();
        while(iter.hasNext()){
            TableColumn tc = (TableColumn)iter.next();
            TableColumn currColumn= null;
            try{//some other RP may have changed the dummy RP...
                //therefore the column may not exist anymore...
                currColumn = table.getColumn(tc.getIdentifier());
            }catch (IllegalArgumentException e){//if col does not exist...skip
                continue;
            }
            int newWidth = ((Integer)map.get(tc)).intValue();
            currColumn.setPreferredWidth(newWidth);
        }
    }

    /** Creates this.table.
     *     @modifies this.table */
    private void createTreeTable() {
        table = new JMultilineToolTipTreeTable(dataModel);
        

		JTree tree = table.getTree();
        if(CommonUtils.isMacOSX()) {
            tree.setShowsRootHandles(false);
        }

		table.setLeafIcon(null);
		table.setClosedIcon(null);
		table.setOpenIcon(null);
        table.setDefaultRenderer(ChatHolder.class, new ChatRenderer());
        table.setDefaultRenderer(QualityHolder.class,new QualityRenderer());
        table.setDefaultRenderer(CachingEndpoint.class,
                                 new CachingEndpointRenderer());
        table.setDefaultRenderer(ResultSpeed.class,new ResultSpeedRenderer());

        sizeListener = new ColumnSizeChangeListener(this);

        for(int i = 0; i<dataModel.getColumnCount();i++){
            int size = dataModel.getPreferredWidth(i);
            TableColumn tc = table.getColumnModel().getColumn(i);
            tc.setPreferredWidth(size);
            tc.addPropertyChangeListener(sizeListener);
            tc.setHeaderRenderer( LimeTableColumn.HEADER_RENDERER );
            tc.setIdentifier(dataModel.getColumnId(i));
        }

        MouseInputListener listMouseListener = new HeaderMouseListener(
            new HeaderMouseObserver() {
                private Object getId(int col) {
                    return table.getColumnModel().getColumn(col).getIdentifier();
                }

                public void handleHeaderColumnLeftClick(Point p) {
                    JTableHeader th = table.getTableHeader();
                    int viewCol = th.columnAtPoint(p);
                    int column = table.convertColumnIndexToModel(viewCol);
                    table.sortColumn( getId(column), !ascending);
                    sort(column);
                 }

                 public void handleHeaderColumnPressed(Point p) {
                    JTableHeader th = table.getTableHeader();
                    int viewCol = th.columnAtPoint(p);
                    int column = table.convertColumnIndexToModel(viewCol);
                    if ( column != -1 ) {
                        table.setPressedColumnId(getId(column));
                        // force the table to redraw the column header
                        th.repaint( th.getHeaderRect(viewCol) );
                    }
                 }

                 public void handleHeaderColumnReleased(Point p) {
                    table.setPressedColumnId(null);

                    JTableHeader th = table.getTableHeader();
                    int viewCol = th.columnAtPoint(p);
                    int column = table.convertColumnIndexToModel(viewCol);
                    if ( column != -1 ) {
                        // force the table to redraw the column header
                        th.repaint( th.getHeaderRect(viewCol) );
                    }
                 }

                 public void handleHeaderPopupMenu(Point p) {
                    JPopupMenu menu = dataModel.getColumnSelectionMenu();
                    menu.show( ResultPanel.this, p.x, p.y);
                 }
            }
        );

        JTableHeader th = table.getTableHeader();
        th.addMouseListener(listMouseListener);
        th.addMouseMotionListener(listMouseListener);
        table.addTreeExpansionListener(new ResultExpansionListener()); 
        addListeners();
    }

    /**
     * @requires line not already in this.  If group!=null, then group is
     *  already a child of root and similar to line.
     * @modifies this
     * @effects Adds line to the appropriate place in this.If group is non-null
     *  and this is grouped, then line will be added as a child of group.
     *  <P>
     * Sumeet Thadani(8/15/01) added functionality. This adds the XMLTableLine
     * to the appropriate list of tableLines depending on the schema of the
     * table line. The normal TableLines do not have a schema and are kept
     * in a different list
     */
    void addLine(TableLine line, TableLine group) {
        if(this.isEmpty()) {
            SEARCH_BUTTONS.transformWishlistButton();
        }
        markTimeLastResultReceived();
        dataModel.addLine(line, group, grouped);
        // set the table to have a non-existent column
        // be the sorted one, so the sort icon is removed.
        table.sortColumn("", false);
        table.revalidate();
    }

    /**
     * @modifies this
     * @effects if group, ensures results in this are always
     *  categorized by group; otherwise they are shown in a flat structure.
     */
	private void setGrouped(boolean grouped) {
        //Note that this' lock is held.
        if (this.grouped==grouped)
            return;  //no action needed

        this.grouped=grouped;
        if (this.grouped)
            dataModel.group();
        else
            dataModel.ungroup();
        table.revalidate();
    }

    void launchBrowser(){
        String action = "";//initialize
        Iterator iter = getSelectedRows();
        TableLine line = null;
        if(iter.hasNext())//grab the first row
            line = (TableLine)iter.next();
        else//no rows selected? Get out
            return;

        LimeXMLDocument doc = line.getXMLDocument();
		if(doc == null) return;

        action = doc.getAction();
        try{
            GUIMediator.openURL(action);
        }catch(IOException ioe){//could not open brower -- give up
        }
    }


    /** Returns the table line grouper for this.  Only one thread may modify
     *  this at a time, but it need not be thread safe.  Groups should be added
     *  to the grouper BEFORE adding to this.  This is a hack to work around the
     *  problem with grouping being too expensive to be in the Swing thread. */
    TableLineGrouper getGrouper() {
        return grouper;
    }

    /** Returns true if this is responsible for results with the given GUID */
    boolean matches(GUID guid) {
        return this.guid.equals(guid);
    }

    /**
     * @modifies this
     * @effects sets this' guid.  This is needed for browse host functionality.
     */
    void setGUID(GUID guid) {
        this.guid=guid;
    }

    /** Returns the query string this is responsible for. */
    String getQuery() {
        return query;
    }

    /** Returns the richquery string this is responsible for. */
    String getRichQuery() {
        return richQuery;
    }

    /** Returns the guid this is responsible for. */
    byte[] getGUID() {
        return guid.bytes();
    }

    /** Returns the media type this is responsible for. */
    MediaType getMediaType() {
        return type;
    }

    /** @return Whether or not this RP is the result of a browse host request or
     * not.
     */
    boolean isBrowseHostTab() {
        return this.isBrowseHostTab;
    }

    /**
     * @requires 0<=column<=4
     * @modifies this
     * @effects sorts this (in place) by the given column
     */
    synchronized void sort(int column) {
        //Note the lock above.
        dataModel.sort(column, ascending);
        table.revalidate();
        ascending=!ascending;
    }

    /** Returns the number of (real) results in this panel,
     *  including duplicates */
    int numResults() {
		return dataModel.numLeaves();
    }

    /**
     * returns the popup menu.
     * before the menu is sent to the parent for display, we disable parts of
     * it that are not relavent to the selected tableLines
     */
    JPopupMenu getPopup() {
        return SearchResultMenu.createMenu(getFirstSelectedTableLine(), 
                                           !areRowsSelected(), this);
    }


    /**
     * @requires this not modified while iterator in use
     * @effects returns an unmodifiable iterator of all selected lines in this,
     *  each a TableLine, in depth-first order, without duplicates.
     */
    Iterator getSelectedRows() {
        return new SelectedRowIterator();
    }


    private class SelectedRowIterator extends UnmodifiableIterator {
        private int next=0;
        private int[] selectedRows=table.getSelectedRows();
        public boolean hasNext() {
            return next<selectedRows.length;
        }
        public Object next() throws NoSuchElementException {
            try {
                int row=selectedRows[next];
                next++;
                TableLine ret=(TableLine)table.nodeForRow(row);
                return ret;
            } catch (IndexOutOfBoundsException e) {
                throw new NoSuchElementException();
            }
        }
    }

    /** Used to implement the following policy:whenever an expanded non-leaf is
     *  selected in table (other than root), all children are selected as
     *  well. This works in conjunction with ResultSelectionListener. */
    private class ResultSelectionListener implements ListSelectionListener {
        //TODO1: test cross-platform.
        public void valueChanged(ListSelectionEvent lse) {
			try {
				//Ignore extra messages
				if (lse.getValueIsAdjusting()) {
					SEARCH_BUTTONS.setButtonsEnabled(false);
					return;
				}
				ListSelectionModel lsm = (ListSelectionModel)lse.getSource();

				int index = lsm.getMinSelectionIndex();
				if(index != -1) {
					TableLine curLine = (TableLine)table.nodeForRow(index);
					if(curLine == null) return;
					SEARCH_BUTTONS.setButtonEnabled(
					    SearchButtons.CHAT_BUTTON_INDEX,
						curLine.getChatEnabled());
					SEARCH_BUTTONS.setButtonEnabled(
					    SearchButtons.DOWNLOAD_BUTTON_INDEX,
						true);
					SEARCH_BUTTONS.setButtonEnabled(
					    SearchButtons.BROWSE_BUTTON_INDEX,
						curLine.getBrowseHostEnabled());

				}

				else {
					// there is nothing selected, so disable the buttons that
					// require lines to be selected
					SEARCH_BUTTONS.setButtonsEnabled(false);
				}

				for (int i=lse.getFirstIndex(); i<=lse.getLastIndex(); i++) {
					if (lsm.isSelectedIndex(i)) {
						TableLine parent=(TableLine)table.nodeForRow(i);
						//Apparently this method can be called while sorting rows.
						//Under these circumstances, the lookup can fail.  So we
						//program defensively.
						if (parent==null)
							continue;
						if ((! dataModel.isLeaf(parent))
							&& table.isExpanded(i)) {
							int start=i+1;
							int stop=i+dataModel.getChildCount(parent);
							//This doesn't cause an infinite loop because children
							//leaves.
							lsm.addSelectionInterval(start, stop);
						}
					}
				}
			} catch(Throwable e) {
				GUIMediator.showInternalError(e, "Search.ListValueChanged");
			}
        }
    }

	/**
	 * Returns whether or not this <tt>ResultPanel</tt> is the currently
	 * selected panel.
	 *
	 * @return <tt>true</tt> if this panel is currently selected, <tt>false</tt>
	 *  otherwise
	 */
	//private boolean isSelectedPanel() {
	//return SearchMediator.isSelectedPanel(this);
	//}

    private void debug(String out){
        if(debug)
            System.out.println(out);
    }

    void columnsChanged(){
        ((TreeTableModelAdapter)table.getModel()).fireTableStructureChanged();
        for(int i = 0; i<dataModel.getColumnCount();i++) {
            int size = dataModel.getPreferredWidth(i);
            table.getColumnModel().getColumn(i).setPreferredWidth(size);
            TableColumn tc = table.getColumnModel().getColumn(i);
            tc.removePropertyChangeListener(sizeListener);
            tc.addPropertyChangeListener(sizeListener);
            tc.setHeaderRenderer( LimeTableColumn.HEADER_RENDERER );
            tc.setIdentifier(dataModel.getColumnId(i));
        }
        table.getTableHeader().setDraggedColumn(null);
        commitColumnSizeChange();
        if(this!=SearchMediator.getDummyResultPanel())//if I am not the dummy
            SearchMediator.columnsChanged();//ask parent to update dummy
        table.refreshTreeTableColumn();
        table.revalidate();
    }


    /**
     * called from StandardSearch view. performs the same as the columnsChanged
     * except that it does not call the method that calls it (otherwise
     * we have a never ending loop of repeated calls
     */
    void dummyColumnsChanged(){
        //System.out.println("Sumeet: OK. dummy updated");
        resetPanel();//create a new model
        ((TreeTableModelAdapter)table.getModel()).fireTableStructureChanged();
        for(int i = 0; i<dataModel.getColumnCount();i++){
            int size = dataModel.getPreferredWidth(i);
            table.getColumnModel().getColumn(i).setPreferredWidth(size);
        }
        table.revalidate();
    }

    /**
     * Called when the user is attempting to turn off the last column
     * <p>
     * package access.
     */
    void showColumnWarning(){
        GUIMediator.showError("ERROR_REMOVE_LAST_COL",
            QuestionsHandler.REMOVE_LAST_COLUMN);
    }

    /** Ensures that all lines unselected when collapsing tree. This
     *  works in conjunction with ResultSelectionListener to make sure
     *  selections work correctly when collapsing selected trees. */
    private class ResultExpansionListener implements TreeExpansionListener {
        public void treeCollapsed(TreeExpansionEvent event) {
            try {
                table.removeRowSelectionInterval(0, table.getRowCount()-1);
            } catch (IllegalArgumentException e) {
                //A crude way of dealing with race conditions!
            }
        }

        public void treeExpanded(TreeExpansionEvent event) {
            //Do nothing.
        }
    }

	/**
	 * Returns whether or not there are rows currently selected in the panel.
	 *
	 * @return <tt>true</tt> if there are rows currently selected, <tt>false</tt>
	 *  otherwise
	 */
	final boolean areRowsSelected() {
		return (table.getSelectedRow() != -1);
	}

	/**
	 * Returns whether or not this is an empty result panel with no results.
	 *
	 * @return <tt>true</tt> if there are no results in this <tt>ResultPanel</tt>
	 *  instance, <tt>false</tt> otherwise
	 */
	final boolean isEmpty() {
		return (dataModel.getRowCount() == 0);
	}

	/**
	 * Returns the <tt>TableLine</tt> for the currently selected row in the table,
	 * or <tt>null</tt> if no line is selected.
	 *
	 * @return the <tt>TableLine</tt> for the currently selected row in the table,
	 *  or <tt>null</tt> if no line is selected
	 */
	final TableLine getFirstSelectedTableLine() {
		if(!areRowsSelected()) return null;
		return (TableLine)table.nodeForRow(table.getSelectedRow());
	}

	// inherit doc comment
	public void updateTheme() {
		Color tableColor = ThemeFileHandler.TABLE_BACKGROUND_COLOR.getValue();
        _tablePanel.setBackground(tableColor);
        table.updateTheme();
		table.setLeafIcon(null);
		table.setClosedIcon(null);
		table.setOpenIcon(null);        
        scrollpane.getViewport().setBackground(tableColor);
        addListeners();
	}

    /**
     * Adds any necessary listeners.
     */
    private void addListeners() {
        //Enable fancy selection.  When clicking on a non-leaf node (other
        //than root), all children are selected.
        ListSelectionModel rowSM = table.getSelectionModel();
        rowSM.addListSelectionListener(new ResultSelectionListener());        
    }
}




















