package com.limegroup.gnutella.gui.search;

import com.limegroup.gnutella.*;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.xml.*;
import com.sun.java.util.collections.*;

/** 
 * The model for a result panel's JTreeTable.  Stores each search result as a
 * TableLine.  <i>This is not synchronized.</i>
 * <p>
 * It's an abstrtact class, that implements the basic functionality of 
 * a model, the specialized functions (which are not implemented by this class
 * are implemented by the it's subclasses.
 */
abstract class ResultPanelModel extends AbstractTreeTableModel {
    /** This is the fake grouped root, which we always maintain since it 
     *  is expensive to compute.  Note that fakeGroupedRoot's children may be 
     *  the children of the real root, but fakeGroupedRoot is never actually 
     *  the root itself.  The ungrouped root is computed as needed since it's
     *  cheap.
     *
     *  INVARIANTS: 
     *       fakeGroupedRoot!=getRoot()     
     *       grouped => fakeGroupedRoot.children=getRoot().children
     *       !grouped => fakeGroupedRoot.children!=getRoot().children
     *       !grouped => fakeGroupedRoot contains COPIES of the same elements
     *                   as getRoot, but the former and not the latter has
     *                   recursive structure.
     *       
     *  Also the invariants of TableLine must be maintained, namely that
     *  childName invariant.
     *
     *  TODO: this would be more robust if the grouped state were explicitly
     *  maintained here instead of in ResultPanel.
     */
    private final TableLine fakeGroupedRoot=new TableLine();
    /** INVARIANT: leaves==number of leaves in this. */
    private int leaves=0;

    /**
     * Creates a new ResultPanelModel rooted at root.  The majority of results
     * (i.e., unique files or non-leaf nodes) are direct children of root.
     * The results are grouped iff grouped==true.
     */
    ResultPanelModel(TableLine root, boolean grouped) {
        super(root); 
        if (grouped)
            fakeGroupedRoot.children=root.children;  //set up aliasing.
    }

    ////////////////////// TableModel interface ////////////////////////
    public boolean isCellEditable(int row, int col) {
        return (false);
    }

    public void setValueAt(Object aValue, int row, int column) {
        Assert.that(false, "Table not editable!");
    }

    ////////////////////// TreeModel interface ////////////////////////
    public int getChildCount(Object node) { 
        return ((TableLine)node).children.size();
    }

    public Object getChild(Object node, int i) { 
        return ((TableLine)node).children.get(i);
    }

    public boolean isLeaf(Object node) {
        return ((TableLine)node).children==null;
	}

    //////////////////// My Own MutableTreeNode-like Interface ////////////
    //See TableLine for an explanation of why these methods are here.//////
    ///////////////////////////////////////////////////////////////////////
    /**
     * @modifies parent, child
     * @effects add child to parent, as the last child.  
     *  If this is currently a leaf, it will become a non-leaf.
     */
    private void add(TableLine parent, TableLine child) {
        if (parent.children==null)
            parent.children=new ArrayList();
        parent.children.add(child);
        child.childName=parent.children.size()-1;
    }
    
    /** 
     * @requires line not already in this.  If group!=null, then group is
     *  already a child of root and similar to line.
     * @modifies this
     * @effects Adds line to the appropriate place in this. If group isnon-null
     *  and this is grouped, then line will be added as a child of group.
     *  If !grouped, a copy of TableLine may actually be added to this.
     */
    void addLine(TableLine line, TableLine group, boolean grouped) {        
        //repOk();
        //If grouped just add this normally
        if (grouped) {
            Assert.that(
                ((TableLine)getRoot()).children==fakeGroupedRoot.children,
                "No aliasing of fakedGroupedRoot's children.");
            addLineInternal(((TableLine)getRoot()), line, group);
        } 
        //If not, we need to update the real root without grouping as well
        //as the saved group root with grouping.  
        else {
            Assert.that(
                ((TableLine)getRoot()).children!=fakeGroupedRoot.children,
                "Aliasing of fakedGroupedRoot's children.");
            addLineInternal(fakeGroupedRoot, line, group);
            //Copy line to prevent aliasing.  This can't be done for the above
            //step because it screws up the grouping.  (We need the copies
            //getName value above, which is undefined.)
            //System.out.println("Sumeet ungrouped");
            addLineInternal(((TableLine)getRoot()), line.createClone(),
                            null); 
        }
        leaves++;
        //repOk();
    }
        
    /** 
     * Like add(line, grouped), but (1) root is considered to be the root and
     * (2) listeners are notified of model changes only if doNotify==true. 
     * Note that if group==null, line is not grouped.
     *     @modifies root.children or group, line
     */
    private void addLineInternal(TableLine root,
                                 TableLine line, TableLine group) {
        if (group!=null) {
            try {
                Assert.that(getChild(root, group.childName)==group,
                    "Group is not the "+group.childName+"'th child of root");
            } catch (IndexOutOfBoundsException e) {
                Assert.that(false,
                    "Group is not the "+group.childName+"'th child of root");
            }
        }

        //1. If line is grouped, add as a child of group
        if (group!=null) {
            //a) Some node is not a leaf and already has duplicates;
            //   append.
            if (! isLeaf(group)) {
                line.parentLine = group; //mantain invariant
                add(group, line);
                //Set parent speed to max, as needed.
                //I'm not quite sure which event to fire here.
                if (group.getSpeed().compareTo(line.getSpeed())<0)
                    group.setSpeed(line.getSpeed());
                //Set parent quality to max, as needed.
                //firing same event as above
                if (group.getQuality() < line.getQuality())
                    group.setQuality(line.getQuality());

				//Set parent chat status to enabled if needed,
				//firing same event as above
				if (!group.getChatEnabled() && line.getChatEnabled()) 
					group.setChatEnabled(true);
                setGroupXMLDocuments(group,line);
            } 
            //b) Some node matches but is a leaf.  Mutate that leaf into a
            //   multiple leaf and add a copy of the original as a child.  It is
            //   critical that we not remove group, as that will screw with the
            //   table line grouper.
            //   *IMPORTANT* We unMutate a TableLine sometimes. If we change
            //   the things we mutate here, we must also make the appropriate
            //   changes in TableLine.unMutate
            else {
                TableLine groupCopy=group.createClone(); //child
				// maintain invariants
				line.parentLine = group;
				groupCopy.parentLine = group;
				group.parentLine = null;
                add(group, groupCopy);
                add(group, line);                    
                //Even though we actually just mutated group, its as if we
                //removed it and added a new subtree.  TODO: is this right?
                group.setLocation(new CachingEndpoint(this, group)); 
                group.setSpeed(ResultSpeed.max(group.getSpeed(),
                                               line.getSpeed()));
                // set the quality  to be the max value
                group.setQuality(Math.max(group.getQuality(),
                                          line.getQuality()));
				group.setChatEnabled(group.getChatEnabled() || line.getChatEnabled());
                setGroupXMLDocuments(group,line);
            }
        }
        //2. No luck?  Insert as new leaf.
        else {
            line.parentLine = null;
            add(root, line);
        }
    }
    
    private void setGroupXMLDocuments(TableLine group, TableLine line) {
        LimeXMLDocument groupDoc = group.getXMLDocument();
        if(groupDoc==null) {//no doc in group
            LimeXMLDocument lineDoc = line.getXMLDocument();
            if(lineDoc!=null)
                group.setXMLDocument(lineDoc);
        }
    }


    /**
     * @requires this is ungrouped.  (This avoids the need for any recursion.)
     * @modifies this
     * @effects groups all duplicates together.  This should be called if 
     *  the USE_RESULT_GROUPING property has been changed; otherwise add(..)
     *  will do the right thing.
     */
    void group() {
        //repOk();
        //1. We've already done the grouping, so just set the current roots
        //children to be the grouped root. 
        TableLine root=(TableLine)getRoot();
        root.children=fakeGroupedRoot.children;
    }

    /**
     * @requires this is grouped
     * @modifies this
     * @effects converts duplicate groups into a flat structure.  This should be
     *  called if the USE_RESULT_GROUPING property has been changed; otherwise
     *  add(..)  will do the right thing.
     */
    void ungroup() {
        //repOk();
        //1. Set the current root to be a new, empty list.
        TableLine root=(TableLine)getRoot();
        root.children=new ArrayList(leavesCount());
        //2. Copy children of saved faked root into it.  This updates the
        //the childName variables, so we have to copy each TableLine.
        Iterator iter=fakeGroupedRoot.children.iterator();
        while ( iter.hasNext() ) {
            TableLine child=(TableLine)iter.next();
            if (isLeaf(child))
                addLineInternal(root, child.createClone(), null);
            else {
                for (int i=0; i<child.children.size(); i++) {
                    TableLine grandChild=(TableLine)child.children.get(i);
                    Assert.that(isLeaf(grandChild));
                    addLineInternal(root, grandChild.createClone(),
                                    null);
                }
            }
        }
    }

    /** 
     * @modifies this
     * @effects Sorts this by the given column.  If column isn't valid,
     *  this is unmodified.  This sort isn't recursive; it only sorts the
     *  children of the root.
     * <p>
     * @param pair a pair of (schema, offset) used to find exact field
     */
    boolean sort(Pair pair, boolean ascending) {
        
        TableLine root=(TableLine)getRoot();
        Collections.sort(root.children,
                         TableLine.newComparator(pair, ascending));
        //Update childName values to maintain invariant.  No need to update
        //grandchildren since sort isn't recursive.
        List rootChildren=root.children;
        for (int i=0; i<rootChildren.size(); i++) {
            TableLine line=(TableLine)rootChildren.get(i);
            line.childName=i;
        }
        return true;
    }

    /** Returns the number of leaves (i.e., results) in this. */
    private int leavesCount() {
        return leaves;
    }

    int getRowCount(){
        TableLine root = (TableLine)getRoot();
        return root.children.size();
    }

    Object getRow(int rowIndex){
        TableLine root = (TableLine)getRoot();
        return root.children.get(rowIndex);
    }


    /** Checks the internal representation, asserting false if any problems. */
    //boolean issuedWarning=false;
    //void repOk() {
//          if (! issuedWarning) {
//              System.out.println("*******************************************");
//              System.out.println("Calling repOk.  This may slow your program.");
//              System.out.println("*******************************************");
//              issuedWarning=true;
//          }
//          TableLine root=(TableLine)getRoot();
//          Assert.that(root!=null, "Null root");
//          int found=repOk(root);
//          Assert.that(leaves==found, "Expected "+leaves+", got "+found);
    //}

    /** Checks the internal representation, starting with parent 
     *  Returns the number of leaf nodes encountered, including parent. */
    private int repOk(TableLine parent) {
        if (parent.children==null)
            return 1;
        int ret=0;
        for (int i=0; i<parent.children.size(); i++) {
            TableLine child=(TableLine)parent.children.get(i);
            Assert.that(child!=null, "Null children");
            Assert.that(child.childName==i,
                "Child's name is "+child.childName+" but I expected "+i);
            ret+=repOk(child);  //always terminates since no cycles
        }
        return ret;
    }

}

