package com.limegroup.gnutella.update.gui;

import com.limegroup.gnutella.update.*;
import com.limegroup.gnutella.gui.*;
import javax.swing.border.*;
import javax.swing.*;
import java.awt.event.*;
import java.awt.*;

/** 
 * This class constructs a dialog window that shows the progress of 
 * an update. It implements <tt>Runnable</tt> so that it is easy to 
 * call on the event dispatch thread. 
 */
final class ProgressWindow extends JDialog implements Runnable {
	
	private final JProgressBar PROGRESS_BAR = new JProgressBar();
	private final JLabel LABEL = new JLabel();;
	
	private double _downloadSize;
	private double _amountRead;

	/** 
	 * Creates the JDialog and sets the initial
	 * values for amount read, file size, etc. 
	 */
	public ProgressWindow() {
		setSize(460, 140);
		
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				cancelUpdate();
			}
		});
		
		// get the screen size to center the component
		Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
		
		// center the window on the screen
		setLocation((screenSize.width-350)/2, (screenSize.height-140)/2);
		
		// panel for all of the components
		JPanel panel = new JPanel();
		
		// panel for the label
		JPanel labelPanel = new JPanel();
		
		// use box layout for both panels
		panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
		labelPanel.setLayout(new BoxLayout(labelPanel, BoxLayout.X_AXIS));
		Border border = BorderFactory.createEmptyBorder(12,12,12,12);
		panel.setBorder(border);
		PROGRESS_BAR.setStringPainted(true);
		PROGRESS_BAR.setString("0%");
		
		LABEL.setText("Contacting the LimeWire server...");
		
		// add the label to its panel
		labelPanel.add(LABEL);
		labelPanel.add(Box.createHorizontalGlue());
		
		_downloadSize = 1000;
		_amountRead = 0;
		
		
		String[] buttonLabelKeys = {
			"UPDATE_CANCEL_BUTTON_LABEL"
		};
		String[] toolTipKeys = {
			"UPDATE_CANCEL_BUTTON_TIP"
		};
		// add the cancel button's listener
		ActionListener buttonListener = new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				cancelUpdate();
			}
		};
		ActionListener[] listeners = {
			buttonListener
		};
		ButtonRow br = new ButtonRow(buttonLabelKeys, toolTipKeys, listeners);
		panel.add(labelPanel);
		panel.add(Box.createVerticalStrut(6));
		panel.add(PROGRESS_BAR);
		panel.add(Box.createVerticalStrut(6));
		panel.add(br);
		getContentPane().add(panel);
	}		
	
	/** 
	 * Implements <tt>Runnable</tt> interface.
	 * Calculates the new percent read and sets the progress bar 
	 * to that value. 
	 */
	public void run() {
		int percent = (int)(100*(_amountRead/_downloadSize));	
		PROGRESS_BAR.setValue(percent);
		PROGRESS_BAR.setString(Integer.toString(percent)+"%");
	}
	
	/** 
	 * Sets the name of the currently downloading file.
	 *
	 * @param fileName the name of the currently downloading file
	 */
	void setDownloadFile(String fileName) {
		LABEL.setText("Downloading: " + fileName + "...");
	}
		
	/** 
	 * Sets the amount read in bytes. 
	 */
	void setAmountRead(int amountRead) {
		_amountRead = (double)amountRead;
	}

	/**
	 * Sets the size of the jar file we are downloading to use for 
	 * calculating the percentage of the download that is complete.
	 */
	void setDownloadSize(int size) {
		_downloadSize = (double)size;
	}

	/**
	 * Interrupts the thread peforming the update and reverts back to 
	 * the previous version of the application.
	 */
	void cancelUpdate() {
		UpdateManager.instance().cancelUpdate();
	}
}
