/****************************************************************************
**
** $Id: TagDialogImpl.cpp,v 1.21 2004/01/11 22:30:25 riemer Exp $
**
** Copyright (C) 2001-2004 The LinCVS development team.
**    Tilo Riemer <riemer@lincvs.org>
**    Falk Brettschneider <gigafalk@yahoo.com>
**    Frank Hemer <frank@hemer.org>
**    Wim Delvaux <wim.delvaux@chello.be>
**    Jose Hernandez <joseh@tesco.net>
**    Helmut Koll <HelmutKoll@web.de>
**    Tom Mishima <tmishima@mail.at-m.or.jp>
**    Joerg Preiss <auba@auba.de>
**    Sven Trogisch <trogisch@iapp.de>
**
**
**----------------------------------------------------------------------------
**
**----------------------------------------------------------------------------
**
** LinCVS is available under two different licenses:
**
** If LinCVS is linked against the GPLed version of Qt 
** LinCVS is released under the terms of GPL also.
**
** If LinCVS is linked against a nonGPLed version of Qt 
** LinCVS is released under the terms of the 
** LinCVS License for non-Unix platforms (LLNU)
**
**
** LinCVS License for non-Unix platforms (LLNU):
**
** Redistribution and use in binary form, without modification, 
** are permitted provided that the following conditions are met:
**
** 1. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 2. It is not permitted to distribute the binary package under a name
**    different than LinCVS.
** 3. The name of the authors may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.
** 4. The source code is the creative property of the authors.
**    Extensions and development under the terms of the Gnu Public License
**    are limited to the Unix platform. Any distribution or compilation of 
**    the source code against libraries licensed other than gpl requires 
**    the written permission of the authors.
**
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR 
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
** WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
** ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY 
** DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL 
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
** NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
** SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
**
**
** LinCVS License for Unix platforms:
**
** This program is free software; you can redistribute it and/or modify 
** it under the terms of the GNU General Public License as published by 
** the Free Software Foundation; either version 2 of the License, or 
** (at your  option) any later version.  This program is distributed in 
** the hope that it will be useful, but WITHOUT ANY WARRANTY; without 
** even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
** PARTICULAR PURPOSE.
**
** See the GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software Foundation,
** Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
*****************************************************************************/

#include "config.h"

#include <qcombobox.h>
#include <qcheckbox.h>
#include <qdatetimeedit.h>
#include <qdatetime.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qlineedit.h>
#include <qstringlist.h>
#include <qpushbutton.h>
#include <qwhatsthis.h>

#include "TagDialogImpl.h"
#include "pixmapcache.h"
#include "globals.h"
#include "Validators.h"


/**
 * Constructs a TagDialogImpl which is a child of 'parent', with the name
 * 'name' and widget flags set to 'f'.  The dialog will by default be
 * modeless, unless you set 'modal' to TRUE to construct a modal dialog.
 */

TagDialogImpl::TagDialogImpl( QString module, QStringList * list, QString moduleName, 
			      const QIconSet &whatsThisIconSet,
			      QWidget * parent,  const char* name, bool, WFlags fl)
  : TagDialog(parent, name, true, fl),
    m_module(module),
    m_pSelectedTagList(list),
    m_topModuleName(moduleName),
    m_fetchOverall(TRUE),
    m_firstTagListOnEdit(FALSE),
    m_secondTagListOnEdit(FALSE),
    m_pOldFirstTagPixmap(NULL),
    m_pOldSecondTagPixmap(NULL),
    m_lastFirstTagLen(0),
    m_lastSecondTagLen(0)
{

  m_pWhatsThis->setIconSet(whatsThisIconSet);
  m_pWhatsThis->setMaximumWidth(m_pWhatsThis->height());

  if (bUseSmallIcons) setIcon( findEmbeddedPixmap( "Tag16x16" ) );
  else setIcon( findEmbeddedPixmap( "Tag32x32" ) );

  m_DateTime->dateEdit()->setOrder(LookAndFeel::g_dateTimeOrder);
  m_DateTime->setDateTime(QDateTime::currentDateTime());

  if (module.isNull()) RTagFrame->hide();

  m_pFirstTag->setAutoCompletion(FALSE);
  m_pFirstTag->setValidator(new TagValidator(m_pFirstTag));
  connect( m_pFirstTag, SIGNAL(textChanged(const QString &)), this, SLOT(textChangedFirstTag(const QString &)));
  connect( m_pFirstTag, SIGNAL(activated(const QString &)), this, SLOT(activatedFirstTag(const QString &)));
  connect( m_pFirstTag->lineEdit(), SIGNAL(selectionChanged()), this, SLOT(setSelectionFirstTag()));

  m_pSecondTag->setAutoCompletion(FALSE);
  m_pSecondTag->setValidator(new TagValidator(m_pSecondTag));
  connect( m_pSecondTag, SIGNAL(textChanged(const QString &)), this, SLOT(textChangedSecondTag(const QString &)));
  connect( m_pSecondTag, SIGNAL(activated(const QString &)), this, SLOT(activatedSecondTag(const QString &)));
  connect( m_pSecondTag->lineEdit(), SIGNAL(selectionChanged()), this, SLOT(setSelectionSecondTag()));

  m_Revision->setValidator(new RevisionValidator(m_Revision));

  connect( parent, SIGNAL(tagListFetched()), this, SLOT(tagListUpdated()));
  connect( this, SIGNAL(getProjectTags()), parent, SLOT(updateProjectTagList()) );
  connect( this, SIGNAL(getSelectedTags()), parent, SLOT(readCurrentTagList()));

  tagListUpdated();

  setCaption (name);
}

/** Destroys the object and frees any allocated resources */
TagDialogImpl::~TagDialogImpl()
{
    // no need to delete child widgets, Qt does it all for us
}

QString TagDialogImpl::getCommand() {

  QString command;
  if (!RTagButton->isChecked()) command = "tag ";
  else command = "rtag ";
  if (RevisionButton->isChecked()) command += "-r "+m_Revision->text()+" ";
  if (TagButton->isChecked()) command += "-r "+m_pSecondTag->currentText()+" ";
  if (DateButton->isChecked()) command += "-D \""+m_DateTime->dateTime().toString("yyyy-MM-dd hh:mm")+"\" ";

  switch( TagOperationGroup->id(TagOperationGroup->selected()) ) {

    case 0:
      command += m_pFirstTag->currentText();
      break;

    case 1:
      command += "-b " + m_pFirstTag->currentText();
      break;
      
    case 2:
      command += "-F " + m_pFirstTag->currentText();
      break;
      
    case 3:
      command += "-d " + m_pFirstTag->currentText();
      break;
  }

  if (RTagButton->isChecked()) command += " " + m_module;

  return command;
}

void TagDialogImpl::readProjectNameList() {
  m_ProjectTagList.clear();
  projectSettings->get(m_topModuleName,PROJECTTAGLIST,m_ProjectTagList);
}

void TagDialogImpl::fetchClicked() {

  if (m_fetchOverall ^= TRUE) {//overall mode
    emit getProjectTags();
  } else {
    emit getSelectedTags();
  }
}

void TagDialogImpl::tagListUpdated() {

  m_firstTagListOnUpdate = TRUE;
  m_secondTagListOnUpdate = TRUE;
  m_firstTagListOnEdit = FALSE;
  m_secondTagListOnEdit = FALSE;
  m_pFirstTag->clear();
  m_pSecondTag->clear();
  m_pFirstTag->insertItem("T: HEAD");
  m_pSecondTag->insertItem("T: HEAD");
  if (m_fetchOverall) {
    FetchButton->setText( tr("Fetch selected"));
    readProjectNameList();
    m_pFirstTag->insertStringList(m_ProjectTagList);
    m_pSecondTag->insertStringList(m_ProjectTagList);
  } else {
    FetchButton->setText( tr("Fetch all"));
    m_pFirstTag->insertStringList(*m_pSelectedTagList);
    m_pSecondTag->insertStringList(*m_pSelectedTagList);
  }
  setPixmaps(m_pFirstTag);
  m_oldFirstTagText = m_pFirstTag->currentText();
  m_firstTagListOnUpdate = FALSE;
  setPixmaps(m_pSecondTag);
  m_oldSecondTagText = m_pFirstTag->currentText();
  m_secondTagListOnUpdate = FALSE;
  m_pFirstTag->lineEdit()->selectAll();
}

void TagDialogImpl::setPixmaps(QComboBox * box) {
  int i;
  for (i = 0; i < box->count(); ++i) {
    QString txt = box->text(i);
    if (txt.startsWith("T: ")) box->changeItem(findEmbeddedPixmap("TagTag30x16"),txt.mid(3),i);
    else if (txt.startsWith("B: ")) box->changeItem(findEmbeddedPixmap("BranchTag30x16"),txt.mid(3),i);
  }
  if (box == m_pFirstTag) m_oldFirstTagIdx = 0;
  else if (box == m_pSecondTag) m_oldSecondTagIdx = 0;

  if (i == 0) {
    box->insertItem(findEmbeddedPixmap("Tag30x16"),"",box == m_pFirstTag ? m_oldFirstTagIdx : m_oldSecondTagIdx);
  }
  box->setCurrentItem(box == m_pFirstTag ? m_oldFirstTagIdx : m_oldSecondTagIdx);//removes T: or B: for the lineEdit
}

void TagDialogImpl::setSelectionFirstTag() {

  if (m_firstTagListOnUpdate) return;

  int start;
  int end;
  if (m_pFirstTag->lineEdit()->getSelection(&start,&end)) {
    m_lastFirstTagLen = start;
  }
}

void TagDialogImpl::setSelectionSecondTag() {

  if (m_secondTagListOnUpdate) return;

  int start;
  int end;
  if (m_pSecondTag->lineEdit()->getSelection(&start,&end)) {
    m_lastSecondTagLen = start;
  }
}

void TagDialogImpl::activatedFirstTag(const QString & s) {

  if (m_pOldFirstTagPixmap) {
    m_pFirstTag->changeItem(*m_pOldFirstTagPixmap,m_oldFirstTagText,m_oldFirstTagIdx);
    m_pOldFirstTagPixmap = NULL;
  }
  m_oldFirstTagText = s;
  m_oldFirstTagIdx = m_pFirstTag->currentItem();
  m_firstTagListOnEdit = FALSE;
  m_pFirstTag->lineEdit()->selectAll();
}

void TagDialogImpl::activatedSecondTag(const QString & s) {

  if (m_pOldSecondTagPixmap) {
    m_pSecondTag->changeItem(*m_pOldSecondTagPixmap,m_oldSecondTagText,m_oldSecondTagIdx);
    m_pOldSecondTagPixmap = NULL;
  }
  m_oldSecondTagText = s;
  m_oldSecondTagIdx = m_pSecondTag->currentItem();
  m_secondTagListOnEdit = FALSE;
  m_pSecondTag->lineEdit()->selectAll();
}

void TagDialogImpl::textChangedFirstTag(const QString &txt) {

  if (m_firstTagListOnUpdate) return;

  int idx = -1;
  for (int i = 0; i<m_pFirstTag->count(); ++i) {
    if (m_pFirstTag->text(i).startsWith(txt)) {
      idx = i;
      break;
    }
  }

  if (idx > -1 && (m_lastFirstTagLen < (int)txt.length()) || (m_pFirstTag->text(idx).length() == txt.length()) ) {

    m_firstTagListOnUpdate = TRUE;
    int cursorPos = m_pFirstTag->lineEdit()->cursorPosition();
    m_pFirstTag->setCurrentItem(idx);
    QString current = m_pFirstTag->currentText();
    activatedFirstTag(current);
    if (current.length() != txt.length()) m_pFirstTag->lineEdit()->setSelection(txt.length(),current.length()-txt.length());
    else m_pFirstTag->lineEdit()->setCursorPosition(cursorPos);
    m_firstTagListOnUpdate = FALSE;

  } else if (m_oldFirstTagIdx == m_pFirstTag->currentItem()) {

    if (!m_firstTagListOnEdit) {
      m_firstTagListOnEdit = TRUE;
      QPixmap const * p = m_pFirstTag->pixmap(m_oldFirstTagIdx);
      if (p) {
	m_pOldFirstTagPixmap = new QPixmap(*p);
      } else {
	m_pOldFirstTagPixmap = NULL;
      }
      m_pFirstTag->changeItem(findEmbeddedPixmap("Tag30x16"),m_oldFirstTagText,m_oldFirstTagIdx);
    }

  }

  m_lastFirstTagLen = txt.length();
}

void TagDialogImpl::textChangedSecondTag(const QString &txt) {

  if (m_secondTagListOnUpdate) return;

  int idx = -1;
  for (int i = 0; i<m_pSecondTag->count(); ++i) {
    if (m_pSecondTag->text(i).startsWith(txt)) {
      idx = i;
      break;
    }
  }

  if (idx > -1 && (m_lastSecondTagLen < (int)txt.length()) || (m_pSecondTag->text(idx).length() == txt.length()) ) {

    m_secondTagListOnUpdate = TRUE;
    int cursorPos = m_pSecondTag->lineEdit()->cursorPosition();
    m_pSecondTag->setCurrentItem(idx);
    QString current = m_pSecondTag->currentText();
    activatedSecondTag(current);
    if (current.length() != txt.length()) m_pSecondTag->lineEdit()->setSelection(txt.length(),current.length()-txt.length());
    else m_pSecondTag->lineEdit()->setCursorPosition(cursorPos);
    m_secondTagListOnUpdate = FALSE;

  } else if (m_oldSecondTagIdx == m_pSecondTag->currentItem()) {

    if (!m_secondTagListOnEdit) {
      m_secondTagListOnEdit = TRUE;
      QPixmap const * p = m_pSecondTag->pixmap(m_oldSecondTagIdx);
      if (p) {
	m_pOldSecondTagPixmap = new QPixmap(*p);
      } else {
	m_pOldSecondTagPixmap = NULL;
      }
      m_pSecondTag->changeItem(findEmbeddedPixmap("Tag30x16"),m_oldSecondTagText,m_oldSecondTagIdx);
    }
  }

  m_lastSecondTagLen = txt.length();
}

void TagDialogImpl::revisionToggled(bool state) {
  if (state) TagButton->setChecked(!state);
  if (state) DateButton->setChecked(!state);
}

void TagDialogImpl::tagToggled(bool state) {
  if (state) RevisionButton->setChecked(!state);
  if (state) DateButton->setChecked(!state);
}

void TagDialogImpl::dateToggled(bool state) {
  if (state) RevisionButton->setChecked(!state);
  if (state) TagButton->setChecked(!state);
}

void TagDialogImpl::okClicked () {
  if (m_pFirstTag->currentText().isEmpty()) ValidatorWarning::TagWarning(this);
  else if (RevisionButton->isChecked() && m_Revision->text().isEmpty()) ValidatorWarning::RevisionWarning(this);
  else if (TagButton->isChecked() && m_pSecondTag->currentText().isEmpty()) ValidatorWarning::TagWarning(this);
  else accept();
}

void TagDialogImpl::enterWhatsThisMode()
{
   QWhatsThis::enterWhatsThisMode();
}
