#include "oruby.h"
#include "stdarg.h"

#define RUBY_HASH     1
#define RUBY_ARRAY    2

static int _lvl = -1;
static int _lvl_type[256];

// Indentation functions

static int indent_level = 0;
static char *indentation_string = "                                        ";

char*
indentation() {
  return indentation_string + 40 - 2 * indent_level;
}

void
indent_inc() { indent_level++; }

void
indent_dec() { indent_level--; }

// Ruby output functions

void oruby_end_low(char c) {
  indent_dec();
  printf("%s%c%s\n", indentation(), c, (_lvl >= 0) ? "," : "");
}

void oruby_end() {
  switch(_lvl_type[_lvl--]) {
    case RUBY_HASH:
      oruby_end_low('}');
      break;
    case RUBY_ARRAY:
      oruby_end_low(']');
      break;
    default:
      fprintf(stderr, "%s(%d): Unknown block type\n", __FILE__, __LINE__);
      exit(1);
      break;
  }
}

void oruby_start_low(char c) {
  printf("%s%c\n",indentation(), c);
  indent_inc();
}

void oruby_start_array() {
  oruby_start_low('[');
  _lvl_type[++_lvl] = RUBY_ARRAY;
}

void oruby_start_hash() {
  oruby_start_low('{');
  _lvl_type[++_lvl] = RUBY_HASH;
}

void
oruby_hash_entry_1(const char *name) {
  printf("%s%s =>\n", indentation(), name);
  indent_inc();
}

void
oruby_hash_entry_array(const char *name) {
  oruby_hash_entry_1(name);
  oruby_start_array();
}

void oruby_hash_entry(const char *name, char *fmt, ...) {
  va_list ap;

  oruby_hash_entry_1(name);
  printf(indentation());
  va_start(ap, fmt);
  vprintf(fmt, ap);
  va_end(ap);
  printf(",\n");
  indent_dec();
}

void
oruby_array_entry(char *fmt, ...) {
  va_list ap;

  printf(indentation());
  va_start(ap, fmt);
  vprintf(fmt, ap);
  va_end(ap);
  printf(",\n");
}

void oruby_print(struct dvd_info *dvd_info) {
	int j, i;

	oruby_start_hash();
	oruby_hash_entry(":device", "'%s'", dvd_info->discinfo.device);
	oruby_hash_entry(":title", "'%s'", dvd_info->discinfo.disc_title);
	oruby_hash_entry(":vmg_id", "'%.12s'", dvd_info->discinfo.vmg_id);
	oruby_hash_entry(":provider_id", "'%.32s'", dvd_info->discinfo.provider_id);

  oruby_hash_entry_array(":tracks");

	for (j=0; j < dvd_info->title_count; j++)
	{
    if ( opt_t == j+1 || opt_t == 0 ) {

      // GENERAL
      if (dvd_info->titles[j].enabled) {

        oruby_start_hash();
        oruby_hash_entry(":ix", "%d", j+1);
        oruby_hash_entry(":length", "%.3f", dvd_info->titles[j].general.length);
        oruby_hash_entry(":vts_id", "'%.12s'", dvd_info->titles[j].general.vts_id);

        if (dvd_info->titles[j].parameter.format != NULL ) {
          oruby_hash_entry(":vts", "%d", dvd_info->titles[j].parameter.vts);
          oruby_hash_entry(":ttn", "%d", dvd_info->titles[j].parameter.ttn);
          oruby_hash_entry(":fps", "%.2f", dvd_info->titles[j].parameter.fps);
          oruby_hash_entry(":format", "'%s'", dvd_info->titles[j].parameter.format);
          oruby_hash_entry(":aspect", "'%s'", dvd_info->titles[j].parameter.aspect);
          oruby_hash_entry(":width", "%s", dvd_info->titles[j].parameter.width);
          oruby_hash_entry(":height", "%s", dvd_info->titles[j].parameter.height);
          oruby_hash_entry(":df", "'%s'", dvd_info->titles[j].parameter.df);
        }

        // PALETTE
        if (dvd_info->titles[j].palette != NULL) {
          oruby_hash_entry_array(":palette");
          for (i=0; i < 16; i++) {
            oruby_array_entry("'%06x'",  dvd_info->titles[j].palette[i]);
          }
          oruby_end();
          indent_dec();
        }

        // ANGLES
        if (dvd_info->titles[j].angle_count) { // poor check, but there's no other info anyway.
          oruby_hash_entry(":angles", "%d", dvd_info->titles[j].angle_count);
        }

        // AUDIO
        if (dvd_info->titles[j].audiostreams != NULL ) {
          oruby_hash_entry_array(":audio");
          for (i=0; i<dvd_info->titles[j].audiostream_count; i++)
          {
            oruby_start_hash();
            oruby_hash_entry(":ix", "%d", i+1);
            oruby_hash_entry(":langcode", "'%s'", dvd_info->titles[j].audiostreams[i].langcode);
            oruby_hash_entry(":language", "'%s'", dvd_info->titles[j].audiostreams[i].language);
            oruby_hash_entry(":format", "'%s'", dvd_info->titles[j].audiostreams[i].format);
            oruby_hash_entry(":frequency", "%s", dvd_info->titles[j].audiostreams[i].frequency);
            oruby_hash_entry(":quantization", "'%s'", dvd_info->titles[j].audiostreams[i].quantization);
            oruby_hash_entry(":channels", "%d", dvd_info->titles[j].audiostreams[i].channels);
            oruby_hash_entry(":ap_mode", "%d", dvd_info->titles[j].audiostreams[i].ap_mode);
            oruby_hash_entry(":content", "'%s'", dvd_info->titles[j].audiostreams[i].content);
            oruby_hash_entry(":streamid", "'0x%x'", dvd_info->titles[j].audiostreams[i].streamid);
            oruby_end();
          }
          oruby_end(); 
          indent_dec();
        }

        // CHAPTERS
        if (dvd_info->titles[j].chapters != NULL) {
          oruby_hash_entry_array(":chapters");
          for (i=0; i<dvd_info->titles[j].chapter_count; i++)
          {
            oruby_start_hash();
            oruby_hash_entry(":ix", "%d", i+1);
            oruby_hash_entry(":length", "%.3f", dvd_info->titles[j].chapters[i].length);
            oruby_hash_entry(":startcell", "%d", dvd_info->titles[j].chapters[i].startcell);
            oruby_end();
          }
          oruby_end();
          indent_dec();
        }

        // CELLS
        if (dvd_info->titles[j].cells != NULL) {
          oruby_hash_entry_array(":cell");
          for (i=0; i<dvd_info->titles[j].cell_count; i++)
          {
            oruby_start_hash();
            oruby_hash_entry(":ix", "%d", i+1);
            oruby_hash_entry(":length", "%.3f", dvd_info->titles[j].cells[i].length);
            oruby_end();
          }
          oruby_end();
          indent_dec();
        }

        // SUBTITLES
        if (dvd_info->titles[j].subtitles != NULL) {
          oruby_hash_entry_array(":subp");
          for (i=0; i<dvd_info->titles[j].subtitle_count; i++)
          {
            oruby_start_hash(0);
            oruby_hash_entry(":ix", "%d", i+1);
            oruby_hash_entry(":langcode", "'%s'", dvd_info->titles[j].subtitles[i].langcode);
            oruby_hash_entry(":language", "'%s'", dvd_info->titles[j].subtitles[i].language);
            oruby_hash_entry(":content", "'%s'", dvd_info->titles[j].subtitles[i].content);
            oruby_hash_entry(":streamid", "'0x%x'", dvd_info->titles[j].subtitles[i].streamid);
            oruby_end();
          }
          oruby_end();
          indent_dec();
        }
        oruby_end();
      }
    }
  }
  oruby_end();
  indent_dec();
  if (! opt_t) {
    oruby_hash_entry(":longest_track", "%d", dvd_info->longest_track);
  }
  oruby_end();
}
