/*
    libfame - Fast Assembly MPEG Encoder Library
    Copyright (C) 2000-2001 Vivien Chappelier

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/****** compute mean absolute difference of pixels in block ******************/
#ifndef __MAD_INT_H__
#define __MAD_INT_H__

#include "mean_int.h"

/*  mad_withoutmask                                                          */
/*                                                                           */
/*  Description:                                                             */
/*    Compute mean absolute difference of pixels values in an 8x8 block.     */
/*                                                                           */
/*  Arguments:                                                               */
/*    unsigned char *input: the input 8x8 block                              */
/*    short pitch: number of pixels to the next line                         */
/*    unsigned long *mad : the mean absolute difference.                     */
/*                                                                           */
/*  Return value:                                                            */


static unsigned short inline mad_withoutmask(unsigned char *input,
					     short pitch,
					     unsigned long *mad)
{
  int i, j;
  unsigned long s, m;
  unsigned short c;

  c = mean_withoutmask(input, pitch, &m);
  s = 0;
  for(i = 0; i < 8; i++) {
    for(j = 0; j < 8; j++)
      s += abs(input[j] - m);
    input += pitch;
  }
  *mad = s;

  return(c);
}

/*  mad_withmask                                                             */
/*                                                                           */
/*  Description:                                                             */
/*    Compute mean absolute difference of pixels values in an 8x8 block      */
/*    using a binary mask.                                                   */
/*                                                                           */
/*  Arguments:                                                               */
/*    unsigned char *input: the input 8x8 block                              */
/*    unsigned char *mask: the input 8x8 mask                                */
/*    short pitch: number of pixels to the next line                         */
/*                                                                           */
/*  Return value:                                                            */
/*    unsigned short : the mean absolute difference.                         */

static unsigned short inline mad_withmask(unsigned char *input,
					  unsigned char *mask,
					  short pitch,
					  unsigned long *mad)
{
  int i, j;
  unsigned long s, m;
  unsigned short c;

  c = mean_withmask(input, mask, pitch, &m);

  s = 0;
  for(i = 0; i < 8; i++) {
    for(j = 0; j < 8; j++) {
      if(mask[j]) {
	s += abs(input[j] - m);
      }
    }
    input += pitch;
    mask += pitch;
  }
  
  *mad = s;
  
  return(c);
}
#endif
