/* WebGraphFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.factories.graphs.standard;

import org.grinvin.Embedding;
import org.grinvin.Graph;
import org.grinvin.Vertex;
import org.grinvin.factories.FactoryParameterException;
import org.grinvin.factories.graphs.AbstractGraphFactory;

/**
 * Factory that creates a webgraph with given number of concentric cycles of given length.
 */
public class WebGraphFactory extends AbstractGraphFactory {
    
    //
    @Override protected void createGraph(Graph graph, Embedding embedding) {
        int n = ((Integer)values[0]).intValue();
        int r = ((Integer)values[1]).intValue();
        embedding.setDimension(2);
        Vertex[][] vertices = new Vertex[r][];
        for(int i=0; i < r; i++)
            vertices[i] = createCycle(graph, embedding, n, (i+1.0)/r);
        for(int i=0; i < r-1; i++) {
            for(int j=0; j < n; j++) {
                graph.addNewEdge(vertices[i][j], vertices[i+1][j], null);
                graph.addNewEdge(vertices[i][j], vertices[i][(j+1)%n], null);
            }
        }
        for(int j=0; j < n; j++)
            graph.addNewEdge(vertices[r-1][j], vertices[r-1][(j+1)%n], null);
        
    }
    
    //
    @Override protected void checkParameters(Object[] values) throws FactoryParameterException {
        super.checkParameters(values);
        int n = ((Integer)values[0]).intValue();
        int r = ((Integer)values[1]).intValue();
        if ( n < 3 || r < 1)
            throw new FactoryParameterException("Length of cycle has to be at least 3 and there must be a positive number of concentric cycles.");
    }
    
    /**
     * Create a graph and embedding with vertices spaced
     * equally along the circumference of a centered circle.
     * @param graph graph to which the given vertices will be assigned
     * @param embedding embedding which will hold the coordinates
     * @param nr number of vertices to create
     * @param radius radius of circle
     * @return The array of vertices created
     */
    protected static Vertex[] createCycle(Graph graph, Embedding embedding, int nr, double radius) {
        embedding.setDimension(2);
        Vertex[] result = new Vertex[nr];
        for(int i = 0; i < nr; i++) {
            double angle = 2 * i * Math.PI / nr;
            double[] coords = new double[2];
            coords[0] = radius * Math.cos(angle);
            coords[1] = radius * Math.sin(angle);
            result[i] = graph.addNewVertex(null);
            embedding.setCoordinates(result[i], coords);
        }
        return result;
    }
}
