/* RandomGraphGenerator.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.generators.graphs.standard;

import java.util.Random;

import org.grinvin.factories.FactoryParameterException;
import org.grinvin.generators.graphs.AbstractGraphGenerator;
import org.grinvin.generators.graphs.AbstractGraphGeneratorInstance;
import org.grinvin.generators.graphs.GraphGeneratorInstance;
import org.grinvin.generators.graphs.GraphGeneratorSink;
import org.grinvin.graphs.Graph;
import org.grinvin.graphs.GraphBundle;
import org.grinvin.graphs.Vertex;

/**
 * Generator for random graphs.
 */
public class RandomGraphGenerator extends AbstractGraphGenerator {
    
    //
    @Override protected void checkParameters(Object[] values) throws FactoryParameterException {
        super.checkParameters(values);
        int n0 = (Integer)values[0];
        int n1 = (Integer)values[1];
        double p0 = (Double)values[2];
        double p1 = (Double)values[3];
        int count = (Integer)values[4];
        if (n0 > n1 || n0 <= 0)
            throw new FactoryParameterException("Invalid order range");
        if (p0 < 0.0 || p1 < 0.0 || p0 > 1.0 || p1 > 1.0 || p0 > p1)
            throw new FactoryParameterException("Invalid edge probablity range");
        if (count < 1)
            throw new FactoryParameterException("Graph count must be at least one");
    }
    
    //
    public GraphGeneratorInstance createInstance() {
        return new RandomGraphGeneratorInstance(this,
                ((Integer)values[0]).intValue(),
                ((Integer)values[1]).intValue(),
                ((Double)values[2]).doubleValue(),
                ((Double)values[3]).doubleValue(),
                ((Integer)values[4]).intValue()
                );
    }

    //
    private static final Random RG = new Random();
        
    
    //
    private class RandomGraphGeneratorInstance extends AbstractGraphGeneratorInstance {
        
        // lower order
        private final int n0;
        
        // higher order
        private final int n1;
        
        // lower probability
        private final double p0;
        
        // higher probability
        private final double p1;
        
        // count
        private final int count;
        
        //
        public RandomGraphGeneratorInstance(AbstractGraphGenerator agg,
                int n0, int n1, double p0, double p1, int count) {
            super(agg);
            this.n0 = n0;
            this.n1 = n1;
            this.p0 = p0;
            this.p1 = p1;
            this.count = count;
        }
        
        /**
         * Generate random graphs.
         */
        public void generate(GraphGeneratorSink sink) {
            for (int c=0; c < count; c++) {
                GraphBundle gb = sink.createGraphBundle();
                Graph graph = gb.createGraph();
                int order = n0 + RG.nextInt(n1+1-n0);
                double edgeProbability = p0 + (p1-p0)*RG.nextDouble();
                // TODO
                for (int i=0; i < order; i++) {
                    Vertex v = graph.addNewVertex();
                    for (int j=0; j < i; j++)
                        if (RG.nextDouble() <= edgeProbability)
                            graph.addNewEdge(v, graph.getVertex(j));
                }
                createCyclicEmbedding(gb);
                sink.receiveGraphBundle(gb);
            }
        }
    }
    
}
