/* NonBipartiteMatchingNumber.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard.matching;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.HashSet;
import java.util.Map;

/**
 * This class computes the matching number(maximum size of matching) of a
 * non-biparite graph.
 */
public class NonBipartiteMatchingNumber extends GeneralMatchingNumber {
    
    // PRIVATE VARIABLES
    
    /** HashMap to represent the graph. */
    private Map<Integer,PseudoVertex> graph;
    /** Set of vertices reached along edges in maxMatching. */
    private Set<PseudoVertex> s;
    /** Set of vertices reached along edges not in maxMatching. */
    private Set<PseudoVertex> t;
    /** List with unexplored vertices in S. */
    private LinkedList<PseudoVertex> unexplored;
    
    /**
     * Creates a new instance of NonBipartiteMatchingNumber
     */
    public NonBipartiteMatchingNumber() {
        super();
        graph = null;
        s = null;
        t = null;
        unexplored = null;
    }
    
    /**
     * Computes the invariant value(the maximum matching number)
     * for a general graph.
     */
    public int compute(int[][] adjlist) {
        super.adjlist = adjlist;
        initializeMatching();
        // Build (original) graph
        initializeGraph();
        // Initialize some aiding variables
        s = new HashSet<PseudoVertex>();
        t = new HashSet<PseudoVertex>();
        unexplored = new LinkedList<PseudoVertex>(); // ToDo: linkedlist?!!!
        int maxMatchingSize = adjlist.length/2;
        for(int u = 0; u < adjlist.length && matchingSize < maxMatchingSize; u++) {
            if(matching[u] == UNSATURATED) { // is u unsaturated?
                PseudoVertex uVertex = graph.get(new Integer(u));
                uVertex.setParent(null); // Root of alternating tree!
                s.clear();
                s.add(uVertex);
                t.clear();
                unexplored.clear();
                unexplored.add(uVertex);
                PseudoVertex lastNode = findAugmentingPath();
                if(lastNode != null) {
                    updateMaxMatching(lastNode);
                }
                cleanUpBlossoms();
            }
        }
        return matchingSize;
    }
    
    /**
     * Initializes the graph. A graph is represented as a HashMap.
     * The key is the id-number(the index in the adjacencylist) and
     * the value is the corresponding PseudoVertex-object.
     */
    private void initializeGraph() {
        graph = new HashMap<Integer,PseudoVertex>();
        for(int i = 0; i < adjlist.length; i++)
            graph.put(new Integer(i), new PseudoVertex(i));
        for(int i = 0; i < adjlist.length; i++) {
            PseudoVertex v = graph.get(new Integer(i));
            if(matching[i] != UNSATURATED)
                v.setSaturated(true);
            for(int j = 0; j < adjlist[i].length; j++)
                v.addNeighbour(graph.get(new Integer(adjlist[i][j])));
        }
    }
    
    /**
     * This method searches an augmenting path in the current graph and
     * returns the last node on that path if one is found(null otherwise).
     * @return Last node of augmenting path(null if none found).
     */
    private PseudoVertex findAugmentingPath() {
        int highestIndex = adjlist.length; // For 'new' vertices: blossoms
        PseudoVertex selected = unexplored.peek();
        while(!unexplored.isEmpty()) {
            PseudoVertex v = unexplored.poll();
            Iterator<PseudoVertex> it = v.getNeighbours().iterator();
            boolean blossomFound = false;
            while(it.hasNext() && !blossomFound) { // For all neighbours of v
                PseudoVertex neighbour = it.next();
                if(unsaturatedEdge(v, neighbour) && !t.contains(neighbour)
                && !blossomFound && neighbour != selected) { // Via unsaturated edge!
                    if(neighbour.isSaturated()) {
                        if(!s.contains(neighbour)) {
                            neighbour.setParent(v);
                            PseudoVertex w = searchIncidentVertex(neighbour);
                            w.setParent(neighbour);
                            t.add(neighbour);
                            s.add(w);
                            unexplored.add(w);
                        } else {
                            blossomFound = true;
                            contractBlossom(v, neighbour, highestIndex++);
                        }
                    }
                    else { // found augmenting path!
                        neighbour.setParent(v);
                        return neighbour;
                    }
                }
            }
        }
        return null;
    }
    
    /**
     * This method searches for the incident vertex of the specified
     * saturated vertex in the current matching.
     * @param vertex A saturated vertex.
     * @return The incident vertex in the matching.
     */
    private PseudoVertex searchIncidentVertex(PseudoVertex vertex) {
        if(vertex.isBlossom())
            return vertex.getParent();
        else
            return graph.get(matching[vertex.getId()]);
    }
    
    /**
     * This method checks if the specified edge is unsaturated.
     * And edge is represented by his two endpoints.
     * @param firstEndpoint The 'first' endpoint of the edge
     * @param secondEndpoint The 'second' endpoint of the edge
     * @return true if the specified edge is unsaturated
     */
    private boolean unsaturatedEdge(PseudoVertex firstEndpoint, 
                                    PseudoVertex secondEndpoint) {
        boolean firstBlossom = firstEndpoint.isBlossom();
        boolean secondBlossom = secondEndpoint.isBlossom();
        if(!firstBlossom && !secondBlossom) {
            return matching[firstEndpoint.getId()] != secondEndpoint.getId();
        }
        else if(firstBlossom && !secondBlossom) {
            return unsaturatedEdge(firstEndpoint.getBase(), secondEndpoint);
        }
        else if(!firstBlossom) {
            return unsaturatedEdge(firstEndpoint, secondEndpoint.getBase());
        }
        else {
            return unsaturatedEdge(firstEndpoint.getBase(), secondEndpoint.getBase());
        }
    }
    
    /**
     * This method performs a contraction of a blossom into a new PseudoVertex.
     * @param first The last vertex of the first path.
     * @param scd The last vertex of the second path.
     * @param highestIndex Id of the new PseudoVertex-object
     */
    private void contractBlossom(PseudoVertex first, PseudoVertex scd, 
                                 int highestIndex) {
        Integer hIndex = new Integer(highestIndex);
        Set<PseudoVertex> set = new HashSet<PseudoVertex>();
        LinkedList<PseudoVertex> bList = new LinkedList<PseudoVertex>();
        // Explore both paths to the vertex and discover the correct base
        explorePath(first, set, bList, true);
        PseudoVertex base = explorePath(scd, set, bList, false); // The base of the blossom
        // BlossomList should now contain the vertices in the correct order
        // with the base of the blossom at the end of the list.
        for(PseudoVertex vertex : set) {
            s.remove(vertex);
            t.remove(vertex);
            unexplored.remove(vertex);
        }
        PseudoVertex b = new PseudoVertex(highestIndex, bList);
        b.setParent(base.getParent());
        b.setSaturated(true);
        s.add(b);
        unexplored.addFirst(b);
        graph.put(hIndex, b);
    }
    
    /**
     * This method will explore the path from the given vertex till the root
     * of the alternating tree, trying to discover the blossom, stem and base
     * of the flower.
     * @param v The last vertex on the path.
     * @param set The set of vertices currently in the flower(=blossom at the end).
     * @param bList The list of vertices belonging to the blossom in the correct order!
     * @param addLast Add new vertices at the end of the blossomlist?
     */
    private PseudoVertex explorePath(PseudoVertex v, Set<PseudoVertex> set,
                                     LinkedList<PseudoVertex> bList, boolean addLast) {
        PseudoVertex base = null;
        while(v != null) { // till root
            if(!set.add(v)) {
                if(base != null) { // v belongs to the stem
                    set.remove(v);
                    bList.removeLast();
                }
                else { // v is base!
                    base = v;
                }
            }
            else if(addLast) {
                bList.addLast(v);
            }
            else {
                bList.addFirst(v);
            }
            v = v.getParent();
        }
        return base;
    }
    
    /**
     * This method performs the actual update of the current matching when
     * an augmenting path is found in the graph during the execution of the
     * algorithm.
     * @param lastNode The current node being processed on the augmenting path.
     */
    private void updateMaxMatching(PseudoVertex lastNode) {
        lastNode.setSaturated(true);
        while(lastNode.getParent() != null) { // till root
            lastNode = performUpdate(lastNode);
        }
        lastNode.setSaturated(true);
        matchingSize++; // Current matching contains one edge more then the former
    }
    
    /**
     * Recursive method to perform the update of the current matching.
     * This method also expand the blossoms on the augmenting path.
     * @param node The current node to process.
     * @return The next node to be processed.
     */
    private PseudoVertex performUpdate(PseudoVertex node) {
        PseudoVertex parent = node.getParent();
        if(parent.isBlossom()) {
            node.setParent(expandBlossom(parent, node));
            return performUpdate(node);
        } else {
            int parentId = parent.getId();
            int nodeId = node.getId();
            if(matching[parentId] != nodeId) {
                matching[nodeId] = parentId;
                matching[parentId] = nodeId;
            }
            return parent;
        }
    }
    
    /**
     * This method expands the specified blossom and returns
     * as a result the parent PseudoVertex object of the child in the blossom.
     * @param blossom The blossom to be expanded.
     * @param child The child of the blossom on the augmenting path.
     * @return The new parent of the child(instead of the blossom).
     */
    private PseudoVertex expandBlossom(PseudoVertex blossom, PseudoVertex child) {
        LinkedList<PseudoVertex> bList = blossom.expandBlossom();
        graph.remove(blossom.getId());
        // Search for the real parent of the child in the blossom
        int parentIndex = findRealParent(blossom, child);
        PseudoVertex newParent = bList.get(parentIndex);
        // Search for the correct "stopvertex"
        int stopVertexIndex = findStopVertex(bList, blossom);
        int startIndex = 0;
        int stopIndex = 0;
        boolean correctPathDirection = true;
        if(parentIndex == (bList.size()-1)) { // parent = base
            startIndex = stopVertexIndex; 
            stopIndex = parentIndex; // = base
            correctPathDirection = false; 
        }
        else { // parent != base
            startIndex = parentIndex;
            stopIndex = stopVertexIndex;
            correctPathDirection = true;
        }
        if(startIndex == stopIndex) { // parent = base & stopvertex = base
            newParent.setParent(blossom.getParent());
            bList.remove(startIndex); // removeLast() should be the same
        }
        else {
            updatePath(bList, blossom, startIndex, stopIndex, correctPathDirection);
        }
        expandRemainingBlossoms(bList);
        return newParent;
    }
    
    /**
     * This method updates the path through the blossom.
     * @param bList The blossomlist of the blossom.
     * @param blossom The blossom.
     * @param startIndex The index in the list to start the updating.
     * @param stopIndex The index in the list to stop the updating.
     * @param correctPathDirection Go through path in correct or reverse direction?
     */
    private void updatePath(LinkedList<PseudoVertex> bList, PseudoVertex blossom,
                            int startIndex, int stopIndex, 
                            boolean correctPathDirection) {
        ListIterator<PseudoVertex> lit = bList.listIterator(startIndex);
        PseudoVertex prev = lit.next(); // prev = startVertex
        lit.remove();
        PseudoVertex vertex = lit.next();
        // start with saturatedEdge!
        if(!unsaturatedEdge(prev, vertex)) { // forward direction!
            lit.remove();
            if(!correctPathDirection) {
                prev.setParent(blossom.getParent());
                vertex.setParent(prev);
            } else {
                prev.setParent(vertex);
            }
            while(lit.hasNext() && (lit.previousIndex() != stopIndex)) {
                prev = vertex;
                vertex = lit.next();
                lit.remove();
                if(correctPathDirection) {
                    prev.setParent(vertex);
                } else {
                    vertex.setParent(prev);
                }
            }
        }
        else { // backward direction!
            lit.previous(); // wrong direction, so go 1(startVertex deleted!) back
            if(!correctPathDirection) {
                prev.setParent(blossom.getParent());
            }
            while(lit.hasPrevious() && (lit.nextIndex() != stopIndex)) {
                vertex = lit.previous();
                lit.remove();
                if(correctPathDirection) {
                    prev.setParent(vertex);
                } else {
                    vertex.setParent(prev);
                }
                prev = vertex;
            }
            if(lit.nextIndex() != stopIndex) {
                vertex = bList.removeLast(); // vertex = base
                if(correctPathDirection) {
                    prev.setParent(vertex);
                } else {
                    vertex.setParent(prev);
                }
            }
        }
        if(correctPathDirection) {
            vertex.setParent(blossom.getParent());
        }
    }
    
    /**
     * This method searches for the real parent of the vertex child in the blossom.
     * @param bList The blossomlist
     * @param child The child.
     * @return The index of real parent of child in the specified blossomlist
     */
    private int findRealParent(PseudoVertex blossom, PseudoVertex child) {
        ListIterator<PseudoVertex> lit = blossom.getBlossomList().listIterator();
        boolean parentFound = false;
        boolean unsaturatedEdge = unsaturatedEdge(blossom, child);
        int parentIndex = 0; // Index of the new parent of child in the blossomlist
        while(lit.hasNext() && !parentFound) {
            PseudoVertex vertex = lit.next();
            if(vertex.getNeighbours().contains(child) &&
               (unsaturatedEdge(vertex, child)==unsaturatedEdge)) {
                parentFound = true;
                parentIndex = lit.previousIndex();
            }
        }
        return parentIndex;
    }
    
    /**
     * This method searches the "stopvertex" where the path through the blossom
     * should stop. This is the first vertex who has a neighbour equal to the parent 
     * of the blossom. Normally this will be the base of the blossom.
     * @param The blossom.
     * @return The index in the blossomlist of the "stopvertex".
     */
    private int findStopVertex(List<PseudoVertex> bList, PseudoVertex blossom) {
        PseudoVertex parentOfBlossom = blossom.getParent();
        ListIterator<PseudoVertex> lit = bList.listIterator(bList.size());
        boolean found = false;
        int index = bList.size()-1;
        while(!found && lit.hasPrevious()) {
            if(lit.previous().getNeighbours().contains(parentOfBlossom)) {
                found = true;
                index = lit.nextIndex();
            }
        }
        return index;
    }
    
    /**
     * This recursive method expands remaining blossoms in the blossomlist of 
     * a blossom after updating the path with the method updatePath().
     * @param bList The blossomList with the remaining vertices from the list.
     */
    private void expandRemainingBlossoms(LinkedList<PseudoVertex> bList) {
        for(PseudoVertex vertex : bList) {
            if(vertex.isBlossom()) {
                graph.remove(vertex.getId());
                expandRemainingBlossoms(vertex.expandBlossom());
            }
        }
    }
    
    /**
     * Clean up all the remaining blossoms!
     */
    private void cleanUpBlossoms() {
        int index = adjlist.length;
        while(graph.size() > adjlist.length) {
            PseudoVertex blossom = graph.get(index);
            if(blossom != null) {
                expandRemainingBlossoms(blossom.expandBlossom());
                graph.remove(index);
            }
            index++;
        }
    }
}
