/* EdgeConnectivityFlow.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.invariants.values.IntegerValue;

/**
 * Invariant computer which returns the edge-connectivity using flows.
 */
public class EdgeConnectivityFlow extends AbstractInvariantComputer {
    
    public IntegerValue compute(GraphBundleView bundle) {
        //first calculate minimum degree (i.e. upperbound for edge-connectivity)
        int[][] adjlist = bundle.adjacencyList();
        if(adjlist.length<2)
            return new IntegerValue(0, this);

        // sort degrees
        int[] degrees = new int[adjlist.length];
        degrees[0] = adjlist[0].length;
        int vertexMinimumDegree = 0;
        for(int i = 1; i < degrees.length; i++){
            int j = i-1;
            while(j>=0 && adjlist[i].length < degrees[j]){
                degrees[j+1] = degrees[j];
                j--;
            }
            degrees[j+1]=adjlist[i].length;
            if(j+1==0)
                vertexMinimumDegree=i;
        }
        
        //calculate lower bound
        double lowerbound = adjlist.length % 2 == 0 ? 0 : degrees[adjlist.length/2];
        for(int i = 0; i < adjlist.length/2; i++)
            lowerbound += 2*degrees[i];
        lowerbound -= adjlist.length*(adjlist.length*0.5 - 1);
        lowerbound *= 0.5;
        
        //optimalisation
        int mindeg = degrees[0];
        
        if(mindeg==adjlist.length-1)
            //complete graph
            return new IntegerValue(mindeg, this);
        
        if(mindeg >= adjlist.length*0.5)
            //optimalisation
            return new IntegerValue(mindeg, this);
        
        
        int minimumCutSize = mindeg;
        for(int i = 0; i < adjlist.length; i++){
            if(i!=vertexMinimumDegree)
                minimumCutSize = findMaxFlowInSTNetwork(adjlist, vertexMinimumDegree, i, minimumCutSize);
            if(minimumCutSize <= lowerbound)
                //optimalisation
                return new IntegerValue(minimumCutSize, this);
        }

        return new IntegerValue(minimumCutSize, this);
    }
    
    //returns the minimum of the maxflow of the st-network and satisfied
    private int findMaxFlowInSTNetwork(int[][] adjlist, int source, int target, int satisfied){
        int[][] paths = new int[adjlist.length][adjlist.length];
        int pathCount = 0;
        while(findPath(adjlist, source, target, paths, new boolean[adjlist.length]) && pathCount < satisfied)
            pathCount++;
        return pathCount;
    }
    
    //try to find a path from the current vertex to the target
    private boolean findPath(int[][] adjlist, int currentVertex, int target, int[][] paths, boolean[] currentPath) {
        currentPath[currentVertex]=true;
        if(currentVertex==target)
            return true;
        for(int nextVertex : adjlist[currentVertex]){
            if(paths[currentVertex][nextVertex]-paths[nextVertex][currentVertex]<=0 && !currentPath[nextVertex]){
                paths[currentVertex][nextVertex]++;
                if(findPath(adjlist, nextVertex, target, paths, currentPath))
                    return true;
                else
                    paths[currentVertex][nextVertex]--;
            }
        }
        currentPath[currentVertex]=false;
        return false;
    }
        
    public String getInvariantId() {
        return "org.grinvin.invariants.EdgeConnectivity";
    }
    
}
