/*

Sega Model 1 Hardware Overview
------------------------------

Note! This document is a Work-In-Progress and covers all the dumped Sega Model 1 games, including....

Star Wars Arcade (C) Sega, 1994
Virtua Fighter   (C) Sega, 1993
Virtua Racing    (C) Sega, 1992
Wing War         (C) Sega, 1994

The Sega Model 1 system comprises several PCB's....
CPU PCB   - This is the main PCB. It holds the main CPU, some surface-mounted ROMs and the 3D co-processors
            (manufactured by Fujitsu and known as 'TGP'). The PCB is the same for all Model 1 games, but some
            games are not swappable due to a different internal 3D co-processor program.
Video PCB - There is only one known revision of this PCB so any game can use this PCB. It contains a very large
            quantity of RAM, some surface-mounted ROMs and some custom Sega video chips. The RGB and Sync is also
            outputted from this PCB. The horizontal sync is 24kHz and requires a medium resolution (24kHz) monitor.
            The video PCB is plugged in below the CPU PCB.
ROM PCB   - This PCB is the same for all games, only the type and number of ROMs differ per game. The ROM PCB is
            plugged in above the CPU PCB.
I/O PCB   - This PCB connects the CPU PCB to the input and output controls via some multi-wire connectors.
Comm PCB  - This PCB is used for communication when linking 2 PCBs together (generally for Virtua Racing only).
            It plugs in below the Video PCB.
Sound PCB - This PCB generates the sound for the game. There is no amplification of the sound here, that is done
            via the Amp PCB.
Motor PCB - This PCB is used to control the motors and steering column clutch used for the force feedback steering.
Amp PCB   - This PCB amplifies the sound generated by the sound PCB.


ROM PCB
-------

MODEL 1 MEMORY BD 837-7893 171-6079B (C) SEGA 1992
|--------------------------------------------------------------------------------------------|
|J1                                                                                          |
|J2      IC5            IC14                    IC26                   IC27                  |
|J3                                                                                          |
|J4      IC4            IC15                                                                 |
|J5                                             IC28                   IC29                  |
|J12     IC13           IC12                                                                 |
|                                                                                            |
|        IC11           IC10                    IC30                   IC31                  |
|                                                                                            |
| CN1    IC9            IC8       CN2                                                    CN3 |
|                                               IC32                   IC33                  |
|        IC7            IC6                                                                  |
|                                               84256                  84256              J11|
|        IC41           IC42                                                              J10|
|                                               84256                  84256              J9 |
|        IC39           IC40                                                              J8 |
|                                                                                         J7 |
|                                                                     315-5478            J6 |
|--------------------------------------------------------------------------------------------|
Notes:
      J1, J2, J4    - Jumpers to configure ROM sizes, all set to 2-3
      J3, J5, J6,   \
      J7, J8, J9    \
      J10, J11,     | Jumpers to configure ROM sizes, all set to 1-2
      J12           /
      CN1, CN2, CN3 - Connectors for joining ROM PCB to CPU PCB
      84256         - Fujitsu 84256 32k x8 SRAM (x4, DIP28)
      315-5478      - Lattice GAL16V8A (DIP20)

      IC4, IC5      - Hitachi HN27C1024 128k x16bit EPROM (DIP40)

      IC14, IC15    - Intel 27C040 512k x 8bit EPROM (DIP32)

      IC6, IC7, IC8,\
      IC9, IC10,    \
      IC11, IC12,   | 834000 4M MASKROM (DIP32)
      IC13,         |
      IC39, IC40,   /
      IC41, IC42    /

      IC26, IC27,   \
      IC28, IC29,   | 8316200 16M MASKROM (DIP42)
      IC30, IC31,   /
      IC32, IC33    /

      IC#   PCB
            Label   Virtua Racing    Virtua Fighter    Wing War       Star Wars Arcade
      --------------------------------------------------------------------------------
      4     B       EPR-14878A       EPR-16080         EPR-16951      -
      5     A       EPR-14879A       EPR-16081         EPR-16950      EPR-16467
      14    LO      EPR-14882        EPR-16082         EPR-16729      EPR-16468
      15    HI      EPR-14883        EPR-16083         EPR-16730      EPR-16469
      6     LO      MPR-14880        MPR-16084         MPR-16738      -
      7     HI      MPR-14881        MPR-16085         MPR-16737      -
      8     LO      MPR-14884        MPR-16086         MPR-16736      -
      9     HI      MPR-14885        MPR-16087         MPR-16735      -
      10    LO      MPR-14886        MPR-16088         MPR-16734      -
      11    HI      MPR-14887        MPR-16089         MPR-16733      -
      12    LO      MPR-14888        MPR-16090         -              -
      13    HI      MPR-14889        MPR-16091         -              -
      39    LO      MPR-14898        -                 MPR-16741      MPR-16472
      40    ML      MPR-14899        -                 MPR-16742      MPR-16473
      41    MH      MPR-14900        -                 MPR-16739      MPR-16474
      42    HI      MPR-14901        -                 MPR-16740      MPR-16475
      26    LO      MPR-14890        MPR-16096         MPR-16743      MPR-16476
      27    HI      MPR-14891        MPR-16097         MPR-16744      MPR-16477
      28    LO      MPR-14892        MPR-16098         MPR-16745      MPR-16478
      29    HI      MPR-14893        MPR-16099         MPR-16746      MPR-16479
      30    LO      MPR-14894        MPR-16100         MPR-16747      MPR-16480
      31    HI      MPR-14895        MPR-16101         MPR-16748      MPR-16481
      32    LO      MPR-14896        MPR-16102         MPR-16749      -
      33    HI      MPR-14897        MPR-16103         MPR-16750      -

      - denotes socket not populated.


CPU PCB
-------

MODEL-1 CPU BOARD 837-8886171-6298C (C) SEGA 1992
|---------------------------------------------------------------------------------------------------------------------------------|
|                        CN2                                    CN1                                   CN4                         |
|         |---------------------------------|   |---------------------------------|            |---------------|                  |
|         |---------------------------------|   |---------------------------------|            |---------------|                  |
|                                                                                                                                 |
|                       OPR-14746.68    |-|                                    5.5V_BATT                                         |-|
|       |-----------|                   | |                                                                          PC910       | |
|       |           |   OPR-14747.69    | |                                                              *1                      | |
|       | SEGA      |                   | |                                    84256     HM658128                    PC910       | |
|       | 315-5572  |                   | |CN13   |-----------|                                                                  | |CN3
|       |           |                   | |       |NEC        |                84256     HM658128                    PC910       | |
|       |           |                   | |       |D70615GD-16|                                                                  | |
|       |-----------|                   |-|       |V60        |                                                                  |-|
|                                                 |           |                                                                   |
|                                                 |           |                                                                   |
|                      32MHz                      |-----------|            |---------|     |---------|                            |
|                                                                          |SEGA     |     |SEGA     |     MB8421                 |
|       OPR-14745.65                                                       |315-5465 |     |315-5338A|                   *2       |
|                                                                          |         |     |         |                            |
|                                                                          |---------|     |---------|                            |
|       OPR-14744.64            40MHz                                                                                             |
|                                                                                      J6             CN5                         |
|                                                                                      J5      |---------------|                  |
|                                                                                      J4      |---------------|                  |
|                         3771                                                                                                    |
|       |-----------|     3771                   |-----------|                                                                    |
|       |           |                            |           |                                                                    |
|       | SEGA      |                            | SEGA      |                                                           D71051   |
|       | 315-5572  |                            | 315-5573  |                  |-----------|                                     |
|       |           |                            |           |                  |           |     M5M5178     OPR-14743.45        |
|       |           |                            |           |                  | SEGA      |     M5M5178                         |
|       |-----------|                            |-----------|                  | 315-5464  |     M5M5178     OPR-14742.44        |
|                                                                               |           |     M5M5178                         |
|                                                                               |           |                                     |
|                                                                               |-----------|                           315-5546A |
|                                                                                                                                 |
|       |-----------|                            |-----------|                                                |-----------|       |
|       |           |                            |           |                                    84256       |           |       |
|       | SEGA      |                            | SEGA      |                                    84256       | SEGA      |       |
|       | 315-5571  |                            | 315-5571  |                                    84256       | 315-5463  |       |
|       |           |                            |           |                                    84256       |           |       |
|       |           |                            |           |                                                |           |       |
|       |-----------|                            |-----------|                                                |-----------|       |
|                                                                                                                                 |
|                                                                                                     CN6                         |
|                     LED1 LED2 LED3 LED4 LED5                                                 |---------------|                  |
|       LED11         LED6 LED7 LED8 LED9 LED10                                                |---------------|                  |
|---------------------------------------------------------------------------------------------------------------------------------|
Notes:
      84256         - Fujitsu 84256A-70LL 32k x8 SRAM (x6, DIP28)
      HM658128      - Hitachi HM658128ALP-10 16k x8 SRAM (x2, DIP32)
      M5M5178       - Mitsubishi M5M5178AP-25 8k x8 SRAM (x4, DIP28)
      CN1, CN2      - Connectors to join CPU board to Video board
      CN3           - Connector for power and input/output controls (joins small adapter board to a larger filter board)
      CN4, CN5, CN6 - Connectors to join ROM board to CPU board
      CN13          - Connector for optional daughterboard (not used)
      3771          - Fujitsu MB3771 master reset IC (x2, DIP8)
      5.5_BATT      - NEC 5.5 volt battery
      PC910         - Sharp PC910 opto-isolator (x3, DIP8)
      D71051        - NEC uPD71051C-10 UART (DIP28)
      MB8421        - Fujitsu MB8421-12LP RAM? (SDIP52)
      *1            - Unpopulated position for Fujitsu MB8421
      D70615GD-16   - NEC uPD70615GD-16-S V60 CPU, running at 16.000MHz (QFP120, clock 32 / 2)
      315-5546A     - Lattice GAL16V8A (DIP20)
      315-5571      - Sega Custom (IC57/IC58, QFP160)  \
      315-5572      - Sega Custom (IC60/IC66, QFP160)  / According to test mode, these chips are the TGPs
      315-5573      - Sega Custom (QFP160)
      315-5463      - Sega Custom (QFP160)
      315-5464      - Sega Custom (QFP160)
      315-5465      - Sega Custom (QFP100)
      315-5338A     - Sega Custom (QFP100)
      *2            - Unpopulated position for 315-5338A
      J4, J5, J6    - Jumpers, all set to 2-3

      OPR-14742.44  \
      OPR-14743.45  - 1M SOP40 MASKROMs, tied to 315-5464

      OPR-14744.64  \
      OPR-14745.65  - 1M SOP40 MASKROMs, tied to both 315-5572's

      OPR-14746.68  \
      OPR-14747.69  - 1M SOP40 MASKROMs, tied to 315-5572 @ IC66


VIDEO PCB
---------

837-7894 171-6080D (C) SEGA 1992
|---------------------------------------------------------------------------------------------------------------------------------|
|                        CN2                                    CN1                                                               |
|         |---------------------------------|   |---------------------------------|                                               |
|         |---------------------------------|   |---------------------------------|                                               |
|                                                                                                                     HM65256     |
|                                                                                     J3                                          |
|   HM658512       HM658512                                        HM658128           J2                              HM65256     |
|                                        |-----------|                                      |-----------|                         |
|                                        |           |             HM658128                 |           |                         |
|                                        | SEGA      |                                      | SEGA      |             HM658128    |
|   HM654128       HM654128              | 315-5422  |                                      | 315-5292  |                         |
|                                        |           |                                      |           |             HM658128    |
|                                        |           |             M5M5178                  |           |                         |
|                                        |-----------|                                      |-----------|             HM658128    |
|   HM654128       HM654128                                        M5M5178                                                        |
|                                      |-----------|                                                                  HM658128    |
|   CY7B185           J1               |           |                                             J4                               |
|   CY7B185                            | SEGA      |                                                                              |
|   CY7B185                            | 315-5423  |        OPR-14748.16                315-5483                                  |
|   CY7B185                            |           |                                                                              |
|                                      |           |                                    315-5484                                  |
|                                      |-----------|                                                                              |
|                                                                                                                                 |
|        |-----------|                 |-----------|                                              M5M5178                         |
|        |           |                 |           |                                                                              |
|        | SEGA      |                 | SEGA      |                                              M5M5178                         |
|        | 315-5425  |                 | 315-5424  |        OPR-14748.15                                                          |
|        |           |                 |           |                                                                              |
|        |           |                 |           |                                                                              |
|        |-----------|                 |-----------|                                                                              |
|                                                                                                                                 |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                                     M5M5178                            |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256            315-5485                 M5M5178                            |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                                     M5M5178                            |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256            36MHz        315-5486                                       |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                         315-5486                                       |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                         315-5486                                       |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                                                                        |
|                                                                                                                                 |
|  HM65256         HM65256          HM65256        HM65256                                        CN4          CN3      LED1      |
|---------------------------------------------------------------------------------------------------------------------------------|
Notes:
      HM65256  - Hitachi HM65256BLSP-10 32k x8 SRAM (x34, DIP28)
      HM658128 - Hitachi HM658128ALP-10 16k x8 SRAM (x10, DIP32)
      HM658512 - Hitachi HM658512LP-8 64k x8 SRAM (x2, DIP32)
      M5M5178  - Mitsubishi M5M5178AP-25 8k x8 SRAM (x7, DIP28)
      CY7B185  - Cypress CY7B185 8k x8 SRAM (x4, DIP28)
      J1       - Jumper, set to 2-3
      J2       - Jumper, set to 1-2
      J3       - Jumper, set to 2-3
      J4       - Jumper, set to 1-2
      CN1, CN2 - Connectors to join Video board to CPU board
      CN3      - Connector for R/G/B/Sync output (@ 24kHz)
      CN4      - 20 pin IDC flat-cable connector (purpose unknown)
      315-5483 - PAL CK2605 (DIP20)
      315-5484 - Lattice GAL16V8A (DIP20)
      315-5485 - Lattice GAL16V8A (DIP20)
      315-5486 - Lattice GAL16V8A (x3, DIP20)
      315-5422 - Sega Custom (QFP160)
      315-5423 - Sega Custom (QFP160)
      315-5424 - Sega Custom (QFP160)
      315-5425 - Sega Custom (QFP160)
      315-5292 - Sega Custom (QFP160)

      OPR-14748.15  \
      OPR-14748.16  - 1M SOP40 MASKROMs, tied to 315-5423 & 315-5424. Note both ROMs are identical.


I/O PCB
-------

837-8950-01 (C) SEGA 1992
|-------------------------------------------|
| CN6                           J3   J2     |
|                                        CN5|
|                      DSW3       LED1      |
|                                           |
| SW7    |---------|                        |
|  32MHz |SEGA     |   DSW1                 |
| SW6    |315-5338A|                        |
|        |         |                        |
| SW5    |---------|   DSW2                 |
|                                        CN1|
| SW4      MB8464                           |
|          14869.25                         |
|   3771                                    |
|          Z80                              |
|   93C45                                   |
|                               PC910 PC910 |
|   LED2                           J1       |
|      M6253                                |
| CN3              CN2          CN4     TL1 |
|-------------------------------------------|
Notes:
      315-5338A - Sega Custom (QFP100)
      Z80       - Zilog Z0840004PSC Z80 CPU, running at 4.000MHz (DIP40, clock 32 / 8)
      14869.25  - ST Microelectronics M27C512 64k x8 EPROM (DIP28, labelled 'EPR-14869')
                  There is an alternative revision B 'EPR-14869B' also
      MB8464    - Fujitsu MB8464 8k x8 SRAM (DIP28)
      93C45     - 128bytes x8 EEPROM (DIP8)
      M6253     - OKI M6253 (DIP18)
      3771      - Fujitsu MB3771 Master Reset IC (DIP8)
      PC910     - Sharp PC910 opto-isolator (x2, DIP8)
      DSW1/2/3  - 8-position Dip Switch (x3)
      J1        - Jumper, set to 2-3
      J2, J3    - Jumper, both set to 1-2
      CN1       - 50 pin connector (joins to control panel assembly)
      CN2       - 26 pin connector (joins to foot pedal assembly)
      CN3       - 10 pin connector for power input
      CN4       - 6 pin connector (joins to sound PCB -> CN2, used for sound communication from Main PCB to Sound PCB)
      CN5       - 12 pin connector for input/output controls
      CN6       - 12 pin connector (joins to Motor PCB)
      TL1       - Connector for network optical cable link
      SW7       - Push Button Service Switch
      SW6       - Push Button Test Switch
      SW5, SW4  - Push Button Switches (purpose unknown)


Motor PCB
---------

SJ25-0155-01 838-9081
|------------------------------------------------------|
|          CN6                      CN7        CN8     |
|CN1                                    6A             |
|                         DSW2                         |
|        |---------|                 2A    C4149       |
|        |SEGA     |      DSW1                      CN9|
|        |315-5296 |                                   |
|        |         |                               CN10|
|        |---------|                MB3759             |
|                                                   C11|
|CN2                   |---------|                     |
|                      |SEGA     |                     |
|          8MHz        |315-5296 |                     |
|                      |         |                     |
|      315-5625        |---------|                     |
|           3771                                       |
|CN3            3773                                   |
|                                    2A                |
|              Z80                          C4149      |
|                                                      |
|              15111.12                            CN12|
|                                   MB3759             |
|           M6253   6264                               |
|                                                      |
|CN4        CN5      7-SEG                             |
|------------------------------------------------------|
Notes:
      There are also many capacitors/transistors and other power-related components which are not shown.

      315-5296 - Sega Custom (QFP100)
      7-SEG    - 7-Segment Display (x4), used for PCB status/diagnostics
      15111.12 - ST Microelectronics M27C512 64k x8 EPROM (DIP28, labelled 'EPR-15111')
      Z80      - Zilog Z0840004PSC Z80 CPU, running at 4.000MHz (DIP40, clock 8 / 2)
      MB8464   - Fujitsu MB8464 8k x8 SRAM (DIP28)
      3771     - Fujitsu MB3771 Master Reset IC (DIP8)
      3773     - Fujitsu MB3771 Master Reset IC with Watchdog Timer (DIP8)
      2A       - Fuse, 2 Amp
      6A       - Fuse, 6 Amp
      MB3759   - Fujitsu MB3759 Switching Regulator Controller (x2, DIP16)
      CN1, CN2 - Connects to Left Side I/O PCB
      CN3, CN4 - Connects to Right Side I/O PCB
      CN5      - Connects to Steering Voltage Regulator (Left and Right Sides)
      CN6      - Connects to Power Supply (+5V)
      CN7, CN8 - Connects to 28VAC transformer
      CN9      - 2 pin connector (not used?)
      CN10     - 2 pin connector (not used?)
      CN11     - 7 pin connector (joins to Motor and Clutch Left Side)
      CN12     - 7 pin connector (joins to Motor and Clutch Right Side)
      DSW1/2   - 8-position Dip Switch (x2)
                 +-------------------------------------------------+
                 |DSW1               1   2   3   4   5   6   7   8 |
                 +---------+-------+---+---+---+---+---+---+---+---+
                 |Feedback | Light |OFF|OFF|OFF|   |   |   |   |   |
                 |Weight   |       +---+---+---+   |   |   |   |   |
                 |PL1      |  /\   |ON |OFF|OFF|   |   |   |   |   |
                 |         |  ||   +---+---+---+   |   |   |   |   |
                 |         |       |OFF|ON |OFF|   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         |       |ON |ON |OFF|   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         |       |OFF|OFF|ON |   |   |   |   |   |
                 |         |  ||   +---+---+---+   |   |   |   |   |
                 |         |  \/   |ON |OFF|ON |   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         | Heavy |ON |ON |ON |   |   |   |   |   |
                 +---------+-------+---+---+---+---+   |   |   |   |
                 |Power On | Yes   |           |OFF|   |   |   |   |
                 |Check    +-------+-----------+---+   |   |   |   |
                 |         | No    |           |ON |   |   |   |   |
                 +---------+-------+-----------+---+---+---+   |   |
                 |Not Used |                       |OFF|OFF|   |   |
                 +---------+-----------------------+---+---+---+---+
                 |Not Used |                               |OFF|OFF|
                 +---------+-------------------------------+---+---+

                 +-------------------------------------------------+
                 |DSW2               1   2   3   4   5   6   7   8 |
                 +---------+-------+---+---+---+---+---+---+---+---+
                 |Feedback | Light |OFF|OFF|OFF|   |   |   |   |   |
                 |Weight   |       +---+---+---+   |   |   |   |   |
                 |PL2      |  /\   |ON |OFF|OFF|   |   |   |   |   |
                 |         |  ||   +---+---+---+   |   |   |   |   |
                 |         |       |OFF|ON |OFF|   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         |       |ON |ON |OFF|   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         |       |OFF|OFF|ON |   |   |   |   |   |
                 |         |  ||   +---+---+---+   |   |   |   |   |
                 |         |  \/   |ON |OFF|ON |   |   |   |   |   |
                 |         |       +---+---+---+   |   |   |   |   |
                 |         | Heavy |ON |ON |ON |   |   |   |   |   |
                 +---------+-------+---+---+---+---+   |   |   |   |
                 |Power On | Yes   |           |OFF|   |   |   |   |
                 |Check    +-------+-----------+---+   |   |   |   |
                 |         | No    |           |ON |   |   |   |   |
                 +---------+-------+-----------+---+---+---+---+---+
                 |7-SEG LED|Display V.R. Value 00-80-FF    |OFF|OFF|
                 |         +-------------------------------+---+---+
                 |         |Display Clutch Value           |ON |OFF|
                 |         +-------------------------------+---+---+
                 |         |Data from the I/O Board (00-FF)|OFF|ON |
                 |         +-------------------------------+---+---+
                 |         |Data to the I/O Board (00-FF)  |ON |ON |
                 +---------+-------------------------------+---+---+


Comm PCB
--------

MODEL-1 COMMUNICATION BD 837-8842 171-6293B (C) SEGA 1992
|--------------------------------------------------------------------------------|
|                                                                                |
|    MB89237A            MB89374                                                 |
|       JP4                                                                 LED1 |
|    15112.17            Z80                                                     |
|    JP2  JP3                                                       75179        |
|    MB8464              315-5624                                     JP6        |
|                                                       315-5547                 |
|        315-5611                                            SW1    PC910     CN4|
|                                                                                |
|                                                                   PC910     CN5|
|     MB8421             MB8431                                JP7               |
|                                                                   JP5          |
|        JP8                                                                  CN7|
|                CN1                                    CN2                      |
| |---------------------------------|   |---------------------------------|   CN6|
| |---------------------------------|   |---------------------------------|      |
|--------------------------------------------------------------------------------|
Notes:
      15112.17 - AMD AM27C100 128k x8 EPROM (DIP32, labelled 'EPR-15112')
      Z80      - Zilog Z0840004PSC Z80 CPU, running at 4.000MHz (DIP40)
      MB8464   - Fujitsu MB8464 8k x8 SRAM (DIP28)
      MB8421   - Fujitsu MB8421-12LP RAM? (SDIP52)
      MB8431   - Fujitsu MB8431-90LP RAM? (SDIP52)
      MB89237A - Fujitsu MB89237A ? (DIP20)
      MB89374  - Fujitsu MB89374 ? (SDIP42)
      75179    - Texas Instruments SN75179 Differential Driver and Receiver Pair (DIP8)
      315-5547 - AMI 18CV8PC-25 PAL (DIP20)
      315-5624 - MMI PAL16L8BCN PAL (DIP20)
      315-5611 - Lattice GAL16V8A PAL (DIP20)
      PC910    - Sharp PC910 opto-isolator (x2, DIP8)
      SW1      - Push Button Switch (purpose unknown)
      CN1, CN2 - Connectors to join Comm board to Video board
      CN4      - 8 pin connector (purpose unknown)
      CN5      - 6 pin connector (purpose unknown)
      CN6, CN7 - Connectors for network optical cable link
      JP2      - Jumper, set to 2-3
      JP3      - Jumper, set to 1-2
      JP4      - Jumper, set to 1-2
      JP5      - Jumper, shorted
      JP6      - Jumper, not shorted
      JP7      - Jumper, not shorted
      JP8      - Jumper, set to 1-2


Sound PCB
---------

MODEL-1 SOUND BD 837-8679 (C) SEGA 1992
|-----------------------------------------------------------|   SOUND BD OPTION 837-8680 (C) SEGA 1992
| CN2                    CN1                 CN5  CN6  CN7  |   |--------------------------|
|                   LED5               CN9                  |   |  CN2    CN3  CN4 CN5     |
| PC910                                               TL062 |   |              uPC844C     |
|                JP5                           TL062        |   |     LC78820        TL062 |
| JP3     IC8    JP6                                        |   |                          |
| JP4                MB8464    82C51                 uPC844C|   |             IC5          |
|                    MB8464                YM3438           |   |                          |
| JP1     IC7                                               |   |    IC3                   |
| JP2                                                       |   |             IC4          |
|                                                  LC78820  |   | LED                      |
|                  315-5578                16MHz            |   |JP6           |---------| |
|                                                           |   |JP5           |SEGA     | |
|                  315-5577                                 |   |JP4           |315-5560 | |
|                                                           |   |JP3  315-5579 |         | |
|                           |---------|                     |   |JP2           |---------| |
| |-----------------|       |SEGA     |    CN8              |   |JP1    CN1                |
| |  TMP68000N-10   |       |315-5560 |          315-5579   |   |--------------------------|
| |                 |       |         |                     |   Notes:
| |-----------------|  JP7  |---------|          IC33       |         JP1, JP2 - Jumpers, both set to 2-3
|                                         IC31              |         JP3, JP4,- Jumpers, all set to 1-2
|   LED1 LED2 LED3 LED4                          IC32       |         JP5, JP6 /
|DSW1                3771      20MHz                        |         CN1, CN2 - Connectors to join Sound Option Board to Main Sound Board
|-----------------------------------------------------------|         CN3      - 5 pin connector, unamplified sound out to Amp PCB
                                                                      CN4, CN5 - Unamplified audio output, left/right RCA audio connectors (not used)
Notes:
      TMP68000N-10- Toshiba TMP68000N-10 CPU, running at 10.000MHz (SDIP64, clock 20 / 2)
      82C51       - Toshiba 82C51AM-10 Programmable 8-bit I/O Serial Interface (SOP28)
      DSW1        - 4 position Dip Switch
      MB8464      - Fujitsu MB8464 8k x8 SRAM (DIP28)
      3771        - Fujitsu MB3771 Master Reset IC (DIP8)
      TL062       - ST Microelectronics Dual Low Power Operational Amplifier (DIP8)
      PC910       - Sharp PC910 opto-isolator (DIP8)
      LC78820     - Sanyo LC78820 2-channel 18-bit D/A Converter (DIP20)
      uPC844C     - NEC uPC844C Quad High Speed Wide Band Operational Amplifier (DIP14)
      315-5560    - Sega Custom (QFP100)
      315-5577    - Lattice GAL16V8A (DIP20)
      315-5578    - Lattice GAL16V8A (DIP20)
      315-5579    - Lattice GAL16V8A (DIP20)
      YM3438      - Yamaha YM3438 sound chip, running at 8.000MHz (DIP24, clock 16 / 2)
      JP1, JP2    - Jumpers to configure ROM size of IC7, both set to 2-3
      JP3, JP4    - Jumpers to configure ROM size of IC8, both set to 2-3
      JP5, JP6    - Jumpers to configure size of RAM, both set to 2-3
      JP7         - Jumper, set to 2-3
      JP13, JP14, \
      JP15, JP16, | Jumpers to configure size of ROMs at IC31, IC32 & IC33
      JP17, JP18  / JP13 & JP14 set to 2-3, others set to 1-2
      CN1         - 10 pin connector for power input (+5V, GND, +12V)
      CN2         - 6 pin connector (joins to main PCB, used for sound communication from Main PCB to Sound PCB)
      CN5         - 5 pin connector, unamplified sound out to Amp PCB
      CN6, CN7    - Unamplified audio output, left/right RCA audio connectors (not used)
      CN8, CN9    - Connectors to join Sound Option Board to Main Sound Board

      Sound Main PCB
      --------------

                    IC#   PCB
                          Label        Virtua Racing    Virtua Fighter    Wing War     Star Wars Arcade
                    -----------------------------------------------------------------------------------
                    7     1024/4096    EPR-14870A       EPR-16120         EPR-16751    !
                    8     1024/4096    -                EPR-16121         EPR-16752    ?
                    31    27C4001      -                -                 -            ?
                    32    23C16000     MPR-14873        MPR-16122         MPR-16753    !
                    33    23C16000     -                MPR-16123         MPR-16754    ?

      Sound Sub PCB
      -------------

                    IC#   PCB
                          Label        Virtua Racing    Virtua Fighter    Wing War     Star Wars Arcade
                    -----------------------------------------------------------------------------------
                    3     27C4001      -                -                 -            ?
                    4     23C16000     MPR-14876        MPR-16124         MPR-16755    ?
                    5     23C16000     -                MPR-16125         MPR-16756    ?

                    - denotes socket known to be not populated.
                    ! denotes known to be not dumped.
                    ? denotes unknown if socket is populated or not, but not dumped either way.


Audio Mix PCB
-------------

839-0542 (C) SEGA 1992
|-------------|
|             |
|             |
|CN1  CN3  CN2|
|             |
|             |
|-------------|
Notes:
      This is a very small PCB with only 3 connectors on it and 4 resistors
      which simply mixes the left/right audio into one overall output which
      is fed into the Amp PCB.
      CN1 - 5 pin connector, right channel audio input
      CN2 - 5 pin connectos, mixed audio output to Amp PCB
      CN3 - 5 pin connector, left channel audio input


Amp PCB
-------

838-10018 SJ25-0168-03
|----------------------------------|
|             CN1                  |
|                        7812   CN2|
|CN3                               |
|                                  |
|                        TA8225H   |
|                                  |
|                               CN4|
|                                  |
|                                  |
|                        TA8225H   |
|                                  |
|LCH                               |
|RCH                               |
|----------------------------------|
Notes:
      There are also many capacitors/transistors and other power-related
      components which are not shown.

      7812     - 12V voltage regulator
      TA8225H  - Toshiba TA8225H 45W power amplifier (x2)
      CN1      - 3 pin connector, 12VAC power input
      CN2      - 2 pin 12VDC power output
      CN3      - 7 pin connector, joins to credit board and coin mech (power/credit related outputs)
      CN4      - 4 pin connector, amplified dual (left/right) speaker audio outputs
      RCH, LCH - RCA connectors, left/right sound input from CN2 of Audio Mix PCB


*/

#include "emu.h"
#include "cpu/v60/v60.h"
#include "deprecat.h"
#include "video/segaic24.h"
#include "cpu/m68000/m68000.h"
#include "cpu/mb86233/mb86233.h"
#include "sound/multipcm.h"
#include "sound/2612intf.h"
#include "machine/nvram.h"
#include "includes/model1.h"



static READ16_HANDLER( io_r )
{
	static const char *const analognames[] = { "AN0", "AN1", "AN2", "AN3", "AN4", "AN5", "AN6", "AN7" };
	static const char *const inputnames[] = { "IN0", "IN1", "IN2" };

	if(offset < 0x8)
		return input_port_read_safe(space->machine(), analognames[offset], 0x00);

	if(offset < 0x10)
	{
		offset -= 0x8;
		if(offset < 3)
			return input_port_read(space->machine(), inputnames[offset]);
		return 0xff;
	}

	logerror("IOR: %02x\n", offset);
	return 0xffff;
}

static READ16_HANDLER( fifoin_status_r )
{
	return 0xffff;
}

static WRITE16_HANDLER( bank_w )
{
	if(ACCESSING_BITS_0_7) {
		switch(data & 0xf) {
		case 0x1: // 100000-1fffff data roms banking
			memory_set_bankptr(space->machine(), "bank1", space->machine().region("maincpu")->base() + 0x1000000 + 0x100000*((data >> 4) & 0xf));
			logerror("BANK %x\n", 0x1000000 + 0x100000*((data >> 4) & 0xf));
			break;
		case 0x2: // 200000-2fffff data roms banking (unused, all known games have only one bank)
			break;
		case 0xf: // f00000-ffffff program roms banking (unused, all known games have only one bank)
			break;
		}
	}
}



static void irq_raise(running_machine &machine, int level)
{
	model1_state *state = machine.driver_data<model1_state>();
	//  logerror("irq: raising %d\n", level);
	//  irq_status |= (1 << level);
	state->m_last_irq = level;
	cputag_set_input_line(machine, "maincpu", 0, HOLD_LINE);
}

static IRQ_CALLBACK(irq_callback)
{
	model1_state *state = device->machine().driver_data<model1_state>();
	return state->m_last_irq;
}
// vf
// 1 = fe3ed4
// 3 = fe3f5c
// other = fe3ec8 / fe3ebc

// vr
// 1 = fe02bc
// other = f302a4 / fe02b0

// swa
// 1 = ff504
// 3 = ff54c
// other = ff568/ff574

static void irq_init(running_machine &machine)
{
	cputag_set_input_line(machine, "maincpu", 0, CLEAR_LINE);
	device_set_irq_callback(machine.device("maincpu"), irq_callback);
}

static INTERRUPT_GEN(model1_interrupt)
{
	model1_state *state = device->machine().driver_data<model1_state>();
	if (cpu_getiloops(device))
	{
		irq_raise(device->machine(), 1);
	}
	else
	{
		irq_raise(device->machine(), state->m_sound_irq);

		// if the FIFO has something in it, signal the 68k too
		if (state->m_fifo_rptr != state->m_fifo_wptr)
		{
			cputag_set_input_line(device->machine(), "audiocpu", 2, HOLD_LINE);
		}
	}
}

static MACHINE_RESET(model1)
{
	model1_state *state = machine.driver_data<model1_state>();
	memory_set_bankptr(machine, "bank1", machine.region("maincpu")->base() + 0x1000000);
	irq_init(machine);
	model1_tgp_reset(machine, !strcmp(machine.system().name, "swa") || !strcmp(machine.system().name, "wingwar") || !strcmp(machine.system().name, "wingwaru") || !strcmp(machine.system().name, "wingwarj"));
	if (!strcmp(machine.system().name, "swa"))
	{
		state->m_sound_irq = 0;
	}
	else
	{
		state->m_sound_irq = 3;
	}

	// init the sound FIFO
	state->m_fifo_rptr = state->m_fifo_wptr = 0;
	memset(state->m_to_68k, 0, sizeof(state->m_to_68k));
}

static MACHINE_RESET(model1_vr)
{
	model1_state *state = machine.driver_data<model1_state>();
	memory_set_bankptr(machine, "bank1", machine.region("maincpu")->base() + 0x1000000);
	irq_init(machine);
	model1_vr_tgp_reset(machine);
	state->m_sound_irq = 3;

	// init the sound FIFO
	state->m_fifo_rptr = state->m_fifo_wptr = 0;
	memset(state->m_to_68k, 0, sizeof(state->m_to_68k));
}

static READ16_HANDLER( network_ctl_r )
{
	if(offset)
		return 0x40;
	else
		return 0x00;
}

static WRITE16_HANDLER( network_ctl_w )
{
}

static WRITE16_HANDLER(md1_w)
{
	model1_state *state = space->machine().driver_data<model1_state>();
	COMBINE_DATA(state->m_display_list1+offset);
	if(0 && offset)
		return;
	if(1 && state->m_dump)
		logerror("TGP: md1_w %x, %04x @ %04x (%x)\n", offset, data, mem_mask, cpu_get_pc(&space->device()));
}

static WRITE16_HANDLER(md0_w)
{
	model1_state *state = space->machine().driver_data<model1_state>();
	COMBINE_DATA(state->m_display_list0+offset);
	if(0 && offset)
		return;
	if(1 && state->m_dump)
		logerror("TGP: md0_w %x, %04x @ %04x (%x)\n", offset, data, mem_mask, cpu_get_pc(&space->device()));
}

static WRITE16_HANDLER(p_w)
{
	UINT16 old = space->machine().generic.paletteram.u16[offset];
	paletteram16_xBBBBBGGGGGRRRRR_word_w(space, offset, data, mem_mask);
	if(0 && space->machine().generic.paletteram.u16[offset] != old)
		logerror("XVIDEO: p_w %x, %04x @ %04x (%x)\n", offset, data, mem_mask, cpu_get_pc(&space->device()));
}

static WRITE16_HANDLER(mr_w)
{
	model1_state *state = space->machine().driver_data<model1_state>();
	COMBINE_DATA(state->m_mr+offset);
	if(0 && offset == 0x1138/2)
		logerror("MR.w %x, %04x @ %04x (%x)\n", offset*2+0x500000, data, mem_mask, cpu_get_pc(&space->device()));
}

static WRITE16_HANDLER(mr2_w)
{
	model1_state *state = space->machine().driver_data<model1_state>();
	COMBINE_DATA(state->m_mr2+offset);
#if 0
	if(0 && offset == 0x6e8/2) {
		logerror("MR.w %x, %04x @ %04x (%x)\n", offset*2+0x400000, data, mem_mask, cpu_get_pc(&space->device()));
	}
	if(offset/2 == 0x3680/4)
		logerror("MW f80[r25], %04x%04x (%x)\n", state->m_mr2[0x3680/2+1], state->m_mr2[0x3680/2], cpu_get_pc(&space->device()));
	if(offset/2 == 0x06ca/4)
		logerror("MW fca[r19], %04x%04x (%x)\n", state->m_mr2[0x06ca/2+1], state->m_mr2[0x06ca/2], cpu_get_pc(&space->device()));
	if(offset/2 == 0x1eca/4)
		logerror("MW fca[r22], %04x%04x (%x)\n", state->m_mr2[0x1eca/2+1], state->m_mr2[0x1eca/2], cpu_get_pc(&space->device()));
#endif

	// wingwar scene position, pc=e1ce -> d735
	if(offset/2 == 0x1f08/4)
		logerror("MW  8[r10], %f (%x)\n", *(float *)(state->m_mr2+0x1f08/2), cpu_get_pc(&space->device()));
	if(offset/2 == 0x1f0c/4)
		logerror("MW  c[r10], %f (%x)\n", *(float *)(state->m_mr2+0x1f0c/2), cpu_get_pc(&space->device()));
	if(offset/2 == 0x1f10/4)
		logerror("MW 10[r10], %f (%x)\n", *(float *)(state->m_mr2+0x1f10/2), cpu_get_pc(&space->device()));
}

static READ16_HANDLER( snd_68k_ready_r )
{
	int sr = cpu_get_reg(space->machine().device("audiocpu"), M68K_SR);

	if ((sr & 0x0700) > 0x0100)
	{
		device_spin_until_time(&space->device(), attotime::from_usec(40));
		return 0;	// not ready yet, interrupts disabled
	}

	return 0xff;
}

static WRITE16_HANDLER( snd_latch_to_68k_w )
{
	model1_state *state = space->machine().driver_data<model1_state>();
	state->m_to_68k[state->m_fifo_wptr] = data;
	state->m_fifo_wptr++;
	if (state->m_fifo_wptr >= ARRAY_LENGTH(state->m_to_68k)) state->m_fifo_wptr = 0;

	// signal the 68000 that there's data waiting
	cputag_set_input_line(space->machine(), "audiocpu", 2, HOLD_LINE);
	// give the 68k time to reply
	device_spin_until_time(&space->device(), attotime::from_usec(40));
}

static ADDRESS_MAP_START( model1_mem, AS_PROGRAM, 16 )
	AM_RANGE(0x000000, 0x0fffff) AM_ROM
	AM_RANGE(0x100000, 0x1fffff) AM_ROMBANK("bank1")
	AM_RANGE(0x200000, 0x2fffff) AM_ROM

	AM_RANGE(0x400000, 0x40ffff) AM_RAM_WRITE(mr2_w) AM_BASE_MEMBER(model1_state, m_mr2)
	AM_RANGE(0x500000, 0x53ffff) AM_RAM_WRITE(mr_w)  AM_BASE_MEMBER(model1_state, m_mr)

	AM_RANGE(0x600000, 0x60ffff) AM_RAM_WRITE(md0_w) AM_BASE_MEMBER(model1_state, m_display_list0)
	AM_RANGE(0x610000, 0x61ffff) AM_RAM_WRITE(md1_w) AM_BASE_MEMBER(model1_state, m_display_list1)
	AM_RANGE(0x680000, 0x680003) AM_READWRITE(model1_listctl_r, model1_listctl_w)

	AM_RANGE(0x700000, 0x70ffff) AM_DEVREADWRITE_MODERN("tile", segas24_tile, tile_r, tile_w)
	AM_RANGE(0x720000, 0x720001) AM_WRITENOP		// Unknown, always 0
	AM_RANGE(0x740000, 0x740001) AM_WRITENOP		// Horizontal synchronization register
	AM_RANGE(0x760000, 0x760001) AM_WRITENOP		// Vertical synchronization register
	AM_RANGE(0x770000, 0x770001) AM_WRITENOP		// Video synchronization switch
	AM_RANGE(0x780000, 0x7fffff) AM_DEVREADWRITE_MODERN("tile", segas24_tile, char_r, char_w)

	AM_RANGE(0x900000, 0x903fff) AM_RAM_WRITE(p_w) AM_BASE_GENERIC(paletteram)
	AM_RANGE(0x910000, 0x91bfff) AM_RAM  AM_BASE_MEMBER(model1_state, m_color_xlat)

	AM_RANGE(0xc00000, 0xc0003f) AM_READ(io_r) AM_WRITENOP

	AM_RANGE(0xc00040, 0xc00043) AM_READWRITE(network_ctl_r, network_ctl_w)

	AM_RANGE(0xc00200, 0xc002ff) AM_RAM AM_SHARE("nvram")

	AM_RANGE(0xc40000, 0xc40001) AM_WRITE(snd_latch_to_68k_w)
	AM_RANGE(0xc40002, 0xc40003) AM_READ(snd_68k_ready_r)

	AM_RANGE(0xd00000, 0xd00001) AM_READWRITE(model1_tgp_copro_adr_r, model1_tgp_copro_adr_w)
	AM_RANGE(0xd20000, 0xd20003) AM_WRITE(model1_tgp_copro_ram_w )
	AM_RANGE(0xd80000, 0xd80003) AM_WRITE(model1_tgp_copro_w) AM_MIRROR(0x10)
	AM_RANGE(0xdc0000, 0xdc0003) AM_READ(fifoin_status_r)

	AM_RANGE(0xe00000, 0xe00001) AM_WRITENOP        // Watchdog?  IRQ ack? Always 0x20, usually on irq
	AM_RANGE(0xe00004, 0xe00005) AM_WRITE(bank_w)
	AM_RANGE(0xe0000c, 0xe0000f) AM_WRITENOP

	AM_RANGE(0xfc0000, 0xffffff) AM_ROM
ADDRESS_MAP_END

static ADDRESS_MAP_START( model1_io, AS_IO, 16 )
	AM_RANGE(0xd20000, 0xd20003) AM_READ(model1_tgp_copro_ram_r)
	AM_RANGE(0xd80000, 0xd80003) AM_READ(model1_tgp_copro_r)
ADDRESS_MAP_END

static ADDRESS_MAP_START( model1_vr_mem, AS_PROGRAM, 16 )
	AM_RANGE(0x000000, 0x0fffff) AM_ROM
	AM_RANGE(0x100000, 0x1fffff) AM_ROMBANK("bank1")
	AM_RANGE(0x200000, 0x2fffff) AM_ROM

	AM_RANGE(0x400000, 0x40ffff) AM_RAM_WRITE(mr2_w) AM_BASE_MEMBER(model1_state, m_mr2)
	AM_RANGE(0x500000, 0x53ffff) AM_RAM_WRITE(mr_w)  AM_BASE_MEMBER(model1_state, m_mr)

	AM_RANGE(0x600000, 0x60ffff) AM_RAM_WRITE(md0_w) AM_BASE_MEMBER(model1_state, m_display_list0)
	AM_RANGE(0x610000, 0x61ffff) AM_RAM_WRITE(md1_w) AM_BASE_MEMBER(model1_state, m_display_list1)
	AM_RANGE(0x680000, 0x680003) AM_READWRITE(model1_listctl_r, model1_listctl_w)

	AM_RANGE(0x700000, 0x70ffff) AM_DEVREADWRITE_MODERN("tile", segas24_tile, tile_r, tile_w)
	AM_RANGE(0x720000, 0x720001) AM_WRITENOP		// Unknown, always 0
	AM_RANGE(0x740000, 0x740001) AM_WRITENOP		// Horizontal synchronization register
	AM_RANGE(0x760000, 0x760001) AM_WRITENOP		// Vertical synchronization register
	AM_RANGE(0x770000, 0x770001) AM_WRITENOP		// Video synchronization switch
	AM_RANGE(0x780000, 0x7fffff) AM_DEVREADWRITE_MODERN("tile", segas24_tile, char_r, char_w)

	AM_RANGE(0x900000, 0x903fff) AM_RAM_WRITE(p_w) AM_BASE_GENERIC(paletteram)
	AM_RANGE(0x910000, 0x91bfff) AM_RAM  AM_BASE_MEMBER(model1_state, m_color_xlat)

	AM_RANGE(0xc00000, 0xc0003f) AM_READ(io_r) AM_WRITENOP

	AM_RANGE(0xc00040, 0xc00043) AM_READWRITE(network_ctl_r, network_ctl_w)

	AM_RANGE(0xc00200, 0xc002ff) AM_RAM AM_SHARE("nvram")

	AM_RANGE(0xc40000, 0xc40001) AM_WRITE(snd_latch_to_68k_w)
	AM_RANGE(0xc40002, 0xc40003) AM_READ(snd_68k_ready_r)

	AM_RANGE(0xd00000, 0xd00001) AM_READWRITE(model1_tgp_vr_adr_r, model1_tgp_vr_adr_w)
	AM_RANGE(0xd20000, 0xd20003) AM_WRITE(model1_vr_tgp_ram_w )
	AM_RANGE(0xd80000, 0xd80003) AM_WRITE(model1_vr_tgp_w) AM_MIRROR(0x10)
	AM_RANGE(0xdc0000, 0xdc0003) AM_READ(fifoin_status_r)

	AM_RANGE(0xe00000, 0xe00001) AM_WRITENOP        // Watchdog?  IRQ ack? Always 0x20, usually on irq
	AM_RANGE(0xe00004, 0xe00005) AM_WRITE(bank_w)
	AM_RANGE(0xe0000c, 0xe0000f) AM_WRITENOP

	AM_RANGE(0xfc0000, 0xffffff) AM_ROM
ADDRESS_MAP_END

static ADDRESS_MAP_START( model1_vr_io, AS_IO, 16 )
	AM_RANGE(0xd20000, 0xd20003) AM_READ(model1_vr_tgp_ram_r)
	AM_RANGE(0xd80000, 0xd80003) AM_READ(model1_vr_tgp_r)
ADDRESS_MAP_END

static READ16_HANDLER( m1_snd_68k_latch_r )
{
	model1_state *state = space->machine().driver_data<model1_state>();
	UINT16 retval;

	retval = state->m_to_68k[state->m_fifo_rptr];

	state->m_fifo_rptr++;
	if (state->m_fifo_rptr >= ARRAY_LENGTH(state->m_to_68k)) state->m_fifo_rptr = 0;

	return retval;
}

static READ16_HANDLER( m1_snd_v60_ready_r )
{
	return 1;
}

static WRITE16_DEVICE_HANDLER( m1_snd_mpcm_bnk_w )
{
	multipcm_set_bank(device, 0x100000 * (data & 3), 0x100000 * (data & 3));
}

static WRITE16_HANDLER( m1_snd_68k_latch1_w )
{
}

static WRITE16_HANDLER( m1_snd_68k_latch2_w )
{
}

static ADDRESS_MAP_START( model1_snd, AS_PROGRAM, 16 )
	AM_RANGE(0x000000, 0x0bffff) AM_ROM
	AM_RANGE(0xc20000, 0xc20001) AM_READWRITE( m1_snd_68k_latch_r, m1_snd_68k_latch1_w )
	AM_RANGE(0xc20002, 0xc20003) AM_READWRITE( m1_snd_v60_ready_r, m1_snd_68k_latch2_w )
	AM_RANGE(0xc40000, 0xc40007) AM_DEVREADWRITE8( "sega1", multipcm_r, multipcm_w, 0x00ff )
	AM_RANGE(0xc40012, 0xc40013) AM_WRITENOP
	AM_RANGE(0xc50000, 0xc50001) AM_DEVWRITE( "sega1", m1_snd_mpcm_bnk_w )
	AM_RANGE(0xc60000, 0xc60007) AM_DEVREADWRITE8( "sega2", multipcm_r, multipcm_w, 0x00ff )
	AM_RANGE(0xc70000, 0xc70001) AM_DEVWRITE( "sega2", m1_snd_mpcm_bnk_w )
	AM_RANGE(0xd00000, 0xd00007) AM_DEVREADWRITE8( "ymsnd", ym3438_r, ym3438_w, 0x00ff )
	AM_RANGE(0xf00000, 0xf0ffff) AM_RAM
ADDRESS_MAP_END


static INPUT_PORTS_START( vf )
	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_SERVICE_NO_TOGGLE(0x0004, IP_ACTIVE_LOW)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("IN1")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("IN2")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

static INPUT_PORTS_START( vr )
	PORT_START("AN0")	/* Steering */
	PORT_BIT( 0xff, 0x80, IPT_PADDLE ) PORT_SENSITIVITY(100) PORT_KEYDELTA(3)

	PORT_START("AN1")	/* Accel / Decel */
	PORT_BIT( 0xff, 0x30, IPT_PEDAL ) PORT_MINMAX(1,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(16)

	PORT_START("AN2")	/* Brake */
	PORT_BIT( 0xff, 0x30, IPT_PEDAL2 ) PORT_MINMAX(1,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(16)

	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_SERVICE_NO_TOGGLE(0x0004, IP_ACTIVE_LOW)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_NAME("VR1 (Red)") PORT_PLAYER(1)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_NAME("VR2 (Blue)") PORT_PLAYER(1)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_NAME("VR3 (Yellow)") PORT_PLAYER(1)
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("IN1")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_NAME("VR4 (Green)") PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON7 ) PORT_NAME("Shift Down") PORT_PLAYER(1)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON8 ) PORT_NAME("Shift Up") PORT_PLAYER(1)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("IN2")
	PORT_BIT( 0x00ff, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

static INPUT_PORTS_START( wingwar )
	PORT_START("AN0")	/* X */
	PORT_BIT( 0xff, 0x80, IPT_AD_STICK_X ) PORT_SENSITIVITY(100) PORT_KEYDELTA(4)

	PORT_START("AN1")	/* Y */
	PORT_BIT( 0xff, 0x80, IPT_AD_STICK_Y ) PORT_SENSITIVITY(100) PORT_KEYDELTA(4) PORT_REVERSE

	PORT_START("AN2")	/* Throttle */
	PORT_BIT( 0xff, 0x01, IPT_PEDAL ) PORT_MINMAX(1,0xff) PORT_SENSITIVITY(100) PORT_KEYDELTA(16)

	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE_NO_TOGGLE(0x0004, IP_ACTIVE_LOW)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("IN1")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_PLAYER(1)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_PLAYER(1)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON7 ) PORT_PLAYER(1)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("IN2")
	PORT_BIT( 0x00ff, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

static INPUT_PORTS_START( swa )
	PORT_START("AN0")	/* X */
	PORT_BIT( 0xff, 127, IPT_AD_STICK_X ) PORT_MINMAX(27,227) PORT_SENSITIVITY(100) PORT_KEYDELTA(4) PORT_REVERSE

	PORT_START("AN1")	/* Y */
	PORT_BIT( 0xff, 127, IPT_AD_STICK_Y ) PORT_MINMAX(27,227) PORT_SENSITIVITY(100) PORT_KEYDELTA(4) PORT_REVERSE

	PORT_START("AN2")	/* Throttle */
	PORT_BIT( 0xff, 228, IPT_PEDAL ) PORT_MINMAX(28,228) PORT_SENSITIVITY(100) PORT_KEYDELTA(16) PORT_REVERSE

	PORT_START("AN4")	/* X */
	PORT_BIT( 0xff, 127, IPT_AD_STICK_X ) PORT_MINMAX(27,227) PORT_SENSITIVITY(100) PORT_KEYDELTA(4) PORT_REVERSE PORT_PLAYER(2)

	PORT_START("AN5")	/* Y */
	PORT_BIT( 0xff, 127, IPT_AD_STICK_Y ) PORT_MINMAX(27,227) PORT_SENSITIVITY(100) PORT_KEYDELTA(4) PORT_PLAYER(2)

	PORT_START("IN0")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_SERVICE_NO_TOGGLE(0x0004, IP_ACTIVE_LOW)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x00c0, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("IN1")
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x00e0, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START("IN2")
	PORT_BIT( 0x00ff, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

#define MODEL1_CPU_BOARD \
	ROM_REGION( 0xe0000, "user5", 0 ) \
	ROM_LOAD32_WORD("opr14742.bin",  0x000000,  0x20000, CRC(446a1085) SHA1(51b3f4d3a35a36087ea0ba4e26d6e7d17b6418e2) ) \
	ROM_LOAD32_WORD("opr14743.bin",  0x000002,  0x20000, CRC(e8953554) SHA1(1499f8e30ac15affc66e6f04ae031bb8680d9260) ) \
	ROM_LOAD("opr14744.bin",   0x040000,  0x20000, CRC(730ea9e0) SHA1(651f1db4089a400d073b19ada299b4b08b08f372) ) \
	ROM_LOAD("opr14745.bin",   0x060000,  0x20000, CRC(4c934d96) SHA1(e3349ece0e47f684d61ad11bfea4a90602287350) ) \
	ROM_LOAD("opr14746.bin",   0x080000,  0x20000, CRC(2a266cbd) SHA1(34e047a93459406c22acf4c25089d1a4955f94ca) ) \
	ROM_LOAD("opr14747.bin",   0x0a0000,  0x20000, CRC(a4ad5e19) SHA1(7d7ec300eeb9a8de1590011e37108688c092f329) ) \
	ROM_LOAD("opr14748.bin",   0x0c0000,  0x20000, CRC(4a532cb8) SHA1(23280ebbcd6b2bc8a8e643a2d07a58d6598301b8) ) \


ROM_START( vf )

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16082.14", 0x200000, 0x80000, CRC(b23f22ee) SHA1(9fd5b5a5974703a60a54de3d2bce4301bfc0e533) )
	ROM_LOAD16_BYTE( "epr-16083.15", 0x200001, 0x80000, CRC(d12c77f8) SHA1(b4aeba8d5f1ab4aec024391407a2cb58ce2e94b0) )

	ROM_LOAD( "epr-16080.4", 0xfc0000, 0x20000, CRC(3662E1A5) SHA1(6bfceb1a7c1c7912679c907f2b7516ae9c7dda67) )
	ROM_LOAD( "epr-16081.5", 0xfe0000, 0x20000, CRC(6DEC06CE) SHA1(7891544456bccd2fc647bccd058945ad50466636) )

	ROM_LOAD16_BYTE( "mpr-16084.6", 0x1000000, 0x80000, CRC(483f453b) SHA1(41a5527be73f5dd1c87b2a8113235bdd247ec049) )
	ROM_LOAD16_BYTE( "mpr-16085.7", 0x1000001, 0x80000, CRC(5fa01277) SHA1(dfa7ddff0a7daf29071431f26b93dd8e8e5793b6) )
	ROM_LOAD16_BYTE( "mpr-16086.8", 0x1100000, 0x80000, CRC(deac47a1) SHA1(3a8016124e4dc579d4aae745d4af1905ad0e4fbd) )
	ROM_LOAD16_BYTE( "mpr-16087.9", 0x1100001, 0x80000, CRC(7a64daac) SHA1(da6a9cad4b0cb2af4299e664c0889f3fbdc25530) )
	ROM_LOAD16_BYTE( "mpr-16088.10", 0x1200000, 0x80000, CRC(fcda2d1e) SHA1(0f7d0f604d429a1da0d1c3f31694520bada49680) )
	ROM_LOAD16_BYTE( "mpr-16089.11", 0x1200001, 0x80000, CRC(39befbe0) SHA1(362c493092cd0536fadee7326ecc7f973e23fb58) )
	ROM_LOAD16_BYTE( "mpr-16090.12", 0x1300000, 0x80000, CRC(90c76831) SHA1(5a3c25f2a131cfbb2ad067bef1ab7b1c95645d41) )
	ROM_LOAD16_BYTE( "mpr-16091.13", 0x1300001, 0x80000, CRC(53115448) SHA1(af798d5b1fcb720d7288a5ac48839d9ace16a2f2) )

	ROM_REGION( 0xc0000, "audiocpu", 0 )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-16120.7", 0x00000, 0x20000, CRC(2bff8378) SHA1(854b08ab983e4e98cb666f2f44de9a6829b1eb52) )
	ROM_LOAD16_WORD_SWAP( "epr-16121.8", 0x20000, 0x20000, CRC(ff6723f9) SHA1(53498b8c103745883657dfd6efe27edfd48b356f) )
	ROM_RELOAD( 0x80000, 0x20000)

	ROM_REGION( 0x400000, "sega1", 0 ) /* Samples */
	ROM_LOAD( "mpr-16122.32", 0x000000, 0x200000, CRC(568bc64e) SHA1(31fd0ef8319efe258011b4621adebb790b620770) )
	ROM_LOAD( "mpr-16123.33", 0x200000, 0x200000, CRC(15d78844) SHA1(37c17e38604cf7004a951408024941cd06b1d93e) )

	ROM_REGION( 0x400000, "sega2", 0 ) /* Samples */
	ROM_LOAD( "mpr-16124.4", 0x000000, 0x200000, CRC(45520ba1) SHA1(c33e3c12639961016e5fa6b5025d0a67dff28907) )
	ROM_LOAD( "mpr-16125.5", 0x200000, 0x200000, CRC(9b4998b6) SHA1(0418d9b0acf79f35d0f7575c21f1be9a0ea343da) )

	ROM_REGION32_LE( 0x1000000, "user1", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16096.26", 0x000000, 0x200000, CRC(a92b0bf3) SHA1(fd3adff5f41f0b0be98df548c848eda04fc0da48) )
	ROM_LOAD32_WORD( "mpr-16097.27", 0x000002, 0x200000, CRC(0232955a) SHA1(df934fb6d022032620932571ff5ed176d5dcb017) )
	ROM_LOAD32_WORD( "mpr-16098.28", 0x400000, 0x200000, CRC(cf2e1b84) SHA1(f3d16c72344f7f218a792ce7f1dd7cad910a8c97) )
	ROM_LOAD32_WORD( "mpr-16099.29", 0x400002, 0x200000, CRC(20e46854) SHA1(423d3642bd2f14e68d29029c027b791de2c1ec53) )
	ROM_LOAD32_WORD( "mpr-16100.30", 0x800000, 0x200000, CRC(e13e983d) SHA1(120637caa2404ad4124b676fd6fcd721f30948df) )
	ROM_LOAD32_WORD( "mpr-16101.31", 0x800002, 0x200000, CRC(0dbed94d) SHA1(df1cddcc1d3976816bd786c2d6211a8563f6f690) )
	ROM_LOAD32_WORD( "mpr-16102.32", 0xc00000, 0x200000, CRC(4cb41fb6) SHA1(4a07bfad4f221508de8c931861424dcc5be3f46a) )
	ROM_LOAD32_WORD( "mpr-16103.33", 0xc00002, 0x200000, CRC(526d1c76) SHA1(edc8dafc9261cd0e970c3b50e3c1ca51a32a4cdf) )
ROM_END

ROM_START( vr )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-14882.14", 0x200000, 0x80000, CRC(547D75AD) SHA1(a57c11966886c37de1d7df131ad60457669231dd) )
	ROM_LOAD16_BYTE( "epr-14883.15", 0x200001, 0x80000, CRC(6BFAD8B1) SHA1(c1f780e456b405abd42d92f4e03e40aad88f8c22) )

	ROM_LOAD( "epr-14878a.4", 0xfc0000, 0x20000, CRC(6D69E695) SHA1(12d3612d3dfd474b8020cdfb8ffc5dcc64e2e1a3) )
	ROM_LOAD( "epr-14879a.5", 0xfe0000, 0x20000, CRC(D45AF9DD) SHA1(48f2bf940c78e3ae4273a56e9f32371d870e41e0) )

	ROM_LOAD16_BYTE( "mpr-14880.6",  0x1000000, 0x80000, CRC(ADC7C208) SHA1(967b6f522011f17fd2821ccbe20bcb6d4680a4a0) )
	ROM_LOAD16_BYTE( "mpr-14881.7",  0x1000001, 0x80000, CRC(E5AB89DF) SHA1(873dea86628457e69f732158e3efb05d133eaa44) )
	ROM_LOAD16_BYTE( "mpr-14884.8",  0x1100000, 0x80000, CRC(6CF9C026) SHA1(f4d717958dba6b6402f5ffe7638fe0bf353b61ed) )
	ROM_LOAD16_BYTE( "mpr-14885.9",  0x1100001, 0x80000, CRC(F65C9262) SHA1(511a22bcfaf199737bfc5a809fd66cb4439dd386) )
	ROM_LOAD16_BYTE( "mpr-14886.10", 0x1200000, 0x80000, CRC(92868734) SHA1(e1dfb134dc3ba7e0b1d40681621e09ac5a222518) )
	ROM_LOAD16_BYTE( "mpr-14887.11", 0x1200001, 0x80000, CRC(10C7C636) SHA1(c77d55460bba4354349e473e129f21afeed05e91) )
	ROM_LOAD16_BYTE( "mpr-14888.12", 0x1300000, 0x80000, CRC(04BFDC5B) SHA1(bb8788a761620d0440a62ae51c3b41f70a04b5e4) )
	ROM_LOAD16_BYTE( "mpr-14889.13", 0x1300001, 0x80000, CRC(C49F0486) SHA1(cc2bb9059c016ba2c4f6e7508bd1687df07b8b48) )

	ROM_REGION( 0xc0000, "audiocpu", 0 )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-14870a.7", 0x00000, 0x20000, CRC(919d9b75) SHA1(27be79881cc9a2b5cf37e18f1e2d87251426b428) )

	ROM_REGION( 0x400000, "sega1", 0 ) /* Samples */
	ROM_LOAD( "mpr-14873.32", 0x000000, 0x200000, CRC(b1965190) SHA1(fc47e9ed4a44d48477bd9a35e42c26508c0f4a0c) )

	ROM_REGION( 0x400000, "sega2", 0 ) /* Samples */
	ROM_LOAD( "mpr-14876.4", 0x000000, 0x200000, CRC(ba6b2327) SHA1(02285520624a4e612cb4b65510e3458b13b1c6ba) )

	ROM_REGION32_LE( 0x1000000, "user1", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-14890.26", 0x000000, 0x200000, CRC(dcbe006b) SHA1(195be7fabec405ca1b4e1338d3b8d7bb4a06dd73) )
	ROM_LOAD32_WORD( "mpr-14891.27", 0x000002, 0x200000, CRC(25832b38) SHA1(a8d74538149c92bae661334e327b031eaca2a8f2) )
	ROM_LOAD32_WORD( "mpr-14892.28", 0x400000, 0x200000, CRC(5136f3ba) SHA1(ce8312975764db09bbfa2068b99559a5c1798a36) )
	ROM_LOAD32_WORD( "mpr-14893.29", 0x400002, 0x200000, CRC(1c531ada) SHA1(8b373ac97a3649c64f48eb3d1dd95c5833f330b6) )
	ROM_LOAD32_WORD( "mpr-14894.30", 0x800000, 0x200000, CRC(830a71bc) SHA1(884378e8a5afeb819daf5285d0d205986d566340) )
	ROM_LOAD32_WORD( "mpr-14895.31", 0x800002, 0x200000, CRC(af027ac5) SHA1(523f03d90358ddb7d0e96fd06b9a65cebfc09f24) )
	ROM_LOAD32_WORD( "mpr-14896.32", 0xc00000, 0x200000, CRC(382091dc) SHA1(efa266f0f6bfe36ad1c365e588fff33b01e166dd) )
	ROM_LOAD32_WORD( "mpr-14879.33", 0xc00002, 0x200000, CRC(74873195) SHA1(80705ec577d14570f9bba77cc26766f831c41f42) )

	ROM_REGION32_LE( 0x200000, "user2", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-14898.39", 0x000000, 0x80000, CRC(61da2bb6) SHA1(7a12ba522d64a1aeec1ca6f5a87ee063692131f9) )
	ROM_LOAD32_BYTE( "mpr-14899.40", 0x000001, 0x80000, CRC(2cd58bee) SHA1(73defec823de4244a387af55fea7210edc1b314f) )
	ROM_LOAD32_BYTE( "mpr-14900.41", 0x000002, 0x80000, CRC(aa7c017d) SHA1(0fa2b59a8bb5f5907b2b2567e69d11c73b398dc1) )
	ROM_LOAD32_BYTE( "mpr-14901.42", 0x000003, 0x80000, CRC(175b7a9a) SHA1(c86602e771cd49bab425b4ba7926d2f44858bd39) )

	ROM_REGION( 0x2000, "tgp", 0 ) /* TGP program rom */
	// this is the Daytona TGP program with some modifications needed for Virtua Racing
	// the real TGP program is an internal ROM and still needs dumping
	ROM_LOAD("vr-tgp.bin", 0x000000, 0x2000, BAD_DUMP CRC(3de33c7f) SHA1(acecc779c9d8fe39ded6c22492be5b7c25fd52db) )
ROM_END

ROM_START( vformula )
	MODEL1_CPU_BOARD

	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-15638.14", 0x200000, 0x80000, CRC(b9db21a2) SHA1(db58c047977f5fc37f278afe7159a78e3fa6c015) )
	ROM_LOAD16_BYTE( "epr-15639.15", 0x200001, 0x80000, CRC(4c3796f5) SHA1(1bf312a4999a15fbc5d194627f9c0ad9dbc1f2c0) )

	ROM_LOAD( "epr-15623.4", 0xfc0000, 0x20000, CRC(155fa5be) SHA1(a0a3fd8980c52279adbc1c64aa22e42a0b196dd9) )
	ROM_LOAD( "epr-15622.5", 0xfe0000, 0x20000, CRC(18007f6f) SHA1(61573742627ec027abd2cc700e79f04da5215bfd) )

	ROM_LOAD16_BYTE( "epr-15641.6",  0x1000000, 0x80000, CRC(bf01e4d5) SHA1(53ad9ecd2de2ea1d7b446f9db61352e10a55ea05) )
	ROM_LOAD16_BYTE( "epr-15640.7",  0x1000001, 0x80000, CRC(3e6d83dc) SHA1(62aa552a38ee193e0dfab5d1261756fe237db42c) )
	ROM_LOAD16_BYTE( "mpr-14884.8",  0x1100000, 0x80000, CRC(6CF9C026) SHA1(f4d717958dba6b6402f5ffe7638fe0bf353b61ed) )
	ROM_LOAD16_BYTE( "mpr-14885.9",  0x1100001, 0x80000, CRC(F65C9262) SHA1(511a22bcfaf199737bfc5a809fd66cb4439dd386) )
	ROM_LOAD16_BYTE( "mpr-14886.10", 0x1200000, 0x80000, CRC(92868734) SHA1(e1dfb134dc3ba7e0b1d40681621e09ac5a222518) )
	ROM_LOAD16_BYTE( "mpr-14887.11", 0x1200001, 0x80000, CRC(10C7C636) SHA1(c77d55460bba4354349e473e129f21afeed05e91) )
	ROM_LOAD16_BYTE( "mpr-14888.12", 0x1300000, 0x80000, CRC(04BFDC5B) SHA1(bb8788a761620d0440a62ae51c3b41f70a04b5e4) )
	ROM_LOAD16_BYTE( "mpr-14889.13", 0x1300001, 0x80000, CRC(C49F0486) SHA1(cc2bb9059c016ba2c4f6e7508bd1687df07b8b48) )

	ROM_REGION( 0xc0000, "audiocpu", 0 )  /* 68K code */
	ROM_LOAD16_WORD_SWAP( "epr-14870a.7", 0x00000, 0x20000, CRC(919d9b75) SHA1(27be79881cc9a2b5cf37e18f1e2d87251426b428) )

	ROM_REGION( 0x400000, "sega1", 0 ) /* Samples */
	ROM_LOAD( "mpr-14873.32", 0x000000, 0x200000, CRC(b1965190) SHA1(fc47e9ed4a44d48477bd9a35e42c26508c0f4a0c) )

	ROM_REGION( 0x400000, "sega2", 0 ) /* Samples */
	ROM_LOAD( "mpr-14876.4", 0x000000, 0x200000, CRC(ba6b2327) SHA1(02285520624a4e612cb4b65510e3458b13b1c6ba) )

	ROM_REGION32_LE( 0x1000000, "user1", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-14890.26", 0x000000, 0x200000, CRC(dcbe006b) SHA1(195be7fabec405ca1b4e1338d3b8d7bb4a06dd73) )
	ROM_LOAD32_WORD( "mpr-14891.27", 0x000002, 0x200000, CRC(25832b38) SHA1(a8d74538149c92bae661334e327b031eaca2a8f2) )
	ROM_LOAD32_WORD( "mpr-14892.28", 0x400000, 0x200000, CRC(5136f3ba) SHA1(ce8312975764db09bbfa2068b99559a5c1798a36) )
	ROM_LOAD32_WORD( "mpr-14893.29", 0x400002, 0x200000, CRC(1c531ada) SHA1(8b373ac97a3649c64f48eb3d1dd95c5833f330b6) )
	ROM_LOAD32_WORD( "mpr-14894.30", 0x800000, 0x200000, CRC(830a71bc) SHA1(884378e8a5afeb819daf5285d0d205986d566340) )
	ROM_LOAD32_WORD( "mpr-14895.31", 0x800002, 0x200000, CRC(af027ac5) SHA1(523f03d90358ddb7d0e96fd06b9a65cebfc09f24) )
	ROM_LOAD32_WORD( "mpr-14896.32", 0xc00000, 0x200000, CRC(382091dc) SHA1(efa266f0f6bfe36ad1c365e588fff33b01e166dd) )
	ROM_LOAD32_WORD( "mpr-14879.33", 0xc00002, 0x200000, CRC(74873195) SHA1(80705ec577d14570f9bba77cc26766f831c41f42) )

	ROM_REGION32_LE( 0x200000, "user2", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-14898.39", 0x000000, 0x80000, CRC(61da2bb6) SHA1(7a12ba522d64a1aeec1ca6f5a87ee063692131f9) )
	ROM_LOAD32_BYTE( "mpr-14899.40", 0x000001, 0x80000, CRC(2cd58bee) SHA1(73defec823de4244a387af55fea7210edc1b314f) )
	ROM_LOAD32_BYTE( "mpr-14900.41", 0x000002, 0x80000, CRC(aa7c017d) SHA1(0fa2b59a8bb5f5907b2b2567e69d11c73b398dc1) )
	ROM_LOAD32_BYTE( "mpr-14901.42", 0x000003, 0x80000, CRC(175b7a9a) SHA1(c86602e771cd49bab425b4ba7926d2f44858bd39) )

	ROM_REGION( 0x20000, "user3", 0 ) /* Comms Board */
	ROM_LOAD( "epr-15624.17", 0x00000, 0x20000, CRC(9b3ba315) SHA1(0cd0983cc8b2f2d6b41617d0d0a24cc6c188e62a) )

	ROM_REGION( 0x2000, "tgp", 0 ) /* TGP program rom */
	// this is the Daytona TGP program with some modifications needed for Virtua Racing
	// the real TGP program is an internal ROM and still needs dumping
	ROM_LOAD("vr-tgp.bin", 0x000000, 0x2000, BAD_DUMP CRC(3de33c7f) SHA1(acecc779c9d8fe39ded6c22492be5b7c25fd52db) )
ROM_END


ROM_START( swa )
	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16468.14", 0x200000, 0x80000, CRC(681d03c0) SHA1(4d21e26ce211466d429b84bca69a8147ff31ec6c) )
	ROM_LOAD16_BYTE( "epr-16469.15", 0x200001, 0x80000, CRC(6f281f7c) SHA1(6a9179e48d14838bb2a1a3f63fdd3a68ed009e03) )

	ROM_LOAD( "epr-16467.5", 0xf80000, 0x80000, CRC(605068f5) SHA1(99d7e171ce3353477c282d7567dedb9947206f14) )
	ROM_RELOAD(          0x000000, 0x80000 )
	ROM_RELOAD(          0x080000, 0x80000 )

	ROM_REGION( 0xc0000, "audiocpu", 0 )  /* 68K code */
        ROM_LOAD16_WORD_SWAP( "epr-16470.bin", 0x000000, 0x020000, CRC(7da18cf7) SHA1(bd432d882d217277faee120e2577357a32eb4a6e) )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, "sega1", 0 ) /* Samples */
        ROM_LOAD( "mpr-16486.bin", 0x000000, 0x200000, CRC(7df50533) SHA1(f2fb876738e37d70eb9005e5629a9ae89aa413a8) )
        ROM_LOAD( "mpr-16487.bin", 0x200000, 0x200000, CRC(31b28dfa) SHA1(bd1ac11bf2f9161f61f8af3b9ff4c2709b7ee700) )

	ROM_REGION( 0x400000, "sega2", 0 ) /* Samples */
        ROM_LOAD( "mpr-16484.bin", 0x000000, 0x200000, CRC(9d4c334d) SHA1(8b4d903f14559fed425d225bb23ccfe8da23cbd3) )
        ROM_LOAD( "mpr-16485.bin", 0x200000, 0x200000, CRC(95aadcad) SHA1(4276db655db9834692c3843eb96a3e3a89cb7252) )

	ROM_REGION( 0x20000, "cpu2", 0 ) /* Z80 DSB code */
        ROM_LOAD( "epr-16471.bin", 0x000000, 0x020000, CRC(f4ee84a4) SHA1(f12b214e6f195b0e5f49ba9f41d8e54bfcea9acc) )

	ROM_REGION( 0x400000, "mpeg", 0 ) /* DSB MPEG data */
        ROM_LOAD( "mpr-16514.bin", 0x000000, 0x200000, CRC(3175b0be) SHA1(63649d053c8c17ce1746d16d0cc8202be20c302f) )
        ROM_LOAD( "mpr-16515.bin", 0x000000, 0x200000, CRC(3114d748) SHA1(9ef090623cdd2a1d06b5d1bc4b9a07ab4eff5b76) )

	ROM_REGION32_LE( 0xc00000, "user1", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16476.26", 0x000000, 0x200000, CRC(d48609ae) SHA1(8c8686a5c9ca4837447a7f70ed194e2f1882b66d) )
// original dump (which one is right?)
//  ROM_LOAD32_WORD( "mpr-16477.27", 0x000002, 0x200000, CRC(b979b082) SHA1(0c60d259093e987f3856730b57b43bde7e9562e3) )
// new dump
        ROM_LOAD32_WORD( "mpr16477.bin", 0x000002, 0x200000, CRC(971ff194) SHA1(9665ede3ca22885489f1f1b5865ccfac42364206) )
	ROM_LOAD32_WORD( "mpr-16478.28", 0x400000, 0x200000, CRC(80c780f7) SHA1(2f57c5373b02765d302bcd81e24f7b7bc4181387) )
	ROM_LOAD32_WORD( "mpr-16479.29", 0x400002, 0x200000, CRC(e43183b3) SHA1(4e62c67cdf7a6fdac0ded86d5f9e81044b9dea8d) )
	ROM_LOAD32_WORD( "mpr-16480.30", 0x800000, 0x200000, CRC(3185547a) SHA1(9871937372c2c755717802117a3ad39e1a11410e) )
	ROM_LOAD32_WORD( "mpr-16481.31", 0x800002, 0x200000, CRC(ce8d76fe) SHA1(0406f0500d19d6707515627b4143f92a9a5db769) )

	ROM_REGION32_LE( 0x200000, "user2", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16472.39", 0x000000, 0x80000, CRC(5a0d7553) SHA1(ba8e08e5a0c6b7fbc10084ad7ad3edf61efb0d70) )
	ROM_LOAD32_BYTE( "mpr-16473.40", 0x000001, 0x80000, CRC(876c5399) SHA1(be7e40c77a385600941f11c24852cd73c71696f0) )
	ROM_LOAD32_BYTE( "mpr-16474.41", 0x000002, 0x80000, CRC(5864a26f) SHA1(be0c22dfff37408f6b401b1970f7fcc6fc7fbcd2) )
	ROM_LOAD32_BYTE( "mpr-16475.42", 0x000003, 0x80000, CRC(b9266be9) SHA1(cf195cd89c9d191b9eb8c5299f8cc154c2b4bd82) )
ROM_END


ROM_START( wingwar )
	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16729.14", 0x200000, 0x80000, CRC(7edec2cc) SHA1(3e423a868ca7c8475fbb5bc1a10526e69d94d865) )
	ROM_LOAD16_BYTE( "epr-16730.15", 0x200001, 0x80000, CRC(bab24dee) SHA1(26c95139c1aa7f34b6a5cce39e5bd1dd2ef0dd49) )

	ROM_LOAD( "epr16953.4", 0xfc0000, 0x20000, CRC(c821a920) SHA1(7fc9ea5d828aac664514fa6d38f708f1ffd26220) )
	ROM_RELOAD(          0x000000, 0x20000 )
	ROM_LOAD( "epr16952.5", 0xfe0000, 0x20000, CRC(03a3ecc5) SHA1(5c4aa221302b0a0800e1af99a41ab46fe4325184) )
	ROM_RELOAD(          0x020000, 0x20000 )

	ROM_LOAD16_BYTE( "mpr-16738.6",  0x1000000, 0x80000, CRC(51518ffa) SHA1(e4674ddfed4205957b14e133c6fdf6454872f324) )
	ROM_LOAD16_BYTE( "mpr-16737.7",  0x1000001, 0x80000, CRC(37b1379c) SHA1(98620c324268e1dd906c077ac8a8cd903b9de1f7) )
	ROM_LOAD16_BYTE( "mpr-16736.8",  0x1100000, 0x80000, CRC(10b6a025) SHA1(7a4f624ceb7c0b92044a5db8ff55440562ef836b) )
	ROM_LOAD16_BYTE( "mpr-16735.9",  0x1100001, 0x80000, CRC(c82fd198) SHA1(d9e53ae1e14dfc8e84a14c0026ef0b904863bb1b) )
	ROM_LOAD16_BYTE( "mpr-16734.10", 0x1200000, 0x80000, CRC(f76371c1) SHA1(0ff082db3877383d0dd977dc60c932b725e3d164) )
	ROM_LOAD16_BYTE( "mpr-16733.11", 0x1200001, 0x80000, CRC(e105847b) SHA1(8489a6c91fd6d1e9ba81e8eaf36c514da30dccbe) )

	ROM_REGION( 0xc0000, "audiocpu", 0 )  /* 68K code */
	ROM_LOAD16_WORD_SWAP("epr-17126.7", 0x000000, 0x20000, CRC(50178e40) SHA1(fb01aecfbe4e90adc997de0d45a63c16ef353b37) )
	ROM_LOAD16_WORD_SWAP("epr-16752.8", 0x020000, 0x20000, CRC(6541c48f) SHA1(9341eff160e31a8574b9545fafc1c4059323fa0c) )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, "sega1", 0 ) /* Samples */
	ROM_LOAD("mpr-16753.32", 0x000000, 0x200000, CRC(324a8333) SHA1(960342e08db637c6f72615d49cffd9fb0889620b) )
	ROM_LOAD("mpr-16754.33", 0x200000, 0x200000, CRC(144f3cf5) SHA1(d2f8cc9086affbbc5ef2195272200230f724e5d1) )

	ROM_REGION( 0x400000, "sega2", 0 ) /* Samples */
	ROM_LOAD("mpr-16755.4", 0x000000, 0x200000, CRC(4baaf878) SHA1(661d4ea9be6a4952852d0ef94becee7ed42bf4a1) )
	ROM_LOAD("mpr-16756.5", 0x200000, 0x200000, CRC(d9c40672) SHA1(83e6f1156b30888d3a00103f079dc74f4fca8446) )

	ROM_REGION32_LE( 0x1000000, "user1", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16743.26", 0x000000, 0x200000, CRC(a710d33c) SHA1(1d0184545b34789ed511caaa25d57db3cd9a8e2f) )
	ROM_LOAD32_WORD( "mpr-16744.27", 0x000002, 0x200000, CRC(de796e1f) SHA1(397efb86a21b178770f29d2464bacf5f893619a0) )
	ROM_LOAD32_WORD( "mpr-16745.28", 0x400000, 0x200000, CRC(905b689c) SHA1(685dec2a65d5b3a386bda0af1bb5ae7e2b73a01a) )
	ROM_LOAD32_WORD( "mpr-16746.29", 0x400002, 0x200000, CRC(163b312e) SHA1(6b45007d6a9d17c8a0b46d81ec84ce9bfefb1695) )
	ROM_LOAD32_WORD( "mpr-16747.30", 0x800000, 0x200000, CRC(7353bb12) SHA1(608c5d561e909b8ba31d53db18e6e199855eaaec) )
	ROM_LOAD32_WORD( "mpr-16748.31", 0x800002, 0x200000, CRC(8ce98d3a) SHA1(1978776a0e2ea817508e30ba232d5f8d9c158f70) )
	ROM_LOAD32_WORD( "mpr-16749.32", 0xc00000, 0x200000, CRC(0e36dc1a) SHA1(4939177a6ac51ca57d0acd118ff14af4f4e438bb) )
	ROM_LOAD32_WORD( "mpr-16750.33", 0xc00002, 0x200000, CRC(e4f0b98d) SHA1(e69de2e5ccea2834fb8326bdd61fc6b517bc60b7) )

	ROM_REGION32_LE( 0x200000, "user2", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16741.39", 0x000000, 0x80000, CRC(84b2ffd8) SHA1(0eba3855d20b88567c6fa08046e12429664d87cb) )
	ROM_LOAD32_BYTE( "mpr-16742.40", 0x000001, 0x80000, CRC(e9cc12bb) SHA1(40c83c968be3b11fad193a00e7b760f074450683) )
	ROM_LOAD32_BYTE( "mpr-16739.41", 0x000002, 0x80000, CRC(6c73e98f) SHA1(7b31e62922ab6d0df97c3ecc52b78e6d086c8635) )
	ROM_LOAD32_BYTE( "mpr-16740.42", 0x000003, 0x80000, CRC(44b31007) SHA1(4bb265fea25a7bbcbb8ab080fdcf09849b18f1de) )
ROM_END

ROM_START( wingwaru )
	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16729.14", 0x200000, 0x80000, CRC(7edec2cc) SHA1(3e423a868ca7c8475fbb5bc1a10526e69d94d865) )
	ROM_LOAD16_BYTE( "epr-16730.15", 0x200001, 0x80000, CRC(bab24dee) SHA1(26c95139c1aa7f34b6a5cce39e5bd1dd2ef0dd49) )

	ROM_LOAD( "epr-16951.4", 0xfc0000, 0x20000, CRC(8df5a798) SHA1(ef2756f237933ecf429dab0f362e572eb1965f4d) )
	ROM_RELOAD(          0x000000, 0x20000 )
	ROM_LOAD( "epr-16950.5", 0xfe0000, 0x20000, CRC(841e2195) SHA1(66f465aaf71955496e6f83335f7b836ad1d8c724) )
	ROM_RELOAD(          0x020000, 0x20000 )

	ROM_LOAD16_BYTE( "mpr-16738.6",  0x1000000, 0x80000, CRC(51518ffa) SHA1(e4674ddfed4205957b14e133c6fdf6454872f324) )
	ROM_LOAD16_BYTE( "mpr-16737.7",  0x1000001, 0x80000, CRC(37b1379c) SHA1(98620c324268e1dd906c077ac8a8cd903b9de1f7) )
	ROM_LOAD16_BYTE( "mpr-16736.8",  0x1100000, 0x80000, CRC(10b6a025) SHA1(7a4f624ceb7c0b92044a5db8ff55440562ef836b) )
	ROM_LOAD16_BYTE( "mpr-16735.9",  0x1100001, 0x80000, CRC(c82fd198) SHA1(d9e53ae1e14dfc8e84a14c0026ef0b904863bb1b) )
	ROM_LOAD16_BYTE( "mpr-16734.10", 0x1200000, 0x80000, CRC(f76371c1) SHA1(0ff082db3877383d0dd977dc60c932b725e3d164) )
	ROM_LOAD16_BYTE( "mpr-16733.11", 0x1200001, 0x80000, CRC(e105847b) SHA1(8489a6c91fd6d1e9ba81e8eaf36c514da30dccbe) )

	ROM_REGION( 0xc0000, "audiocpu", 0 )  /* 68K code */
	ROM_LOAD16_WORD_SWAP("epr-16751.7", 0x000000, 0x20000, CRC(23ba5ebc) SHA1(b98aab546c5e980baeedbada4e7472eb4c588260) )
	ROM_LOAD16_WORD_SWAP("epr-16752.8", 0x020000, 0x20000, CRC(6541c48f) SHA1(9341eff160e31a8574b9545fafc1c4059323fa0c) )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, "sega1", 0 ) /* Samples */
	ROM_LOAD("mpr-16753.32", 0x000000, 0x200000, CRC(324a8333) SHA1(960342e08db637c6f72615d49cffd9fb0889620b) )
	ROM_LOAD("mpr-16754.33", 0x200000, 0x200000, CRC(144f3cf5) SHA1(d2f8cc9086affbbc5ef2195272200230f724e5d1) )

	ROM_REGION( 0x400000, "sega2", 0 ) /* Samples */
	ROM_LOAD("mpr-16755.4", 0x000000, 0x200000, CRC(4baaf878) SHA1(661d4ea9be6a4952852d0ef94becee7ed42bf4a1) )
	ROM_LOAD("mpr-16756.5", 0x200000, 0x200000, CRC(d9c40672) SHA1(83e6f1156b30888d3a00103f079dc74f4fca8446) )

	ROM_REGION32_LE( 0x1000000, "user1", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16743.26", 0x000000, 0x200000, CRC(a710d33c) SHA1(1d0184545b34789ed511caaa25d57db3cd9a8e2f) )
	ROM_LOAD32_WORD( "mpr-16744.27", 0x000002, 0x200000, CRC(de796e1f) SHA1(397efb86a21b178770f29d2464bacf5f893619a0) )
	ROM_LOAD32_WORD( "mpr-16745.28", 0x400000, 0x200000, CRC(905b689c) SHA1(685dec2a65d5b3a386bda0af1bb5ae7e2b73a01a) )
	ROM_LOAD32_WORD( "mpr-16746.29", 0x400002, 0x200000, CRC(163b312e) SHA1(6b45007d6a9d17c8a0b46d81ec84ce9bfefb1695) )
	ROM_LOAD32_WORD( "mpr-16747.30", 0x800000, 0x200000, CRC(7353bb12) SHA1(608c5d561e909b8ba31d53db18e6e199855eaaec) )
	ROM_LOAD32_WORD( "mpr-16748.31", 0x800002, 0x200000, CRC(8ce98d3a) SHA1(1978776a0e2ea817508e30ba232d5f8d9c158f70) )
	ROM_LOAD32_WORD( "mpr-16749.32", 0xc00000, 0x200000, CRC(0e36dc1a) SHA1(4939177a6ac51ca57d0acd118ff14af4f4e438bb) )
	ROM_LOAD32_WORD( "mpr-16750.33", 0xc00002, 0x200000, CRC(e4f0b98d) SHA1(e69de2e5ccea2834fb8326bdd61fc6b517bc60b7) )

	ROM_REGION32_LE( 0x200000, "user2", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16741.39", 0x000000, 0x80000, CRC(84b2ffd8) SHA1(0eba3855d20b88567c6fa08046e12429664d87cb) )
	ROM_LOAD32_BYTE( "mpr-16742.40", 0x000001, 0x80000, CRC(e9cc12bb) SHA1(40c83c968be3b11fad193a00e7b760f074450683) )
	ROM_LOAD32_BYTE( "mpr-16739.41", 0x000002, 0x80000, CRC(6c73e98f) SHA1(7b31e62922ab6d0df97c3ecc52b78e6d086c8635) )
	ROM_LOAD32_BYTE( "mpr-16740.42", 0x000003, 0x80000, CRC(44b31007) SHA1(4bb265fea25a7bbcbb8ab080fdcf09849b18f1de) )
ROM_END

ROM_START( wingwarj )
	ROM_REGION( 0x2000000, "maincpu", ROMREGION_ERASEFF ) /* v60 code */
	ROM_LOAD16_BYTE( "epr-16729.14", 0x200000, 0x80000, CRC(7edec2cc) SHA1(3e423a868ca7c8475fbb5bc1a10526e69d94d865) )
	ROM_LOAD16_BYTE( "epr-16730.15", 0x200001, 0x80000, CRC(bab24dee) SHA1(26c95139c1aa7f34b6a5cce39e5bd1dd2ef0dd49) )

	ROM_LOAD( "epr-16728.4", 0xfc0000, 0x20000, CRC(194d58ff) SHA1(281201cbb9bb148a06fe88edeccf680f837dcc10) )
	ROM_RELOAD(          0x000000, 0x20000 )
	ROM_LOAD( "epr-16727.5", 0xfe0000, 0x20000, CRC(430380eb) SHA1(25bd81a69a9b4cdc644f912283b65b3eab988327) )
	ROM_RELOAD(          0x020000, 0x20000 )

	ROM_LOAD16_BYTE( "mpr-16738.6",  0x1000000, 0x80000, CRC(51518ffa) SHA1(e4674ddfed4205957b14e133c6fdf6454872f324) )
	ROM_LOAD16_BYTE( "mpr-16737.7",  0x1000001, 0x80000, CRC(37b1379c) SHA1(98620c324268e1dd906c077ac8a8cd903b9de1f7) )
	ROM_LOAD16_BYTE( "mpr-16736.8",  0x1100000, 0x80000, CRC(10b6a025) SHA1(7a4f624ceb7c0b92044a5db8ff55440562ef836b) )
	ROM_LOAD16_BYTE( "mpr-16735.9",  0x1100001, 0x80000, CRC(c82fd198) SHA1(d9e53ae1e14dfc8e84a14c0026ef0b904863bb1b) )
	ROM_LOAD16_BYTE( "mpr-16734.10", 0x1200000, 0x80000, CRC(f76371c1) SHA1(0ff082db3877383d0dd977dc60c932b725e3d164) )
	ROM_LOAD16_BYTE( "mpr-16733.11", 0x1200001, 0x80000, CRC(e105847b) SHA1(8489a6c91fd6d1e9ba81e8eaf36c514da30dccbe) )

	ROM_REGION( 0xc0000, "audiocpu", 0 )  /* 68K code */
	ROM_LOAD16_WORD_SWAP("epr-16751.7", 0x000000, 0x20000, CRC(23ba5ebc) SHA1(b98aab546c5e980baeedbada4e7472eb4c588260) )
	ROM_LOAD16_WORD_SWAP("epr-16752.8", 0x020000, 0x20000, CRC(6541c48f) SHA1(9341eff160e31a8574b9545fafc1c4059323fa0c) )
	ROM_RELOAD(0x80000, 0x20000)

	ROM_REGION( 0x400000, "sega1", 0 ) /* Samples */
	ROM_LOAD("mpr-16753.32", 0x000000, 0x200000, CRC(324a8333) SHA1(960342e08db637c6f72615d49cffd9fb0889620b) )
	ROM_LOAD("mpr-16754.33", 0x200000, 0x200000, CRC(144f3cf5) SHA1(d2f8cc9086affbbc5ef2195272200230f724e5d1) )

	ROM_REGION( 0x400000, "sega2", 0 ) /* Samples */
	ROM_LOAD("mpr-16755.4", 0x000000, 0x200000, CRC(4baaf878) SHA1(661d4ea9be6a4952852d0ef94becee7ed42bf4a1) )
	ROM_LOAD("mpr-16756.5", 0x200000, 0x200000, CRC(d9c40672) SHA1(83e6f1156b30888d3a00103f079dc74f4fca8446) )

	ROM_REGION32_LE( 0x1000000, "user1", 0 ) /* TGP model roms */
	ROM_LOAD32_WORD( "mpr-16743.26", 0x000000, 0x200000, CRC(a710d33c) SHA1(1d0184545b34789ed511caaa25d57db3cd9a8e2f) )
	ROM_LOAD32_WORD( "mpr-16744.27", 0x000002, 0x200000, CRC(de796e1f) SHA1(397efb86a21b178770f29d2464bacf5f893619a0) )
	ROM_LOAD32_WORD( "mpr-16745.28", 0x400000, 0x200000, CRC(905b689c) SHA1(685dec2a65d5b3a386bda0af1bb5ae7e2b73a01a) )
	ROM_LOAD32_WORD( "mpr-16746.29", 0x400002, 0x200000, CRC(163b312e) SHA1(6b45007d6a9d17c8a0b46d81ec84ce9bfefb1695) )
	ROM_LOAD32_WORD( "mpr-16747.30", 0x800000, 0x200000, CRC(7353bb12) SHA1(608c5d561e909b8ba31d53db18e6e199855eaaec) )
	ROM_LOAD32_WORD( "mpr-16748.31", 0x800002, 0x200000, CRC(8ce98d3a) SHA1(1978776a0e2ea817508e30ba232d5f8d9c158f70) )
	ROM_LOAD32_WORD( "mpr-16749.32", 0xc00000, 0x200000, CRC(0e36dc1a) SHA1(4939177a6ac51ca57d0acd118ff14af4f4e438bb) )
	ROM_LOAD32_WORD( "mpr-16750.33", 0xc00002, 0x200000, CRC(e4f0b98d) SHA1(e69de2e5ccea2834fb8326bdd61fc6b517bc60b7) )

	ROM_REGION32_LE( 0x200000, "user2", 0 ) /* TGP data roms */
	ROM_LOAD32_BYTE( "mpr-16741.39", 0x000000, 0x80000, CRC(84b2ffd8) SHA1(0eba3855d20b88567c6fa08046e12429664d87cb) )
	ROM_LOAD32_BYTE( "mpr-16742.40", 0x000001, 0x80000, CRC(e9cc12bb) SHA1(40c83c968be3b11fad193a00e7b760f074450683) )
	ROM_LOAD32_BYTE( "mpr-16739.41", 0x000002, 0x80000, CRC(6c73e98f) SHA1(7b31e62922ab6d0df97c3ecc52b78e6d086c8635) )
	ROM_LOAD32_BYTE( "mpr-16740.42", 0x000003, 0x80000, CRC(44b31007) SHA1(4bb265fea25a7bbcbb8ab080fdcf09849b18f1de) )
ROM_END


static MACHINE_CONFIG_START( model1, model1_state )
	MCFG_CPU_ADD("maincpu", V60, 16000000)
	MCFG_CPU_PROGRAM_MAP(model1_mem)
	MCFG_CPU_IO_MAP(model1_io)
	MCFG_CPU_VBLANK_INT_HACK(model1_interrupt, 2)

	MCFG_CPU_ADD("audiocpu", M68000, 10000000)	// verified on real h/w
	MCFG_CPU_PROGRAM_MAP(model1_snd)

	MCFG_MACHINE_START(model1)
	MCFG_MACHINE_RESET(model1)
	MCFG_NVRAM_ADD_0FILL("nvram")

	MCFG_S24TILE_DEVICE_ADD("tile", 0x3fff)

	MCFG_VIDEO_ATTRIBUTES(VIDEO_UPDATE_AFTER_VBLANK )

	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_RAW_PARAMS(XTAL_16MHz, 656, 0/*+69*/, 496/*+69*/, 424, 0/*+25*/, 384/*+25*/)
	MCFG_SCREEN_FORMAT(BITMAP_FORMAT_RGB15)
	MCFG_SCREEN_UPDATE(model1)
	MCFG_SCREEN_EOF(model1)

	MCFG_PALETTE_LENGTH(8192)

	MCFG_VIDEO_START(model1)

	MCFG_SPEAKER_STANDARD_STEREO("lspeaker", "rspeaker")

	MCFG_SOUND_ADD("ymsnd", YM3438, 8000000)
	MCFG_SOUND_ROUTE(0, "lspeaker", 0.60)
	MCFG_SOUND_ROUTE(1, "rspeaker", 0.60)

	MCFG_SOUND_ADD("sega1", MULTIPCM, 8000000)
	MCFG_SOUND_ROUTE(0, "lspeaker", 1.0)
	MCFG_SOUND_ROUTE(1, "rspeaker", 1.0)

	MCFG_SOUND_ADD("sega2", MULTIPCM, 8000000)
	MCFG_SOUND_ROUTE(0, "lspeaker", 1.0)
	MCFG_SOUND_ROUTE(1, "rspeaker", 1.0)
MACHINE_CONFIG_END

static MACHINE_CONFIG_START( model1_vr, model1_state )
	MCFG_CPU_ADD("maincpu", V60, 16000000)
	MCFG_CPU_PROGRAM_MAP(model1_vr_mem)
	MCFG_CPU_IO_MAP(model1_vr_io)
	MCFG_CPU_VBLANK_INT_HACK(model1_interrupt, 2)

	MCFG_CPU_ADD("audiocpu", M68000, 10000000)	// verified on real h/w
	MCFG_CPU_PROGRAM_MAP(model1_snd)

	MCFG_CPU_ADD("tgp", MB86233, 16000000)
	MCFG_CPU_CONFIG(model1_vr_tgp_config)
	MCFG_CPU_PROGRAM_MAP(model1_vr_tgp_map)

	MCFG_MACHINE_START(model1)
	MCFG_MACHINE_RESET(model1_vr)
	MCFG_NVRAM_ADD_0FILL("nvram")

	MCFG_S24TILE_DEVICE_ADD("tile", 0x3fff)

	MCFG_VIDEO_ATTRIBUTES(VIDEO_UPDATE_AFTER_VBLANK )

	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_RAW_PARAMS(XTAL_16MHz, 656, 0/*+69*/, 496/*+69*/, 424, 0/*+25*/, 384/*+25*/)
	MCFG_SCREEN_FORMAT(BITMAP_FORMAT_RGB15)
	MCFG_SCREEN_UPDATE(model1)
	MCFG_SCREEN_EOF(model1)

	MCFG_PALETTE_LENGTH(8192)

	MCFG_VIDEO_START(model1)

	MCFG_SPEAKER_STANDARD_STEREO("lspeaker", "rspeaker")

	MCFG_SOUND_ADD("ymsnd", YM3438, 8000000)
	MCFG_SOUND_ROUTE(0, "lspeaker", 0.60)
	MCFG_SOUND_ROUTE(1, "rspeaker", 0.60)

	MCFG_SOUND_ADD("sega1", MULTIPCM, 8000000)
	MCFG_SOUND_ROUTE(0, "lspeaker", 1.0)
	MCFG_SOUND_ROUTE(1, "rspeaker", 1.0)

	MCFG_SOUND_ADD("sega2", MULTIPCM, 8000000)
	MCFG_SOUND_ROUTE(0, "lspeaker", 1.0)
	MCFG_SOUND_ROUTE(1, "rspeaker", 1.0)
MACHINE_CONFIG_END

GAME( 1993, vf,       0,       model1,    vf,       0, ROT0, "Sega", "Virtua Fighter", GAME_IMPERFECT_GRAPHICS )
GAME( 1992, vr,       0,       model1_vr, vr,       0, ROT0, "Sega", "Virtua Racing", GAME_IMPERFECT_GRAPHICS )
GAME( 1993, vformula, vr,      model1_vr, vr,       0, ROT0, "Sega", "Virtua Formula", GAME_IMPERFECT_GRAPHICS )
GAME( 1993, swa,      0,       model1,    swa,      0, ROT0, "Sega", "Star Wars Arcade", GAME_NOT_WORKING | GAME_IMPERFECT_SOUND )
GAME( 1994, wingwar,  0,       model1,    wingwar,  0, ROT0, "Sega", "Wing War (World)", GAME_NOT_WORKING )
GAME( 1994, wingwaru, wingwar, model1,    wingwar,  0, ROT0, "Sega", "Wing War (US)", GAME_NOT_WORKING )
GAME( 1994, wingwarj, wingwar, model1,    wingwar,  0, ROT0, "Sega", "Wing War (Japan)", GAME_NOT_WORKING )
